package net.aihelp.data.track.statistic;

import android.text.TextUtils;

import net.aihelp.data.model.rpa.msg.bot.Faq;
import net.aihelp.data.logic.cs.TicketStatusTracker;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class StatisticTracker {

    public JSONArray trackEventArray = new JSONArray();
    public TrackEntity trackEntity = new TrackEntity("", "");
    public List<String> entranceTrackedEvents = new ArrayList<>();

    // region 单例模式
    private StatisticTracker() {
        trackEventArray = new JSONArray();
        trackEntity = new TrackEntity("", "");
        entranceTrackedEvents = new ArrayList<>();
    }

    public static StatisticTracker getInstance() {
        return StatisticTracker.LazyHolder.HOLDER;
    }

    private static class LazyHolder {
        static final StatisticTracker HOLDER = new StatisticTracker();

        private LazyHolder() {
        }
    }
    // endregion

    // region 初始化方法
    public void setupTrackHandler(String entranceId, String activeId) {
        trackEventArray = new JSONArray();
        entranceTrackedEvents = new ArrayList<>();
        trackEntity = new TrackEntity(entranceId, activeId);
    }

    public JSONObject getExtendDataForLogin() {
        JSONObject extendData = new JSONObject();
        try {
            if (trackEntity != null) {
                extendData.put("entranceId", trackEntity.getEntranceId());
                extendData.put("activeId", trackEntity.getActiveId());
                extendData.put("faqMainId", trackEntity.getFaqMainId());
                extendData.put("faqContentId", trackEntity.getFaqContentId());
                extendData.put("sectionId", trackEntity.getSectionId());
            }
        } catch (Exception e) {
            // ignored
        }
        return extendData;
    }
    // endregion

    // region DurationTracker
    public void setWhenMessageSent() {
        if (trackEntity != null) {
            trackEntity.setWhenMessageSent(System.currentTimeMillis());
        }
    }

    public void calculateDurationForWaiting() {
        if (trackEntity != null) {
            long waitingTime = System.currentTimeMillis() - trackEntity.getWhenMessageSent();
            if (waitingTime > 0) {
                JSONObject customerServiceDuration = new JSONObject();
                TrackHelper.addTrackData(customerServiceDuration, "stayTime", waitingTime);
                trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_DURATION_USER_WAITING,
                        trackEntity, customerServiceDuration));
            }
        }
    }

    public void calculateDurationInCustomerService() {
        if (trackEntity != null) {
            trackEntity.calculateTotalCustomerServiceDuration();
        }
    }

    public void calculateDurationInAIHelp() {
        if (trackEntity != null) {
            // ignore all the rest logic if the AIHelp page is never shown
            if (trackEntity.getWhenAIHelpVisible() <= 0) return;

            // customer service duration
            long totalCustomerServiceDuration = trackEntity.getTotalCustomerServiceDuration();
            if (totalCustomerServiceDuration >= 1000) {
                JSONObject customerServiceDuration = new JSONObject();
                TrackHelper.addTrackData(customerServiceDuration, "stayTime", totalCustomerServiceDuration);
                trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_DURATION_CUSTOMER_SERVICE,
                        trackEntity, customerServiceDuration));
            }

            // sdk total duration
            long totalSDKDuration = System.currentTimeMillis() - trackEntity.getWhenAIHelpVisible();
            if (totalSDKDuration >= 1000) {
                JSONObject sdkDuration = new JSONObject();
                TrackHelper.addTrackData(sdkDuration, "stayTime", totalSDKDuration);
                trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_DURATION_AIHELP, trackEntity, sdkDuration));
            }

            // help center duration = sdk total duration - customer service duration
            long totalHelpCenterDuration = totalSDKDuration - totalCustomerServiceDuration;
            if (totalHelpCenterDuration >= 1000) {
                JSONObject helpCenterDuration = new JSONObject();
                TrackHelper.addTrackData(helpCenterDuration, "stayTime", totalHelpCenterDuration);
                trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_DURATION_HELP_CENTER, trackEntity, helpCenterDuration));
            }
        }

        // batch sync all data to server after leaving AIHelp page
        TrackHelper.syncTrack(trackEventArray);
    }
    // endregion

    // region FaqEventTracker
    public void checkedFAQInBot(int faqSource, String mainId, String contentId) {
        JSONObject jsonObject = new JSONObject();
        TrackHelper.addTrackData(jsonObject, "mainId", mainId);
        TrackHelper.addTrackData(jsonObject, "contentId", contentId);
        TrackHelper.addTrackData(jsonObject, "ticketId", TicketStatusTracker.currentTicketId);
        int trackType = faqSource == Faq.FAQ_SOURCE_RPA ? TrackType.TRACK_FAQ_CHECKED_IN_RPA_BOT
                : TrackType.TRACK_FAQ_CHECKED_IN_ANSWER_BOT;
        trackEventArray.put(TrackHelper.getCommonParams(trackType, trackEntity, jsonObject));
        if (trackEntity != null) {
            trackEntity.setFaqMainId(mainId);
            trackEntity.setFaqContentId(contentId);
        }
    }

    public void checkedFAQ(String mainId, String contentId, String title) {
        JSONObject jsonObject = new JSONObject();
        TrackHelper.addTrackData(jsonObject, "mainId", mainId);
        TrackHelper.addTrackData(jsonObject, "contentId", contentId);
        TrackHelper.addTrackData(jsonObject, "title", title);
        trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_FAQ_CHECKED, trackEntity, jsonObject));
        if (trackEntity != null) {
            trackEntity.setFaqMainId(mainId);
            trackEntity.setFaqContentId(contentId);
        }
    }

    public void markedHelpful(String mainId, String contentId, String title) {
        JSONObject jsonObject = new JSONObject();
        TrackHelper.addTrackData(jsonObject, "mainId", mainId);
        TrackHelper.addTrackData(jsonObject, "contentId", contentId);
        TrackHelper.addTrackData(jsonObject, "title", title);
        trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_FAQ_MARKED_HELPFUL, trackEntity, jsonObject));
    }

    public void markedUnhelpful(String mainId, String contentId, String title) {
        JSONObject jsonObject = new JSONObject();
        TrackHelper.addTrackData(jsonObject, "mainId", mainId);
        TrackHelper.addTrackData(jsonObject, "contentId", contentId);
        TrackHelper.addTrackData(jsonObject, "title", title);
        trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_FAQ_MARKED_UNHELPFUL, trackEntity, jsonObject));
    }

    public void submitSuggestion(String mainId, String contentId, String title, String feedback) {
        JSONObject jsonObject = new JSONObject();
        TrackHelper.addTrackData(jsonObject, "mainId", mainId);
        TrackHelper.addTrackData(jsonObject, "contentId", contentId);
        TrackHelper.addTrackData(jsonObject, "title", title);
        TrackHelper.addTrackData(jsonObject, "feedback", feedback);
        trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_FAQ_SUBMIT_SUGGESTION, trackEntity, jsonObject));
    }

    public void clickServiceEntranceInFaq(String faqContentVisitId) {
        String clickServiceEvent = String.format("%s-%s", TrackType.TRACK_FAQ_CLICK_CUSTOMER_SERVICE, faqContentVisitId);
        if (!entranceTrackedEvents.contains(clickServiceEvent)) {
            trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_FAQ_CLICK_CUSTOMER_SERVICE, trackEntity));
            entranceTrackedEvents.add(clickServiceEvent);
        }
    }

    public void trackSearchQuery(String searchContent) {
        if (TextUtils.isEmpty(searchContent)) return;
        JSONObject jsonObject = new JSONObject();
        TrackHelper.addTrackData(jsonObject, "content", searchContent);
        trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_FAQ_SEARCH_CONTENT, trackEntity, jsonObject));
    }

    public void trackSearchResult(String searchContent) {
        if (TextUtils.isEmpty(searchContent)) return;
        JSONObject jsonObject = new JSONObject();
        TrackHelper.addTrackData(jsonObject, "content", searchContent);
        trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_FAQ_SEARCH_RESULT, trackEntity, jsonObject));
    }

    public void trackSearchEmpty(String searchContent) {
        if (TextUtils.isEmpty(searchContent)) return;
        JSONObject jsonObject = new JSONObject();
        TrackHelper.addTrackData(jsonObject, "content", searchContent);
        trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_FAQ_SEARCH_EMPTY, trackEntity, jsonObject));
    }
    // endregion

    // region FormEventTracker
    public void onFormClicked(String formUrl) {
        JSONObject jsonObject = new JSONObject();
        TrackHelper.addTrackData(jsonObject, "formId", getFormIdFromUrl(formUrl));
        trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_FORM_ACTION_CLICKED, trackEntity, jsonObject));
    }

    public void onFormSubmitted(String formId) {
        JSONObject jsonObject = new JSONObject();
        TrackHelper.addTrackData(jsonObject, "formId", formId);
        trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_FORM_ACTION_SUBMITTED, trackEntity, jsonObject));
    }

    private String getFormIdFromUrl(String formUrl) {
        Matcher formIdFilter = Pattern.compile("(?<=(formId=))((?!&).)*").matcher(formUrl);
        if (formIdFilter.find()) {
            return formIdFilter.group();
        }
        return "";
    }
    // endregion

    // region HomeEntranceEventTracker
    public void clickSearch() {
        String searchClickEvent = String.format("%s", TrackType.TRACK_ENTRANCE_CLICK_SEARCH);
        if (!entranceTrackedEvents.contains(searchClickEvent)) {
            trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_ENTRANCE_CLICK_SEARCH, trackEntity));
            entranceTrackedEvents.add(searchClickEvent);
        }
    }

    public void clickNotification(String mainId, String contentId, String title) {
        clickSomeFaq(TrackType.TRACK_ENTRANCE_CLICK_NOTIFICATION, mainId, contentId, title);
    }

    public void clickHotTopic(String mainId, String contentId, String title) {
        clickSomeFaq(TrackType.TRACK_ENTRANCE_CLICK_HOT_TOPIC, mainId, contentId, title);
    }

    public void clickFaq(String mainId, String contentId, String title) {
        clickSomeFaq(TrackType.TRACK_ENTRANCE_CLICK_FAQ, mainId, contentId, title);
    }

    public void clickSection(String sectionId, String title) {
        String clickSectionEvent = String.format("%s-%s", TrackType.TRACK_ENTRANCE_CLICK_SECTION, sectionId);
        if (!entranceTrackedEvents.contains(clickSectionEvent)) {
            JSONObject jsonObject = new JSONObject();
            TrackHelper.addTrackData(jsonObject, "sectionId", sectionId);
            TrackHelper.addTrackData(jsonObject, "title", title);
            trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_ENTRANCE_CLICK_SECTION, trackEntity, jsonObject));
            entranceTrackedEvents.add(clickSectionEvent);
        }
        if (trackEntity != null) {
            trackEntity.setSectionId(sectionId);
        }
    }

    private void clickSomeFaq(int type, String mainId, String contentId, String title) {
        String clickHotTopicEvent = String.format("%s-%s", type, contentId);
        if (!entranceTrackedEvents.contains(clickHotTopicEvent)) {
            JSONObject jsonObject = new JSONObject();
            TrackHelper.addTrackData(jsonObject, "mainId", mainId);
            TrackHelper.addTrackData(jsonObject, "contentId", contentId);
            TrackHelper.addTrackData(jsonObject, "title", title);
            trackEventArray.put(TrackHelper.getCommonParams(type, trackEntity, jsonObject));
            entranceTrackedEvents.add(clickHotTopicEvent);
        }
    }

    public void clickServiceEntrance(String faqContentVisitId) {
        // 记录进入客服系统事件
        showCustomerService(false);
        // 记录按钮点击操作
        String clickServiceEvent = String.format("%s", TrackType.TRACK_ENTRANCE_CLICK_CUSTOMER_SERVICE);
        if (!entranceTrackedEvents.contains(clickServiceEvent)) {
            trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_ENTRANCE_CLICK_CUSTOMER_SERVICE, trackEntity));
            entranceTrackedEvents.add(clickServiceEvent);
        }
        // 单独记录按钮在详情页的点击：一次进入详情多次点击按钮，只记录一次
        clickServiceEntranceInFaq(faqContentVisitId);
    }
    // endregion

    // region PageVisibleEventTracker
    public void showAIHelp() {
        trackEntity.setWhenAIHelpVisible(System.currentTimeMillis());
        trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_ACTION_SHOW_AIHELP, trackEntity));
    }

    public void showHelpCenter() {
        trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_ACTION_SHOW_HELP_CENTER, trackEntity));
    }

    /**
     * 进入客服页面的打点：
     * 1、通过入口打开的客服入口，每次进入都打点；
     * 2、通过 FAQ 页面（不论哪个页面）进入的客服入口，多次进出客服页面只记录一次
     *
     * @param alwaysTrack 是否忽略历史记录，始终操作本次打点
     */
    public void showCustomerService(boolean alwaysTrack) {
        trackEntity.setWhenCustomerServiceVisible(System.currentTimeMillis());
        if (alwaysTrack || !trackEntity.isTracked()) {
            trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_ACTION_SHOW_CUSTOMER_SERVICE, trackEntity));
            trackEntity.setTracked(true);
        }
    }

    // endregion

    // region UserAskEventTracker
    public void selectSmartIntent(String intentId) {
        JSONObject jsonObject = new JSONObject();
        TrackHelper.addTrackData(jsonObject, "intentId", intentId);
        TrackHelper.addTrackData(jsonObject, "ticketId", TicketStatusTracker.currentTicketId);
        trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_ASK_SELECT_INTENT, trackEntity, jsonObject));
    }

    public void onUserInput() {
        if (!TicketStatusTracker.isTicketActive || TicketStatusTracker.isTicketServingByAnswerBot()) {
            JSONObject jsonObject = new JSONObject();
            TrackHelper.addTrackData(jsonObject, "ticketId", TicketStatusTracker.currentTicketId);
            trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_ASK_USER_INPUT, trackEntity, jsonObject));
        }
    }

    public void onAnswerBotSelected() {
        JSONObject jsonObject = new JSONObject();
        TrackHelper.addTrackData(jsonObject, "ticketId", TicketStatusTracker.currentTicketId);
        trackEventArray.put(TrackHelper.getCommonParams(TrackType.TRACK_ASK_SELECT_ANSWER_BOT, trackEntity, jsonObject));
    }
    // endregion

}
