package net.aihelp.init;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.content.Intent;
import android.text.TextUtils;
import android.util.Log;

import net.aihelp.BuildConfig;
import net.aihelp.common.ConfigValues;
import net.aihelp.common.Const;
import net.aihelp.common.UserProfile;
import net.aihelp.config.AIHelpContext;
import net.aihelp.config.ApiConfig;
import net.aihelp.common.IntentValues;
import net.aihelp.config.UserConfig;
import net.aihelp.config.enums.PublishCountryOrRegion;
import net.aihelp.config.enums.PushPlatform;
import net.aihelp.config.enums.ShowConversationMoment;
import net.aihelp.core.net.http.config.HttpConfig;
import net.aihelp.core.net.monitor.NetworkMonitorManager;
import net.aihelp.core.ui.ActivityManager;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.core.util.crash.AIHelpCrashHandler;
import net.aihelp.data.localize.config.ProcessEntranceHelper;
import net.aihelp.data.track.event.EventTracker;
import net.aihelp.data.track.event.utils.EventType;
import net.aihelp.data.logic.common.InitPresenter;
import net.aihelp.data.model.config.ProcessEntity;
import net.aihelp.exception.AIHelpInitException;
import net.aihelp.ui.helper.BreakReleaseHelper;
import net.aihelp.ui.listener.OnAIHelpInitializedCallback;
import net.aihelp.ui.listener.OnAIHelpSessionCloseCallback;
import net.aihelp.ui.listener.OnAIHelpSessionOpenCallback;
import net.aihelp.ui.listener.OnSpecificUrlClickedCallback;
import net.aihelp.ui.listener.OnMessageCountArrivedCallback;
import net.aihelp.ui.listener.OnNetworkCheckResultCallback;
import net.aihelp.ui.listener.OnSpecificFormSubmittedCallback;
import net.aihelp.ui.webkit.WebViewUtil;
import net.aihelp.utils.AppInfoUtil;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.SchemaUtil;
import net.aihelp.utils.TLog;
import net.aihelp.utils.ToastUtil;

import java.lang.ref.WeakReference;
import java.lang.reflect.Method;

class AIHelpCore {

    private Object refWatcher; // DO NOT DELETE THIS

    private WeakReference<Activity> mActivity;
    private Context mAppContext;
    private InitPresenter mInitPresenter;

    public void init(Context context, String appKey, String domain, String appId, String language) throws AIHelpInitException {

        try {
            final String trimApiKey = !TextUtils.isEmpty(appKey) ? appKey.trim() : appKey;
            final String trimDomain = !TextUtils.isEmpty(domain) ? domain.trim() : domain;
            final String trimAppId = generateAppIdFromAppKey(appKey, domain, appId);
            SchemaUtil.validateInitializeCredentials(context, trimApiKey, trimDomain, trimAppId);

            // The init job must be started on UI thread, as we're initializing handler instance under the hood.
            ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    if (context instanceof Activity) {
                        mActivity = new WeakReference<>((Activity) context);
                    }
                    mAppContext = context.getApplicationContext();
                    AIHelpContext.getInstance().setContext(mAppContext);

                    AIHelpCrashHandler.INSTANCE.init(mAppContext);
                    WebViewUtil.prepareDataDirectorySuffix(mAppContext);

                    mInitPresenter = new InitPresenter(mAppContext, trimApiKey, trimDomain, trimAppId, language);

                    if (BreakReleaseHelper.isBreak()) {
                        TLog.initLog(true);
                    }

                    NetworkMonitorManager.getInstance().init(mAppContext);
                    InitHelper.getInstance().runInitRelatedTask(mAppContext);
                }
            });
        } catch (AIHelpInitException e) {
            throw e;
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private String generateAppIdFromAppKey(String appKey, String domain, String appId) {
        if (TextUtils.isEmpty(appId) && !TextUtils.isEmpty(appKey) && !TextUtils.isEmpty(domain)) {
            String updatedDomain = domain.replace("https://", "").replace("http://", "");
            String newAppKey = HttpConfig.md5(appKey.toLowerCase() + "AIHelpSDKToAndroidAppId");
            String newDomain = updatedDomain.toLowerCase().substring(0, updatedDomain.indexOf("."));
            return newDomain + "_platform_" + newAppKey;
        }
        return appId.trim();
    }

    boolean show(ApiConfig apiConfig) {
        if (isInitStillInProgress() || apiConfig == null) return false;
        EventTracker.INSTANCE.log(EventType.API_SHOW, apiConfig.desc());
        ProcessEntity process = ProcessEntranceHelper.INSTANCE.getProcess(apiConfig.getEntranceId());
        Const.CUSTOM_WELCOME_MSG = apiConfig.getWelcomeMessage();
        switch (process.getIntent()) {
            case ConfigValues.ENTRANCE_HELP_CENTER:
                showHelpCenter(process);
                break;
            case ConfigValues.ENTRANCE_FAQ_SECTION:
                showFAQSection(process);
                break;
            case ConfigValues.ENTRANCE_SINGLE_FAQ:
                showSingleFAQ(process);
                break;
            case ConfigValues.ENTRANCE_CUSTOMER_SERVICE:
                showCustomerService();
                break;
            default:
                showErrorEntrancePage();
                break;
        }
        return true;
    }

    private void showErrorEntrancePage() {
        ProcessEntranceHelper.INSTANCE.fetchPlaceholderTemplateData();
        Intent errorEntranceIntent = IntentValues.getShowErrorEntranceIntent(mAppContext);
        if (mActivity != null && mActivity.get() != null) {
            mActivity.get().startActivity(errorEntranceIntent);
        } else {
            mAppContext.startActivity(IntentValues.wrapForApplicationContext(errorEntranceIntent));
        }
    }

    void showHelpCenter(ProcessEntity process) {
        Intent showFAQIntent = IntentValues.getShowFAQIntent(mAppContext, process);
        if (mActivity != null && mActivity.get() != null) {
            mActivity.get().startActivity(showFAQIntent);
        } else {
            mAppContext.startActivity(IntentValues.wrapForApplicationContext(showFAQIntent));
        }
    }

    void showFAQSection(ProcessEntity process) {
        Intent showFAQSectionIntent = IntentValues.getShowFAQSectionIntent(mAppContext, process);
        if (mActivity != null && mActivity.get() != null) {
            mActivity.get().startActivity(showFAQSectionIntent);
        } else {
            mAppContext.startActivity(IntentValues.wrapForApplicationContext(showFAQSectionIntent));
        }
    }

    void showSingleFAQ(ProcessEntity process) {
        Intent showSingleFAQIntent = IntentValues.getShowSingleFAQIntent(mAppContext, process);
        if (mActivity != null && mActivity.get() != null) {
            mActivity.get().startActivity(showSingleFAQIntent);
        } else {
            mAppContext.startActivity(IntentValues.wrapForApplicationContext(showSingleFAQIntent));
        }
    }

    void showSingleFAQ(String faqId, ShowConversationMoment moment) {
        ProcessEntranceHelper.INSTANCE.fetchPlaceholderTemplateData();
        EventTracker.INSTANCE.log(EventType.API_SHOW_SINGLE_FAQ, faqId);
        Intent showSingleFAQIntent = IntentValues.getShowSingleFAQIntent(mAppContext, faqId, moment);
        if (mActivity != null && mActivity.get() != null) {
            mActivity.get().startActivity(showSingleFAQIntent);
        } else {
            mAppContext.startActivity(IntentValues.wrapForApplicationContext(showSingleFAQIntent));
        }
    }

    void showCustomerService() {
        Intent showSupportIntent = IntentValues.getShowCustomerServiceIntent(mAppContext);
        if (mActivity != null && mActivity.get() != null) {
            mActivity.get().startActivity(showSupportIntent);
        } else {
            mAppContext.startActivity(IntentValues.wrapForApplicationContext(showSupportIntent));
        }
    }

    void updateSDKLanguage(String sdkLanguage) {
        try {
            if (isInitStillInProgress()) return;
            EventTracker.INSTANCE.log(EventType.API_UPDATE_SDK_LANGUAGE, sdkLanguage);
            mInitPresenter.updateSDKLanguage(sdkLanguage);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void showUrl(String url) {
        ProcessEntranceHelper.INSTANCE.fetchPlaceholderTemplateData();
        if (mAppContext != null && !TextUtils.isEmpty(url)) {
            Intent showUrlIntent = IntentValues.getUrlIntent(mAppContext, url);
            if (mActivity != null && mActivity.get() != null) {
                mActivity.get().startActivity(showUrlIntent);
            } else {
                mAppContext.startActivity(IntentValues.wrapForApplicationContext(showUrlIntent));
            }
        } else {
            TLog.d("AIHelp", "AIHelpSupport showUrl illegal argument");
        }
    }

    void updateUserInfo(UserConfig config) {
        if (isPresenterStillNull() || config == null) return;
        mInitPresenter.updateUserProfile(config);
    }

    void setUploadLogPath(String logPath) {
        if (isPresenterStillNull()) return;
        mInitPresenter.setUploadLogPath(logPath);
    }

    void setNetworkCheckHostAddress(String hostAddress, OnNetworkCheckResultCallback listener) {
        if (isPresenterStillNull()) return;
        mInitPresenter.setNetworkCheckHostAddress(hostAddress, listener);
    }

    void setPushTokenAndPlatform(String pushToken, PushPlatform pushPlatform) {
        if (isPresenterStillNull() || pushPlatform == null) return;
        UserProfile.PUSH_TOKEN = pushToken;
        UserProfile.PUSH_PLATFORM = pushPlatform.getValue();
        mInitPresenter.postCrmPushTokenInfo();
    }

    void additionalSupportFor(PublishCountryOrRegion countryOrRegion) {
        Const.countryOrRegion = countryOrRegion;
    }

    // Make sure this API is called on UI thread, as we're initializing handler instance inside.
    void startUnreadMessageCountPolling(OnMessageCountArrivedCallback listener) {
        if (isPresenterStillNull()) return;
        mInitPresenter.startUnreadMessagePolling(listener);
    }

    void fetchUnreadMessageCount(OnMessageCountArrivedCallback listener) {
        if (isPresenterStillNull()) return;
        mInitPresenter.fetchUnreadMessageCount(listener);
    }

    void stopUnreadMessageCountPolling() {
        if (isPresenterStillNull()) return;
        mInitPresenter.stopUnreadMessagePolling();
    }

    void setOnSpecificFormSubmittedCallback(OnSpecificFormSubmittedCallback callback) {
        Const.sSpecificFormSubmittedListener = callback;
    }

    void setOnAIHelpSessionOpenCallback(OnAIHelpSessionOpenCallback callback) {
        Const.sSessionOpenListener = callback;
    }

    void setOnAIHelpSessionCloseCallback(OnAIHelpSessionCloseCallback callback) {
        Const.sSessionCloseListener = callback;
    }

    void setOnAIHelpInitializedCallback(OnAIHelpInitializedCallback listener) {
        Const.sInitializedListener = listener;
    }

    void setOnSpecificUrlClickedCallback(OnSpecificUrlClickedCallback callback) {
        Const.sOnSpecificUrlClickedListener = callback;
    }

    void close() {
        ActivityManager.INSTANCE.finishAll();
    }

    void enableLogging(boolean enable) {
        TLog.initLog(enable);
    }

    private boolean isInitStillInProgress() {
        if (!AppInfoUtil.isNetworkAvailable(mAppContext)) {
            ToastUtil.INSTANCE.makeRawToast(mAppContext, ResResolver.getString("aihelp_network_no_connect"), false);
            return true;
        }
        if (!AIHelpContext.successfullyInit.get()) {
            Log.e("AIHelp", "There seems to be something wrong with the initialization of AIHelp, please try again later.");
            return true;
        }
        return false;
    }

    private boolean isPresenterStillNull() {
        if (mInitPresenter == null) {
            Log.e("AIHelp", "Please ensure AIHelpSupport#init is called at the very first place before you call any other API.");
            return true;
        }
        return false;
    }

    // DO NOT DELETE THIS
    private void installLeakCanary(Application app) {
        if (app != null && BuildConfig.DEBUG) {
            try {
                Class<?> leakCanary = Class.forName("com.squareup.leakcanary.LeakCanary");
                Method installMethod = leakCanary.getDeclaredMethod("install", Application.class);
                installMethod.setAccessible(true);
                refWatcher = installMethod.invoke(null, app);
            } catch (Exception e) {
                // ignored
            }
        }
    }

    // ===================== getInstance =======================

    public static AIHelpCore getInstance() {
        return Holder.INSTANCE;
    }

    private AIHelpCore() {
    }

    private static class Holder {
        private static final AIHelpCore INSTANCE = new AIHelpCore();
    }

}
