package net.aihelp.ui.adapter.cs;

import android.content.Context;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.text.Spannable;
import android.text.SpannableStringBuilder;
import android.text.Spanned;
import android.text.TextPaint;
import android.text.TextUtils;
import android.text.method.LinkMovementMethod;
import android.text.style.ClickableSpan;
import android.text.style.UnderlineSpan;
import android.view.View;
import android.widget.TextView;

import net.aihelp.common.CustomConfig;
import net.aihelp.core.ui.adapter.ItemViewDelegate;

import net.aihelp.data.model.rpa.msg.base.FileMessage;
import net.aihelp.data.model.rpa.msg.base.Message;
import net.aihelp.data.model.rpa.msg.bot.FormUrl;
import net.aihelp.ui.adapter.MessageListAdapter;
import net.aihelp.utils.AppInfoUtil;
import net.aihelp.utils.RegexUtil;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.Styles;
import net.aihelp.utils.URLEncodeUtil;
import net.aihelp.utils.UploadFileHelper;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import androidx.fragment.app.Fragment;

public abstract class BaseMsgAdapter implements ItemViewDelegate<Message> {

    protected final Context mContext;
    protected final Fragment mFragment;

    protected MessageListAdapter.OnClickedListenerWrapper mWrapper;
    protected List<Message> dataList = new ArrayList<>();

    protected boolean isCurrentRtl;

    public BaseMsgAdapter(Context context) {
        this(context, null);
    }

    public BaseMsgAdapter(Context context, Fragment fragment) {
        this.mContext = context;
        this.mFragment = fragment;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            isCurrentRtl = context.getResources()
                    .getConfiguration().getLayoutDirection() == View.LAYOUT_DIRECTION_RTL;
        }
    }

    protected View.OnClickListener getRetryListener(int position, Message message) {
        return new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                if (mWrapper != null) {
                    message.setTimestamp(System.currentTimeMillis());
                    // 重发时将消息的状态重置，MediaMessage 为发送中状态，普通消息为正常状态
                    message.setMsgStatus(message instanceof FileMessage ? Message.STATUS_SENDING : Message.STATUS_NORMAL);
                    mWrapper.onRetrySendingMessage(position, message);
                }
            }
        };
    }

    protected View.OnClickListener getFileRetryListener(int position, Message message) {
        return new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                if (mWrapper != null) {
                    message.setTimestamp(System.currentTimeMillis());
                    message.setMsgStatus(Message.STATUS_SENDING);
                    mWrapper.onRetrySendingMessage(position, message);
                    UploadFileHelper.INSTANCE.setOnUploadFileListener(new UploadFileHelper.OnUploadFileListener() {
                        @Override
                        public void onFileUploaded(String fileUrl) {
                            boolean isUrlValid = !TextUtils.isEmpty(fileUrl);
                            message.setContent(isUrlValid ? fileUrl : message.getContent());
                            message.setMsgStatus(isUrlValid ? Message.STATUS_NORMAL : Message.STATUS_FAILURE);
                            mWrapper.onRetrySendingMessage(position, message);
                        }
                    }).performUpload(new File(message.getContent()));
                }
            }
        };
    }

    protected Drawable getAdminBackgroundDrawable(boolean isRtl) {
        int colorFromHex = Styles.getColorWithAlpha(CustomConfig.CommonSetting.upperBackgroundColor,
                CustomConfig.CommonSetting.upperBackgroundAlpha);
        return isRtl ? Styles.getDrawableWithCorner(colorFromHex, 15, 0, 15, 15) :
                Styles.getDrawableWithCorner(colorFromHex, 0, 15, 15, 15);
    }

    public void setOnClickedListenerWrapper(MessageListAdapter.OnClickedListenerWrapper mWrapper) {
        this.mWrapper = mWrapper;
    }

    protected TextView getMsg(String botMsg, boolean enableInteraction) {
        if (!TextUtils.isEmpty(botMsg) && botMsg.contains("&formUrlTitle=")) {
            Matcher matcher = Pattern.compile("(http:|https:)(//)((?!\").)*?(&formUrlTitle=)*?").matcher(botMsg);
            if (matcher.matches()) {
                botMsg = botMsg + "&fromSdk=1&isCustom=1";
                Matcher subMatcher = Pattern.compile("(?<=(formUrlTitle=)).*(?=(&fromSdk=1&isCustom=1))").matcher(botMsg);
                String formTitle = botMsg;
                if (subMatcher.find()) {
                    formTitle = URLEncodeUtil.decode(subMatcher.group());
                }
                FormUrl formUrl = new FormUrl(formTitle, botMsg);
                return getHighlightedClickableTextView(formUrl.getTitle(), new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        String urlLink = formUrl.getLink();
                        if (AppInfoUtil.isUrlStillNeedResponding(mContext, urlLink)) {
                            if (mWrapper != null) {
                                mWrapper.onUrlClicked(true, urlLink);
                            }
                        }
                    }
                });
            }
        }

        TextView textView = new TextView(mContext);
        Styles.reRenderTextView(textView, "");
        textView.setLineSpacing(dip2px(mContext, 3), 1.0f);
        textView.setTextIsSelectable(true);

        boolean isMIUI12 = ("XiaoMi".equalsIgnoreCase(Build.MANUFACTURER) || "RedMi".equalsIgnoreCase(Build.MANUFACTURER)) &&
                Build.VERSION.SDK_INT == Build.VERSION_CODES.R;
        boolean isSamsungOreo = "samsung".equalsIgnoreCase(Build.MANUFACTURER) && Build.VERSION.SDK_INT <= Build.VERSION_CODES.O;

        if (isMIUI12 || isSamsungOreo) {
            textView.setText(botMsg);
        } else {
            textView.setText(getUrlSupportedText(botMsg));
            textView.setMovementMethod(LinkMovementMethod.getInstance());
        }

        return textView;
    }

    protected TextView getHighlightedClickableTextView(String text, View.OnClickListener onClickListener) {
        TextView textView = new TextView(mContext);
        Styles.reRenderTextView(textView, text,
                Color.parseColor(CustomConfig.CommonSetting.interactElementTextColor), true, 14
        );
        textView.setOnClickListener(onClickListener);
        textView.getPaint().setFlags(Paint.UNDERLINE_TEXT_FLAG);
        return textView;
    }

    @Override
    public void onDataSourceUpdated(List<Message> data) {
    }

    protected SpannableStringBuilder getUrlSupportedText(String msg) {
        return getUrlSupportedText(msg, CustomConfig.CommonSetting.interactElementTextColor);
    }

    protected SpannableStringBuilder getUrlSupportedText(String msg, String urlColor) {
        if (TextUtils.isEmpty(msg)) return new SpannableStringBuilder("");

        SpannableStringBuilder ssb = new SpannableStringBuilder(msg);

        Matcher matcher = Pattern.compile(RegexUtil.REGEX_RICH_TEXT).matcher(ssb);
        while (matcher.find()) {
            int start = matcher.start();
            int end = matcher.end();
            final String urlStr = matcher.group();

            ClickableSpan clickableSpan = new ClickableSpan() {
                @Override
                public void onClick(View view) {
                    if (AppInfoUtil.isUrlStillNeedResponding(mContext, urlStr)) {
                        if (mWrapper != null) {
                            mWrapper.onUrlClicked(false, urlStr);
                        }
                    }
                }

                @Override
                public void updateDrawState(TextPaint dp) {
                    dp.setColor(Color.parseColor(urlColor));
                }
            };
            ssb.setSpan(clickableSpan, start, end, Spannable.SPAN_EXCLUSIVE_INCLUSIVE);
            ssb.setSpan(new UnderlineSpan(), start, end, Spanned.SPAN_EXCLUSIVE_INCLUSIVE);
        }
        return ssb;
    }

    protected int getViewId(String name) {
        return ResResolver.getViewId(name);
    }

    protected int dip2px(Context context, double dpValue) {
        float density = context.getResources().getDisplayMetrics().density;
        return (int) (dpValue * density + 0.5);
    }

}
