package net.aihelp.ui.adapter.faq;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Color;
import android.os.Bundle;
import android.text.Spannable;
import android.text.SpannableString;
import android.text.TextUtils;
import android.text.style.ForegroundColorSpan;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.TextView;


import net.aihelp.common.CustomConfig;
import net.aihelp.data.track.event.EventTracker;
import net.aihelp.data.track.event.utils.EventType;
import net.aihelp.data.model.faq.FaqListEntity;
import net.aihelp.ui.faq.BaseFaqFragment;
import net.aihelp.ui.faq.IFaqEventListener;
import net.aihelp.ui.widget.AIHelpServiceEntrance;
import net.aihelp.ui.wrapper.FaqSelectedListenerWrapper;
import net.aihelp.utils.ListUtil;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.Styles;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.GridLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

public class FaqCardLayoutAdapter extends RecyclerView.Adapter<RecyclerView.ViewHolder> {

    private static final int VIEW_TYPE_CS_ENTRANCE = 1;

    private final Context context;
    private boolean isGridLayout;
    private final boolean isFooterVisible;

    private Bundle bundle;
    private IFaqEventListener faqEventListener;
    private BaseFaqFragment faqFragment;

    public void setup(Bundle bundle, IFaqEventListener faqEventListener, BaseFaqFragment faqFragment) {
        this.bundle = bundle;
        this.faqEventListener = faqEventListener;
        this.faqFragment = faqFragment;
    }

    private final List<FaqListEntity> mDataSources = new ArrayList<>();

    public FaqCardLayoutAdapter(Context context) {
        this(context, false);
    }

    public FaqCardLayoutAdapter(Context context, boolean isFooterVisible) {
        this.context = context;
        this.isFooterVisible = isFooterVisible;
    }

    @SuppressLint("NotifyDataSetChanged")
    public void update(List<FaqListEntity> dataSources) {
        this.mDataSources.clear();
        if (!ListUtil.isListEmpty(dataSources)) {
            this.mDataSources.addAll(dataSources);
        }
        notifyDataSetChanged();
    }

    private FaqSelectedListenerWrapper mListener;

    public void setOnFaqSelectedListener(FaqSelectedListenerWrapper mListener) {
        this.mListener = mListener;
    }

    @Override
    public int getItemViewType(int position) {
        if (isFooterVisible && position == getItemCount() - 1) {
            return VIEW_TYPE_CS_ENTRANCE;
        }
        return super.getItemViewType(position);
    }

    @NonNull
    @Override
    public RecyclerView.ViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        if (viewType == VIEW_TYPE_CS_ENTRANCE) {
            int layoutId = ResResolver.getLayoutId("aihelp_ada_faq_cs_entrance");
            View itemView = LayoutInflater.from(context).inflate(layoutId, parent, false);
            return new CSEntranceViewHolder(itemView);
        }

        int layoutId = ResResolver.getLayoutId(isGridLayout ? "aihelp_ada_faq_grid_list" :
                "aihelp_ada_faq_linear_list");
        View itemView = LayoutInflater.from(context).inflate(layoutId, parent, false);
        itemView.setBackground(Styles.getClickableDrawableForList());

        return new ItemViewHolder(itemView);
    }

    @Override
    public void onBindViewHolder(@NonNull RecyclerView.ViewHolder viewHolder, int position) {
        if (viewHolder instanceof ItemViewHolder) {
            FaqListEntity faq = mDataSources.get(position);
            switch (faq.getDisplayType()) {
                case FaqListEntity.FAQ_DISPLAY_HOT_TOPICS:
                case FaqListEntity.FAQ_DISPLAY_SECTION:
                    convertFaqHomeList(viewHolder, faq);
                    break;
                case FaqListEntity.FAQ_DISPLAY_SUB_SECTION:
                case FaqListEntity.FAQ_DISPLAY_QUESTION_LIST:
                    convertFaqSectionOrQuestionList(viewHolder, faq);
                    break;
                case FaqListEntity.FAQ_DISPLAY_SEARCH:
                    convertSearchMatchingList(viewHolder, faq);
                    break;
            }

            viewHolder.itemView.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    if (mListener != null) {
                        switch (faq.getDisplayType()) {
                            case FaqListEntity.FAQ_DISPLAY_SECTION:
                            case FaqListEntity.FAQ_DISPLAY_SUB_SECTION:
                                mListener.onIntentToSubSectionOrQuestionList(faq);
                                break;
                            case FaqListEntity.FAQ_DISPLAY_HOT_TOPICS:
                            case FaqListEntity.FAQ_DISPLAY_NOTIFICATION:
                            case FaqListEntity.FAQ_DISPLAY_QUESTION_LIST:
                            case FaqListEntity.FAQ_DISPLAY_SEARCH:
                                mListener.onIntentToQuestionContent(faq);
                                break;
                        }
                    }
                }
            });
        }

        if (viewHolder instanceof CSEntranceViewHolder) {
            CSEntranceViewHolder holder = (CSEntranceViewHolder) viewHolder;
            holder.csService.setup(bundle, faqEventListener, faqFragment);
        }
    }

    private void convertFaqHomeList(RecyclerView.ViewHolder viewHolder, final FaqListEntity faq) {
        ItemViewHolder holder = (ItemViewHolder) viewHolder;

        boolean hotTopicIcon = faq.getDisplayType() == FaqListEntity.FAQ_DISPLAY_HOT_TOPICS && CustomConfig.HelpCenter.isFaqHotTopicItemIconVisible;
        boolean sectionIcon = faq.getDisplayType() == FaqListEntity.FAQ_DISPLAY_SECTION && CustomConfig.HelpCenter.isFaqSectionItemIconVisible;
        Styles.loadIcon(holder.ivTitle, faq.getIconUrl(),
                hotTopicIcon || sectionIcon, "aihelp_svg_ic_placeholder");
        Styles.reRenderTextView(holder.tvTitle, faq.getTitle());
    }

    private void convertFaqSectionOrQuestionList(RecyclerView.ViewHolder viewHolder, final FaqListEntity faq) {
        ItemViewHolder holder = (ItemViewHolder) viewHolder;
        holder.ivTitle.setVisibility(View.GONE);
        Styles.reRenderTextView(holder.tvTitle, faq.getTitle());
    }

    private void convertSearchMatchingList(RecyclerView.ViewHolder viewHolder, final FaqListEntity faq) {
        try {
            ItemViewHolder holder = (ItemViewHolder) viewHolder;
            holder.ivTitle.setVisibility(View.GONE);
            Styles.reRenderTextView(holder.tvTitle, faq.getTitle());

            String query = faq.getQuery();
            String sectionName = faq.getTitle();
            if (!TextUtils.isEmpty(query)) {
                String matchedWords = query.toLowerCase();
                String title = sectionName.toLowerCase();
                int highlightColor = Color.parseColor(CustomConfig.CommonSetting.highlightedColor);
                Spannable spannedTitle = new SpannableString(sectionName);
                for (int index = TextUtils.indexOf(title, matchedWords, 0); index >= 0; index = TextUtils.indexOf(title, matchedWords, index + matchedWords.length())) {
                    spannedTitle.setSpan(new ForegroundColorSpan(highlightColor), index, Math.min(index + matchedWords.length(), title.length()), 33);
                }
                Styles.reRenderTextView(holder.tvTitle, spannedTitle.toString());
            } else {
                Styles.reRenderTextView(holder.tvTitle, sectionName);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onAttachedToRecyclerView(@NonNull RecyclerView recyclerView) {
        super.onAttachedToRecyclerView(recyclerView);
        isGridLayout = recyclerView.getLayoutManager() instanceof GridLayoutManager;
    }

    @Override
    public int getItemCount() {
        if (isFooterVisible) {
            return mDataSources.size() + 1;
        }
        return mDataSources.size();
    }

    public static class ItemViewHolder extends RecyclerView.ViewHolder {

        ImageView ivTitle;
        TextView tvTitle;

        public ItemViewHolder(@NonNull View itemView) {
            super(itemView);
            ivTitle = itemView.findViewById(ResResolver.getViewId("aihelp_iv_title"));
            tvTitle = itemView.findViewById(ResResolver.getViewId("aihelp_tv_title"));
        }
    }

    public static class CSEntranceViewHolder extends RecyclerView.ViewHolder {

        AIHelpServiceEntrance csService;

        public CSEntranceViewHolder(@NonNull View itemView) {
            super(itemView);
            csService = itemView.findViewById(ResResolver.getViewId("aihelp_cs_entrance"));
        }

    }

}
