package net.aihelp.ui.cs;

import android.annotation.SuppressLint;
import android.content.res.Configuration;
import android.graphics.Color;
import android.os.Bundle;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;


import net.aihelp.common.Const;
import net.aihelp.common.CustomConfig;
import net.aihelp.core.mvp.IPresenter;
import net.aihelp.core.net.http.config.HttpConfig;
import net.aihelp.core.net.mqtt.callback.IMqttCallback;
import net.aihelp.core.ui.BaseFragment;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.data.event.SearchViewVisibilityChangeEvent;
import net.aihelp.data.event.UpdateTitleEvent;
import net.aihelp.data.logic.common.MqttCallbackImpl;
import net.aihelp.data.model.rpa.msg.base.Message;
import net.aihelp.ui.adapter.MessageListAdapter;
import net.aihelp.ui.widget.AIHelpChatRecyclerView;
import net.aihelp.utils.KeyboardChangeHelper;
import net.aihelp.utils.ListUtil;
import net.aihelp.utils.SoftInputUtil;
import net.aihelp.utils.Styles;

import java.util.List;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.LinearLayoutManager;


public abstract class BaseCSFragment<P extends IPresenter> extends BaseFragment<P> {

    AIHelpChatRecyclerView rvMsgList;
    MessageListAdapter mAdapter;

    ViewGroup bottomContainer;
    IMqttCallback mqttCallback;

    private KeyboardChangeHelper helper;

    private Message mTimeStampMsg;

    private String titleIcon;
    private String titleText;

    @Override
    protected void getBundleBeforeDataPrepared(Bundle extras) {
        titleIcon = CustomConfig.CustomerService.csNavigationBarTitleIcon;
        titleText = CustomConfig.CustomerService.csNavigationTitle;
    }

    @Override
    public void onResume() {
        super.onResume();
        EventBus.getDefault().post(new UpdateTitleEvent(UpdateTitleEvent.CUSTOM_SERVICE, titleIcon, titleText));
        EventBus.getDefault().post(new SearchViewVisibilityChangeEvent(false));
    }

    @Override
    protected void initEventAndData(View contentView) {

        mqttCallback = MqttCallbackImpl.getInstance();

        ViewGroup bottomRootLayout = get("aihelp_chat_bottom_layout");
        bottomRootLayout.setBackgroundColor(Styles.getColorWithAlpha(CustomConfig.CommonSetting.upperBackgroundColor,
                CustomConfig.CommonSetting.upperBackgroundAlpha));
        bottomContainer = get("aihelp_bottom_container");

        TextView tvLogo = get("aihelp_logo");
        tvLogo.setVisibility(Const.TOGGLE_HIDE_BRAND ? View.GONE : View.VISIBLE);

        rvMsgList = get("aihelp_rv_msg_list");
        rvMsgList.setBackgroundColor(Color.TRANSPARENT);
        mAdapter = new MessageListAdapter(getContext(), this);
        rvMsgList.setAdapter(mAdapter);
        LinearLayoutManager linearLayoutManager = new LinearLayoutManager(getContext());
        rvMsgList.setLayoutManager(linearLayoutManager);

        helper = new KeyboardChangeHelper(getView());
        helper.addListener();
        helper.addOnKeyboardShowListener(new KeyboardChangeHelper.OnKeyboardShowListener() {
            @Override
            public void onKeyboardShow() {
                if (!linearLayoutManager.getStackFromEnd()) {
                    int firstVisibleItemPosition = linearLayoutManager.findFirstCompletelyVisibleItemPosition();
                    if (firstVisibleItemPosition > 0) linearLayoutManager.setStackFromEnd(true);
                }
                scrollRecyclerViewToEnd();
                onKeyboardStatusChanged(true);
            }

            @Override
            public void onKeyboardHide() {
                linearLayoutManager.setStackFromEnd(false);
                scrollRecyclerViewToEnd();
                onKeyboardStatusChanged(false);
            }
        });

    }

    protected void onKeyboardStatusChanged(boolean keyboardShow) {

    }

    public void updateChatList(Message msg) {
        if (msg.isEmptyMessage()) {
            return;
        }

        if (CustomConfig.CustomerService.isMessageTimestampVisible &&
                (msg.isUserMessage() || msg.isAgentMessage())) {
            prepareTimeStamp(msg.getTimestamp());
        }

        // 如果当前是玩家发送的消息，先插入消息，再插入 loading
        if (msg.isUserMessage()) {
            mAdapter.update(msg);
            if (msg.isDuringRPAProcedure()) {
                mAdapter.updateAgentTypingStatus(true);
            }
        } else if (msg.isAgentMessage()) {
            if (msg.isDuringRPAProcedure()) {
                // 如果是 agent 的消息，需要先把最后一条 loading 移除，再添加新消息
                mAdapter.updateAgentTypingStatus(false);
            }
            mAdapter.update(msg);
        }

        scrollRecyclerViewToEnd();
    }

    public void updateChatList(List<Message> msgList, boolean isClear) {
        if (!ListUtil.isListEmpty(msgList)) {
            // Try to find the latest timestamp message to avoid displaying time separator within 5 minutes
            for (int i = msgList.size() - 1; i >= 0; i--) {
                Message msg = msgList.get(i);
                if (msg != null && msg.isTimeStampMessage()) {
                    mTimeStampMsg = msg;
                    break;
                }
            }
            Message lastMessage = msgList.get(msgList.size() - 1);
            if (lastMessage != null && lastMessage.isAgentMessage() && lastMessage.isDuringRPAProcedure()) {
                mAdapter.updateAgentTypingStatus(false);
            }
        }
        mAdapter.update(msgList, isClear);
        scrollRecyclerViewToEnd();
    }

    protected void scrollRecyclerViewToEnd() {
        rvMsgList.post(new Runnable() {
            @Override
            public void run() {
                rvMsgList.scrollToPosition(mAdapter.getItemCount() - 1);
            }
        });
    }

    @SuppressLint("NotifyDataSetChanged")
    @Override
    public void onConfigurationChanged(@NonNull Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        if (rvMsgList != null) {
            SoftInputUtil.hideSoftInput(getContext(), rvMsgList);
            rvMsgList.setBackgroundColor(Color.TRANSPARENT);
            if (mAdapter != null) mAdapter.notifyDataSetChanged();
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        if (helper != null) {
            helper.removeListener();
        }
    }

    @Override
    protected boolean isBindEventBus() {
        return true;
    }

    private void prepareTimeStamp(long timeStamp) {
        if (mTimeStampMsg == null || timeStamp - Long.parseLong(mTimeStampMsg.getContent()) > 5 * 60 * 1000) {
            mTimeStampMsg = new Message(Message.TYPE_TIMESTAMP);
            mTimeStampMsg.setTimestamp(timeStamp - 1);
            mTimeStampMsg.setContent(String.valueOf(timeStamp));
            mAdapter.update(mTimeStampMsg);
        }
    }

    protected void countDownLoadingStatus(long timestamp) {
        android.os.Message message = android.os.Message.obtain();
        message.obj = timestamp;
        message.what = getMessageWhatFromTimeStamp(String.valueOf(timestamp));
        mHandler.sendMessageDelayed(message, 1000 * HttpConfig.TIME_OUT_LIMIT);
    }

    protected int getMessageWhatFromTimeStamp(String timeStamp) {
        return Integer.parseInt(timeStamp.substring(timeStamp.length() / 5 * 2));
    }

}
