package net.aihelp.ui.cs;

import android.content.Intent;
import android.os.Bundle;
import android.text.TextUtils;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

import net.aihelp.common.CustomConfig;
import net.aihelp.common.IntentValues;
import net.aihelp.core.net.monitor.NetworkMonitor;
import net.aihelp.core.net.monitor.NetworkState;
import net.aihelp.core.net.mqtt.AIHelpMqtt;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.permission.AIHelpPermissions;
import net.aihelp.data.event.PageHoppingEvent;
import net.aihelp.data.model.faq.FaqListEntity;
import net.aihelp.data.model.rpa.msg.UserMessage;
import net.aihelp.data.track.event.EventTracker;
import net.aihelp.data.track.event.utils.EventType;
import net.aihelp.data.logic.cs.CustomerServicePresenter;
import net.aihelp.data.logic.cs.MessagePoller;
import net.aihelp.data.model.rpa.msg.BotMessage;
import net.aihelp.data.model.rpa.msg.base.FileMessage;
import net.aihelp.data.model.rpa.msg.base.Message;
import net.aihelp.data.model.rpa.step.RPAStep;
import net.aihelp.data.model.rpa.msg.bot.Faq;
import net.aihelp.data.track.statistic.StatisticTracker;
import net.aihelp.ui.adapter.MessageListAdapter;
import net.aihelp.ui.adapter.faq.FaqAlertAdapter;
import net.aihelp.ui.cs.bottom.BottomActionPickerView;
import net.aihelp.ui.cs.bottom.BottomAttachmentView;
import net.aihelp.ui.cs.bottom.BottomBaseView;
import net.aihelp.ui.cs.bottom.BottomBotInputView;
import net.aihelp.ui.cs.bottom.BottomDatePickerView;
import net.aihelp.ui.cs.bottom.BottomEvaluateFaqView;
import net.aihelp.ui.cs.bottom.BottomEvaluateServiceView;
import net.aihelp.ui.cs.bottom.BottomFillFormView;
import net.aihelp.ui.cs.bottom.BottomManualInputView;
import net.aihelp.ui.cs.bottom.BottomNewConversationView;
import net.aihelp.ui.cs.bottom.BottomResolveConfirmView;
import net.aihelp.ui.cs.bottom.BottomSelfServiceView;
import net.aihelp.ui.cs.middle.MiddleHandlerView;
import net.aihelp.ui.cs.middle.predict.PredictRecyclerView;
import net.aihelp.data.logic.cs.rpa.factory.UserMessageFactory;
import net.aihelp.data.attachment.AttachmentPicker;
import net.aihelp.data.logic.cs.TicketStatusTracker;
import net.aihelp.utils.AppInfoUtil;
import net.aihelp.utils.ListUtil;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.Styles;
import net.aihelp.utils.TLog;
import net.aihelp.utils.ToastUtil;

import java.util.List;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import androidx.swiperefreshlayout.widget.SwipeRefreshLayout;

public class CustomerServiceFragment extends BaseCSFragment<CustomerServicePresenter> implements IServiceEventListener {

    private String userInput;
    private boolean shouldRefreshSkipView;

    private boolean isTicketAlreadyFinished;
    private SwipeRefreshLayout mRefreshLayout;
    private RPAStep mCurrentStep;

    private MiddleHandlerView mHandlerView;
    private PredictRecyclerView mPredictionView;
    private FaqAlertAdapter mFaqAlertAdapter;
    private BottomBaseView bottomBaseView = null;

    public static CustomerServiceFragment newInstance(Bundle bundle) {
        CustomerServiceFragment customerServiceFragment = new CustomerServiceFragment();
        customerServiceFragment.setArguments(bundle);
        return customerServiceFragment;
    }

    @Override
    protected void initEventAndData(View contentView) {
        super.initEventAndData(contentView);

        EventTracker.INSTANCE.log(EventType.CS_PAGE_OPENED);

        mHandlerView = get("aihelp_middle_view");
        mHandlerView.setServiceEventListener(this);

        mPredictionView = get("aihelp_rv_predict");
        mPredictionView.setLayoutManager(new LinearLayoutManager(getContext()));
        mFaqAlertAdapter = new FaqAlertAdapter(getContext(), new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mFaqAlertAdapter.clear();
                String input = ((TextView) v).getText().toString();
                UserMessage userMessage = Message.getUserTextMsg(input);
                userMessage.setRequestParams(input, UserMessage.INPUT_FORMAT_TEXT, UserMessage.INPUT_SOURCE_PREDICT);
                onUserAction(userMessage);
            }
        });
        mPredictionView.setAdapter(mFaqAlertAdapter);

        mRefreshLayout = get("aihelp_refresh_layout");
        mRefreshLayout.setEnabled(CustomConfig.CustomerService.isHistoryChatEnable);
        mRefreshLayout.setColorSchemeColors(Styles.getColor(CustomConfig.CommonSetting.interactElementTextColor));
        if (CustomConfig.CustomerService.isHistoryChatEnable) {
            mRefreshLayout.setOnRefreshListener(new SwipeRefreshLayout.OnRefreshListener() {
                @Override
                public void onRefresh() {
                    mPresenter.getLastConversation();
                }
            });
        }

        mAdapter.setOnClickedListener(new MessageListAdapter.OnClickedListenerWrapper() {

            // 此处回调的是人工客服发送的表单链接或者普通链接
            @Override
            public void onUrlClicked(boolean isForm, String url) {
                super.onUrlClicked(isForm, url);
                Bundle bundle = new Bundle();
                bundle.putString(IntentValues.INTENT_URL, url);
                EventBus.getDefault().post(new PageHoppingEvent(IntentValues.PAGE_HOPPING_FORM, bundle));
            }

            @Override
            public void onBotAnswerSelected(Message message) {
                onUserAction(message);
            }

            @Override
            public void onRetrySendingMessage(int position, Message msg) {
                if (!AppInfoUtil.validateNetwork(getContext())) {
                    return;
                } else if (isTicketAlreadyFinished) {
                    ToastUtil.INSTANCE.makeRawToast(getContext(), ResResolver.getString("aihelp_ticket_closed"));
                    return;
                }
                // If msg's status is failure, then the message is user file message, leave it alone,
                // let onUserAction() update its status, do not remove the message itself.
                // Check net.aihelp.ui.adapter.cs.user.UserFileAdapter.getRetryListener for more information.
                if (msg.getMsgStatus() != Message.STATUS_FAILURE) {
                    mAdapter.remove(position);
                }
                onUserAction(msg);
            }

        });

        mqttCallback.showMqttLoading();
        mPresenter.requestLogin();
        mPresenter.refreshUnreadMessageCount();

    }

    public void prepareMqtt(int connectionType) {
        mqttCallback.updateHostView(this);
        mPresenter.prepareMqtt(mqttCallback, connectionType);
    }

    public void onLogin(List<Message> msgList, Message message, RPAStep rpaStep) {
        updateChatList(msgList, true);
        updateBottomLayout(message, rpaStep);
        MessagePoller.INSTANCE.startMessagePolling(this);
        prepareMqtt(MqttConfig.TYPE_CONNECTION_LOGIN);
    }

    public void onFormSubmitted(String formContent) {
        shouldRefreshSkipView = true;
        updateChatList(UserMessageFactory.getUserFormMsgList(formContent), false);
        if (!TicketStatusTracker.isTicketActive && !(bottomContainer.getChildAt(0) instanceof BottomManualInputView)) {
            RPAStep rpaStep = new RPAStep();
            rpaStep.setNextStep(RPAStep.STEP_MANUAL_INPUT);
            updateBottomLayout(new Message(), rpaStep);
        }
    }

    public void onLastConversationRetrieved(List<Message> msgList) {
        mRefreshLayout.setRefreshing(false);
        if (!ListUtil.isListEmpty(msgList)) {
            mAdapter.insertHistoryConversation(msgList);
            RecyclerView.LayoutManager layoutManager = rvMsgList.getLayoutManager();
            if (layoutManager instanceof LinearLayoutManager) {
                LinearLayoutManager linearManager = (LinearLayoutManager) layoutManager;
                int lastItemPosition = linearManager.findLastVisibleItemPosition();
                rvMsgList.scrollToPosition(msgList.size() - 1 + lastItemPosition);
                rvMsgList.smoothScrollBy(0, -100);
            }
        }
    }

    public void onTicketFinishedOrRejected() {
        isTicketAlreadyFinished = true;
        RPAStep rpaStep = new RPAStep();
        rpaStep.setNextStep(RPAStep.STEP_NEW_CONVERSATION);
        updateBottomLayout(new Message(), rpaStep);
        mHandlerView.setVisibility(View.GONE);
        mAdapter.updateAgentTypingStatus(false);
    }

    public void onTicketAssignStatusChanged() {
        shouldRefreshSkipView = true;
        mHandlerView.updateViewVisibility(null);
    }

    public void updateMessageStatus(boolean updateToNormal, long timestamp, long updatedTimestamp) {
        for (int i = mAdapter.getDataList().size() - 1; i >= 0; i--) {
            Message msg = mAdapter.getDataList().get(i);
            if (msg.getTimestamp() == timestamp) {
                msg.setTimestamp(updatedTimestamp);
                msg.setMsgStatus(updateToNormal ? Message.STATUS_NORMAL : Message.STATUS_FAILURE);
                mAdapter.notifyItemChanged(i);
                // FIXME: 这里可能还有其他问题，需要再确定一下
                if (!updateToNormal) {
                    mAdapter.updateAgentTypingStatus(false);
                }
                break;
            }
        }
    }

    public void notifyMessageWithdrawn(long startTime, long endTime) {
        for (int i = mAdapter.getDataList().size() - 1; i >= 0; i--) {
            Message msg = mAdapter.getDataList().get(i);
            if (msg.getTimestamp() >= startTime && msg.getTimestamp() <= endTime) {
                mAdapter.remove(i);
                mPresenter.updateCachedUnreadMessageCount(msg.isAgentMessage(), true);
                if (startTime == endTime) break;
            }
        }
    }

    public void notifyMessageWithdrawn(long withdrawTimeStamp) {
        notifyMessageWithdrawn(withdrawTimeStamp, withdrawTimeStamp);
    }

    @Override
    public void updateChatList(Message msg) {
        super.updateChatList(msg);
        updateCacheUnreadMessageCount(msg);
    }

    @Override
    public void updateChatList(List<Message> msgList, boolean isClear) {
        super.updateChatList(msgList, isClear);
        if (!isClear) { // ignore the login scenario, otherwise the welcome message will mess up the cached count
            for (int i = 0; i < msgList.size(); i++) {
                updateCacheUnreadMessageCount(msgList.get(i));
            }
        }
    }

    private void updateCacheUnreadMessageCount(Message message) {
        if (message != null) {
            mPresenter.updateCachedUnreadMessageCount(message.isAgentMessage(), false);
        }
    }

    @NetworkMonitor
    public void onNetworkStateChanged(NetworkState networkState) {
        if (networkState != NetworkState.NONE && !MqttConfig.getInstance().isConnected()) {
            AIHelpMqtt.getInstance().prepare(mqttCallback, MqttConfig.TYPE_CONNECTION_RECONNECT);
        }
    }

    public void updateBottomLayout(Message message, RPAStep rpaStep) {
        updateBottomLayout(message, rpaStep, true);
    }

    public void updateBottomLayout(Message message, RPAStep rpaStep, boolean scrollToEnd) {
        // todo: make sure whether to use RPAStep to check duplicate step,
        // and do not forget to reset cachedStep to null in onUserAction and onUserGoBack
        mHandler.removeCallbacksAndMessages(null); // remove all pending event since we've obtained next action
        mPredictionView.setVisibility(View.GONE); // dismiss the predicting list after step changed
        mHandlerView.updateViewVisibility(rpaStep); // update the middle handle view according to current step
        mCurrentStep = rpaStep; // remember current step to refresh skip button after submitting a form
        bottomBaseView = null; // reset the bottom base view to avoid adding twice
        Bundle bundle = new Bundle();
        switch (rpaStep.getNextStep()) {
            case RPAStep.STEP_BOT_INPUT_TEXT:
            case RPAStep.STEP_BOT_INPUT_MAIL:
            case RPAStep.STEP_BOT_INPUT_NUMBER:
                bottomBaseView = new BottomBotInputView(getContext());
                break;
            case RPAStep.STEP_ACTION_PICKER:
                bottomBaseView = new BottomActionPickerView(getContext());
                break;
            case RPAStep.STEP_DATE_PICKER:
                bottomBaseView = new BottomDatePickerView(getContext());
                break;
            case RPAStep.STEP_ADDING_ATTACHMENT:
                bottomBaseView = new BottomAttachmentView(getContext());
                break;
            case RPAStep.STEP_FILLING_FORM:
                if (message instanceof BotMessage && ((BotMessage) message).hasFormUrl()) {
                    bundle.putString(IntentValues.INTENT_URL, ((BotMessage) message).getFormUrl().getLink());
                    bottomBaseView = new BottomFillFormView(getContext());
                } else {
                    bottomBaseView = new BottomBotInputView(getContext());
                }
                break;
            case RPAStep.STEP_SELF_SERVICE:
                if (message instanceof BotMessage && ((BotMessage) message).hasSelfService()) {
                    bundle.putParcelable(IntentValues.BOTTOM_SELF_SERVICE, ((BotMessage) message).getSelfService());
                    bottomBaseView = new BottomSelfServiceView(getContext());
                } else {
                    bottomBaseView = new BottomBotInputView(getContext());
                }
                break;
            case RPAStep.STEP_EVALUATE_FAQ:
                if (message instanceof BotMessage && ((BotMessage) message).hasFaq()) {
                    List<Faq.FaqData> faqDataList = ((BotMessage) message).getFaq().getFaqDataList();
                    if (faqDataList != null && faqDataList.size() == 1) {
                        bundle.putLong(IntentValues.BOTTOM_FAQ_CONTENT_ID, faqDataList.get(0).getContentId());
                    }
                }
                bottomBaseView = new BottomEvaluateFaqView(getContext());
                break;
            case RPAStep.STEP_MANUAL_INPUT:
                bundle.putBoolean(IntentValues.BOTTOM_TICKET_FINISHED, isTicketAlreadyFinished);
                bundle.putBoolean(IntentValues.BOTTOM_DURING_PROCEDURE, false);
                bottomBaseView = new BottomManualInputView(getContext());
                break;
            case RPAStep.STEP_RESOLVE_CONFIRM:
                bottomBaseView = new BottomResolveConfirmView(getContext());
                break;
            case RPAStep.STEP_EVALUATE_SERVICE:
                bottomBaseView = new BottomEvaluateServiceView(getContext());
                break;
            case RPAStep.STEP_NEW_CONVERSATION:
                mAdapter.updateAgentTypingStatus(false);
                bottomBaseView = new BottomNewConversationView(getContext());
                break;
        }
        if (bottomBaseView != null && !isBottomViewAlreadyAdded(bottomBaseView, bundle, rpaStep)) {
            if (TicketStatusTracker.isTicketServingByRPA() || TicketStatusTracker.isTicketServingByAgent()) {
                mAdapter.updateAgentTypingStatus(false);
            }
            bottomBaseView.setBottomViewEventListener(bundle, rpaStep, this);
            bottomContainer.removeAllViews();
            bottomContainer.addView(bottomBaseView);
        }
        if (scrollToEnd) scrollRecyclerViewToEnd();
    }

    private boolean isBottomViewAlreadyAdded(BottomBaseView baseView, Bundle bundle, RPAStep rpaStep) {
        if (baseView != null) {
            String targetBottomViewName = baseView.getClass().getSimpleName();
            BottomBaseView currentBottomView = (BottomBaseView) bottomContainer.getChildAt(0);
            if (currentBottomView != null) {
                String currentBottomViewName = currentBottomView.getClass().getSimpleName();
                if (targetBottomViewName.equals(currentBottomViewName)) {
                    currentBottomView.setBottomViewEventListener(bundle, rpaStep, this);
                    bottomBaseView = currentBottomView;
                    return true;
                }
            }
        }
        return false;
    }

    public long getLastAgentMessageTimestamp() {
        List<Message> dataList = mAdapter.getDataList();
        for (int i = dataList.size() - 1; i >= 0; i--) {
            Message message = dataList.get(i);
            if (message.isAgentMessage()) {
                return message.getTimestamp();
            }
        }
        return System.currentTimeMillis();
    }

    @Override
    public void onUserAction(Message message) {

        // 玩家做出具体的交互后，隐藏中间操作层
        mHandlerView.setVisibility(View.GONE);

        if (message instanceof FileMessage) {
            if (message.getMsgStatus() == Message.STATUS_SENDING) {
                updateChatList(message);
            } else {
                updateMessageStatus(message.getMsgStatus() == Message.STATUS_NORMAL,
                        message.getTimestamp(), message.getTimestamp());
            }
        } else {
            updateChatList(message);
        }

        if (message.getMsgStatus() == Message.STATUS_NORMAL) {
            mPresenter.chatWithSupport(message.getRequestType(), message.getTimestamp(), message.getRequestParams());
        }

        // 当用户处于 RPA 流程中，用户完成操作后直接将底部区域样式移除，以免重复操作引发其他问题
        // 如果当前是人工客服逻辑，那就不需要移除
        if (message.isDuringRPAProcedure()) {
            bottomContainer.removeAllViews();
            countDownLoadingStatus(message.getTimestamp());
        }

    }

    @Override
    protected void handleMsg(android.os.Message msg) {
        mAdapter.updateAgentTypingStatus(false);
        List<Message> dataList = mAdapter.getDataList();
        Message message = dataList.get(dataList.size() - 1);
        if (message != null && message.isUserMessage()) {
            message.setMsgStatus(Message.STATUS_FAILURE);
            mAdapter.notifyItemChanged(dataList.size() - 1);
        }
    }

    @Override
    public void onUserGoBack() {
        mHandlerView.setVisibility(View.GONE);
        bottomContainer.removeAllViews();
        mPresenter.goBackToPreviousStep();
    }

    @Override
    public void onNewConversationStarted() {
        isTicketAlreadyFinished = false;
        mPresenter.logout();
        mPresenter.requestLogin();
    }

    @Override
    public Fragment getHostFragment() {
        return this;
    }

    @Override
    public void scrollToBottom() {
        scrollRecyclerViewToEnd();
    }

    @Override
    public void scrollBy(int height) {
        rvMsgList.scrollBy(0, height);
    }

    @Override
    protected void onKeyboardStatusChanged(boolean keyboardShow) {
        if (TextUtils.isEmpty(userInput)) {
            // Screen orientation change would trigger global layout changes,
            // and cause this callback to be called with a misleading true for the `keyboardShow` parameter.
            // so we just ignore the keyboard show case for the landscape scenario,
            // since there will not have any keyboard show in the landscape mode
            if (keyboardShow && !Styles.isLandscape()) {
                mHandlerView.setVisibility(View.GONE);
            } else {
                // DO NOT update the view visibility if there is no element in the bottom container,
                // as this should be the user action case, where all the handler views have been removed in `onUserAction`,
                // otherwise users could send other messages when there is a loading on the screen which is not what we want
                if (bottomContainer != null && bottomContainer.getChildCount() > 0) {
                    mHandlerView.updateViewVisibility(mCurrentStep);
                }
            }
        }
    }

    @Override
    public void onMiddleHandlerViewVisibilityChanged() {
        boolean middleViewVisible = mHandlerView != null && mHandlerView.isVisible();
        // todo: try make this responsive according to the middle view's height change
        // get rid of the dynamic margin top solution
        int targetMarginBottom = middleViewVisible ? mHandlerView.getMeasuredHeight() : 0;
        ViewGroup.LayoutParams layoutParams = mRefreshLayout.getLayoutParams();
        if (layoutParams instanceof ViewGroup.MarginLayoutParams) {
            ViewGroup.MarginLayoutParams params = (ViewGroup.MarginLayoutParams) layoutParams;
            if (params.bottomMargin != targetMarginBottom) {
                params.bottomMargin = targetMarginBottom;
                mRefreshLayout.setLayoutParams(params);
                scrollRecyclerViewToEnd();
            }
        }
    }

    @Override
    public void onPredictResult(CharSequence query, List<FaqListEntity> faqs) {
        userInput = query.toString();
        if (!ListUtil.isListEmpty(faqs)) {
            mPredictionView.setVisibility(View.VISIBLE);
            mHandlerView.setVisibility(View.GONE);
            mFaqAlertAdapter.update(faqs, true);
        } else {
            mPredictionView.setVisibility(View.GONE);
        }
    }

    @Override
    public void onEditTextEditDone() {
        onKeyboardStatusChanged(false);
    }

    @Override
    public void onResume() {
        super.onResume();

        if (!MqttConfig.getInstance().isConnected()) {
            AIHelpMqtt.getInstance().prepare(mqttCallback, MqttConfig.TYPE_CONNECTION_RECONNECT);
            return;
        }

        // After form submitted or ticket assign status changed, scroll the message list to the end to display form content
        if (shouldRefreshSkipView) {
            scrollRecyclerViewToEnd();
            // 提交表单后返回客服页面，刷新跳过按钮的状态
            mHandlerView.updateViewVisibility(mCurrentStep);
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        if (mPresenter != null) mPresenter.logout();
        AIHelpMqtt.getInstance().disconnect();
        MessagePoller.INSTANCE.stopMessagePolling();
        StatisticTracker.getInstance().calculateDurationInCustomerService();
        EventTracker.INSTANCE.log(EventType.CS_PAGE_CLOSED);
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        AIHelpPermissions.getInstance().onRequestPermissionsResult(permissions, grantResults);
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, @Nullable Intent intent) {
        super.onActivityResult(requestCode, resultCode, intent);
        AttachmentPicker.INSTANCE.onAttachmentRequestResult(requestCode, resultCode, intent);
    }

    @Override
    protected int getLayout() {
        return ResResolver.getLayoutId("aihelp_fra_customer_service");
    }

    @Override
    protected int getLoadingTargetViewId() {
        return ResResolver.getViewId("aihelp_customer_service_root");
    }

}
