package net.aihelp.ui.cs.bottom;

import android.Manifest;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Build;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.View;

import net.aihelp.BuildConfig;
import net.aihelp.common.CustomConfig;
import net.aihelp.common.IntentValues;
import net.aihelp.common.SpKeys;
import net.aihelp.core.util.permission.AIHelpPermissions;
import net.aihelp.core.util.permission.IPermissionCallback;
import net.aihelp.core.util.permission.Permission;
import net.aihelp.data.attachment.IAttachmentPickerListener;
import net.aihelp.data.model.rpa.msg.base.FileMessage;
import net.aihelp.data.model.rpa.msg.base.Message;
import net.aihelp.data.model.rpa.step.RPAStep;
import net.aihelp.data.model.rpa.msg.UserMessage;
import net.aihelp.ui.helper.BreakReleaseHelper;
import net.aihelp.ui.preview.PreviewActivity;
import net.aihelp.data.attachment.AttachmentPicker;
import net.aihelp.ui.cs.IServiceEventListener;
import net.aihelp.ui.preview.data.PreviewInfo;
import net.aihelp.ui.widget.AIHelpButton;
import net.aihelp.ui.widget.snackbar.Snackbar;
import net.aihelp.utils.AppInfoUtil;
import net.aihelp.utils.MediaUtils;
import net.aihelp.utils.RegexUtil;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.SpUtil;
import net.aihelp.utils.ToastUtil;
import net.aihelp.utils.UploadFileHelper;

import org.json.JSONObject;

import java.io.File;
import java.lang.reflect.Method;

import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;

public class BottomAttachmentView extends BottomBaseView implements View.OnClickListener, IAttachmentPickerListener {

    public BottomAttachmentView(Context context) {
        this(context, null);
    }

    public BottomAttachmentView(Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public BottomAttachmentView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        View.inflate(context, ResResolver.getLayoutId("aihelp_bottom_attachment"), this);

        AIHelpButton addAttachment = findViewById(ResResolver.getViewId("aihelp_btn_add_attachment"));
        addAttachment.setText(ResResolver.getString("aihelp_upload_attachment"));
        addAttachment.setOnClickListener(this);
    }

    @Override
    public void onClick(View v) {
        if (AppInfoUtil.validateNetwork(getContext()) && v.getId() == ResResolver.getViewId("aihelp_btn_add_attachment")) {
            if (mListener != null) {
                Fragment hostFragment = mListener.getHostFragment();
                if (hostFragment != null) {
                    String[] permissionArray = new String[]{Manifest.permission.READ_EXTERNAL_STORAGE};
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                        permissionArray = new String[]{
                                Manifest.permission.READ_MEDIA_IMAGES,
                                Manifest.permission.READ_MEDIA_VIDEO
                        };
                    }
                    AIHelpPermissions.getInstance()
                            .setFragment(hostFragment)
                            .setPermissionHandler(this)
                            .setRequestCode(Permission.REQUEST_CODE_CUSTOMER_SERVICE)
                            .setRequestPermission(permissionArray)
                            .request(getContext(), AttachmentPicker.ATTACHMENT_TYPE_RPA);
                }
            }
        }
    }

    @Override
    public void setBottomViewEventListener(Bundle bundle, RPAStep step, IServiceEventListener listener) {
        super.setBottomViewEventListener(bundle, step, listener);
        if (step != null) {
            CustomConfig.UploadLimit.rpaAttachmentTypes = step.getAttachmentTypes();
        }
    }

    @Permission(requestCode = Permission.REQUEST_CODE_CUSTOMER_SERVICE)
    public void onPermissionRequested(Permission.Result result, IPermissionCallback callback, int requestType) {
        if (mListener != null) {
            Fragment hostFragment = mListener.getHostFragment();
            switch (result) {
                case GRANTED:
                case NONE:
                    AttachmentPicker.INSTANCE.setPickerHost(hostFragment).setAttachmentPickerListener(this)
                            .launchPicker(requestType);
                    break;
                case DENIED:
                    ToastUtil.INSTANCE.showRawSnackBar(hostFragment.getActivity(), ResResolver.getString("aihelp_permission_denied"), Snackbar.LENGTH_LONG);
                    break;
                case RATIONAL:
                    ToastUtil.INSTANCE.showRawSnackBar(hostFragment.getActivity(), ResResolver.getString("aihelp_permission_denied"),
                            ResResolver.getString("aihelp_yes"), Snackbar.LENGTH_LONG, new View.OnClickListener() {
                                @Override
                                public void onClick(View v) {
                                    callback.onPermissionRational();
                                }
                            });
                    break;
                case GO_SETTING:
                    ToastUtil.INSTANCE.showRawSnackBar(hostFragment.getActivity(), ResResolver.getString("aihelp_permission_ignored"),
                            ResResolver.getString("aihelp_permission_settings"), Snackbar.LENGTH_LONG, new View.OnClickListener() {
                                @Override
                                public void onClick(View v) {
                                    callback.onPermissionIgnored();
                                }
                            });
                    break;
            }
        }
    }

    @Override
    public void onPickSuccess(File pickedFile) {
        String path = pickedFile.getAbsolutePath();
        if (mListener != null) {
            Fragment hostFragment = mListener.getHostFragment();
            if (hostFragment != null) {
                PreviewInfo previewInfo = PreviewInfo.get(path, pickedFile.getName(), pickedFile.length(), true);
                PreviewActivity.startAct(hostFragment, previewInfo);
            }
        }
    }

    @Override
    public void onPickFailure(int errCode) {
        ToastUtil.INSTANCE.makeRawToast(getContext(), ResResolver.getString("aihelp_resource_not_support"));
    }

    @Override
    public void onPreviewCanceled(int attachmentType) {
        if (mListener != null) {
            Fragment hostFragment = mListener.getHostFragment();
            if (hostFragment != null) {
                AttachmentPicker.INSTANCE.setPickerHost(hostFragment)
                        .setAttachmentPickerListener(this).launchPicker(attachmentType);
            }
        }
    }

    @Override
    public void onPreviewConfirmed(String path) {
        File file = new File(path);
        int msgType = getMsgTypeByPath(file.getName());
        if (msgType > 0) {
            if (mListener != null) {
                FileMessage fileMsg = new FileMessage(msgType, file.getPath());
                fileMsg.setFileInfo(file.getName(), file.length());
                fileMsg.setMsgStatus(Message.STATUS_SENDING);
                fileMsg.setDuringRPAProcedure(bundle.getBoolean(IntentValues.BOTTOM_DURING_PROCEDURE, true));
                mListener.onUserAction(fileMsg);

                UploadFileHelper.INSTANCE.setOnUploadFileListener(new UploadFileHelper.OnUploadFileListener() {
                    @Override
                    public void onFileUploaded(String fileUrl) {
                        boolean isUrlValid = !TextUtils.isEmpty(fileUrl);
                        fileMsg.setMsgStatus(isUrlValid ? Message.STATUS_NORMAL : Message.STATUS_FAILURE);
                        fileMsg.setDuringRPAProcedure(bundle.getBoolean(IntentValues.BOTTOM_DURING_PROCEDURE, true));
                        fileMsg.setRequestParams(getUploadRequestParams(msgType, fileUrl, file));
                        if (mListener != null) {
                            mListener.onUserAction(fileMsg);
                        }
                    }
                }).performUpload(file);
            }
        } else {
            if (BreakReleaseHelper.isBreak()) {
                ToastUtil.INSTANCE.makeText(getContext(), file.getAbsolutePath(), true);
            } else {
                ToastUtil.INSTANCE.makeRawToast(getContext(), ResResolver.getString("aihelp_resource_not_support"));
            }
        }
    }

    private int getMsgTypeByPath(String filePath) {
        if (CustomConfig.UploadLimit.isImageEnableUploading && RegexUtil.isMatch(filePath, RegexUtil.AIHELP_SUPPORTED_IMAGE)) {
            if (RegexUtil.isMatch(filePath, RegexUtil.ANDROID_SUPPORTED_IMAGE)) {
                return Message.TYPE_USER_IMAGE;
            } else {
                return Message.TYPE_USER_FILE;
            }
        }
        if (CustomConfig.UploadLimit.isVideoEnableUploading && RegexUtil.isMatch(filePath, RegexUtil.AIHELP_SUPPORTED_VIDEO)) {
            if (RegexUtil.isMatch(filePath, RegexUtil.ANDROID_SUPPORTED_VIDEO)) {
                return Message.TYPE_USER_VIDEO;
            } else {
                return Message.TYPE_USER_FILE;
            }
        }
        if (CustomConfig.UploadLimit.isFileEnableUploading && isSupportedFileType(filePath)) {
            return Message.TYPE_USER_FILE;
        }
        return -1;
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        UploadFileHelper.INSTANCE.setOnUploadFileListener(null);
    }

    private JSONObject getUploadRequestParams(int msgType, String fileUrl, File file) {
        switch (msgType) {
            case Message.TYPE_USER_IMAGE:
                Bitmap bitmap = BitmapFactory.decodeFile(file.getPath());
                return UserMessage.getRequestParams(fileUrl, file.getName(), file.length(),
                        getSafeValue(bitmap, "getWidth", 0), getSafeValue(bitmap, "getHeight", 0));
            case Message.TYPE_USER_VIDEO:
                Bitmap thumbnail = BitmapFactory.decodeFile(MediaUtils.getImageForVideoSync(file.getPath()));
                return UserMessage.getRequestParams(fileUrl, file.getName(), file.length(),
                        getSafeValue(thumbnail, "getWidth", 0), getSafeValue(thumbnail, "getHeight", 0));
            case Message.TYPE_USER_FILE:
                return UserMessage.getRequestParams(fileUrl, file.getName(), file.length(), 0, 0);
        }
        return null;
    }

    private <In, Out> Out getSafeValue(In value, String methodName, Out fallback) {
        if (value == null) return fallback;
        try {
            Class<?> clazz = value.getClass();
            Method declaredMethod = clazz.getDeclaredMethod(methodName);
            declaredMethod.setAccessible(true);
            return (Out) declaredMethod.invoke(value);
        } catch (Exception e) {
            return fallback;
        }
    }

    private boolean isSupportedFileType(String filePath) {
        String fileTypes = CustomConfig.UploadLimit.fileTypes;
        if (TextUtils.isEmpty(filePath) || TextUtils.isEmpty(fileTypes)) {
            return false;
        }
        return fileTypes.contains(filePath.substring(filePath.indexOf(".") + 1));
    }
}
