package net.aihelp.ui.cs.middle;

import android.content.Context;
import android.graphics.Color;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.View;
import android.widget.RelativeLayout;

import net.aihelp.common.CustomConfig;
import net.aihelp.data.model.config.IntentEntity;
import net.aihelp.data.model.rpa.msg.base.Message;
import net.aihelp.data.model.rpa.step.RPAStep;
import net.aihelp.data.model.rpa.msg.UserMessage;
import net.aihelp.ui.cs.IServiceEventListener;
import net.aihelp.ui.cs.middle.intent.IOnIntentViewEventListener;
import net.aihelp.ui.cs.middle.intent.SmartIntentView;
import net.aihelp.ui.widget.AIHelpButton;
import net.aihelp.utils.AppInfoUtil;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.Styles;

public class MiddleHandlerView extends RelativeLayout implements View.OnClickListener, IOnIntentViewEventListener {

    private final SmartIntentView intentView;
    private final AIHelpButton tvPrevStep;
    private final AIHelpButton tvSkip;
    private IServiceEventListener serviceEventListener;

    public void setServiceEventListener(IServiceEventListener serviceEventListener) {
        this.serviceEventListener = serviceEventListener;
    }

    public MiddleHandlerView(Context context) {
        this(context, null);
    }

    public MiddleHandlerView(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public MiddleHandlerView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        View.inflate(context, ResResolver.getLayoutId("aihelp_layout_middle_handler"), this);
        intentView = findViewById(ResResolver.getViewId("aihelp_smart_intent"));
        tvPrevStep = findViewById(ResResolver.getViewId("aihelp_tv_previous"));
        tvSkip = findViewById(ResResolver.getViewId("aihelp_tv_skip"));
        tvPrevStep.setOnClickListener(this);
        tvSkip.setOnClickListener(this);
        intentView.setOnIntentViewEventListener(this);
    }

    @Override
    public void onClick(View v) {
        if (!AppInfoUtil.validateNetwork(getContext())) return;
        if (v.getId() == ResResolver.getViewId("aihelp_tv_previous")) {
            if (serviceEventListener != null) {
                serviceEventListener.onUserGoBack();
            }
        }
        if (v.getId() == ResResolver.getViewId("aihelp_tv_skip")) {
            if (serviceEventListener != null) {
                String skip = tvSkip.getText().toString();
                UserMessage userMessage = Message.getUserTextMsg(skip);
                userMessage.setRequestParams(skip, true, UserMessage.INPUT_FORMAT_BUTTON,
                        "", UserMessage.INPUT_SOURCE_SKIP);
                serviceEventListener.onUserAction(userMessage);
                tvSkip.setVisibility(GONE);
                notifyVisibilityChanged();
            }
        }
    }

    public void updateViewVisibility(RPAStep rpaStep) {
        setVisibility(VISIBLE);

        if (rpaStep != null && rpaStep.isEnableSkip() && rpaStep.getNextStep() != RPAStep.STEP_ACTION_PICKER) {
            updateButtonStyle(tvSkip, rpaStep.getSkipHint(), Styles.getScreenWidth(getContext()) * 0.3f);
        }

        // intentView and tvPrevStep are mutually exclusive, they cannot be displayed at the same time
        if (rpaStep != null && rpaStep.isEnablePrevStep()) {
            updateButtonStyle(tvPrevStep, rpaStep.getPrevStepHint(), Styles.getScreenWidth(getContext()) * 0.6f);
        } else {
            intentView.updateViewVisibility();
        }

        notifyVisibilityChanged();
    }

    @Override
    public void setVisibility(int visibility) {
        super.setVisibility(visibility);

        // hide all child views whenever user tries to update MiddleHandlerView's visibility,
        // which is totally fine since every child view will take care of their own visibility later
        tvSkip.setVisibility(GONE);
        tvPrevStep.setVisibility(GONE);
        intentView.setVisibility(GONE);

        if (visibility != getVisibility()) {
            notifyVisibilityChanged();
        }
    }

    public void notifyVisibilityChanged() {
        if (serviceEventListener != null) {
            serviceEventListener.onMiddleHandlerViewVisibilityChanged();
        }
    }

    public boolean isVisible() {
        measure(0, 0);
        return getVisibility() == View.VISIBLE && (intentView.getVisibility() == View.VISIBLE || tvPrevStep.getVisibility() == View.VISIBLE
                || tvSkip.getVisibility() == View.VISIBLE);
    }

    @Override
    public void onIntentViewVisibilityChanged(int visibility) {
        notifyVisibilityChanged();
    }

    @Override
    public void onIntentSelected(IntentEntity intent) {
        String intentName = intent.getIntentName();
        String intentId = String.valueOf(intent.getIntentId());
        UserMessage userMessage = Message.getUserTextMsg(intentName);
        userMessage.setRequestParams(intentName, false, UserMessage.INPUT_FORMAT_INTENT, intentId,
                UserMessage.INPUT_SOURCE_INTENT);
        if (serviceEventListener != null) {
            serviceEventListener.onUserAction(userMessage);
        }
    }

    private void updateButtonStyle(AIHelpButton targetView, String text, float maxWidth) {
        if (!TextUtils.isEmpty(text)) {
            targetView.setVisibility(View.VISIBLE);
            targetView.setMaxWidth((int) maxWidth);
            targetView.setBackground(Styles.getDrawable(Color.parseColor(CustomConfig.CommonSetting.interactElementTextColor), 8));
            targetView.setText(text);
        } else {
            targetView.setVisibility(View.GONE);
        }
    }

}
