package net.aihelp.ui.faq;

import android.graphics.Color;
import android.os.Bundle;
import android.text.TextUtils;
import android.view.View;
import android.widget.FrameLayout;

import net.aihelp.common.API;
import net.aihelp.common.ConfigValues;
import net.aihelp.common.CustomConfig;
import net.aihelp.common.IntentValues;
import net.aihelp.common.UserProfile;
import net.aihelp.core.net.http.AIHelpRequest;
import net.aihelp.core.net.http.callback.ReqCallback;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.data.event.PageHoppingEvent;
import net.aihelp.data.localize.config.ProcessEntranceHelper;
import net.aihelp.data.localize.data.FaqHelper;
import net.aihelp.data.logic.cs.TicketStatusTracker;
import net.aihelp.data.logic.cs.rpa.helper.ResponseHelper;
import net.aihelp.data.track.event.EventTracker;
import net.aihelp.data.track.event.utils.EventType;
import net.aihelp.data.logic.faq.FaqPresenter;

import net.aihelp.data.model.config.ProcessEntity;
import net.aihelp.data.model.faq.FaqContentEntity;
import net.aihelp.data.track.statistic.StatisticTracker;
import net.aihelp.ui.HostActivity;
import net.aihelp.ui.helper.WebViewInjector;
import net.aihelp.ui.webkit.AIHelpWebChromeClient;
import net.aihelp.ui.webkit.AIHelpWebProgress;
import net.aihelp.ui.webkit.AIHelpWebView;
import net.aihelp.ui.webkit.AIHelpWebViewClient;
import net.aihelp.ui.widget.AIHelpButton;
import net.aihelp.ui.widget.AIHelpEvaluateView;
import net.aihelp.ui.widget.AIHelpServiceEntrance;
import net.aihelp.utils.AppInfoUtil;
import net.aihelp.utils.DomainSupportHelper;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.Styles;

import org.json.JSONObject;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class FaqContentFragment extends BaseFaqFragment<FaqPresenter> {

    private boolean showEntranceAfterEvaluated;
    private AIHelpWebView mWebView;
    private AIHelpWebProgress mProgressBar;
    private AIHelpButton mAttachedForm;
    private AIHelpEvaluateView mEvaluateView;
    private AIHelpServiceEntrance mServiceEntrance;
    private String currentSectionId;

    public static FaqContentFragment newInstance(Bundle bundle) {
        FaqContentFragment faqFragment = new FaqContentFragment();
        faqFragment.setArguments(bundle);
        return faqFragment;
    }

    @Override
    protected void getBundleBeforeDataPrepared(Bundle extras) {
        super.getBundleBeforeDataPrepared(extras);
        currentSectionId = extras.getString(IntentValues.SECTION_ID, "");
    }

    @Override
    protected void initEventAndData(View view) {
        FrameLayout flContent = get("aihelp_ll_root");
        int colorFromHex = Styles.getColorWithAlpha(CustomConfig.CommonSetting.upperBackgroundColor,
                CustomConfig.CommonSetting.upperBackgroundAlpha);
        flContent.setBackgroundColor(colorFromHex);

        mProgressBar = get("aihelp_progress_bar");
        mAttachedForm = get("aihelp_btn_form");
        mEvaluateView = get("aihelp_evaluate_faq");
        mServiceEntrance = get("aihelp_cs_entrance");

        mWebView = get("aihelp_web_view");
        mWebView.setBackgroundColor(Color.TRANSPARENT);
        AIHelpWebViewClient webViewClient = new AIHelpWebViewClient(getContext(), mProgressBar, true);
        mWebView.setWebViewClient(webViewClient);
        WebViewInjector.getInstance().setHostFragment(this).inject(mWebView, webViewClient);
        mWebView.setWebChromeClient(new AIHelpWebChromeClient(this, mProgressBar));

        // 在 showSingleFAQ 的场景下准备 mqtt 和未读消息数据
        if (intentMode == IntentValues.MODE_SHOW_SINGLE_FAQ) {
            mPresenter.prepareFAQNotification();
        }
    }

    @Override
    protected void getBundleAfterDataPrepared(Bundle extras) {
        String sectionId = extras.getString(IntentValues.SECTION_ID);
        String faqId = extras.getString(IntentValues.FAQ_MAIN_ID);
        String searchTerm = extras.getString(IntentValues.SEARCH_MATCH);
        mPresenter.goFetchQuestionContent(sectionId, faqId, searchTerm);
        if (TextUtils.isEmpty(sectionId) && TextUtils.isEmpty(searchTerm)) {
            titleText = AppInfoUtil.getAppName(getContext());
        }
    }

    @Override
    protected int getLayout() {
        return ResResolver.getLayoutId("aihelp_fra_faq_content");
    }

    @Override
    protected int getLoadingTargetViewId() {
        return ResResolver.getViewId("aihelp_faq_content");
    }

    public void refreshQuestionContent(FaqContentEntity faq, String searchTerm) {
        super.refreshQuestionContent(faq, searchTerm);
        if (faq != null) {

            // 配置绑定的表单
            if (!TicketStatusTracker.isTicketActive && !TextUtils.isEmpty(faq.getFormTitle())
                    && !TextUtils.isEmpty(faq.getFormUrl())) {
                mAttachedForm.setVisibility(View.VISIBLE);
                mAttachedForm.setText(faq.getFormTitle());
                mAttachedForm.setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        Bundle bundle = new Bundle();
                        bundle.putString(IntentValues.INTENT_URL, String.format("%s&userId=%s", faq.getFormUrl(), UserProfile.USER_ID));
                        EventBus.getDefault().post(new PageHoppingEvent(IntentValues.PAGE_HOPPING_FORM, bundle));
                        // be sure to call the login endpoint, otherwise the device info could be lost
                        JSONObject loginParams = ResponseHelper.getLoginParams();
                        AIHelpRequest.getInstance().requestPostByJson(API.LOGIN, loginParams, null);
                    }
                });
            } else {
                mAttachedForm.setVisibility(View.GONE);
            }

            // 配置 EvaluateView
            mEvaluateView.setFaqData(faq.getFaqTitle(), faq.getFaqMainId(), faq.getFaqContentId());
            if (CustomConfig.InitSetting.isEvaluationForAnswerPageEnable &&
                    mPresenter.shouldShowQuestionFooter(faq.getFaqMainId(), faq.getLastUpdateTime())) {
                mEvaluateView.setEvaluateState(AIHelpEvaluateView.STATE_NORMAL);
            } else {
                mEvaluateView.setEvaluateState(AIHelpEvaluateView.STATE_INVISIBLE);
            }
            mEvaluateView.setOnAIHelpEvaluateViewCallback(new AIHelpEvaluateView.OnAIHelpEvaluateViewCallback() {
                @Override
                public void onEvaluated(boolean isHelpful) {
                    if (!isHelpful && getArguments() != null) {
                        String supportMoment = getArguments().getString(IntentValues.FAQ_SUPPORT_MOMENT, "");
                        if (supportMoment.contains(ConfigValues.FAQ_SUPPORT_ENTRANCE_AFTER_MARKING_UNHELPFUL)) {
                            mServiceEntrance.setVisibility(View.VISIBLE);
                            showEntranceAfterEvaluated = true;
                        }
                    }
                    FaqHelper.INSTANCE.afterFaqEvaluated(faq.getFaqMainId(), faq.getLastUpdateTime());
                }
            });

            // 设置 WebView 内容
            String faqContent = DomainSupportHelper.getAdjustedUrl(faq.getFaqContent());
            int[] colorRGB = Styles.getColorRGB(CustomConfig.CommonSetting.textColor);
            String textColor = String.format("color: rgba(%s, %s, %s, %s)", colorRGB[0], colorRGB[1], colorRGB[2], 1);
            String timeColor = String.format("color: rgba(%s, %s, %s, %s)", colorRGB[0], colorRGB[1], colorRGB[2], 0.3);
            faqContent = faqContent.replace("<body>", String.format("<body style=\"background-color: transparent; %s\">", textColor));
            faqContent = faqContent.replace("<div style='font-size:14px;color:#CCCCCC;'>", String.format("<div style='font-size:14px; %s'>", timeColor));
            faqContent = faqContent.replaceAll("(?i)(\\.(mp4|mov))", "$1#t=0.01");
            faqContent = scaleFaqContent(faqContent, "font-size");
//            faqContent = scaleFaqContent(faqContent, "line-height");
            mWebView.loadDataWithBaseURL(null, faqContent, "text/html", "utf-8", null);

            handleTrackLogic(faq, searchTerm);

        } else {
            showEmpty();
        }

    }

    private String scaleFaqContent(String content, String key) {
        StringBuffer output = new StringBuffer();
        try {
            Pattern pattern = Pattern.compile(String.format("(%s:)\\s*(\\d+(?:\\.\\d+)?)([^;]+)?", key));
            Matcher matcher = pattern.matcher(content);
            while (matcher.find()) {
                String firstGroup = matcher.group(1);
                String secondGroup = matcher.group(2);
                String thirdGroup = matcher.group(3);

                if (!TextUtils.isEmpty(firstGroup) && !TextUtils.isEmpty(secondGroup)) {
                    float oldSize = Float.parseFloat(secondGroup);
                    String replacement = String.format("%s%s%s",
                            firstGroup,
                            oldSize * CustomConfig.CommonSetting.scaleFactor,
                            TextUtils.isEmpty(thirdGroup) ? "" : thirdGroup
                    );
                    matcher.appendReplacement(output, replacement);
                }
            }
            matcher.appendTail(output);
            return output.toString();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return content;
    }

    private void handleTrackLogic(FaqContentEntity faq, String searchTerm) {
        // FAQ 点击事件统计统一处理
        switch (currentSectionId) {
            case FaqHelper.FAQ_NOTIFICATION:
                StatisticTracker.getInstance().clickNotification(faq.getFaqMainId(), faq.getFaqContentId(), faq.getFaqTitle());
                break;
            case FaqHelper.FAQ_HOT_TOPICS:
                StatisticTracker.getInstance().clickHotTopic(faq.getFaqMainId(), faq.getFaqContentId(), faq.getFaqTitle());
                break;
            default:
                // 如果当前入口为某个分类，并且当前 FAQ 所属分类 ID 与入口的分类 ID 相同，则说明当前 FAQ 是从该入口的 FAQ 列表点击进入的
                // 这种情况下需要对该次点击事件进行打点
                ProcessEntity currentProcess = ProcessEntranceHelper.INSTANCE.getCurrentProcess();
                if (currentProcess != null && currentProcess.getIntent() == ConfigValues.ENTRANCE_FAQ_SECTION
                        && currentProcess.getSectionId().equals(currentSectionId)) {
                    StatisticTracker.getInstance().clickFaq(faq.getFaqMainId(), faq.getFaqContentId(), faq.getFaqTitle());
                }
                break;
        }
        // FAQ 阅读量统计
        StatisticTracker.getInstance().checkedFAQ(faq.getFaqMainId(), faq.getFaqContentId(), faq.getFaqTitle());
        // FAQ 打开
        EventTracker.INSTANCE.log(EventType.HC_PAGE_CONTENT_OPENED,
                "sectionId", currentSectionId,
                "faqMainId", faq.getFaqMainId(),
                "faqTitle", faq.getFaqTitle(),
                "query", searchTerm
        );
    }

    @Override
    public void onResume() {
        super.onResume();
        if (showEntranceAfterEvaluated) {
            mServiceEntrance.setVisibility(View.VISIBLE);
        }
        // If the user comes back after filling out a form in FAQ, the isTicketActive flag will be true,
        // display the customer service button and jump to customer service page directly
        if (mAttachedForm != null && mAttachedForm.getVisibility() == View.VISIBLE
                && TicketStatusTracker.isTicketActive) {
            mAttachedForm.setVisibility(View.GONE);
            ///Refreshing here seems to be redundant, the user would be navigated to customer service page,
            // they can not see the difference of button change. After user comes back from the customer service page,
            // the service entrance can be refreshed in the parent fragment.
            // So, there is no need to refresh customer service at here.
            if (csEntrance != null) {
                csEntrance.onIntentToCustomerService(getMergedBundle(), getFaqFlowListener(), this);
            }
        }
    }

    @Override
    public void onStop() {
        super.onStop();
        mProgressBar.hide();
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        EventTracker.INSTANCE.log(EventType.HC_PAGE_CONTENT_CLOSED);
    }

    public boolean onBackPressed() {
        if (mWebView.canGoBack()) {
            mWebView.goBack();
            return false;
        }
        return true;
    }

}
