package net.aihelp.ui.widget;

import android.content.Context;
import android.content.res.TypedArray;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.widget.EditText;
import android.widget.FrameLayout;
import android.widget.TextView;

import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.common.CustomConfig;
import net.aihelp.common.UserProfile;
import net.aihelp.core.net.http.AIHelpRequest;
import net.aihelp.core.net.http.callback.ReqCallback;
import net.aihelp.core.ui.dialog.AlertDialog;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.bus.Subscribe;
import net.aihelp.core.util.bus.ThreadMode;
import net.aihelp.core.util.bus.event.EventCenter;
import net.aihelp.data.event.OrientationChangeEvent;
import net.aihelp.data.event.TemplateReadyEvent;
import net.aihelp.data.track.event.EventTracker;
import net.aihelp.data.track.event.utils.EventType;
import net.aihelp.data.track.statistic.StatisticTracker;
import net.aihelp.ui.wrapper.TextWatcherWrapper;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.Styles;
import net.aihelp.utils.ToastUtil;

import org.json.JSONObject;

import java.util.Iterator;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.widget.AppCompatImageView;

public class AIHelpEvaluateView extends FrameLayout implements View.OnClickListener {

    public static final int FAQ_TYPE_BOT_FAQ = 1;
    public static final int FAQ_TYPE_FAQ_DETAIL = 2;
    public static final int FAQ_TYPE_OPERATE_ARTICLE = 3;

    public static final int STATE_NORMAL = 1;
    public static final int STATE_INVISIBLE = 2;
    public static final int STATE_PLAIN_TEXT = 3;
    public static final int STATE_REQUESTING_FEEDBACK = 4;

    private int evaluateTarget = -1;

    private final ViewGroup mEvaluateFaqLayout;

    private final ViewGroup mAfterEvaluateLayout;
    private final AIHelpButton mTvAdvice;
    private final TextView mTvShowThanks;

    private OnAIHelpEvaluateViewCallback listener;
    private AlertDialog feedbackDialog;

    private String title;
    private String mainId;
    private String contentId;

    public void setFaqData(String title, String mainId, String contentId) {
        this.title = title;
        this.mainId = mainId;
        this.contentId = contentId;
    }

    public void setEvaluateState(int evaluateState) {
        setVisibility(VISIBLE);
        switch (evaluateState) {
            case STATE_NORMAL:
                break;
            case STATE_INVISIBLE:
                setVisibility(GONE);
                break;
            case STATE_PLAIN_TEXT:
                mEvaluateFaqLayout.setVisibility(View.GONE);
                mAfterEvaluateLayout.setVisibility(View.VISIBLE);
                mTvAdvice.setVisibility(View.GONE);
                Styles.reRenderTextView(mTvShowThanks, CustomConfig.InitSetting.commonPositiveFeedbackHint, 0.5f);
                break;
            case STATE_REQUESTING_FEEDBACK:
                mEvaluateFaqLayout.setVisibility(View.GONE);
                mAfterEvaluateLayout.setVisibility(View.VISIBLE);
                Styles.reRenderTextView(mTvShowThanks, CustomConfig.InitSetting.commonNegativeFeedbackHint, 0.5f);
                mTvAdvice.setVisibility(CustomConfig.InitSetting.isFaqUnhelpfulFeedbackEnable ? VISIBLE : GONE);
                break;
        }
    }

    public void setOnAIHelpEvaluateViewCallback(OnAIHelpEvaluateViewCallback listener) {
        this.listener = listener;
    }

    public AIHelpEvaluateView(@NonNull Context context) {
        this(context, null);
    }

    public AIHelpEvaluateView(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public AIHelpEvaluateView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);

        int[] evaluateViewStyleable = ResResolver.getStyleable("aihelp_evaluate_view");
        if (evaluateViewStyleable != null) {
            // 获取自定义属性
            TypedArray typedArray = context.obtainStyledAttributes(attrs, evaluateViewStyleable);
            evaluateTarget = typedArray.getInt(ResResolver.getStyleableFieldIndex(
                    "aihelp_evaluate_view", "aihelp_evaluate_target"), -1);
            typedArray.recycle();
        }

        // 加载 xml 布局
        View rootView = View.inflate(context, ResResolver.getLayoutId("aihelp_evaluate_view"), this);

        // 点踩视图
        mEvaluateFaqLayout = rootView.findViewById(ResResolver.getViewId("aihelp_ll_evaluate_faq"));
        AppCompatImageView unhelpfulView = rootView.findViewById(ResResolver.getViewId("aihelp_iv_un_helpful"));
        Styles.reRenderImageView(unhelpfulView, "aihelp_svg_ic_un_helpful", true);

        // 点赞视图
        AppCompatImageView helpfulView = rootView.findViewById(ResResolver.getViewId("aihelp_iv_helpful"));
        Styles.reRenderImageView(helpfulView, "aihelp_svg_ic_helpful", true);

        // 评价后视图
        mAfterEvaluateLayout = rootView.findViewById(ResResolver.getViewId("aihelp_ll_feedback"));
        mTvShowThanks = rootView.findViewById(ResResolver.getViewId("aihelp_tv_thanks"));
        mTvAdvice = rootView.findViewById(ResResolver.getViewId("aihelp_tv_advice"));
        mTvAdvice.setText(ResResolver.getString("aihelp_faq_feedback_suggest"));

        // 给点赞、点踩和提建议设置点击事件
        unhelpfulView.setOnClickListener(this);
        helpfulView.setOnClickListener(this);
        mTvAdvice.setOnClickListener(this);

    }

    @Override
    public void onClick(View v) {
        if (v.getId() == ResResolver.getViewId("aihelp_iv_un_helpful")) {
            setEvaluateState(STATE_REQUESTING_FEEDBACK);
            if (listener != null) {
                listener.onEvaluated(false);
            }
            if (evaluateTarget != FAQ_TYPE_BOT_FAQ) {
                StatisticTracker.getInstance().markedUnhelpful(mainId, contentId, title);
            }
            EventTracker.INSTANCE.log(EventType.HC_ACTION_MARK_FAQ_HELPFUL, false);
        }

        if (v.getId() == ResResolver.getViewId("aihelp_iv_helpful")) {
            setEvaluateState(STATE_PLAIN_TEXT);
            if (listener != null) {
                listener.onEvaluated(true);
            }
            if (evaluateTarget != FAQ_TYPE_BOT_FAQ) {
                StatisticTracker.getInstance().markedHelpful(mainId, contentId, title);
            }
            EventTracker.INSTANCE.log(EventType.HC_ACTION_MARK_FAQ_HELPFUL, true);
        }

        if (v.getId() == ResResolver.getViewId("aihelp_tv_advice")) {
            showAdviceAlert(v.getContext(), evaluateTarget, mainId);
        }
    }

    public void showAdviceAlert(final Context context, final int faqType, final String mainId) {
        feedbackDialog = new AlertDialog.Builder(context)
                .setContentView(ResResolver.getLayoutId("aihelp_dia_advice"))
                .setWidthByDevice()
                .create();

        final EditText etFeedback = feedbackDialog.getView(ResResolver.getViewId("aihelp_et_feedback"));
        final TextView tvTitle = feedbackDialog.getView(ResResolver.getViewId("aihelp_tv_title"));
        final TextView tvCancel = feedbackDialog.getView(ResResolver.getViewId("aihelp_tv_cancel"));
        final TextView tvConfirm = feedbackDialog.getView(ResResolver.getViewId("aihelp_tv_confirm"));

        Styles.reRenderTextView(
                etFeedback,
                ResResolver.getString("aihelp_chat_hint"),
                Styles.getColor(context, ResResolver.getColorId("aihelp_color_text_title")),
                true,
                16
        );
        Styles.reRenderTextView(
                tvTitle,
                ResResolver.getString("aihelp_faq_feedback"),
                Styles.getColor(context, ResResolver.getColorId("aihelp_color_text_title")),
                true,
                15
        );
        Styles.reRenderTextView(
                tvCancel,
                ResResolver.getString("aihelp_no"),
                Styles.getColor(context, ResResolver.getColorId("aihelp_color_text_title")),
                true,
                16
        );
        Styles.reRenderTextView(
                tvConfirm,
                ResResolver.getString("aihelp_yes"),
                Styles.getColor(context, ResResolver.getColorId("aihelp_color_main")),
                true,
                16
        );

        feedbackDialog.setOnClickListener(ResResolver.getViewId("aihelp_tv_cancel"), new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                feedbackDialog.dismiss();
            }
        });
        etFeedback.addTextChangedListener(new TextWatcherWrapper() {
            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                tvConfirm.setEnabled(!TextUtils.isEmpty(s.toString().trim()));
                tvConfirm.setAlpha(TextUtils.isEmpty(s) ? 0.5f : 1);
            }
        });

        tvConfirm.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (TextUtils.isEmpty(etFeedback.getText().toString().trim())) {
                    ToastUtil.INSTANCE.makeRawToast(context, ResResolver.getString("aihelp_faq_feedback"));
                    return;
                }
                feedbackDialog.dismiss();
                postFeedbackOnFaq(faqType, mainId, contentId, etFeedback.getText().toString().trim(), String.valueOf(System.currentTimeMillis()));
                mTvAdvice.setVisibility(View.GONE);
                if (listener != null) {
                    listener.onFeedbackConfirmed();
                }
                EventTracker.INSTANCE.log(EventType.HC_ACTION_SUBMIT_FAQ_FEEDBACK);
            }
        });
        feedbackDialog.show();
    }

    private void postFeedbackOnFaq(int faqType, String mainId, String contentId, String feedbackMsg, String createTime) {
        JSONObject params = new JSONObject();
        try {
            params.put("Language", Const.CORRECT_LANGUAGE);
            params.put("PlayerId", String.format("%s|%s", Const.APP_ID, UserProfile.USER_ID));
            params.put("PlayerName", UserProfile.USER_NAME);
            params.put("FaqId", mainId);
            params.put("contentId", contentId);
            params.put("Message", feedbackMsg);
            params.put("Type", faqType);
            params.put("CreateTime", createTime);

            if (listener != null && listener.requestDataForFeedback() != null) {
                JSONObject dataForFeedback = listener.requestDataForFeedback();
                Iterator<String> keys = dataForFeedback.keys();
                while (keys.hasNext()) {
                    String key = keys.next();
                    params.put(key, dataForFeedback.optString(key));
                }
            }

        } catch (Exception e) {
            e.printStackTrace();
        }
        AIHelpRequest.getInstance().requestPostByJson(API.FAQ_FEEDBACK_URL, params, new ReqCallback<String>() {
            @Override
            public void onReqSuccess(String result) {
                StatisticTracker.getInstance().submitSuggestion(mainId, contentId, title, feedbackMsg);
            }
        });
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        EventBus.getDefault().register(this);
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        EventBus.getDefault().unregister(this);
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEventComing(EventCenter<?> event) {
        if (event instanceof TemplateReadyEvent) {
            AppCompatImageView unhelpfulView = findViewById(ResResolver.getViewId("aihelp_iv_un_helpful"));
            Styles.reRenderImageView(unhelpfulView, "aihelp_svg_ic_un_helpful", true);

            AppCompatImageView helpfulView = findViewById(ResResolver.getViewId("aihelp_iv_helpful"));
            Styles.reRenderImageView(helpfulView, "aihelp_svg_ic_helpful", true);
        }
        if (event instanceof OrientationChangeEvent) {
            if (feedbackDialog != null && feedbackDialog.isShowing()) {
                WindowManager.LayoutParams attributes = feedbackDialog.getWindow().getAttributes();
                attributes.width = (int) (Styles.getScreenWidth(getContext()) * 0.725);
                feedbackDialog.getWindow().setAttributes(attributes);
            }
        }
    }

    public static abstract class OnAIHelpEvaluateViewCallback {
        public void onEvaluated(boolean isHelpful) {
        }

        public JSONObject requestDataForFeedback() {
            return null;
        }

        public void onFeedbackConfirmed() {
        }
    }

}
