package net.aihelp.utils;

import android.content.Context;
import android.net.Uri;
import android.os.ParcelFileDescriptor;
import android.text.TextUtils;

import net.aihelp.config.AIHelpContext;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import androidx.annotation.RestrictTo;

@RestrictTo(RestrictTo.Scope.LIBRARY)
public class FileUtil {

    private static final HashSet<String> imageMimeTypes =
            new HashSet<>(Arrays.asList("image/jpeg", "image/png", "image/gif",
                    "image/x-png", "image/x-citrix-pjpeg", "image/x-citrix-gif", "image/pjpeg"));

    private static final Set<String> resizableImageMimeTypes =
            new HashSet<>(Arrays.asList("image/jpeg", "image/png", "image/x-png", "image/x-citrix-pjpeg", "image/pjpeg"));

    public static boolean isSupportedMimeType(String mimeType) {
        return imageMimeTypes.contains(mimeType);
    }

    public static void saveFile(URL url, File saveFile) {
        InputStream input = null;
        FileOutputStream output = null;

        try {
            input = url.openStream();
            output = new FileOutputStream(saveFile);
            byte[] buffer = new byte[500];

            int bytesRead;
            while ((bytesRead = input.read(buffer, 0, buffer.length)) >= 0) {
                output.write(buffer, 0, bytesRead);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            closeQuietly(input);
            closeQuietly(output);
        }

    }

    private static void closeQuietly(Closeable stream) {
        try {
            if (stream != null) {
                stream.close();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    public static String getMimeType(URL url) {
        try {
            return url.openConnection().getContentType();
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    public static String getMimeType(String fileLocation) {
        try {
            return getMimeType(new URL("file://" + fileLocation));
        } catch (MalformedURLException e) {
            e.printStackTrace();
            return null;
        }
    }

    public static boolean doesFileFromUriExistAndCanRead(Uri uri, Context context) {
        ParcelFileDescriptor parcelFileDescriptor = null;

        try {
            parcelFileDescriptor = context.getContentResolver().openFileDescriptor(uri, "r");
            return parcelFileDescriptor != null;
        } catch (Exception e) {
            e.printStackTrace();
            TLog.d("doesFileFromUriExistAndCanRead exception " + e.toString());
            return false;
        } finally {
            try {
                if (parcelFileDescriptor != null) {
                    parcelFileDescriptor.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }

        }
    }

    public static synchronized String getContentFromFile(String filePath) {
        try {
            File file = new File(filePath);
            if (file.exists()) {
                RandomAccessFile f = new RandomAccessFile(file, "r");
                byte[] bytes = new byte[(int) f.length()];
                f.readFully(bytes);
                f.close();
                return new String(bytes);
            }
            return "";
        } catch (Exception e) {
            return "";
        }
    }

    public static synchronized boolean writeFileToDisk(InputStream ins, String filePath) {
        Context context = AIHelpContext.getInstance().getContext();
        try {
            if (context != null) {
                File targetFile = new File(filePath);

                if (targetFile.exists()) targetFile.delete();

                InputStream inputStream = null;
                OutputStream outputStream = null;

                try {
                    byte[] fileReader = new byte[8192];
                    inputStream = ins;
                    outputStream = new FileOutputStream(targetFile);

                    while (true) {
                        int read = inputStream.read(fileReader);
                        if (read == -1) break;
                        outputStream.write(fileReader, 0, read);
                    }
                    outputStream.flush();
                    return true;
                } catch (Exception e) {
                    // ignored
                } finally {
                    if (inputStream != null) inputStream.close();
                    if (outputStream != null) outputStream.close();
                }
            }
        } catch (Exception e) {
            // ignored
        }
        return false;
    }

    private static boolean createOrExistsDir(final File file) {
        return file != null && (file.exists() ? file.isDirectory() : file.mkdirs());
    }

    public static File copyAndRenameFile(Context context, String sourcePath, String newFileName) {
        if (!TextUtils.isEmpty(sourcePath)) {
            File sourceFile = new File(sourcePath);
            if (sourceFile.exists() && sourceFile.length() > 0) {
                String destinationFolder = context.getFilesDir().getAbsolutePath() + "/AIHelp/event";
                File destinationFile = new File(destinationFolder, newFileName);
                if (destinationFile.exists()) {
                    destinationFile.delete();
                }
                try (FileInputStream fis = new FileInputStream(sourceFile);
                     FileOutputStream fos = new FileOutputStream(destinationFile)) {
                    byte[] buffer = new byte[1024];
                    int bytesRead;
                    while ((bytesRead = fis.read(buffer)) > 0) {
                        fos.write(buffer, 0, bytesRead);
                    }
                    return destinationFile;
                } catch (Exception e) {
                    // ignored
                }
            }
        }
        return null;
    }

}
