package net.aihelp.utils;

import android.app.Activity;
import android.graphics.Rect;
import android.os.Build;
import android.util.DisplayMetrics;
import android.view.View;
import android.view.ViewTreeObserver;
import android.widget.EditText;

import androidx.annotation.RestrictTo;

@RestrictTo(RestrictTo.Scope.LIBRARY)
public class KeyboardChangeHelper implements ViewTreeObserver.OnGlobalLayoutListener {

    private View contentView;
    private EditText targetView;
    private int rootViewVisibleHeight = 0;
    private int offset;
    private OnKeyboardShowListener listener;

    public KeyboardChangeHelper(View contentView) {
        this.contentView = contentView;
        setTranslationOffset(0);
    }

    public static boolean isKeyboardShown(View view) {
        View rootView = view.getRootView();
        final int softKeyboardHeight = 100;
        Rect r = new Rect();
        rootView.getWindowVisibleDisplayFrame(r);
        DisplayMetrics dm = rootView.getResources().getDisplayMetrics();
        int heightDiff = rootView.getBottom() - r.bottom;
        return heightDiff > softKeyboardHeight * dm.density;
    }

    @Override
    public void onGlobalLayout() {
        if (!initTargetView()) {
            return;
        }
        Rect r = new Rect();
        contentView.getWindowVisibleDisplayFrame(r);
        int visibleHeight = r.height();
        if (rootViewVisibleHeight == 0) {
            rootViewVisibleHeight = visibleHeight;
            return;
        }
        //根视图显示高度没有变化，可以看作软键盘显示／隐藏状态没有改变
        if (rootViewVisibleHeight == visibleHeight) {
            return;
        }
        //根视图显示高度变小超过200，可以看作软键盘显示了
        if (rootViewVisibleHeight - visibleHeight > 200) {
            rootViewVisibleHeight = visibleHeight;
            if (null != listener) {
                listener.onKeyboardShow();
            }
            layoutResize(true, r.bottom);
            return;
        }
        //根视图显示高度变大超过200，可以看作软键盘隐藏了
        if (visibleHeight - rootViewVisibleHeight > 200) {
            rootViewVisibleHeight = visibleHeight;
            if (null != listener) {
                listener.onKeyboardHide();
            }
            layoutResize(false, 0);
        }
    }

    private void layoutResize(boolean isKeyboardShow, int currentBottom) {
        int resizeHeight = 0;
        Rect rect = new Rect();
        if (null != targetView) {
            targetView.getGlobalVisibleRect(rect);
            // rect.bottom 获取到的时EditText的底部，加上offset是再往下offset距离的底部
            resizeHeight = rect.bottom + offset - currentBottom;
        }
        if (resizeHeight < 0) {
            return;
        }
        if (isKeyboardShow) {
            contentView.setTranslationY(-resizeHeight);
        } else {
            contentView.setTranslationY(0);
        }
    }

    private boolean initTargetView() {
        View focusedView;
        Activity activity = (Activity) contentView.getContext();
        if (null != activity) {
            focusedView = activity.getCurrentFocus();
        } else {
            return false;
        }
        if (focusedView instanceof EditText) {
            targetView = (EditText)focusedView;
        }
        return true;
    }

    public void addListener() {
        contentView.getViewTreeObserver().addOnGlobalLayoutListener(this);
    }

    public void removeListener() {
        // remember to remove if used
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            contentView.getViewTreeObserver().removeOnGlobalLayoutListener(this);
        }
    }

    public void setTranslationOffset(int offset) {
        this.offset = offset;
    }

    public void addOnKeyboardShowListener(OnKeyboardShowListener listener) {
        this.listener = listener;
    }

    public interface OnKeyboardShowListener {
        void onKeyboardShow();
        void onKeyboardHide();
    }

}
