package net.aihelp.utils;

import android.content.Context;
import android.graphics.Bitmap;
import android.media.MediaMetadataRetriever;
import android.os.AsyncTask;
import android.text.TextUtils;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;

import net.aihelp.core.ui.glide.Glide;
import net.aihelp.core.ui.glide.request.animation.GlideAnimation;
import net.aihelp.core.ui.glide.request.target.SimpleTarget;

import net.aihelp.config.AIHelpContext;
import net.aihelp.ui.helper.BitmapHelper;

import java.io.File;
import java.io.FileOutputStream;
import java.util.HashMap;

import androidx.annotation.RestrictTo;

@RestrictTo(RestrictTo.Scope.LIBRARY)
public class MediaUtils {

    public static final int MEDIA_TYPE_IMAGE = 1;
    public static final int MEDIA_TYPE_VIDEO = 2;

    public static void scaleImageView(String source, ImageView targetView, View paramsView, OnImageScaledListener listener) {
        if (targetView == null || paramsView == null || TextUtils.isEmpty(source)) return;
        String adjustedSource = DomainSupportHelper.getAdjustedUrl(source);
        Context context = targetView.getContext();
        Glide.with(context).load(adjustedSource).asBitmap().into(new SimpleTarget<Bitmap>() {
            @Override
            public void onResourceReady(Bitmap resource, GlideAnimation<? super Bitmap> glideAnimation) {

                Glide.with(context).load(adjustedSource).into(targetView);

                ViewGroup.LayoutParams params = paramsView.getLayoutParams();

                int[] size = BitmapHelper.computeSize(resource.getWidth(), resource.getHeight());
                params.width = Math.max(size[0], Styles.dpToPx(context, 50));
                params.height = Math.max(size[1], Styles.dpToPx(context, 50));
                paramsView.setLayoutParams(params);

                if (listener != null) {
                    listener.onImageScaled();
                }
            }
        });
    }

    public static File getOutputMediaFile(String urlPath) {
        Context context = AIHelpContext.getInstance().getContext();
        if (context != null) {
            File mediaStorageDir = new File(context.getExternalCacheDir() + "/aihelp/image");
            if (!mediaStorageDir.exists()) {
                if (!mediaStorageDir.mkdirs()) {
                    return null;
                }
            }
            String fileName = "IMG_" + System.currentTimeMillis() + ".jpg";
            if (!TextUtils.isEmpty(urlPath)) {
                int lastSeparator = urlPath.lastIndexOf("/");
                int extensionIndex = urlPath.lastIndexOf(".");
                if (lastSeparator != -1 && extensionIndex != -1 && lastSeparator < extensionIndex) {
                    fileName = urlPath.substring(lastSeparator + 1, extensionIndex) + ".jpg";
                }
            }
            return new File(mediaStorageDir.getPath() + File.separator + fileName);
        }
        return null;
    }

    /**
     * 获取视频的第一帧图片
     */
    public static void getImageForVideo(String videoPath, OnLoadVideoImageListener listener) {
        LoadVideoImageTask task = new LoadVideoImageTask(listener);
        task.execute(videoPath);
    }

    public static String getImageForVideoSync(String videoPath) {
        File thumbnail = getVideoThumbnail(videoPath);
        return thumbnail != null ? thumbnail.getAbsolutePath() : "";
    }

    public static class LoadVideoImageTask extends AsyncTask<String, Integer, File> {
        private OnLoadVideoImageListener listener;

        public LoadVideoImageTask(OnLoadVideoImageListener listener) {
            this.listener = listener;
        }

        public void setOnLoadVideoImageListener(OnLoadVideoImageListener listener) {
            this.listener = listener;
        }

        @Override
        protected File doInBackground(String... params) {
            return getVideoThumbnail(params[0]);
        }

        @Override
        protected void onPostExecute(File file) {
            super.onPostExecute(file);
            if (listener != null) {
                listener.onLoadImage(file);
            }
        }
    }

    private static File getVideoThumbnail(String videoPath) {
        try {
            File mediaFile = getOutputMediaFile(videoPath);
            if (mediaFile != null && mediaFile.exists()) {
                return mediaFile;
            }

            MediaMetadataRetriever mmr = new MediaMetadataRetriever();
            if (RegexUtil.isLocalMediaFile(videoPath)) {
                mmr.setDataSource(videoPath);
            } else {
                mmr.setDataSource(videoPath, new HashMap<String, String>());
            }

            Bitmap bitmap = mmr.getFrameAtTime();
            if (mediaFile != null) {
                try {
                    FileOutputStream out = new FileOutputStream(mediaFile);
                    bitmap.compress(Bitmap.CompressFormat.JPEG, 90, out);
                    out.flush();
                    out.close();
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
            mmr.release();
            return mediaFile;
        } catch (Exception e) {
            return new File(videoPath);
        }
    }

    public interface OnLoadVideoImageListener {
        void onLoadImage(File file);
    }

    public interface OnImageScaledListener {
        void onImageScaled();
    }

}
