package net.andreinc.mockneat.abstraction;


import net.andreinc.mockneat.utils.ValidationUtils;

import java.util.function.Supplier;
import java.util.stream.DoubleStream;

import static java.util.stream.IntStream.range;
import static net.andreinc.mockneat.utils.ValidationUtils.isTrue;

public interface MockUnitDouble extends MockUnit<Double> {

    /**
     * <p>Transforms an existing {@code MockUnitDouble} into a {@code MockUnit<DoubleStream>}.</p>
     *
     * <p>The DoubleStream will be "infinite" and will contain values generated by the internal {@code MockUnitDouble} supplier.</p>
     *
     * @return A new {@code MockUnit<DoubleStream>}
     */
    default MockUnit<DoubleStream> doubleStream() {
        Supplier<DoubleStream> supp = () -> DoubleStream.generate(supplier()::get);
        return () -> supp;
    }

    /**
     * <p>Transforms an existing {@code MockUnitDouble} into a {@code MockUnit<double[]>}.</p>
     *
     * <p>The values in the array will be generated using the {@code MockUnitDouble} supplier.</p>
     *
     * @param size The size of the array.
     * @return A new {@code MockUnit<double[]>}
     */
    default MockUnit<double[]> arrayPrimitive(int size) {
        isTrue(size>=0, ValidationUtils.SIZE_BIGGER_THAN_ZERO);
        Supplier<double[]> supp = () -> {
            final double[] result = new double[size];
            range(0, size).forEach(i -> result[i] = val());
            return result;
        };
        return () -> supp;
    }

    /**
     * <p>Transforms an existing {@code MockUnitDouble} into a {@code MockUnit<Double[]>}</p>
     *
     * <p>The values in the array will be generated using the {@code MockUnitDouble} supplier.</p>
     *
     * @param size The size of the array
     * @return A new {@code MockUnit<Double[]>}
     */
    default MockUnit<Double[]> array(int size) {
        isTrue(size>=0, ValidationUtils.SIZE_BIGGER_THAN_ZERO);
        Supplier<Double[]> supp = () -> {
            final Double[] result = new Double[size];
            range(0, size).forEach(i -> result[i] = val());
            return result;
        };
        return () -> supp;
    }
}
