/*
 * Copyright (c) 2018, apexes.net. All rights reserved.
 *
 *         http://www.apexes.net
 *
 */
package net.apexes.commons.json.jackson;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 *
 * @author <a href=mailto:hedyn@foxmail.com>HeDYn</a>
 */
public class AdaptiveDateJsonDeserializer extends JsonDeserializer<Date> {

    public static final String DEFAULT_DATE_FORMAT = "yyyy-MM-dd";
    public static final String DEFAULT_DATETIME_FORMAT = "yyyy-MM-dd HH:mm:ss";
    public static final String ISO8601_DATE_WITHOUT_ZONE_FORMAT = "yyyy-MM-dd'T'HH:mm:ss";

    @Override
    public Date deserialize(JsonParser jp, DeserializationContext ctxt) throws IOException {
        String dateStr = jp.getText();
        String temp = dateStr; // 备份原始数据

        if (dateStr == null || dateStr.isEmpty()) {
            return null;
        }

        String fm = null;
        if (dateStr.contains(":")) {
            if (dateStr.contains("T")) {
                fm = ISO8601_DATE_WITHOUT_ZONE_FORMAT;
            } else if (dateStr.contains(":")) {
                fm = DEFAULT_DATETIME_FORMAT;
            }
        } else if (dateStr.indexOf('-') > 1) {
            fm = DEFAULT_DATE_FORMAT;
        }
        try {
            if (fm != null) {
                return new SimpleDateFormat(fm).parse(dateStr);
            } else {
                return new Date(Long.parseLong(dateStr));
            }
        } catch (ParseException e) {
            throw new JsonDateParseException(temp, e);
        }
    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     *
     */
    public static class JsonDateParseException extends JsonProcessingException {

        private static final long serialVersionUID = 1L;

        protected JsonDateParseException(String msg, Throwable rootCause) {
            super(msg, rootCause);
        }

    }
}
