/*
 * Copyright (c) 2018, apexes.net. All rights reserved.
 *
 *         http://www.apexes.net
 *
 */
package net.apexes.commons.json.jackson;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.CollectionLikeType;
import com.fasterxml.jackson.databind.type.TypeFactory;
import net.apexes.commons.lang.Checks;
import net.apexes.commons.net.JsonHttpCaller;

import java.util.List;

/**
 * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
 */
public final class JacksonHttps {

    private JacksonHttps() {}

    public static JsonHttpCaller.JsonEncoder createEncoder() {
        return createEncoder(null);
    }

    public static JsonHttpCaller.JsonEncoder createEncoder(ObjectMapper objectMapper) {
        return new JacksonEncoder(objectMapper);
    }

    public static <R> JsonHttpCaller.JsonDecoder<R> createDecoder(Class<R> returnClass) {
        return createDecoder(null, returnClass);
    }

    public static <R> JsonHttpCaller.JsonDecoder<R> createDecoder(ObjectMapper objectMapper, Class<R> returnClass) {
        return new JacksonDecoder<>(objectMapper, returnClass);
    }

    public static <R> JsonHttpCaller.JsonDecoder<R> createDecoder(JavaType javaType) {
        return createDecoder(null, javaType);
    }

    public static <R> JsonHttpCaller.JsonDecoder<R> createDecoder(ObjectMapper objectMapper, JavaType javaType) {
        return new JacksonDecoder<>(objectMapper, javaType);
    }

    public static <R> JsonHttpCaller.JsonDecoder<R> createDecoder(ObjectMapper objectMapper, TypeReference<R> typeRef) {
        return new JacksonTypeReferenceDecoder<>(objectMapper, typeRef);
    }

    public static JsonHttpCaller.JsonHttpNoticer forNotice(String url) {
        return forNotice(url, createEncoder());
    }

    public static JsonHttpCaller.JsonHttpNoticer forNotice(String url, JsonHttpCaller.JsonEncoder encoder) {
        return JsonHttpCaller.forNotice(url, encoder);
    }

    public static <R> JsonHttpCaller<R> forRequest(String url, Class<R> returnClass) {
        return forRequest(url, null, returnClass);
    }

    public static <R> JsonHttpCaller<R> forRequest(String url, ObjectMapper objectMapper, Class<R> returnClass) {
        JsonHttpCaller.JsonEncoder encoder = createEncoder(objectMapper);
        JsonHttpCaller.JsonDecoder<R> decoder = createDecoder(objectMapper, returnClass);
        return JsonHttpCaller.forRequest(url, encoder, decoder);
    }

    public static <R> JsonHttpCaller<R> forRequest(String url, JavaType javaType) {
        return forRequest(url, null, javaType);
    }

    public static <R> JsonHttpCaller<R> forRequest(String url, ObjectMapper objectMapper, JavaType javaType) {
        JsonHttpCaller.JsonEncoder encoder = createEncoder(objectMapper);
        JsonHttpCaller.JsonDecoder<R> decoder = createDecoder(objectMapper, javaType);
        return JsonHttpCaller.forRequest(url, encoder, decoder);
    }

    public static <R> JsonHttpCaller<R> forRequest(String url, TypeReference<R> typeRef) {
        return forRequest(url, null, typeRef);
    }

    public static <R> JsonHttpCaller<R> forRequest(String url, ObjectMapper objectMapper, TypeReference<R> typeRef) {
        JsonHttpCaller.JsonEncoder encoder = createEncoder(objectMapper);
        JsonHttpCaller.JsonDecoder<R> decoder = createDecoder(objectMapper, typeRef);
        return JsonHttpCaller.forRequest(url, encoder, decoder);
    }

    public static <R> JsonHttpCaller<R> forRequest(String url, JsonHttpCaller.JsonDecoder<R> decoder) {
        JsonHttpCaller.JsonEncoder encoder = createEncoder();
        return JsonHttpCaller.forRequest(url, encoder, decoder);
    }

    public static <R> JsonHttpCaller<R> forRequest(String url, JsonHttpCaller.JsonEncoder encoder, JsonHttpCaller.JsonDecoder<R> decoder) {
        return JsonHttpCaller.forRequest(url, encoder, decoder);
    }

    public static <D> JavaType constructParametricType(ObjectMapper objectMapper, Class<?> parametrized, Class<D> dataClass) {
        return objectMapper.getTypeFactory().constructParametricType(parametrized, dataClass);
    }

    public static <DE> JavaType constructListParametricType(ObjectMapper objectMapper, Class<?> parametrized, Class<DE> elementClass) {
        TypeFactory factory = objectMapper.getTypeFactory();
        CollectionLikeType likeType = factory.constructCollectionLikeType(List.class, elementClass);
        return factory.constructParametricType(parametrized, likeType);
    }

    public static final ObjectMapper DEFAULT_OBJECT_MAPPER = createObjectMapper();

    private static ObjectMapper createObjectMapper() {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        return objectMapper;
    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     */
    private static class JacksonEncoder implements JsonHttpCaller.JsonEncoder {

        private final ObjectMapper objectMapper;

        JacksonEncoder(ObjectMapper objectMapper) {
            this.objectMapper = objectMapper != null ? objectMapper : DEFAULT_OBJECT_MAPPER;
        }

        @Override
        public String toJson(Object object) throws Exception {
            return objectMapper.writeValueAsString(object);
        }
    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     *
     * @param <R>
     */
    private static class JacksonDecoder<R> implements JsonHttpCaller.JsonDecoder<R> {

        private final ObjectMapper objectMapper;
        private final JavaType javaType;

        JacksonDecoder(ObjectMapper objectMapper, Class<R> returnClass) {
            this(objectMapper, javaType(objectMapper, returnClass));
        }

        JacksonDecoder(ObjectMapper objectMapper, JavaType javaType) {
            Checks.verifyNotNull(javaType, "javaType");
            this.objectMapper = defaultIfNull(objectMapper);
            this.javaType = javaType;
        }

        @Override
        public R fromJson(String json) throws Exception {
            return objectMapper.readValue(json, javaType);
        }

        private static ObjectMapper defaultIfNull(ObjectMapper objectMapper) {
            return objectMapper != null ? objectMapper : DEFAULT_OBJECT_MAPPER;
        }

        private static JavaType javaType(ObjectMapper objectMapper, Class<?> returnClass) {
            objectMapper = defaultIfNull(objectMapper);
            return objectMapper.getTypeFactory().constructType(returnClass);
        }

    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     *
     * @param <R>
     */
    private static class JacksonTypeReferenceDecoder<R> implements JsonHttpCaller.JsonDecoder<R> {

        private final ObjectMapper objectMapper;
        private final TypeReference<R> typeRef;

        JacksonTypeReferenceDecoder(ObjectMapper objectMapper, TypeReference<R> typeRef) {
            Checks.verifyNotNull(typeRef, "typeRef");
            this.objectMapper = objectMapper != null ? objectMapper : DEFAULT_OBJECT_MAPPER;
            this.typeRef = typeRef;
        }

        @Override
        public R fromJson(String json) throws Exception {
            return objectMapper.readValue(json, typeRef);
        }

    }

}
