/*
 * Copyright (c) 2018, apexes.net. All rights reserved.
 *
 *         http://www.apexes.net
 *
 */
package net.apexes.commons.lang;

import java.text.MessageFormat;
import java.util.Random;

/**
 * 
 * @author <a href=mailto:hedyn@foxmail.com>HeDYn</a>
 *
 */
public final class Strings {
    
    private Strings() {}
    
    /**
     * 返回一个定长的随机字符串(只包含大小写字母、数字)
     * @param length 随机字符串长度
     * @return 随机字符串
     */
    public static String randomString(int length) {
        StringBuilder sb = new StringBuilder();
        Random random = new Random();
        for (int i = 0; i < length; i++) {
            sb.append(Radix62.ALPHABET[random.nextInt(Radix62.ALPHABET.length)]);
        }
        return sb.toString();
    }
    
    /**
     * 返回一个定长的随机纯字母字符串(只包含大小写字母)
     * @param length 随机字符串长度
     * @return 随机字符串
     */
    public static String randomLetter(int length) {
        StringBuilder sb = new StringBuilder();
        Random random = new Random();
        for (int i = 0; i < length; i++) {
            sb.append(Radix62.ALPHABET[10 + random.nextInt(Radix62.ALPHABET.length - 10)]);
        }
        return sb.toString();
    }

    /**
     * 生成一个定长的随机纯数字字符串
     * @param length 随机字符串长度
     * @return 随机字符串
     */
    public static String randomNumber(int length) {
        StringBuilder sb = new StringBuilder();
        Random random = new Random();
        for (int i = 0; i < length; i++) {
            sb.append(random.nextInt(10));
        }
        return sb.toString();
    }
    
    /**
     * 去除字符串前面空白
     * @param str 字符串
     * @return 返回去除字符串前面空白后的字符串
     */
    public static String ltrim(String str) {
        int len = str.length();
        int st = 0;
        char[] val = str.toCharArray();
        
        while ((st < len) && (val[st] <= ' ')) {
            st++;
        }
        return (st > 0) ? str.substring(st, len) : str;
    }
    
    /**
     * 去除字符串后面空白
     * @param str 字符串
     * @return 返回去除字符串后面空白后的字符串
     */
    public static String rtrim(String str) {
        int len = str.length();
        char[] val = str.toCharArray();
        
        while ((0 < len) && (val[len - 1] <= ' ')) {
            len--;
        }
        return (len < str.length()) ? str.substring(0, len) : str;
    }
    
    /**
     * 将首字节大写。注意：此方法与JavaBean规范中定义的属性的getter、setter方法名称不同。
     * @param str 字符串
     * @return 返回首字节大写后的字符串
     */
    public static String capitalize(String str) {
        int strLen;
        if (str == null || (strLen = str.length()) == 0) {
            return str;
        }
        return new StringBuilder(strLen)
                .append(Character.toTitleCase(str.charAt(0)))
                .append(str.substring(1))
                .toString();
    }
    
    /**
     * 将首字节小写。注意：此方法与JavaBean规范中定义的属性的getter、setter方法名称不同。
     * @param str 字符串
     * @return 返回首字节小写后的字符串
     */
    public static String uncapitalize(String str) {
        int strLen;
        if ((str == null) || ((strLen = str.length()) == 0)) {
            return str;
        }
        return new StringBuilder(strLen)
                .append(Character.toLowerCase(str.charAt(0)))
                .append(str.substring(1))
                .toString();
    }
    
    /**
     * 将驼峰式字符串转换为下划线小写方式。如果转换前的字符串为空，则返回空字符串。</br> 
     * 转换规格是：在大写字符前增加“_”并转为小写
     * 例如：helloWorld->hello_world
     * @param str 转换前的驼峰式字符串
     * @return 转换后下划线方式字符串
     */
    public static String underscore(String str) {
        if (str == null || str.trim().isEmpty()) {
            return "";
        }
        int len = str.length();
        StringBuilder sb = new StringBuilder(len);
        for (int i = 0; i < len; i++) {
            char ch = str.charAt(i);
            if (Character.isUpperCase(ch)) {
                sb.append("_");
                sb.append(Character.toLowerCase(ch));
            } else {
                sb.append(ch);
            }
        }
        return sb.toString();
    }
    
    /**
     * 将下划线方式字符串转为小驼峰式。如果转换前的字符串为空，则返回空字符串。</br>
     * 转换规格是：去掉“_”并大写其后的一个字符 
     * 例如：hello_world->helloWorld
     * @param str 转换前的下划线方式字符串
     * @return 转换后小驼峰式命名字符串
     */
    public static String camelCase(String str) {
        if (str == null || str.trim().isEmpty()) {
            return "";
        }
        int len = str.length();
        StringBuilder sb = new StringBuilder(len);
        for (int i = 0; i < len; i++) {
            char ch = str.charAt(i);
            if (ch == '_') {
                i++;
                if (i >= len) {
                    sb.append(ch);
                    break;
                }
                sb.append(Character.toUpperCase(str.charAt(i)));
            } else {
                sb.append(ch);
            }
        }
        return sb.toString();
    }
    
    /**
     * 
     * @param format
     * @param params
     * @return
     */
    public static String format(String format, final Object... params) {
        if (params != null) {
            // MessageFormat 要求 It's 中的  ' 是 ''
            return MessageFormat.format(format.replaceAll("'", "''"), params);
        }
        return format;
    }

    /**
     * 按指定长度，省略字符串部分字符
     *
     * @param str 字符串
     * @param length 保留字符串字数
     *
     * @return 省略后的字符串
     */
    public static String omitString(String str, int length) {
        if (str == null) {
            return null;
        }
        if (length <= 0) {
            return "";
        }
        if (str.length() <= length) {
            return str;
        }
        StringBuilder buf1 = new StringBuilder();
        StringBuilder buf2 = new StringBuilder();
        int i1 = 0;
        int i2 = str.length() - 1;
        char c;
        length -= 3;
        while (length > 0) {
            c = str.charAt(i1);
            length--;
            buf1.append(c);
            i1++;

            if (length > 0 ) {
                c = str.charAt(i2);
                length--;
                buf2.insert(0, c);
                i2--;
            }
        }
        buf1.append("...");
        buf1.append(buf2.toString());
        return buf1.toString();
    }

    /**
     * 将指定的字符串转换成适合HTML格式的字符串。
     *
     * @param text
     * @return
     */
    public static String stringToHTMLString(String text) {
        StringBuilder sb = new StringBuilder(text.length());
        // true if last char was blank
        boolean lastWasBlankChar = false;
        int len = text.length();
        char c;

        for (int i = 0; i < len; i++) {
            c = text.charAt(i);
            if (c == ' ') {
                // blank gets extra work,
                // this solves the problem you get if you replace all
                // blanks with &nbsp;, if you do that you loss
                // word breaking
                if (lastWasBlankChar) {
                    lastWasBlankChar = false;
                    sb.append("&nbsp;");
                } else {
                    lastWasBlankChar = true;
                    sb.append(' ');
                }
            } else {
                lastWasBlankChar = false;
                //
                // HTML Special Chars
                if (c == '"') {
                    sb.append("&quot;");
                } else if (c == '&') {
                    sb.append("&amp;");
                } else if (c == '<') {
                    sb.append("&lt;");
                } else if (c == '>') {
                    sb.append("&gt;");
                } else if (c == '\n') {
                    // Handle Newline
                    sb.append("&lt;br/&gt;");
                } else {
                    int ci = 0xffff & c;
                    if (ci < 160) {
                        // nothing special only 7 Bit
                        sb.append(c);
                    } else {
                        // Not 7 Bit use the unicode system
                        sb.append("&#");
                        sb.append(ci);
                        sb.append(';');
                    }
                }
            }
        }
        return sb.toString();
    }

    /**
     * 获取自增型文件名。
     * 如：
     * a.txt --> a_1.txt
     * b_5.txt --> b_6.txt
     * c_2 --> c_3
     *
     * @param filename
     * @return
     */
    public static String autoincrementFilename(String filename) {
        String s1;
        String s2;
        String suffix;
        String tmp;
        int m = filename.lastIndexOf('.');
        if (m >= 0) {
            tmp = filename.substring(0, m);
            suffix = filename.substring(m);
        } else {
            tmp = filename;
            suffix = "";
        }
        int n = tmp.lastIndexOf('_');
        if (n >= 0) {
            s1 = tmp.substring(0, n);
            s2 = tmp.substring(n+1);
        } else {
            s1 = tmp;
            s2 = "";
        }
        int i = 1;
        if (s2.matches("^[1-9]\\d*$")) {
            try {
                i = Integer.parseInt(s2);
            } catch (Exception ex) {
                // ignore
            }
            i++;
            return s1 + "_" + i + suffix;
        }
        if (m != -1) {
            return tmp + "_"+ i + suffix;
        }
        return filename + "_"+ i;
    }
}
