/*
 * Copyright (c) 2018, apexes.net. All rights reserved.
 *
 *         http://www.apexes.net
 *
 */
package net.apexes.commons.lang;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

/**
 *
 * @author <a href=mailto:hedyn@foxmail.com>HeDYn</a>
 */
public final class Zips {

    /**
     * 将指定的zip文件中的内容(不含zip文件名目录)解压缩到descDir目录中
     * @param zipFile
     * @param descDir
     * @throws IOException
     */
    public static void unzip(Path zipFile, String descDir) throws IOException {
        Path descDirFile = Paths.get(descDir);
        if (Files.notExists(descDirFile)) {
            Files.createDirectory(descDirFile);
        }

        try (ZipFile zip = new ZipFile(zipFile.toFile())) {
            Enumeration<? extends ZipEntry> entries = zip.entries();
            while (entries.hasMoreElements()) {
                ZipEntry entry = entries.nextElement();
                String zipEntryName = entry.getName();
                Path entryFile = Paths.get(descDir, zipEntryName);
                Path entryFileParentDir = entryFile.getParent();
                // 判断路径是否存在, 不存在则创建文件路径
                if (Files.notExists(entryFileParentDir)) {
                    Files.createDirectory(entryFileParentDir);
                }

                // 如果是目录就不需要解压
                if (Files.isDirectory(entryFile)) {
                    continue;
                }

                try (InputStream is = zip.getInputStream(entry)) {
                    try (OutputStream os = Files.newOutputStream(entryFile)) {
                        Streams.transfer(is, os);
                    }
                }
            }
        }
    }

    public static void toZip(OutputStream out, File... srcs) throws Exception {
        toZip(out, false, false, srcs);
    }

    public static void toZip(OutputStream out, boolean keepStructure, File... srcs) throws Exception {
        toZip(out, keepStructure, false, srcs);
    }

    public static void toZip(OutputStream out, boolean keepStructure, boolean snapshoot, File... srcs)
            throws Exception {
        toZip(out, keepStructure, snapshoot, Arrays.asList(srcs));
    }

    public static void toZip(OutputStream out, List<File> srcs) throws Exception {
        toZip(out, false, false, srcs);
    }

    public static void toZip(OutputStream out, boolean keepStructure, List<File> srcs) throws Exception {
        toZip(out, keepStructure, false, srcs);
    }

    public static void toZip(OutputStream out, boolean keepStructure, boolean snapshoot, List<File> srcs)
            throws Exception {
        try (ZipOutputStream zos = new ZipOutputStream(out)) {
            for (File src : srcs) {
                compress(src, zos, src.getName(), keepStructure, snapshoot);
            }
            zos.finish();
        }
    }

    private static void compress(File sourceFile,
                                 ZipOutputStream zos,
                                 String name,
                                 boolean keepStructure,
                                 boolean snapshoot) throws Exception {
        if (sourceFile.isFile()) {
            zos.putNextEntry(new ZipEntry(name));
            Streams.transfer(sourceFile, zos, snapshoot);
            zos.closeEntry();
        } else {
            File[] listFiles = sourceFile.listFiles();
            if (listFiles == null || listFiles.length == 0) {
                // 需要保留原来文件结构时,需要对空文件夹进行处理
                if (keepStructure) {
                    zos.putNextEntry(new ZipEntry(name + "/"));
                    zos.closeEntry();
                }

            } else {
                for (File file : listFiles) {
                    if (keepStructure) {
                        // 保留原来文件结构时前面需要带上父文件夹名字
                        compress(file, zos, name + "/" + file.getName(), keepStructure, snapshoot);
                    } else {
                        compress(file, zos, file.getName(), keepStructure, snapshoot);
                    }
                }
            }
        }
    }

    public static ZipCompress keepStructure(OutputStream out) {
        return new ZipCompressImpl(out, true);
    }

    public static ZipCompress notStructure(OutputStream out) {
        return new ZipCompressImpl(out, true);
    }

    /**
     * @author <a href=mailto:hedyn@foxmail.com>HeDYn</a>
     */
    public interface ZipCompress {

        ZipCompress snapshoot();

        ZipCompress addFile(File... files);

        ZipCompress addFiles(List<File> files);

        void compress() throws Exception;

    }

    /**
     * @author <a href=mailto:hedyn@foxmail.com>HeDYn</a>
     */
    private static class ZipCompressImpl implements ZipCompress {

        private final OutputStream out;
        private final boolean keepStructure;
        private final Map<String, File> fileMap;
        private boolean snapshoot;

        private ZipCompressImpl(OutputStream out, boolean keepStructure) {
            this.out = out;
            this.keepStructure = keepStructure;
            this.fileMap = new LinkedHashMap<>();
        }

        @Override
        public ZipCompress snapshoot() {
            this.snapshoot = true;
            return this;
        }

        @Override
        public ZipCompress addFile(File... files) {
            Checks.verifyNotEmpty(files, "files");
            return addFiles(Arrays.asList(files));
        }

        @Override
        public ZipCompress addFiles(List<File> files) {
            Checks.verifyNotEmpty(files, "files");
            for (File file : files) {
                if (file.exists()) {
                    fileMap.put(file.getAbsolutePath(), file);
                }
            }
            return this;
        }

        @Override
        public void compress() throws Exception {
            toZip(out, keepStructure, snapshoot, new ArrayList<>(fileMap.values()));
        }
    }
}
