package net.avalara.avatax.rest.client;

import com.google.gson.reflect.TypeToken;
import net.avalara.avatax.rest.client.models.*;
import net.avalara.avatax.rest.client.enums.*;

import org.apache.commons.codec.binary.Base64;
import org.apache.http.impl.client.HttpClientBuilder;

import java.math.BigDecimal;
import java.util.Date;
import java.util.HashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.ArrayList;

/*
 * AvaTax Software Development Kit for Java JRE based environments
 *
 * (c) 2004-2018 Avalara, Inc.
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 *
 * @author     Dustin Welden <dustin.welden@avalara.com>
 * @copyright  2004-2018 Avalara, Inc.
 * @license    https://www.apache.org/licenses/LICENSE-2.0
 * @link       https://github.com/avadev/AvaTax-REST-V2-JRE-SDK
 */
 
public class AvaTaxClient {

    private final ExecutorService threadPool;
    private RestCallFactory restCallFactory;

    private AvaTaxClient() {
        this(null);
    }

    private AvaTaxClient(ExecutorService threadPool) {
        if (threadPool != null) {
            this.threadPool = threadPool;
        } else {
            this.threadPool = Executors.newFixedThreadPool(3);
        }
    }

    public AvaTaxClient(String appName, String appVersion, String machineName, AvaTaxEnvironment environment) {
        this(appName, appVersion, machineName, environment, null);
    }

    public AvaTaxClient(String appName, String appVersion, String machineName, String environmentUrl) {
        this();
        this.restCallFactory = new RestCallFactory(appName, appVersion, machineName, environmentUrl);
    }

    public AvaTaxClient(String appName, String appVersion, String machineName, AvaTaxEnvironment environment, String proxyHost, int proxyPort, String proxySchema) {
        this(appName, appVersion, machineName, environment, proxyHost, proxyPort, proxySchema, null);
    }

    public AvaTaxClient(String appName, String appVersion, String machineName, String environmentUrl, String proxyHost, int proxyPort, String proxySchema) {
        this();
        this.restCallFactory = new RestCallFactory(appName, appVersion, machineName, environmentUrl, proxyHost, proxyPort, proxySchema);
    }

    public AvaTaxClient(String appName, String appVersion, String machineName, AvaTaxEnvironment environment, ExecutorService threadPool) {
        this(appName, appVersion, machineName, environment == AvaTaxEnvironment.Production ? AvaTaxConstants.Production_Url : AvaTaxConstants.Sandbox_Url, threadPool);
    }

    public AvaTaxClient(String appName, String appVersion, String machineName, AvaTaxEnvironment environment, ExecutorService threadPool, HttpClientBuilder httpClientBuilder) {
        this(appName, appVersion, machineName, environment == AvaTaxEnvironment.Production ? AvaTaxConstants.Production_Url : AvaTaxConstants.Sandbox_Url, threadPool, httpClientBuilder);
    }

    public AvaTaxClient(String appName, String appVersion, String machineName, String environmentUrl, ExecutorService threadPool) {
        this(threadPool);
        this.restCallFactory = new RestCallFactory(appName, appVersion, machineName, environmentUrl);
    }

    public AvaTaxClient(String appName, String appVersion, String machineName, String environmentUrl, ExecutorService threadPool, HttpClientBuilder httpClientBuilder) {
        this(threadPool);
        this.restCallFactory = new RestCallFactory(appName, appVersion, machineName, environmentUrl, httpClientBuilder);
    }

    public AvaTaxClient(String appName, String appVersion, String machineName, AvaTaxEnvironment environment, String proxyHost, int proxyPort, String proxySchema, ExecutorService threadPool) {
        this(appName, appVersion, machineName, environment == AvaTaxEnvironment.Production ? AvaTaxConstants.Production_Url : AvaTaxConstants.Sandbox_Url, proxyHost, proxyPort, proxySchema, threadPool);
    }

    public AvaTaxClient(String appName, String appVersion, String machineName, String environmentUrl, String proxyHost, int proxyPort, String proxySchema, ExecutorService threadPool) {
        this(threadPool);
        this.restCallFactory = new RestCallFactory(appName, appVersion, machineName, environmentUrl, proxyHost, proxyPort, proxySchema);
    }

    public AvaTaxClient withSecurity(String securityHeader) {
        this.restCallFactory.addSecurityHeader(securityHeader);

        return this;
    }

    public AvaTaxClient withSecurity(String username, String password) {
        String header = null;

        try {
            header = Base64.encodeBase64String((username + ":" + password).getBytes("utf-8"));
        } catch (java.io.UnsupportedEncodingException ex) {
            System.out.println("Could not find encoding for UTF-8.");
            ex.printStackTrace();
        }

        return withSecurity(header);
    }


//region Methods
    /**
     * Reset this account's license key
     * 
     * Resets the existing license key for this account to a new key.
     *             
     * To reset your account, you must specify the ID of the account you wish to reset and confirm the action.
     *             
     * This API is only available to account administrators for the account in question, and may only be called after
     * an account has been activated by reading and accepting Avalara's terms and conditions.  To activate your account
     * please log onto the AvaTax website or call the `ActivateAccount` API.
     *             
     * Resetting a license key cannot be undone.  Any previous license keys will immediately cease to work when a new key is created.
     *             
     * When you call this API, all account administrators for this account will receive an email with the newly updated license key.
     * The email will specify which user reset the license key and it will contain the new key to use to update your connectors.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param id The ID of the account you wish to update.
     * @param model A request confirming that you wish to reset the license key of this account.
     * @return LicenseKeyModel
     */
    public LicenseKeyModel accountResetLicenseKey(Integer id, ResetLicenseKeyModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}/resetlicensekey");
        path.applyField("id", id);
        return ((RestCall<LicenseKeyModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<LicenseKeyModel>(){})).call();
    }

    /**
     * Reset this account's license key
     * 
     * Resets the existing license key for this account to a new key.
     *             
     * To reset your account, you must specify the ID of the account you wish to reset and confirm the action.
     *             
     * This API is only available to account administrators for the account in question, and may only be called after
     * an account has been activated by reading and accepting Avalara's terms and conditions.  To activate your account
     * please log onto the AvaTax website or call the `ActivateAccount` API.
     *             
     * Resetting a license key cannot be undone.  Any previous license keys will immediately cease to work when a new key is created.
     *             
     * When you call this API, all account administrators for this account will receive an email with the newly updated license key.
     * The email will specify which user reset the license key and it will contain the new key to use to update your connectors.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param id The ID of the account you wish to update.
     * @param model A request confirming that you wish to reset the license key of this account.
     * @return LicenseKeyModel
     */
    public Future<LicenseKeyModel> accountResetLicenseKeyAsync(Integer id, ResetLicenseKeyModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}/resetlicensekey");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<LicenseKeyModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<LicenseKeyModel>(){}));
    }

    /**
     * Activate an account by accepting terms and conditions
     * 
     * Activate the account specified by the unique accountId number.
     *             
     * This activation request can only be called by account administrators.  You must indicate
     * that you have read and accepted Avalara's terms and conditions to call this API.
     *             
     * Once you have activated your account, use the `AccountResetLicenseKey` API to generate
     * a license key for your account.
     *             
     * If you have not read or accepted the terms and conditions, this API call will return the
     * unchanged account model.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param id The ID of the account to activate
     * @param model The activation request
     * @return AccountModel
     */
    public AccountModel activateAccount(Integer id, ActivateAccountModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}/activate");
        path.applyField("id", id);
        return ((RestCall<AccountModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<AccountModel>(){})).call();
    }

    /**
     * Activate an account by accepting terms and conditions
     * 
     * Activate the account specified by the unique accountId number.
     *             
     * This activation request can only be called by account administrators.  You must indicate
     * that you have read and accepted Avalara's terms and conditions to call this API.
     *             
     * Once you have activated your account, use the `AccountResetLicenseKey` API to generate
     * a license key for your account.
     *             
     * If you have not read or accepted the terms and conditions, this API call will return the
     * unchanged account model.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param id The ID of the account to activate
     * @param model The activation request
     * @return AccountModel
     */
    public Future<AccountModel> activateAccountAsync(Integer id, ActivateAccountModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}/activate");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<AccountModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<AccountModel>(){}));
    }

    /**
     * Retrieve audit history for an account.
     * 
     * Retrieve audit trace history for an account.
     *             
     * Your audit trace history contains a record of all API calls made against the AvaTax REST API.  You can use this API to investigate
     * problems and see exactly what information was sent back and forth between your code and AvaTax.
     *             
     * When specifying a start and end datetime, please include a valid timezone indicator, such as the "Z" present in the examples for the start and end query parameters.
     * You can learn more about valid time zone designators at https://en.wikipedia.org/wiki/ISO_8601#Time_zone_designators.
     *             
     * This API enforces limits to the amount of data retrieved. These limits are subject to change.
     *             
     * * You may request data from a maximum of a one-hour time period.
     * * The amount of data and number of API calls returned by this API are limited and may be adjusted at any time.
     * * Old records may be migrated out of immediately available storage.  To request older data, please contact your account manager.
     * * New records must migrate to available storage before they can be retrieved.  You may need to wait a period of time before newly created records can be fetched.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param id The ID of the account you wish to audit.
     * @param start The start datetime of audit history you with to retrieve, e.g. "2018-06-08T17:00:00Z". Defaults to the past 15 minutes.
     * @param end The end datetime of audit history you with to retrieve, e.g. "2018-06-08T17:15:00Z. Defaults to the current time. Maximum of an hour after the start time.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @return FetchResult<AuditModel>
     */
    public FetchResult<AuditModel> auditAccount(Integer id, Date start, Date end, Integer top, Integer skip) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}/audit");
        path.applyField("id", id);
        path.addQuery("start", start);
        path.addQuery("end", end);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        return ((RestCall<FetchResult<AuditModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<AuditModel>>(){})).call();
    }

    /**
     * Retrieve audit history for an account.
     * 
     * Retrieve audit trace history for an account.
     *             
     * Your audit trace history contains a record of all API calls made against the AvaTax REST API.  You can use this API to investigate
     * problems and see exactly what information was sent back and forth between your code and AvaTax.
     *             
     * When specifying a start and end datetime, please include a valid timezone indicator, such as the "Z" present in the examples for the start and end query parameters.
     * You can learn more about valid time zone designators at https://en.wikipedia.org/wiki/ISO_8601#Time_zone_designators.
     *             
     * This API enforces limits to the amount of data retrieved. These limits are subject to change.
     *             
     * * You may request data from a maximum of a one-hour time period.
     * * The amount of data and number of API calls returned by this API are limited and may be adjusted at any time.
     * * Old records may be migrated out of immediately available storage.  To request older data, please contact your account manager.
     * * New records must migrate to available storage before they can be retrieved.  You may need to wait a period of time before newly created records can be fetched.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param id The ID of the account you wish to audit.
     * @param start The start datetime of audit history you with to retrieve, e.g. "2018-06-08T17:00:00Z". Defaults to the past 15 minutes.
     * @param end The end datetime of audit history you with to retrieve, e.g. "2018-06-08T17:15:00Z. Defaults to the current time. Maximum of an hour after the start time.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @return FetchResult<AuditModel>
     */
    public Future<FetchResult<AuditModel>> auditAccountAsync(Integer id, Date start, Date end, Integer top, Integer skip) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}/audit");
        path.applyField("id", id);
        path.addQuery("start", start);
        path.addQuery("end", end);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        return this.threadPool.submit((RestCall<FetchResult<AuditModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<AuditModel>>(){}));
    }

    /**
     * Retrieve a single account
     * 
     * Get the account object identified by this URL.
     * You may use the '$include' parameter to fetch additional nested data:
     *             
     * * Subscriptions
     * * Users
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param id The ID of the account to retrieve
     * @param include A comma separated list of special fetch options
     * @return AccountModel
     */
    public AccountModel getAccount(Integer id, String include) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}");
        path.applyField("id", id);
        path.addQuery("$include", include);
        return ((RestCall<AccountModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<AccountModel>(){})).call();
    }

    /**
     * Retrieve a single account
     * 
     * Get the account object identified by this URL.
     * You may use the '$include' parameter to fetch additional nested data:
     *             
     * * Subscriptions
     * * Users
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param id The ID of the account to retrieve
     * @param include A comma separated list of special fetch options
     * @return AccountModel
     */
    public Future<AccountModel> getAccountAsync(Integer id, String include) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}");
        path.applyField("id", id);
        path.addQuery("$include", include);
        return this.threadPool.submit((RestCall<AccountModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<AccountModel>(){}));
    }

    /**
     * Get configuration settings for this account
     * 
     * Retrieve a list of all configuration settings tied to this account.
     *             
     * Configuration settings provide you with the ability to control features of your account and of your
     * tax software.  The category names `TaxServiceConfig` and `AddressServiceConfig` are reserved for
     * Avalara internal software configuration values; to store your own account-level settings, please
     * create a new category name that begins with `X-`, for example, `X-MyCustomCategory`.
     *             
     * Account settings are permanent settings that cannot be deleted.  You can set the value of an
     * account setting to null if desired.
     *             
     * Avalara-based account settings for `TaxServiceConfig` and `AddressServiceConfig` affect your account's
     * tax calculation and address resolution, and should only be changed with care.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param id 
     * @return ArrayList<AccountConfigurationModel>
     */
    public ArrayList<AccountConfigurationModel> getAccountConfiguration(Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}/configuration");
        path.applyField("id", id);
        return ((RestCall<ArrayList<AccountConfigurationModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<AccountConfigurationModel>>(){})).call();
    }

    /**
     * Get configuration settings for this account
     * 
     * Retrieve a list of all configuration settings tied to this account.
     *             
     * Configuration settings provide you with the ability to control features of your account and of your
     * tax software.  The category names `TaxServiceConfig` and `AddressServiceConfig` are reserved for
     * Avalara internal software configuration values; to store your own account-level settings, please
     * create a new category name that begins with `X-`, for example, `X-MyCustomCategory`.
     *             
     * Account settings are permanent settings that cannot be deleted.  You can set the value of an
     * account setting to null if desired.
     *             
     * Avalara-based account settings for `TaxServiceConfig` and `AddressServiceConfig` affect your account's
     * tax calculation and address resolution, and should only be changed with care.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param id 
     * @return ArrayList<AccountConfigurationModel>
     */
    public Future<ArrayList<AccountConfigurationModel>> getAccountConfigurationAsync(Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}/configuration");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<AccountConfigurationModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<AccountConfigurationModel>>(){}));
    }

    /**
     * Retrieve all accounts
     * 
     * List all account objects that can be seen by the current user.
     *             
     * This API lists all accounts you are allowed to see.  In general, most users will only be able to see their own account.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * Subscriptions
     * * Users
     *             
     * For more information about filtering in REST, please see the documentation at http://developer.avalara.com/avatax/filtering-in-rest/ .
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param include A comma separated list of objects to fetch underneath this account. Any object with a URL path underneath this account can be fetched by specifying its name.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* subscriptions, users
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<AccountModel>
     */
    public FetchResult<AccountModel> queryAccounts(String include, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts");
        path.addQuery("$include", include);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<AccountModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<AccountModel>>(){})).call();
    }

    /**
     * Retrieve all accounts
     * 
     * List all account objects that can be seen by the current user.
     *             
     * This API lists all accounts you are allowed to see.  In general, most users will only be able to see their own account.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * Subscriptions
     * * Users
     *             
     * For more information about filtering in REST, please see the documentation at http://developer.avalara.com/avatax/filtering-in-rest/ .
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param include A comma separated list of objects to fetch underneath this account. Any object with a URL path underneath this account can be fetched by specifying its name.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* subscriptions, users
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<AccountModel>
     */
    public Future<FetchResult<AccountModel>> queryAccountsAsync(String include, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts");
        path.addQuery("$include", include);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<AccountModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<AccountModel>>(){}));
    }

    /**
     * Change configuration settings for this account
     * 
     * Update configuration settings tied to this account.
     *             
     * Configuration settings provide you with the ability to control features of your account and of your
     * tax software.  The category names `TaxServiceConfig` and `AddressServiceConfig` are reserved for
     * Avalara internal software configuration values; to store your own account-level settings, please
     * create a new category name that begins with `X-`, for example, `X-MyCustomCategory`.
     *             
     * Account settings are permanent settings that cannot be deleted.  You can set the value of an
     * account setting to null if desired.
     *             
     * Avalara-based account settings for `TaxServiceConfig` and `AddressServiceConfig` affect your account's
     * tax calculation and address resolution, and should only be changed with care.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param id 
     * @param model 
     * @return ArrayList<AccountConfigurationModel>
     */
    public ArrayList<AccountConfigurationModel> setAccountConfiguration(Integer id, ArrayList<AccountConfigurationModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}/configuration");
        path.applyField("id", id);
        return ((RestCall<ArrayList<AccountConfigurationModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<AccountConfigurationModel>>(){})).call();
    }

    /**
     * Change configuration settings for this account
     * 
     * Update configuration settings tied to this account.
     *             
     * Configuration settings provide you with the ability to control features of your account and of your
     * tax software.  The category names `TaxServiceConfig` and `AddressServiceConfig` are reserved for
     * Avalara internal software configuration values; to store your own account-level settings, please
     * create a new category name that begins with `X-`, for example, `X-MyCustomCategory`.
     *             
     * Account settings are permanent settings that cannot be deleted.  You can set the value of an
     * account setting to null if desired.
     *             
     * Avalara-based account settings for `TaxServiceConfig` and `AddressServiceConfig` affect your account's
     * tax calculation and address resolution, and should only be changed with care.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param id 
     * @param model 
     * @return ArrayList<AccountConfigurationModel>
     */
    public Future<ArrayList<AccountConfigurationModel>> setAccountConfigurationAsync(Integer id, ArrayList<AccountConfigurationModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}/configuration");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<AccountConfigurationModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<AccountConfigurationModel>>(){}));
    }

    /**
     * Retrieve geolocation information for a specified address
     * 
     * Resolve an address against Avalara's address-validation system.  If the address can be resolved, this API
     * provides the latitude and longitude of the resolved location.  The value 'resolutionQuality' can be used
     * to identify how closely this address can be located.  If the address cannot be clearly located, use the
     * 'messages' structure to learn more about problems with this address.
     * This is the same API as the POST /api/v2/addresses/resolve endpoint.
     * Both verbs are supported to provide for flexible implementation.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AutoAddress.
     * 
     * @param line1 Line 1
     * @param line2 Line 2
     * @param line3 Line 3
     * @param city City
     * @param region State / Province / Region
     * @param postalCode Postal Code / Zip Code
     * @param country Two character ISO 3166 Country Code (see /api/v2/definitions/countries for a full list)
     * @param textCase selectable text case for address validation (See TextCase::* for a list of allowable values)
     * @return AddressResolutionModel
     */
    public AddressResolutionModel resolveAddress(String line1, String line2, String line3, String city, String region, String postalCode, String country, TextCase textCase) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/addresses/resolve");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        path.addQuery("textCase", textCase);
        return ((RestCall<AddressResolutionModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<AddressResolutionModel>(){})).call();
    }

    /**
     * Retrieve geolocation information for a specified address
     * 
     * Resolve an address against Avalara's address-validation system.  If the address can be resolved, this API
     * provides the latitude and longitude of the resolved location.  The value 'resolutionQuality' can be used
     * to identify how closely this address can be located.  If the address cannot be clearly located, use the
     * 'messages' structure to learn more about problems with this address.
     * This is the same API as the POST /api/v2/addresses/resolve endpoint.
     * Both verbs are supported to provide for flexible implementation.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AutoAddress.
     * 
     * @param line1 Line 1
     * @param line2 Line 2
     * @param line3 Line 3
     * @param city City
     * @param region State / Province / Region
     * @param postalCode Postal Code / Zip Code
     * @param country Two character ISO 3166 Country Code (see /api/v2/definitions/countries for a full list)
     * @param textCase selectable text case for address validation (See TextCase::* for a list of allowable values)
     * @return AddressResolutionModel
     */
    public Future<AddressResolutionModel> resolveAddressAsync(String line1, String line2, String line3, String city, String region, String postalCode, String country, TextCase textCase) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/addresses/resolve");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        path.addQuery("textCase", textCase);
        return this.threadPool.submit((RestCall<AddressResolutionModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<AddressResolutionModel>(){}));
    }

    /**
     * Retrieve geolocation information for a specified address
     * 
     * Resolve an address against Avalara's address-validation system.  If the address can be resolved, this API
     * provides the latitude and longitude of the resolved location.  The value 'resolutionQuality' can be used
     * to identify how closely this address can be located.  If the address cannot be clearly located, use the
     * 'messages' structure to learn more about problems with this address.
     * This is the same API as the GET /api/v2/addresses/resolve endpoint.
     * Both verbs are supported to provide for flexible implementation.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AutoAddress.
     * 
     * @param model The address to resolve
     * @return AddressResolutionModel
     */
    public AddressResolutionModel resolveAddressPost(AddressValidationInfo model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/addresses/resolve");
        return ((RestCall<AddressResolutionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<AddressResolutionModel>(){})).call();
    }

    /**
     * Retrieve geolocation information for a specified address
     * 
     * Resolve an address against Avalara's address-validation system.  If the address can be resolved, this API
     * provides the latitude and longitude of the resolved location.  The value 'resolutionQuality' can be used
     * to identify how closely this address can be located.  If the address cannot be clearly located, use the
     * 'messages' structure to learn more about problems with this address.
     * This is the same API as the GET /api/v2/addresses/resolve endpoint.
     * Both verbs are supported to provide for flexible implementation.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AutoAddress.
     * 
     * @param model The address to resolve
     * @return AddressResolutionModel
     */
    public Future<AddressResolutionModel> resolveAddressPostAsync(AddressValidationInfo model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/addresses/resolve");
        return this.threadPool.submit((RestCall<AddressResolutionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<AddressResolutionModel>(){}));
    }

    /**
     * Create a new AvaFileForm
     * 
     * Create one or more AvaFileForms
     * A 'AvaFileForm' represents a form supported by our returns team
     * 
     * ### Security Policies
     * 
     * * This API requires the user role Compliance Root User.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param model The AvaFileForm you wish to create.
     * @return ArrayList<AvaFileFormModel>
     */
    public ArrayList<AvaFileFormModel> createAvaFileForms(ArrayList<AvaFileFormModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/avafileforms");
        return ((RestCall<ArrayList<AvaFileFormModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<AvaFileFormModel>>(){})).call();
    }

    /**
     * Create a new AvaFileForm
     * 
     * Create one or more AvaFileForms
     * A 'AvaFileForm' represents a form supported by our returns team
     * 
     * ### Security Policies
     * 
     * * This API requires the user role Compliance Root User.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param model The AvaFileForm you wish to create.
     * @return ArrayList<AvaFileFormModel>
     */
    public Future<ArrayList<AvaFileFormModel>> createAvaFileFormsAsync(ArrayList<AvaFileFormModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/avafileforms");
        return this.threadPool.submit((RestCall<ArrayList<AvaFileFormModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<AvaFileFormModel>>(){}));
    }

    /**
     * Delete a single AvaFileForm
     * 
     * Marks the existing AvaFileForm object at this URL as deleted.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: Compliance Root User, ComplianceUser, FirmAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param id The ID of the AvaFileForm you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteAvaFileForm(Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/avafileforms/{id}");
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a single AvaFileForm
     * 
     * Marks the existing AvaFileForm object at this URL as deleted.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: Compliance Root User, ComplianceUser, FirmAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param id The ID of the AvaFileForm you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteAvaFileFormAsync(Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/avafileforms/{id}");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Retrieve a single AvaFileForm
     * 
     * Get the AvaFileForm object identified by this URL.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, CompanyAdmin, CompanyUser, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, FirmAdmin, FirmUser, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param id The primary key of this AvaFileForm
     * @return AvaFileFormModel
     */
    public AvaFileFormModel getAvaFileForm(Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/avafileforms/{id}");
        path.applyField("id", id);
        return ((RestCall<AvaFileFormModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<AvaFileFormModel>(){})).call();
    }

    /**
     * Retrieve a single AvaFileForm
     * 
     * Get the AvaFileForm object identified by this URL.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, CompanyAdmin, CompanyUser, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, FirmAdmin, FirmUser, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param id The primary key of this AvaFileForm
     * @return AvaFileFormModel
     */
    public Future<AvaFileFormModel> getAvaFileFormAsync(Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/avafileforms/{id}");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<AvaFileFormModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<AvaFileFormModel>(){}));
    }

    /**
     * Retrieve all AvaFileForms
     * 
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, CompanyAdmin, CompanyUser, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, FirmAdmin, FirmUser, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* outletTypeId
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<AvaFileFormModel>
     */
    public FetchResult<AvaFileFormModel> queryAvaFileForms(String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/avafileforms");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<AvaFileFormModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<AvaFileFormModel>>(){})).call();
    }

    /**
     * Retrieve all AvaFileForms
     * 
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, CompanyAdmin, CompanyUser, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, FirmAdmin, FirmUser, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* outletTypeId
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<AvaFileFormModel>
     */
    public Future<FetchResult<AvaFileFormModel>> queryAvaFileFormsAsync(String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/avafileforms");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<AvaFileFormModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<AvaFileFormModel>>(){}));
    }

    /**
     * Update a AvaFileForm
     * 
     * All data from the existing object will be replaced with data in the object you PUT.  
     * To set a field's value to null, you may either set its value to null or omit that field from the object you post.
     * 
     * ### Security Policies
     * 
     * * This API requires the user role Compliance Root User.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param id The ID of the AvaFileForm you wish to update
     * @param model The AvaFileForm model you wish to update.
     * @return AvaFileFormModel
     */
    public AvaFileFormModel updateAvaFileForm(Integer id, AvaFileFormModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/avafileforms/{id}");
        path.applyField("id", id);
        return ((RestCall<AvaFileFormModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<AvaFileFormModel>(){})).call();
    }

    /**
     * Update a AvaFileForm
     * 
     * All data from the existing object will be replaced with data in the object you PUT.  
     * To set a field's value to null, you may either set its value to null or omit that field from the object you post.
     * 
     * ### Security Policies
     * 
     * * This API requires the user role Compliance Root User.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param id The ID of the AvaFileForm you wish to update
     * @param model The AvaFileForm model you wish to update.
     * @return AvaFileFormModel
     */
    public Future<AvaFileFormModel> updateAvaFileFormAsync(Integer id, AvaFileFormModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/avafileforms/{id}");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<AvaFileFormModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<AvaFileFormModel>(){}));
    }

    /**
     * Create a new batch
     * 
     * Create one or more new batch objects attached to this company.
     * 
     * Each batch object may have one or more file objects (currently only one file is supported).
     * 
     * When a batch is created, it is added to the AvaTax Batch Queue and will be 
     * processed as quickly as possible in the order it was received. To check the 
     * status of a batch, fetch the batch and retrieve the results of the batch 
     * operation.
     * 
     * Because the batch system processes with a degree of concurrency, and
     * because of batch sizes in the queue vary, AvaTax API is unable to accurately 
     * predict when a batch will complete. If high performance processing is 
     * required, please use the 
     * [CreateTransaction API](https://developer.avalara.com/api-reference/avatax/rest/v2/methods/Transactions/CreateTransaction/).
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, CompanyAdmin, CSPTester, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this batch.
     * @param model The batch you wish to create.
     * @return ArrayList<BatchModel>
     */
    public ArrayList<BatchModel> createBatches(Integer companyId, ArrayList<BatchModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<BatchModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<BatchModel>>(){})).call();
    }

    /**
     * Create a new batch
     * 
     * Create one or more new batch objects attached to this company.
     * 
     * Each batch object may have one or more file objects (currently only one file is supported).
     * 
     * When a batch is created, it is added to the AvaTax Batch Queue and will be 
     * processed as quickly as possible in the order it was received. To check the 
     * status of a batch, fetch the batch and retrieve the results of the batch 
     * operation.
     * 
     * Because the batch system processes with a degree of concurrency, and
     * because of batch sizes in the queue vary, AvaTax API is unable to accurately 
     * predict when a batch will complete. If high performance processing is 
     * required, please use the 
     * [CreateTransaction API](https://developer.avalara.com/api-reference/avatax/rest/v2/methods/Transactions/CreateTransaction/).
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, CompanyAdmin, CSPTester, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this batch.
     * @param model The batch you wish to create.
     * @return ArrayList<BatchModel>
     */
    public Future<ArrayList<BatchModel>> createBatchesAsync(Integer companyId, ArrayList<BatchModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<BatchModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<BatchModel>>(){}));
    }

    /**
     * Delete a single batch
     * 
     * Marks the batch identified by this URL as deleted.
     * 
     * If you attempt to delete a batch that is being processed, you will receive an error message.
     * Deleting a batch does not delete any transactions that were created by importing the batch.
     * 
     * Because the batch system processes with a degree of concurrency, and
     * because of batch sizes in the queue vary, AvaTax API is unable to accurately 
     * predict when a batch will complete. If high performance processing is 
     * required, please use the 
     * [CreateTransaction API](https://developer.avalara.com/api-reference/avatax/rest/v2/methods/Transactions/CreateTransaction/).
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: CSPAdmin, CSPTester, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this batch.
     * @param id The ID of the batch to delete.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteBatch(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a single batch
     * 
     * Marks the batch identified by this URL as deleted.
     * 
     * If you attempt to delete a batch that is being processed, you will receive an error message.
     * Deleting a batch does not delete any transactions that were created by importing the batch.
     * 
     * Because the batch system processes with a degree of concurrency, and
     * because of batch sizes in the queue vary, AvaTax API is unable to accurately 
     * predict when a batch will complete. If high performance processing is 
     * required, please use the 
     * [CreateTransaction API](https://developer.avalara.com/api-reference/avatax/rest/v2/methods/Transactions/CreateTransaction/).
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: CSPAdmin, CSPTester, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this batch.
     * @param id The ID of the batch to delete.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteBatchAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Download a single batch file
     * 
     * Download a single batch file identified by this URL.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The ID of the company that owns this batch
     * @param batchId The ID of the batch object
     * @param id The primary key of this batch file object
     * @return String
     */
    public String downloadBatch(Integer companyId, Integer batchId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches/{batchId}/files/{id}/attachment");
        path.applyField("companyId", companyId);
        path.applyField("batchId", batchId);
        path.applyField("id", id);
        return ((RestCall<String>)restCallFactory.createRestCall("get", path, null, new TypeToken<String>(){})).call();
    }

    /**
     * Download a single batch file
     * 
     * Download a single batch file identified by this URL.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The ID of the company that owns this batch
     * @param batchId The ID of the batch object
     * @param id The primary key of this batch file object
     * @return String
     */
    public Future<String> downloadBatchAsync(Integer companyId, Integer batchId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches/{batchId}/files/{id}/attachment");
        path.applyField("companyId", companyId);
        path.applyField("batchId", batchId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<String>)restCallFactory.createRestCall("get", path, null, new TypeToken<String>(){}));
    }

    /**
     * Retrieve a single batch
     * 
     * Get the batch object identified by this URL. A batch object is a large 
     * collection of API calls stored in a compact file.
     * 
     * Use this endpoint to retrieve the results or check the status of a batch.
     * 
     * When a batch is created, it is added to the AvaTax Batch Queue and will be 
     * processed as quickly as possible in the order it was received. To check the 
     * status of a batch, fetch the batch and retrieve the results of the batch 
     * operation.
     * 
     * Because the batch system processes with a degree of concurrency, and
     * because of batch sizes in the queue vary, AvaTax API is unable to accurately 
     * predict when a batch will complete. If high performance processing is 
     * required, please use the 
     * [CreateTransaction API](https://developer.avalara.com/api-reference/avatax/rest/v2/methods/Transactions/CreateTransaction/).
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The ID of the company that owns this batch
     * @param id The primary key of this batch
     * @return BatchModel
     */
    public BatchModel getBatch(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<BatchModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<BatchModel>(){})).call();
    }

    /**
     * Retrieve a single batch
     * 
     * Get the batch object identified by this URL. A batch object is a large 
     * collection of API calls stored in a compact file.
     * 
     * Use this endpoint to retrieve the results or check the status of a batch.
     * 
     * When a batch is created, it is added to the AvaTax Batch Queue and will be 
     * processed as quickly as possible in the order it was received. To check the 
     * status of a batch, fetch the batch and retrieve the results of the batch 
     * operation.
     * 
     * Because the batch system processes with a degree of concurrency, and
     * because of batch sizes in the queue vary, AvaTax API is unable to accurately 
     * predict when a batch will complete. If high performance processing is 
     * required, please use the 
     * [CreateTransaction API](https://developer.avalara.com/api-reference/avatax/rest/v2/methods/Transactions/CreateTransaction/).
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The ID of the company that owns this batch
     * @param id The primary key of this batch
     * @return BatchModel
     */
    public Future<BatchModel> getBatchAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<BatchModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<BatchModel>(){}));
    }

    /**
     * Retrieve all batches for this company
     * 
     * List all batch objects attached to the specified company.
     * 
     * A batch object is a large collection of API calls stored in a compact file.
     * 
     * Search for specific objects using the criteria in the `$filter` parameter; 
     * full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * Use [GetBatch](https://developer.avalara.com/api-reference/avatax/rest/v2/methods/Batches/GetBatch/) 
     * to retrieve the results, or check the status, of an individual batch.
     * 
     * When a batch is created, it is added to the AvaTax Batch Queue and will be 
     * processed as quickly as possible in the order it was received. To check the 
     * status of a batch, fetch the batch and retrieve the results of the batch 
     * operation.
     * 
     * Because the batch system processes with a degree of concurrency, and
     * because of batch sizes in the queue vary, AvaTax API is unable to accurately 
     * predict when a batch will complete. If high performance processing is 
     * required, please use the
     * [CreateTransaction API](https://developer.avalara.com/api-reference/avatax/rest/v2/methods/Transactions/CreateTransaction/).
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The ID of the company that owns these batches
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* files
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<BatchModel>
     */
    public FetchResult<BatchModel> listBatchesByCompany(Integer companyId, String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<BatchModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<BatchModel>>(){})).call();
    }

    /**
     * Retrieve all batches for this company
     * 
     * List all batch objects attached to the specified company.
     * 
     * A batch object is a large collection of API calls stored in a compact file.
     * 
     * Search for specific objects using the criteria in the `$filter` parameter; 
     * full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * Use [GetBatch](https://developer.avalara.com/api-reference/avatax/rest/v2/methods/Batches/GetBatch/) 
     * to retrieve the results, or check the status, of an individual batch.
     * 
     * When a batch is created, it is added to the AvaTax Batch Queue and will be 
     * processed as quickly as possible in the order it was received. To check the 
     * status of a batch, fetch the batch and retrieve the results of the batch 
     * operation.
     * 
     * Because the batch system processes with a degree of concurrency, and
     * because of batch sizes in the queue vary, AvaTax API is unable to accurately 
     * predict when a batch will complete. If high performance processing is 
     * required, please use the
     * [CreateTransaction API](https://developer.avalara.com/api-reference/avatax/rest/v2/methods/Transactions/CreateTransaction/).
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The ID of the company that owns these batches
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* files
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<BatchModel>
     */
    public Future<FetchResult<BatchModel>> listBatchesByCompanyAsync(Integer companyId, String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<BatchModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<BatchModel>>(){}));
    }

    /**
     * Retrieve all batches
     * 
     * Get multiple batch objects across all companies.
     * 
     * A batch object is a large collection of API calls stored in a compact file.
     * 
     * Search for specific objects using the criteria in the `$filter` parameter; 
     * full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * When a batch is created, it is added to the AvaTax Batch Queue and will be 
     * processed as quickly as possible in the order it was received. To check the 
     * status of a batch, fetch the batch and retrieve the results of the batch 
     * operation.
     * 
     * Because the batch system processes with a degree of concurrency, and
     * because of batch sizes in the queue vary, AvaTax API is unable to accurately 
     * predict when a batch will complete. If high performance processing is 
     * required, please use the
     * [CreateTransaction API](https://developer.avalara.com/api-reference/avatax/rest/v2/methods/Transactions/CreateTransaction/).
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* files
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<BatchModel>
     */
    public FetchResult<BatchModel> queryBatches(String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/batches");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<BatchModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<BatchModel>>(){})).call();
    }

    /**
     * Retrieve all batches
     * 
     * Get multiple batch objects across all companies.
     * 
     * A batch object is a large collection of API calls stored in a compact file.
     * 
     * Search for specific objects using the criteria in the `$filter` parameter; 
     * full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * When a batch is created, it is added to the AvaTax Batch Queue and will be 
     * processed as quickly as possible in the order it was received. To check the 
     * status of a batch, fetch the batch and retrieve the results of the batch 
     * operation.
     * 
     * Because the batch system processes with a degree of concurrency, and
     * because of batch sizes in the queue vary, AvaTax API is unable to accurately 
     * predict when a batch will complete. If high performance processing is 
     * required, please use the
     * [CreateTransaction API](https://developer.avalara.com/api-reference/avatax/rest/v2/methods/Transactions/CreateTransaction/).
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* files
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<BatchModel>
     */
    public Future<FetchResult<BatchModel>> queryBatchesAsync(String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/batches");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<BatchModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<BatchModel>>(){}));
    }

    /**
     * Create a CertExpress invitation
     * 
     * Creates an invitation for a customer to self-report certificates using the CertExpress website.
     * 
     * This invitation is delivered by your choice of method, or you can present a hyperlink to the user
     * directly in your connector.  Your customer will be redirected to https://app.certexpress.com/ where
     * they can follow a step-by-step guide to enter information about their exemption certificates.  The
     * certificates entered will be recorded and automatically linked to their customer record.
     * 
     * The [CertExpress website](https://app.certexpress.com/home) is available for customers to use at any time.
     * Using CertExpress with this API will ensure that your certificates are automatically linked correctly into
     * your company so that they can be used for tax exemptions.
     * 
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document 
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that will record certificates
     * @param customerCode The number of the customer where the request is sent to
     * @param model the requests to send out to customers
     * @return ArrayList<CertExpressInvitationStatusModel>
     */
    public ArrayList<CertExpressInvitationStatusModel> createCertExpressInvitation(Integer companyId, String customerCode, ArrayList<CreateCertExpressInvitationModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}/certexpressinvites");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        return ((RestCall<ArrayList<CertExpressInvitationStatusModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<CertExpressInvitationStatusModel>>(){})).call();
    }

    /**
     * Create a CertExpress invitation
     * 
     * Creates an invitation for a customer to self-report certificates using the CertExpress website.
     * 
     * This invitation is delivered by your choice of method, or you can present a hyperlink to the user
     * directly in your connector.  Your customer will be redirected to https://app.certexpress.com/ where
     * they can follow a step-by-step guide to enter information about their exemption certificates.  The
     * certificates entered will be recorded and automatically linked to their customer record.
     * 
     * The [CertExpress website](https://app.certexpress.com/home) is available for customers to use at any time.
     * Using CertExpress with this API will ensure that your certificates are automatically linked correctly into
     * your company so that they can be used for tax exemptions.
     * 
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document 
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that will record certificates
     * @param customerCode The number of the customer where the request is sent to
     * @param model the requests to send out to customers
     * @return ArrayList<CertExpressInvitationStatusModel>
     */
    public Future<ArrayList<CertExpressInvitationStatusModel>> createCertExpressInvitationAsync(Integer companyId, String customerCode, ArrayList<CreateCertExpressInvitationModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}/certexpressinvites");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        return this.threadPool.submit((RestCall<ArrayList<CertExpressInvitationStatusModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<CertExpressInvitationStatusModel>>(){}));
    }

    /**
     * Retrieve a single CertExpress invitation
     * 
     * Retrieve an existing CertExpress invitation sent to a customer.
     * 
     * A CertExpression invitation allows a customer to follow a helpful step-by-step guide to provide information
     * about their certificates.  This step by step guide allows the customer to complete and upload the full 
     * certificate in a convenient, friendly web browser experience.  When the customer completes their certificates,
     * they will automatically be recorded to your company and linked to the customer record.
     * 
     * The [CertExpress website](https://app.certexpress.com/home) is available for customers to use at any time.
     * Using CertExpress with this API will ensure that your certificates are automatically linked correctly into
     * your company so that they can be used for tax exemptions.
     * 
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document 
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that issued this invitation
     * @param customerCode The number of the customer where the request is sent to
     * @param id The unique ID number of this CertExpress invitation
     * @param include OPTIONAL: A comma separated list of special fetch options. No options are defined at this time.
     * @return CertExpressInvitationModel
     */
    public CertExpressInvitationModel getCertExpressInvitation(Integer companyId, String customerCode, Integer id, String include) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}/certexpressinvites/{id}");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        path.applyField("id", id);
        path.addQuery("$include", include);
        return ((RestCall<CertExpressInvitationModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<CertExpressInvitationModel>(){})).call();
    }

    /**
     * Retrieve a single CertExpress invitation
     * 
     * Retrieve an existing CertExpress invitation sent to a customer.
     * 
     * A CertExpression invitation allows a customer to follow a helpful step-by-step guide to provide information
     * about their certificates.  This step by step guide allows the customer to complete and upload the full 
     * certificate in a convenient, friendly web browser experience.  When the customer completes their certificates,
     * they will automatically be recorded to your company and linked to the customer record.
     * 
     * The [CertExpress website](https://app.certexpress.com/home) is available for customers to use at any time.
     * Using CertExpress with this API will ensure that your certificates are automatically linked correctly into
     * your company so that they can be used for tax exemptions.
     * 
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document 
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that issued this invitation
     * @param customerCode The number of the customer where the request is sent to
     * @param id The unique ID number of this CertExpress invitation
     * @param include OPTIONAL: A comma separated list of special fetch options. No options are defined at this time.
     * @return CertExpressInvitationModel
     */
    public Future<CertExpressInvitationModel> getCertExpressInvitationAsync(Integer companyId, String customerCode, Integer id, String include) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}/certexpressinvites/{id}");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        path.applyField("id", id);
        path.addQuery("$include", include);
        return this.threadPool.submit((RestCall<CertExpressInvitationModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<CertExpressInvitationModel>(){}));
    }

    /**
     * List CertExpress invitations
     * 
     * Retrieve CertExpress invitations sent by this company.
     * 
     * A CertExpression invitation allows a customer to follow a helpful step-by-step guide to provide information
     * about their certificates.  This step by step guide allows the customer to complete and upload the full 
     * certificate in a convenient, friendly web browser experience.  When the customer completes their certificates,
     * they will automatically be recorded to your company and linked to the customer record.
     * 
     * The [CertExpress website](https://app.certexpress.com/home) is available for customers to use at any time.
     * Using CertExpress with this API will ensure that your certificates are automatically linked correctly into
     * your company so that they can be used for tax exemptions.
     * 
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document 
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that issued this invitation
     * @param include OPTIONAL: A comma separated list of special fetch options.       No options are defined at this time.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* companyId, customer, coverLetter, exposureZones, exemptReasons, requestLink
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CertExpressInvitationModel>
     */
    public FetchResult<CertExpressInvitationModel> listCertExpressInvitations(Integer companyId, String include, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certexpressinvites");
        path.applyField("companyId", companyId);
        path.addQuery("$include", include);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<CertExpressInvitationModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CertExpressInvitationModel>>(){})).call();
    }

    /**
     * List CertExpress invitations
     * 
     * Retrieve CertExpress invitations sent by this company.
     * 
     * A CertExpression invitation allows a customer to follow a helpful step-by-step guide to provide information
     * about their certificates.  This step by step guide allows the customer to complete and upload the full 
     * certificate in a convenient, friendly web browser experience.  When the customer completes their certificates,
     * they will automatically be recorded to your company and linked to the customer record.
     * 
     * The [CertExpress website](https://app.certexpress.com/home) is available for customers to use at any time.
     * Using CertExpress with this API will ensure that your certificates are automatically linked correctly into
     * your company so that they can be used for tax exemptions.
     * 
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document 
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that issued this invitation
     * @param include OPTIONAL: A comma separated list of special fetch options.       No options are defined at this time.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* companyId, customer, coverLetter, exposureZones, exemptReasons, requestLink
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CertExpressInvitationModel>
     */
    public Future<FetchResult<CertExpressInvitationModel>> listCertExpressInvitationsAsync(Integer companyId, String include, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certexpressinvites");
        path.applyField("companyId", companyId);
        path.addQuery("$include", include);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<CertExpressInvitationModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CertExpressInvitationModel>>(){}));
    }

    /**
     * Create certificates for this company
     * 
     * Record one or more certificates document for this company.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * When you create a certificate, it will be processed by Avalara and will become available for use in
     * calculating tax exemptions when processing is complete.  For a certificate to be used in calculating exemptions,
     * it must have the following:
     *             
     * * A list of exposure zones indicating where the certificate is valid
     * * A link to the customer that is allowed to use this certificate
     * * Your tax transaction must contain the correct customer code
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The ID number of the company recording this certificate
     * @param preValidatedExemptionReason If set to true, the certificate will bypass the human verification process.
     * @param model Certificates to be created
     * @return ArrayList<CertificateModel>
     */
    public ArrayList<CertificateModel> createCertificates(Integer companyId, Boolean preValidatedExemptionReason, ArrayList<CertificateModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates");
        path.applyField("companyId", companyId);
        path.addQuery("$preValidatedExemptionReason", preValidatedExemptionReason);
        return ((RestCall<ArrayList<CertificateModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<CertificateModel>>(){})).call();
    }

    /**
     * Create certificates for this company
     * 
     * Record one or more certificates document for this company.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * When you create a certificate, it will be processed by Avalara and will become available for use in
     * calculating tax exemptions when processing is complete.  For a certificate to be used in calculating exemptions,
     * it must have the following:
     *             
     * * A list of exposure zones indicating where the certificate is valid
     * * A link to the customer that is allowed to use this certificate
     * * Your tax transaction must contain the correct customer code
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The ID number of the company recording this certificate
     * @param preValidatedExemptionReason If set to true, the certificate will bypass the human verification process.
     * @param model Certificates to be created
     * @return ArrayList<CertificateModel>
     */
    public Future<ArrayList<CertificateModel>> createCertificatesAsync(Integer companyId, Boolean preValidatedExemptionReason, ArrayList<CertificateModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates");
        path.applyField("companyId", companyId);
        path.addQuery("$preValidatedExemptionReason", preValidatedExemptionReason);
        return this.threadPool.submit((RestCall<ArrayList<CertificateModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<CertificateModel>>(){}));
    }

    /**
     * Revoke and delete a certificate
     * 
     * Revoke the certificate identified by this URL, then delete it.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Revoked certificates can no longer be used.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @return CertificateModel
     */
    public CertificateModel deleteCertificate(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<CertificateModel>)restCallFactory.createRestCall("delete", path, null, new TypeToken<CertificateModel>(){})).call();
    }

    /**
     * Revoke and delete a certificate
     * 
     * Revoke the certificate identified by this URL, then delete it.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Revoked certificates can no longer be used.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @return CertificateModel
     */
    public Future<CertificateModel> deleteCertificateAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<CertificateModel>)restCallFactory.createRestCall("delete", path, null, new TypeToken<CertificateModel>(){}));
    }

    /**
     * Download an image for this certificate
     * 
     * Download an image or PDF file for this certificate.
     *             
     * This API can be used to download either a single-page preview of the certificate or a full PDF document.
     * To retrieve a preview image, set the `$type` parameter to `Jpeg` and the `$page` parameter to `1`.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param page If you choose `$type`=`Jpeg`, you must specify which page number to retrieve.
     * @param type The data format in which to retrieve the certificate image (See CertificatePreviewType::* for a list of allowable values)
     * @return String
     */
    public String downloadCertificateImage(Integer companyId, Integer id, Integer page, CertificatePreviewType type) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}/attachment");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.addQuery("$page", page);
        path.addQuery("$type", type);
        return ((RestCall<String>)restCallFactory.createRestCall("get", path, null, new TypeToken<String>(){})).call();
    }

    /**
     * Download an image for this certificate
     * 
     * Download an image or PDF file for this certificate.
     *             
     * This API can be used to download either a single-page preview of the certificate or a full PDF document.
     * To retrieve a preview image, set the `$type` parameter to `Jpeg` and the `$page` parameter to `1`.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param page If you choose `$type`=`Jpeg`, you must specify which page number to retrieve.
     * @param type The data format in which to retrieve the certificate image (See CertificatePreviewType::* for a list of allowable values)
     * @return String
     */
    public Future<String> downloadCertificateImageAsync(Integer companyId, Integer id, Integer page, CertificatePreviewType type) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}/attachment");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.addQuery("$page", page);
        path.addQuery("$type", type);
        return this.threadPool.submit((RestCall<String>)restCallFactory.createRestCall("get", path, null, new TypeToken<String>(){}));
    }

    /**
     * Retrieve a single certificate
     * 
     * Get the current certificate identified by this URL.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * You can use the `$include` parameter to fetch the following additional objects for expansion:
     *             
     * * customers - Retrieves the list of customers linked to the certificate.
     * * po_numbers - Retrieves all PO numbers tied to the certificate.
     * * attributes - Retrieves all attributes applied to the certificate.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param include OPTIONAL: A comma separated list of special fetch options. You can specify one or more of the following:      * customers - Retrieves the list of customers linked to the certificate.   * po_numbers - Retrieves all PO numbers tied to the certificate.   * attributes - Retrieves all attributes applied to the certificate.
     * @return CertificateModel
     */
    public CertificateModel getCertificate(Integer companyId, Integer id, String include) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.addQuery("$include", include);
        return ((RestCall<CertificateModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<CertificateModel>(){})).call();
    }

    /**
     * Retrieve a single certificate
     * 
     * Get the current certificate identified by this URL.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * You can use the `$include` parameter to fetch the following additional objects for expansion:
     *             
     * * customers - Retrieves the list of customers linked to the certificate.
     * * po_numbers - Retrieves all PO numbers tied to the certificate.
     * * attributes - Retrieves all attributes applied to the certificate.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param include OPTIONAL: A comma separated list of special fetch options. You can specify one or more of the following:      * customers - Retrieves the list of customers linked to the certificate.   * po_numbers - Retrieves all PO numbers tied to the certificate.   * attributes - Retrieves all attributes applied to the certificate.
     * @return CertificateModel
     */
    public Future<CertificateModel> getCertificateAsync(Integer companyId, Integer id, String include) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.addQuery("$include", include);
        return this.threadPool.submit((RestCall<CertificateModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<CertificateModel>(){}));
    }

    /**
     * Check a company's exemption certificate status.
     * 
     * Checks whether this company is configured to use exemption certificates in AvaTax.
     *             
     * Exemption certificates are tracked through a different auditable data store than the one that
     * holds AvaTax transactions.  To use the AvaTax exemption certificate document store, please call
     * `GetCertificateSetup` to see if your company is configured to use the exemption certificate
     * document store.  To request setup, please call `RequestCertificateSetup` and your company will
     * be configured with data storage in the auditable certificate system.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The company ID to check
     * @return ProvisionStatusModel
     */
    public ProvisionStatusModel getCertificateSetup(Integer companyId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/setup");
        path.applyField("companyId", companyId);
        return ((RestCall<ProvisionStatusModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<ProvisionStatusModel>(){})).call();
    }

    /**
     * Check a company's exemption certificate status.
     * 
     * Checks whether this company is configured to use exemption certificates in AvaTax.
     *             
     * Exemption certificates are tracked through a different auditable data store than the one that
     * holds AvaTax transactions.  To use the AvaTax exemption certificate document store, please call
     * `GetCertificateSetup` to see if your company is configured to use the exemption certificate
     * document store.  To request setup, please call `RequestCertificateSetup` and your company will
     * be configured with data storage in the auditable certificate system.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The company ID to check
     * @return ProvisionStatusModel
     */
    public Future<ProvisionStatusModel> getCertificateSetupAsync(Integer companyId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/setup");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ProvisionStatusModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<ProvisionStatusModel>(){}));
    }

    /**
     * Link attributes to a certificate
     * 
     * Link one or many attributes to a certificate.
     *             
     * A certificate may have multiple attributes that control its behavior.  You may link or unlink attributes to a
     * certificate at any time.  The full list of defined attributes may be found using `ListCertificateAttributes`.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param model The list of attributes to link to this certificate.
     * @return FetchResult<CertificateAttributeModel>
     */
    public FetchResult<CertificateAttributeModel> linkAttributesToCertificate(Integer companyId, Integer id, ArrayList<CertificateAttributeModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}/attributes/link");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FetchResult<CertificateAttributeModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<CertificateAttributeModel>>(){})).call();
    }

    /**
     * Link attributes to a certificate
     * 
     * Link one or many attributes to a certificate.
     *             
     * A certificate may have multiple attributes that control its behavior.  You may link or unlink attributes to a
     * certificate at any time.  The full list of defined attributes may be found using `ListCertificateAttributes`.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param model The list of attributes to link to this certificate.
     * @return FetchResult<CertificateAttributeModel>
     */
    public Future<FetchResult<CertificateAttributeModel>> linkAttributesToCertificateAsync(Integer companyId, Integer id, ArrayList<CertificateAttributeModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}/attributes/link");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FetchResult<CertificateAttributeModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<CertificateAttributeModel>>(){}));
    }

    /**
     * Link customers to a certificate
     * 
     * Link one or more customers to an existing certificate.
     *             
     * Customers and certificates must be linked before a customer can make use of a certificate to obtain
     * a tax exemption in AvaTax.  Since some certificates may cover more than one business entity, a certificate
     * can be connected to multiple customer records using the `LinkCustomersToCertificate` API.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param model The list of customers needed be added to the Certificate for exemption
     * @return FetchResult<CustomerModel>
     */
    public FetchResult<CustomerModel> linkCustomersToCertificate(Integer companyId, Integer id, LinkCustomersModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}/customers/link");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FetchResult<CustomerModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<CustomerModel>>(){})).call();
    }

    /**
     * Link customers to a certificate
     * 
     * Link one or more customers to an existing certificate.
     *             
     * Customers and certificates must be linked before a customer can make use of a certificate to obtain
     * a tax exemption in AvaTax.  Since some certificates may cover more than one business entity, a certificate
     * can be connected to multiple customer records using the `LinkCustomersToCertificate` API.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param model The list of customers needed be added to the Certificate for exemption
     * @return FetchResult<CustomerModel>
     */
    public Future<FetchResult<CustomerModel>> linkCustomersToCertificateAsync(Integer companyId, Integer id, LinkCustomersModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}/customers/link");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FetchResult<CustomerModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<CustomerModel>>(){}));
    }

    /**
     * List all attributes applied to this certificate
     * 
     * Retrieve the list of attributes that are linked to this certificate.
     *             
     * A certificate may have multiple attributes that control its behavior.  You may link or unlink attributes to a
     * certificate at any time.  The full list of defined attributes may be found using [ListCertificateAttributes](https://developer.avalara.com/api-reference/avatax/rest/v2/methods/Definitions/ListCertificateAttributes/) API.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @return FetchResult<CertificateAttributeModel>
     */
    public FetchResult<CertificateAttributeModel> listAttributesForCertificate(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}/attributes");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FetchResult<CertificateAttributeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CertificateAttributeModel>>(){})).call();
    }

    /**
     * List all attributes applied to this certificate
     * 
     * Retrieve the list of attributes that are linked to this certificate.
     *             
     * A certificate may have multiple attributes that control its behavior.  You may link or unlink attributes to a
     * certificate at any time.  The full list of defined attributes may be found using [ListCertificateAttributes](https://developer.avalara.com/api-reference/avatax/rest/v2/methods/Definitions/ListCertificateAttributes/) API.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @return FetchResult<CertificateAttributeModel>
     */
    public Future<FetchResult<CertificateAttributeModel>> listAttributesForCertificateAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}/attributes");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FetchResult<CertificateAttributeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CertificateAttributeModel>>(){}));
    }

    /**
     * List customers linked to this certificate
     * 
     * List all customers linked to this certificate.
     *             
     * Customers must be linked to a certificate in order to make use of its tax exemption features.  You
     * can link or unlink customers to a certificate at any time.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param include OPTIONAL: A comma separated list of special fetch options.   No options are currently available when fetching customers.
     * @return FetchResult<CustomerModel>
     */
    public FetchResult<CustomerModel> listCustomersForCertificate(Integer companyId, Integer id, String include) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}/customers");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.addQuery("$include", include);
        return ((RestCall<FetchResult<CustomerModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CustomerModel>>(){})).call();
    }

    /**
     * List customers linked to this certificate
     * 
     * List all customers linked to this certificate.
     *             
     * Customers must be linked to a certificate in order to make use of its tax exemption features.  You
     * can link or unlink customers to a certificate at any time.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param include OPTIONAL: A comma separated list of special fetch options.   No options are currently available when fetching customers.
     * @return FetchResult<CustomerModel>
     */
    public Future<FetchResult<CustomerModel>> listCustomersForCertificateAsync(Integer companyId, Integer id, String include) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}/customers");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.addQuery("$include", include);
        return this.threadPool.submit((RestCall<FetchResult<CustomerModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CustomerModel>>(){}));
    }

    /**
     * List all certificates for a company
     * 
     * List all certificates recorded by a company
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * You can use the `$include` parameter to fetch the following additional objects for expansion:
     *             
     * * customers - Retrieves the list of customers linked to the certificate.
     * * po_numbers - Retrieves all PO numbers tied to the certificate.
     * * attributes - Retrieves all attributes applied to the certificate.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The ID number of the company to search
     * @param include OPTIONAL: A comma separated list of special fetch options. You can specify one or more of the following:      * customers - Retrieves the list of customers linked to the certificate.   * po_numbers - Retrieves all PO numbers tied to the certificate.   * attributes - Retrieves all attributes applied to the certificate.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* status, pdf, pages
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CertificateModel>
     */
    public FetchResult<CertificateModel> queryCertificates(Integer companyId, String include, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates");
        path.applyField("companyId", companyId);
        path.addQuery("$include", include);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<CertificateModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CertificateModel>>(){})).call();
    }

    /**
     * List all certificates for a company
     * 
     * List all certificates recorded by a company
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * You can use the `$include` parameter to fetch the following additional objects for expansion:
     *             
     * * customers - Retrieves the list of customers linked to the certificate.
     * * po_numbers - Retrieves all PO numbers tied to the certificate.
     * * attributes - Retrieves all attributes applied to the certificate.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The ID number of the company to search
     * @param include OPTIONAL: A comma separated list of special fetch options. You can specify one or more of the following:      * customers - Retrieves the list of customers linked to the certificate.   * po_numbers - Retrieves all PO numbers tied to the certificate.   * attributes - Retrieves all attributes applied to the certificate.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* status, pdf, pages
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CertificateModel>
     */
    public Future<FetchResult<CertificateModel>> queryCertificatesAsync(Integer companyId, String include, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates");
        path.applyField("companyId", companyId);
        path.addQuery("$include", include);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<CertificateModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CertificateModel>>(){}));
    }

    /**
     * Request setup of exemption certificates for this company.
     * 
     * Requests the setup of exemption certificates for this company.
     *             
     * Exemption certificates are tracked through a different auditable data store than the one that
     * holds AvaTax transactions.  To use the AvaTax exemption certificate document store, please call
     * `GetCertificateSetup` to see if your company is configured to use the exemption certificate
     * document store.  To request setup, please call `RequestCertificateSetup` and your company will
     * be configured with data storage in the auditable certificate system.
     *             
     * This API will return the current status of exemption certificate setup for this company.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId 
     * @return ProvisionStatusModel
     */
    public ProvisionStatusModel requestCertificateSetup(Integer companyId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/setup");
        path.applyField("companyId", companyId);
        return ((RestCall<ProvisionStatusModel>)restCallFactory.createRestCall("post", path, null, new TypeToken<ProvisionStatusModel>(){})).call();
    }

    /**
     * Request setup of exemption certificates for this company.
     * 
     * Requests the setup of exemption certificates for this company.
     *             
     * Exemption certificates are tracked through a different auditable data store than the one that
     * holds AvaTax transactions.  To use the AvaTax exemption certificate document store, please call
     * `GetCertificateSetup` to see if your company is configured to use the exemption certificate
     * document store.  To request setup, please call `RequestCertificateSetup` and your company will
     * be configured with data storage in the auditable certificate system.
     *             
     * This API will return the current status of exemption certificate setup for this company.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId 
     * @return ProvisionStatusModel
     */
    public Future<ProvisionStatusModel> requestCertificateSetupAsync(Integer companyId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/setup");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ProvisionStatusModel>)restCallFactory.createRestCall("post", path, null, new TypeToken<ProvisionStatusModel>(){}));
    }

    /**
     * Unlink attributes from a certificate
     * 
     * Unlink one or many attributes from a certificate.
     *             
     * A certificate may have multiple attributes that control its behavior.  You may link or unlink attributes to a
     * certificate at any time.  The full list of defined attributes may be found using `ListCertificateAttributes`.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param model The list of attributes to unlink from this certificate.
     * @return FetchResult<CertificateAttributeModel>
     */
    public FetchResult<CertificateAttributeModel> unlinkAttributesFromCertificate(Integer companyId, Integer id, ArrayList<CertificateAttributeModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}/attributes/unlink");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FetchResult<CertificateAttributeModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<CertificateAttributeModel>>(){})).call();
    }

    /**
     * Unlink attributes from a certificate
     * 
     * Unlink one or many attributes from a certificate.
     *             
     * A certificate may have multiple attributes that control its behavior.  You may link or unlink attributes to a
     * certificate at any time.  The full list of defined attributes may be found using `ListCertificateAttributes`.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param model The list of attributes to unlink from this certificate.
     * @return FetchResult<CertificateAttributeModel>
     */
    public Future<FetchResult<CertificateAttributeModel>> unlinkAttributesFromCertificateAsync(Integer companyId, Integer id, ArrayList<CertificateAttributeModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}/attributes/unlink");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FetchResult<CertificateAttributeModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<CertificateAttributeModel>>(){}));
    }

    /**
     * Unlink customers from a certificate
     * 
     * Unlinks one or more customers from a certificate.
     *             
     * Unlinking a certificate from a customer will prevent the certificate from being used to generate
     * tax exemptions for the customer in the future.  If any previous transactions for this customer had
     * used this linked certificate, those transactions will be unchanged and will still have a link to the
     * exemption certificate in question.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param model The list of customers to unlink from this certificate
     * @return FetchResult<CustomerModel>
     */
    public FetchResult<CustomerModel> unlinkCustomersFromCertificate(Integer companyId, Integer id, LinkCustomersModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}/customers/unlink");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FetchResult<CustomerModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<CustomerModel>>(){})).call();
    }

    /**
     * Unlink customers from a certificate
     * 
     * Unlinks one or more customers from a certificate.
     *             
     * Unlinking a certificate from a customer will prevent the certificate from being used to generate
     * tax exemptions for the customer in the future.  If any previous transactions for this customer had
     * used this linked certificate, those transactions will be unchanged and will still have a link to the
     * exemption certificate in question.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param model The list of customers to unlink from this certificate
     * @return FetchResult<CustomerModel>
     */
    public Future<FetchResult<CustomerModel>> unlinkCustomersFromCertificateAsync(Integer companyId, Integer id, LinkCustomersModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}/customers/unlink");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FetchResult<CustomerModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<CustomerModel>>(){}));
    }

    /**
     * Update a single certificate
     * 
     * Replace the certificate identified by this URL with a new one.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param model The new certificate object that will replace the existing one
     * @return CertificateModel
     */
    public CertificateModel updateCertificate(Integer companyId, Integer id, CertificateModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<CertificateModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<CertificateModel>(){})).call();
    }

    /**
     * Update a single certificate
     * 
     * Replace the certificate identified by this URL with a new one.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param model The new certificate object that will replace the existing one
     * @return CertificateModel
     */
    public Future<CertificateModel> updateCertificateAsync(Integer companyId, Integer id, CertificateModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<CertificateModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<CertificateModel>(){}));
    }

    /**
     * Upload an image or PDF attachment for this certificate
     * 
     * Upload an image or PDF attachment for this certificate.
     *             
     * Image attachments can be of the format `PDF`, `JPEG`, `TIFF`, or `PNG`.  To upload a multi-page image, please
     * use the `PDF` data type.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param file The exemption certificate file you wanted to upload. Accepted formats are: PDF, JPEG, TIFF, PNG.
     * @return String
     */
    public String uploadCertificateImage(Integer companyId, Integer id, String file) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}/attachment");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<String>)restCallFactory.createRestCall("post", path, null, new TypeToken<String>(){})).call();
    }

    /**
     * Upload an image or PDF attachment for this certificate
     * 
     * Upload an image or PDF attachment for this certificate.
     *             
     * Image attachments can be of the format `PDF`, `JPEG`, `TIFF`, or `PNG`.  To upload a multi-page image, please
     * use the `PDF` data type.
     *             
     * A certificate is a document stored in either AvaTax Exemptions or CertCapture.  The certificate document
     * can contain information about a customer's eligibility for exemption from sales or use taxes based on
     * criteria you specify when you store the certificate.  To view or manage your certificates directly, please
     * log onto the administrative website for the product you purchased.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this certificate
     * @param id The unique ID number of this certificate
     * @param file The exemption certificate file you wanted to upload. Accepted formats are: PDF, JPEG, TIFF, PNG.
     * @return String
     */
    public Future<String> uploadCertificateImageAsync(Integer companyId, Integer id, String file) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/certificates/{id}/attachment");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<String>)restCallFactory.createRestCall("post", path, null, new TypeToken<String>(){}));
    }

    /**
     * Change the filing status of this company
     * 
     * Changes the current filing status of this company.
     *             
     * For customers using Avalara's Managed Returns Service, each company within their account can request
     * for Avalara to file tax returns on their behalf.  Avalara compliance team members will review all
     * requested filing calendars prior to beginning filing tax returns on behalf of this company.
     *             
     * The following changes may be requested through this API:
     *             
     * * If a company is in `NotYetFiling` status, the customer may request this be changed to `FilingRequested`.
     * * Avalara compliance team members may change a company from `FilingRequested` to `FirstFiling`.
     * * Avalara compliance team members may change a company from `FirstFiling` to `Active`.
     *             
     * All other status changes must be requested through the Avalara customer support team.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param id 
     * @param model 
     * @return String
     */
    public String changeFilingStatus(Integer id, FilingStatusChangeModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}/filingstatus");
        path.applyField("id", id);
        return ((RestCall<String>)restCallFactory.createRestCall("post", path, model, new TypeToken<String>(){})).call();
    }

    /**
     * Change the filing status of this company
     * 
     * Changes the current filing status of this company.
     *             
     * For customers using Avalara's Managed Returns Service, each company within their account can request
     * for Avalara to file tax returns on their behalf.  Avalara compliance team members will review all
     * requested filing calendars prior to beginning filing tax returns on behalf of this company.
     *             
     * The following changes may be requested through this API:
     *             
     * * If a company is in `NotYetFiling` status, the customer may request this be changed to `FilingRequested`.
     * * Avalara compliance team members may change a company from `FilingRequested` to `FirstFiling`.
     * * Avalara compliance team members may change a company from `FirstFiling` to `Active`.
     *             
     * All other status changes must be requested through the Avalara customer support team.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param id 
     * @param model 
     * @return String
     */
    public Future<String> changeFilingStatusAsync(Integer id, FilingStatusChangeModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}/filingstatus");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<String>)restCallFactory.createRestCall("post", path, model, new TypeToken<String>(){}));
    }

    /**
     * Quick setup for a company with a single physical address
     * 
     * Shortcut to quickly setup a single-physical-location company with critical information and activate it.
     * This API provides quick and simple company setup functionality and does the following things:
     *             
     * * Create a company object with its own tax profile
     * * Add a key contact person for the company
     * * Set up one physical location for the main office
     * * Declare nexus in all taxing jurisdictions for that main office address
     * * Activate the company
     *             
     * This API only provides a limited subset of functionality compared to the 'Create Company' API call.
     * If you need additional features or options not present in this 'Quick Setup' API call, please use the full 'Create Company' call instead.
     * Please allow 1 minute before making transactions using the company.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param model Information about the company you wish to create.
     * @return CompanyModel
     */
    public CompanyModel companyInitialize(CompanyInitializationModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/initialize");
        return ((RestCall<CompanyModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<CompanyModel>(){})).call();
    }

    /**
     * Quick setup for a company with a single physical address
     * 
     * Shortcut to quickly setup a single-physical-location company with critical information and activate it.
     * This API provides quick and simple company setup functionality and does the following things:
     *             
     * * Create a company object with its own tax profile
     * * Add a key contact person for the company
     * * Set up one physical location for the main office
     * * Declare nexus in all taxing jurisdictions for that main office address
     * * Activate the company
     *             
     * This API only provides a limited subset of functionality compared to the 'Create Company' API call.
     * If you need additional features or options not present in this 'Quick Setup' API call, please use the full 'Create Company' call instead.
     * Please allow 1 minute before making transactions using the company.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param model Information about the company you wish to create.
     * @return CompanyModel
     */
    public Future<CompanyModel> companyInitializeAsync(CompanyInitializationModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/initialize");
        return this.threadPool.submit((RestCall<CompanyModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<CompanyModel>(){}));
    }

    /**
     * Create new companies
     * 
     * Create one or more new company objects.
     * A 'company' represents a single corporation or individual that is registered to handle transactional taxes.
     * You may attach nested data objects such as contacts, locations, and nexus with this CREATE call, and those objects will be created with the company.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param model Either a single company object or an array of companies to create
     * @return ArrayList<CompanyModel>
     */
    public ArrayList<CompanyModel> createCompanies(ArrayList<CompanyModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies");
        return ((RestCall<ArrayList<CompanyModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<CompanyModel>>(){})).call();
    }

    /**
     * Create new companies
     * 
     * Create one or more new company objects.
     * A 'company' represents a single corporation or individual that is registered to handle transactional taxes.
     * You may attach nested data objects such as contacts, locations, and nexus with this CREATE call, and those objects will be created with the company.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param model Either a single company object or an array of companies to create
     * @return ArrayList<CompanyModel>
     */
    public Future<ArrayList<CompanyModel>> createCompaniesAsync(ArrayList<CompanyModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies");
        return this.threadPool.submit((RestCall<ArrayList<CompanyModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<CompanyModel>>(){}));
    }

    /**
     * Request managed returns funding setup for a company
     * 
     * This API is available by invitation only.
     * Companies that use the Avalara Managed Returns or the SST Certified Service Provider services are
     * required to setup their funding configuration before Avalara can begin filing tax returns on their
     * behalf.
     * Funding configuration for each company is set up by submitting a funding setup request, which can
     * be sent either via email or via an embedded HTML widget.
     * When the funding configuration is submitted to Avalara, it will be reviewed by treasury team members
     * before approval.
     * This API records that an ambedded HTML funding setup widget was activated.
     * This API requires a subscription to Avalara Managed Returns or SST Certified Service Provider.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param id The unique identifier of the company
     * @param model The funding initialization request
     * @return FundingStatusModel
     */
    public FundingStatusModel createFundingRequest(Integer id, FundingInitiateModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}/funding/setup");
        path.applyField("id", id);
        return ((RestCall<FundingStatusModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FundingStatusModel>(){})).call();
    }

    /**
     * Request managed returns funding setup for a company
     * 
     * This API is available by invitation only.
     * Companies that use the Avalara Managed Returns or the SST Certified Service Provider services are
     * required to setup their funding configuration before Avalara can begin filing tax returns on their
     * behalf.
     * Funding configuration for each company is set up by submitting a funding setup request, which can
     * be sent either via email or via an embedded HTML widget.
     * When the funding configuration is submitted to Avalara, it will be reviewed by treasury team members
     * before approval.
     * This API records that an ambedded HTML funding setup widget was activated.
     * This API requires a subscription to Avalara Managed Returns or SST Certified Service Provider.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param id The unique identifier of the company
     * @param model The funding initialization request
     * @return FundingStatusModel
     */
    public Future<FundingStatusModel> createFundingRequestAsync(Integer id, FundingInitiateModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}/funding/setup");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FundingStatusModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FundingStatusModel>(){}));
    }

    /**
     * Delete a single company
     * 
     * Deleting a company will delete all child companies, and all users attached to this company.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param id The ID of the company you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteCompany(Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}");
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a single company
     * 
     * Deleting a company will delete all child companies, and all users attached to this company.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param id The ID of the company you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteCompanyAsync(Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Check the funding configuration of a company
     * 
     * This API is available by invitation only.
     * Requires a subscription to Avalara Managed Returns or SST Certified Service Provider.
     * Returns the funding configuration of the requested company.
     * .
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique identifier of the company
     * @return FundingConfigurationModel
     */
    public FundingConfigurationModel fundingConfigurationByCompany(Integer companyId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/funding/configuration");
        path.applyField("companyId", companyId);
        return ((RestCall<FundingConfigurationModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FundingConfigurationModel>(){})).call();
    }

    /**
     * Check the funding configuration of a company
     * 
     * This API is available by invitation only.
     * Requires a subscription to Avalara Managed Returns or SST Certified Service Provider.
     * Returns the funding configuration of the requested company.
     * .
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique identifier of the company
     * @return FundingConfigurationModel
     */
    public Future<FundingConfigurationModel> fundingConfigurationByCompanyAsync(Integer companyId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/funding/configuration");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<FundingConfigurationModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FundingConfigurationModel>(){}));
    }

    /**
     * Check the funding configuration of a company
     * 
     * This API is available by invitation only.
     * Requires a subscription to Avalara Managed Returns or SST Certified Service Provider.
     * Returns the funding configuration of the requested company.
     * .
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique identifier of the company
     * @param currency The currency of the funding. USD and CAD are the only valid currencies
     * @return ArrayList<FundingConfigurationModel>
     */
    public ArrayList<FundingConfigurationModel> fundingConfigurationsByCompanyAndCurrency(Integer companyId, String currency) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/funding/configurations");
        path.applyField("companyId", companyId);
        path.addQuery("currency", currency);
        return ((RestCall<ArrayList<FundingConfigurationModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<FundingConfigurationModel>>(){})).call();
    }

    /**
     * Check the funding configuration of a company
     * 
     * This API is available by invitation only.
     * Requires a subscription to Avalara Managed Returns or SST Certified Service Provider.
     * Returns the funding configuration of the requested company.
     * .
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique identifier of the company
     * @param currency The currency of the funding. USD and CAD are the only valid currencies
     * @return ArrayList<FundingConfigurationModel>
     */
    public Future<ArrayList<FundingConfigurationModel>> fundingConfigurationsByCompanyAndCurrencyAsync(Integer companyId, String currency) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/funding/configurations");
        path.applyField("companyId", companyId);
        path.addQuery("currency", currency);
        return this.threadPool.submit((RestCall<ArrayList<FundingConfigurationModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<FundingConfigurationModel>>(){}));
    }

    /**
     * Retrieve a single company
     * 
     * Get the company object identified by this URL.
     * A 'company' represents a single corporation or individual that is registered to handle transactional taxes.
     * You may specify one or more of the following values in the '$include' parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     *  * Contacts
     *  * Items
     *  * Locations
     *  * Nexus
     *  * Settings
     *  * TaxCodes
     *  * TaxRules
     *  * UPC
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param id The ID of the company to retrieve.
     * @param include OPTIONAL: A comma separated list of special fetch options.      * Child objects - Specify one or more of the following to retrieve objects related to each company: "Contacts", "FilingCalendars", "Items", "Locations", "Nexus", "TaxCodes", "NonReportingChildren" or "TaxRules".   * Deleted objects - Specify "FetchDeleted" to retrieve information about previously deleted objects.
     * @return CompanyModel
     */
    public CompanyModel getCompany(Integer id, String include) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}");
        path.applyField("id", id);
        path.addQuery("$include", include);
        return ((RestCall<CompanyModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<CompanyModel>(){})).call();
    }

    /**
     * Retrieve a single company
     * 
     * Get the company object identified by this URL.
     * A 'company' represents a single corporation or individual that is registered to handle transactional taxes.
     * You may specify one or more of the following values in the '$include' parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     *  * Contacts
     *  * Items
     *  * Locations
     *  * Nexus
     *  * Settings
     *  * TaxCodes
     *  * TaxRules
     *  * UPC
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param id The ID of the company to retrieve.
     * @param include OPTIONAL: A comma separated list of special fetch options.      * Child objects - Specify one or more of the following to retrieve objects related to each company: "Contacts", "FilingCalendars", "Items", "Locations", "Nexus", "TaxCodes", "NonReportingChildren" or "TaxRules".   * Deleted objects - Specify "FetchDeleted" to retrieve information about previously deleted objects.
     * @return CompanyModel
     */
    public Future<CompanyModel> getCompanyAsync(Integer id, String include) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}");
        path.applyField("id", id);
        path.addQuery("$include", include);
        return this.threadPool.submit((RestCall<CompanyModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<CompanyModel>(){}));
    }

    /**
     * Get configuration settings for this company
     * 
     * Retrieve a list of all configuration settings tied to this company.
     *             
     * Configuration settings provide you with the ability to control features of your account and of your
     * tax software.  The category name `AvaCertServiceConfig` is reserved for
     * Avalara internal software configuration values; to store your own company-level settings, please
     * create a new category name that begins with `X-`, for example, `X-MyCustomCategory`.
     *             
     * Company settings are permanent settings that cannot be deleted.  You can set the value of a
     * company setting to null if desired and if the particular setting supports it.
     *             
     * Avalara-based company settings for `AvaCertServiceConfig` affect your company's exemption certificate
     * processing, and should be changed with care.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param id 
     * @return ArrayList<CompanyConfigurationModel>
     */
    public ArrayList<CompanyConfigurationModel> getCompanyConfiguration(Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}/configuration");
        path.applyField("id", id);
        return ((RestCall<ArrayList<CompanyConfigurationModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<CompanyConfigurationModel>>(){})).call();
    }

    /**
     * Get configuration settings for this company
     * 
     * Retrieve a list of all configuration settings tied to this company.
     *             
     * Configuration settings provide you with the ability to control features of your account and of your
     * tax software.  The category name `AvaCertServiceConfig` is reserved for
     * Avalara internal software configuration values; to store your own company-level settings, please
     * create a new category name that begins with `X-`, for example, `X-MyCustomCategory`.
     *             
     * Company settings are permanent settings that cannot be deleted.  You can set the value of a
     * company setting to null if desired and if the particular setting supports it.
     *             
     * Avalara-based company settings for `AvaCertServiceConfig` affect your company's exemption certificate
     * processing, and should be changed with care.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param id 
     * @return ArrayList<CompanyConfigurationModel>
     */
    public Future<ArrayList<CompanyConfigurationModel>> getCompanyConfigurationAsync(Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}/configuration");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<CompanyConfigurationModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<CompanyConfigurationModel>>(){}));
    }

    /**
     * Get this company's filing status
     * 
     * Retrieve the current filing status of this company.
     *             
     * For customers using Avalara's Managed Returns Service, each company within their account can request
     * for Avalara to file tax returns on their behalf.  Avalara compliance team members will review all
     * requested filing calendars prior to beginning filing tax returns on behalf of this company.
     *             
     * A company's filing status can be one of the following values:
     *             
     * * `NoReporting` - This company is not configured to report tax returns; instead, it reports through a parent company.
     * * `NotYetFiling` - This company has not yet begun filing tax returns through Avalara's Managed Returns Service.
     * * `FilingRequested` - The company has requested to begin filing tax returns, but Avalara's compliance team has not yet begun filing.
     * * `FirstFiling` - The company has recently filing tax returns and is in a new status.
     * * `Active` - The company is currently active and is filing tax returns via Avalara Managed Returns.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param id 
     * @return String
     */
    public String getFilingStatus(Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}/filingstatus");
        path.applyField("id", id);
        return ((RestCall<String>)restCallFactory.createRestCall("get", path, null, new TypeToken<String>(){})).call();
    }

    /**
     * Get this company's filing status
     * 
     * Retrieve the current filing status of this company.
     *             
     * For customers using Avalara's Managed Returns Service, each company within their account can request
     * for Avalara to file tax returns on their behalf.  Avalara compliance team members will review all
     * requested filing calendars prior to beginning filing tax returns on behalf of this company.
     *             
     * A company's filing status can be one of the following values:
     *             
     * * `NoReporting` - This company is not configured to report tax returns; instead, it reports through a parent company.
     * * `NotYetFiling` - This company has not yet begun filing tax returns through Avalara's Managed Returns Service.
     * * `FilingRequested` - The company has requested to begin filing tax returns, but Avalara's compliance team has not yet begun filing.
     * * `FirstFiling` - The company has recently filing tax returns and is in a new status.
     * * `Active` - The company is currently active and is filing tax returns via Avalara Managed Returns.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param id 
     * @return String
     */
    public Future<String> getFilingStatusAsync(Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}/filingstatus");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<String>)restCallFactory.createRestCall("get", path, null, new TypeToken<String>(){}));
    }

    /**
     * Check managed returns funding status for a company
     * 
     * This API is available by invitation only.
     * Requires a subscription to Avalara Managed Returns or SST Certified Service Provider.
     * Returns a list of funding setup requests and their current status.
     * Each object in the result is a request that was made to setup or adjust funding status for this company.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param id The unique identifier of the company
     * @return ArrayList<FundingStatusModel>
     */
    public ArrayList<FundingStatusModel> listFundingRequestsByCompany(Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}/funding");
        path.applyField("id", id);
        return ((RestCall<ArrayList<FundingStatusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<FundingStatusModel>>(){})).call();
    }

    /**
     * Check managed returns funding status for a company
     * 
     * This API is available by invitation only.
     * Requires a subscription to Avalara Managed Returns or SST Certified Service Provider.
     * Returns a list of funding setup requests and their current status.
     * Each object in the result is a request that was made to setup or adjust funding status for this company.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param id The unique identifier of the company
     * @return ArrayList<FundingStatusModel>
     */
    public Future<ArrayList<FundingStatusModel>> listFundingRequestsByCompanyAsync(Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}/funding");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<FundingStatusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<FundingStatusModel>>(){}));
    }

    /**
     * Retrieve a list of MRS Companies with account
     * 
     * This API is available by invitation only.
     *             
     * Get a list of companies with an active MRS service.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @return FetchResult<MrsCompanyModel>
     */
    public FetchResult<MrsCompanyModel> listMrsCompanies() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/mrs");
        return ((RestCall<FetchResult<MrsCompanyModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<MrsCompanyModel>>(){})).call();
    }

    /**
     * Retrieve a list of MRS Companies with account
     * 
     * This API is available by invitation only.
     *             
     * Get a list of companies with an active MRS service.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @return FetchResult<MrsCompanyModel>
     */
    public Future<FetchResult<MrsCompanyModel>> listMrsCompaniesAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/mrs");
        return this.threadPool.submit((RestCall<FetchResult<MrsCompanyModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<MrsCompanyModel>>(){}));
    }

    /**
     * Retrieve all companies
     * 
     * Get multiple company objects.
     *             
     * A `company` represents a single corporation or individual that is registered to handle transactional taxes.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * Contacts
     * * Items
     * * Locations
     * * Nexus
     * * Settings
     * * TaxCodes
     * * TaxRules
     * * UPC
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param include A comma separated list of objects to fetch underneath this company. Any object with a URL path underneath this company can be fetched by specifying its name.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* contacts, items, locations, nexus, settings, taxCodes, taxRules, upcs, nonReportingChildCompanies, exemptCerts
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CompanyModel>
     */
    public FetchResult<CompanyModel> queryCompanies(String include, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies");
        path.addQuery("$include", include);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<CompanyModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CompanyModel>>(){})).call();
    }

    /**
     * Retrieve all companies
     * 
     * Get multiple company objects.
     *             
     * A `company` represents a single corporation or individual that is registered to handle transactional taxes.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * Contacts
     * * Items
     * * Locations
     * * Nexus
     * * Settings
     * * TaxCodes
     * * TaxRules
     * * UPC
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param include A comma separated list of objects to fetch underneath this company. Any object with a URL path underneath this company can be fetched by specifying its name.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* contacts, items, locations, nexus, settings, taxCodes, taxRules, upcs, nonReportingChildCompanies, exemptCerts
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CompanyModel>
     */
    public Future<FetchResult<CompanyModel>> queryCompaniesAsync(String include, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies");
        path.addQuery("$include", include);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<CompanyModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CompanyModel>>(){}));
    }

    /**
     * Change configuration settings for this company
     * 
     * Update configuration settings tied to this company.
     *             
     * Configuration settings provide you with the ability to control features of your account and of your
     * tax software.  The category names `AvaCertServiceConfig` is reserved for
     * Avalara internal software configuration values; to store your own company-level settings, please
     * create a new category name that begins with `X-`, for example, `X-MyCustomCategory`.
     *             
     * Company settings are permanent settings that cannot be deleted.  You can set the value of a
     * company setting to null if desired and if the particular setting supports it.
     *             
     * Avalara-based company settings for `AvaCertServiceConfig` affect your company's exemption certificate
     * processing, and should be changed with care.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param id 
     * @param model 
     * @return ArrayList<CompanyConfigurationModel>
     */
    public ArrayList<CompanyConfigurationModel> setCompanyConfiguration(Integer id, ArrayList<CompanyConfigurationModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}/configuration");
        path.applyField("id", id);
        return ((RestCall<ArrayList<CompanyConfigurationModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<CompanyConfigurationModel>>(){})).call();
    }

    /**
     * Change configuration settings for this company
     * 
     * Update configuration settings tied to this company.
     *             
     * Configuration settings provide you with the ability to control features of your account and of your
     * tax software.  The category names `AvaCertServiceConfig` is reserved for
     * Avalara internal software configuration values; to store your own company-level settings, please
     * create a new category name that begins with `X-`, for example, `X-MyCustomCategory`.
     *             
     * Company settings are permanent settings that cannot be deleted.  You can set the value of a
     * company setting to null if desired and if the particular setting supports it.
     *             
     * Avalara-based company settings for `AvaCertServiceConfig` affect your company's exemption certificate
     * processing, and should be changed with care.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param id 
     * @param model 
     * @return ArrayList<CompanyConfigurationModel>
     */
    public Future<ArrayList<CompanyConfigurationModel>> setCompanyConfigurationAsync(Integer id, ArrayList<CompanyConfigurationModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}/configuration");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<CompanyConfigurationModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<CompanyConfigurationModel>>(){}));
    }

    /**
     * Update a single company
     * 
     * Replace the existing company object at this URL with an updated object.
     *             
     * A `CompanyModel` represents a single corporation or individual that is registered to handle transactional taxes.
     * All data from the existing object will be replaced with data in the object you PUT.
     *             
     * When calling `UpdateCompany`, you are permitted to update the company itself.  Updates to the nested objects
     * such as contacts, locations, or settings are not permitted.  To update the nested objects
     *             
     * To set a field's value to `null`, you may either set its value to `null` or omit that field from the object you PUT.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param id The ID of the company you wish to update.
     * @param model The company object you wish to update.
     * @return CompanyModel
     */
    public CompanyModel updateCompany(Integer id, CompanyModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}");
        path.applyField("id", id);
        return ((RestCall<CompanyModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<CompanyModel>(){})).call();
    }

    /**
     * Update a single company
     * 
     * Replace the existing company object at this URL with an updated object.
     *             
     * A `CompanyModel` represents a single corporation or individual that is registered to handle transactional taxes.
     * All data from the existing object will be replaced with data in the object you PUT.
     *             
     * When calling `UpdateCompany`, you are permitted to update the company itself.  Updates to the nested objects
     * such as contacts, locations, or settings are not permitted.  To update the nested objects
     *             
     * To set a field's value to `null`, you may either set its value to `null` or omit that field from the object you PUT.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param id The ID of the company you wish to update.
     * @param model The company object you wish to update.
     * @return CompanyModel
     */
    public Future<CompanyModel> updateCompanyAsync(Integer id, CompanyModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<CompanyModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<CompanyModel>(){}));
    }

    /**
     * Create a new contact
     * 
     * Create one or more new contact objects.
     * A 'contact' is a person associated with a company who is designated to handle certain responsibilities of
     * a tax collecting and filing entity.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this contact.
     * @param model The contacts you wish to create.
     * @return ArrayList<ContactModel>
     */
    public ArrayList<ContactModel> createContacts(Integer companyId, ArrayList<ContactModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<ContactModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<ContactModel>>(){})).call();
    }

    /**
     * Create a new contact
     * 
     * Create one or more new contact objects.
     * A 'contact' is a person associated with a company who is designated to handle certain responsibilities of
     * a tax collecting and filing entity.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this contact.
     * @param model The contacts you wish to create.
     * @return ArrayList<ContactModel>
     */
    public Future<ArrayList<ContactModel>> createContactsAsync(Integer companyId, ArrayList<ContactModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<ContactModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<ContactModel>>(){}));
    }

    /**
     * Delete a single contact
     * 
     * Mark the existing contact object at this URL as deleted.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this contact.
     * @param id The ID of the contact you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteContact(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a single contact
     * 
     * Mark the existing contact object at this URL as deleted.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this contact.
     * @param id The ID of the contact you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteContactAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Retrieve a single contact
     * 
     * Get the contact object identified by this URL.
     * A 'contact' is a person associated with a company who is designated to handle certain responsibilities of
     * a tax collecting and filing entity.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param companyId The ID of the company for this contact
     * @param id The primary key of this contact
     * @return ContactModel
     */
    public ContactModel getContact(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ContactModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<ContactModel>(){})).call();
    }

    /**
     * Retrieve a single contact
     * 
     * Get the contact object identified by this URL.
     * A 'contact' is a person associated with a company who is designated to handle certain responsibilities of
     * a tax collecting and filing entity.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param companyId The ID of the company for this contact
     * @param id The primary key of this contact
     * @return ContactModel
     */
    public Future<ContactModel> getContactAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ContactModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<ContactModel>(){}));
    }

    /**
     * Retrieve contacts for this company
     * 
     * List all contact objects assigned to this company.
     * 
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param companyId The ID of the company that owns these contacts
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ContactModel>
     */
    public FetchResult<ContactModel> listContactsByCompany(Integer companyId, String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<ContactModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ContactModel>>(){})).call();
    }

    /**
     * Retrieve contacts for this company
     * 
     * List all contact objects assigned to this company.
     * 
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param companyId The ID of the company that owns these contacts
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ContactModel>
     */
    public Future<FetchResult<ContactModel>> listContactsByCompanyAsync(Integer companyId, String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<ContactModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ContactModel>>(){}));
    }

    /**
     * Retrieve all contacts
     * 
     * Get multiple contact objects across all companies.
     * A 'contact' is a person associated with a company who is designated to handle certain responsibilities of
     * a tax collecting and filing entity.
     * 
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ContactModel>
     */
    public FetchResult<ContactModel> queryContacts(String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/contacts");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<ContactModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ContactModel>>(){})).call();
    }

    /**
     * Retrieve all contacts
     * 
     * Get multiple contact objects across all companies.
     * A 'contact' is a person associated with a company who is designated to handle certain responsibilities of
     * a tax collecting and filing entity.
     * 
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ContactModel>
     */
    public Future<FetchResult<ContactModel>> queryContactsAsync(String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/contacts");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<ContactModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ContactModel>>(){}));
    }

    /**
     * Update a single contact
     * 
     * Replace the existing contact object at this URL with an updated object.
     * A 'contact' is a person associated with a company who is designated to handle certain responsibilities of
     * a tax collecting and filing entity.
     * All data from the existing object will be replaced with data in the object you PUT.  
     * To set a field's value to null, you may either set its value to null or omit that field from the object you post.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that this contact belongs to.
     * @param id The ID of the contact you wish to update
     * @param model The contact you wish to update.
     * @return ContactModel
     */
    public ContactModel updateContact(Integer companyId, Integer id, ContactModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ContactModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<ContactModel>(){})).call();
    }

    /**
     * Update a single contact
     * 
     * Replace the existing contact object at this URL with an updated object.
     * A 'contact' is a person associated with a company who is designated to handle certain responsibilities of
     * a tax collecting and filing entity.
     * All data from the existing object will be replaced with data in the object you PUT.  
     * To set a field's value to null, you may either set its value to null or omit that field from the object you post.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that this contact belongs to.
     * @param id The ID of the contact you wish to update
     * @param model The contact you wish to update.
     * @return ContactModel
     */
    public Future<ContactModel> updateContactAsync(Integer companyId, Integer id, ContactModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ContactModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<ContactModel>(){}));
    }

    /**
     * Create customers for this company
     * 
     * Create one or more customers for this company.
     *             
     * A customer object defines information about a person or business that purchases products from your
     * company.  When you create a tax transaction in AvaTax, you can use the `customerCode` from this
     * record in your `CreateTransaction` API call.  AvaTax will search for this `customerCode` value and
     * identify any certificates linked to this `customer` object.  If any certificate applies to the transaction,
     * AvaTax will record the appropriate elements of the transaction as exempt and link it to the `certificate`.
     *             
     * A nested object such as CustomFields could be specified and created along with the customer object. To fetch the
     * nested object, please call 'GetCustomer' API with appropriate $include parameters.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param model The list of customer objects to be created
     * @return ArrayList<CustomerModel>
     */
    public ArrayList<CustomerModel> createCustomers(Integer companyId, ArrayList<CustomerModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<CustomerModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<CustomerModel>>(){})).call();
    }

    /**
     * Create customers for this company
     * 
     * Create one or more customers for this company.
     *             
     * A customer object defines information about a person or business that purchases products from your
     * company.  When you create a tax transaction in AvaTax, you can use the `customerCode` from this
     * record in your `CreateTransaction` API call.  AvaTax will search for this `customerCode` value and
     * identify any certificates linked to this `customer` object.  If any certificate applies to the transaction,
     * AvaTax will record the appropriate elements of the transaction as exempt and link it to the `certificate`.
     *             
     * A nested object such as CustomFields could be specified and created along with the customer object. To fetch the
     * nested object, please call 'GetCustomer' API with appropriate $include parameters.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param model The list of customer objects to be created
     * @return ArrayList<CustomerModel>
     */
    public Future<ArrayList<CustomerModel>> createCustomersAsync(Integer companyId, ArrayList<CustomerModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<CustomerModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<CustomerModel>>(){}));
    }

    /**
     * Delete a customer record
     * 
     * Deletes the customer object referenced by this URL.
     *             
     * A customer object defines information about a person or business that purchases products from your
     * company.  When you create a tax transaction in AvaTax, you can use the `customerCode` from this
     * record in your `CreateTransaction` API call.  AvaTax will search for this `customerCode` value and
     * identify any certificates linked to this `customer` object.  If any certificate applies to the transaction,
     * AvaTax will record the appropriate elements of the transaction as exempt and link it to the `certificate`.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param customerCode The unique code representing this customer
     * @return CustomerModel
     */
    public CustomerModel deleteCustomer(Integer companyId, String customerCode) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        return ((RestCall<CustomerModel>)restCallFactory.createRestCall("delete", path, null, new TypeToken<CustomerModel>(){})).call();
    }

    /**
     * Delete a customer record
     * 
     * Deletes the customer object referenced by this URL.
     *             
     * A customer object defines information about a person or business that purchases products from your
     * company.  When you create a tax transaction in AvaTax, you can use the `customerCode` from this
     * record in your `CreateTransaction` API call.  AvaTax will search for this `customerCode` value and
     * identify any certificates linked to this `customer` object.  If any certificate applies to the transaction,
     * AvaTax will record the appropriate elements of the transaction as exempt and link it to the `certificate`.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param customerCode The unique code representing this customer
     * @return CustomerModel
     */
    public Future<CustomerModel> deleteCustomerAsync(Integer companyId, String customerCode) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        return this.threadPool.submit((RestCall<CustomerModel>)restCallFactory.createRestCall("delete", path, null, new TypeToken<CustomerModel>(){}));
    }

    /**
     * Retrieve a single customer
     * 
     * Retrieve the customer identified by this URL.
     *             
     * A customer object defines information about a person or business that purchases products from your
     * company.  When you create a tax transaction in AvaTax, you can use the `customerCode` from this
     * record in your `CreateTransaction` API call.  AvaTax will search for this `customerCode` value and
     * identify any certificates linked to this customer object.  If any certificate applies to the transaction,
     * AvaTax will record the appropriate elements of the transaction as exempt and link it to the `certificate`.
     *             
     * You can use the `$include` parameter to fetch the following additional objects for expansion:
     *             
     * * Certificates - Fetch a list of certificates linked to this customer.
     * * CustomFields - Fetch a list of custom fields associated to this customer.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param customerCode The unique code representing this customer
     * @param include Specify optional additional objects to include in this fetch request
     * @return CustomerModel
     */
    public CustomerModel getCustomer(Integer companyId, String customerCode, String include) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        path.addQuery("$include", include);
        return ((RestCall<CustomerModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<CustomerModel>(){})).call();
    }

    /**
     * Retrieve a single customer
     * 
     * Retrieve the customer identified by this URL.
     *             
     * A customer object defines information about a person or business that purchases products from your
     * company.  When you create a tax transaction in AvaTax, you can use the `customerCode` from this
     * record in your `CreateTransaction` API call.  AvaTax will search for this `customerCode` value and
     * identify any certificates linked to this customer object.  If any certificate applies to the transaction,
     * AvaTax will record the appropriate elements of the transaction as exempt and link it to the `certificate`.
     *             
     * You can use the `$include` parameter to fetch the following additional objects for expansion:
     *             
     * * Certificates - Fetch a list of certificates linked to this customer.
     * * CustomFields - Fetch a list of custom fields associated to this customer.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param customerCode The unique code representing this customer
     * @param include Specify optional additional objects to include in this fetch request
     * @return CustomerModel
     */
    public Future<CustomerModel> getCustomerAsync(Integer companyId, String customerCode, String include) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        path.addQuery("$include", include);
        return this.threadPool.submit((RestCall<CustomerModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<CustomerModel>(){}));
    }

    /**
     * Link certificates to a customer
     * 
     * Link one or more certificates to a customer.
     *             
     * A customer object defines information about a person or business that purchases products from your
     * company.  When you create a tax transaction in AvaTax, you can use the `customerCode` from this
     * record in your `CreateTransaction` API call.  AvaTax will search for this `customerCode` value and
     * identify any certificates linked to this `customer` object.  If any certificate applies to the transaction,
     * AvaTax will record the appropriate elements of the transaction as exempt and link it to the `certificate`.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param customerCode The unique code representing this customer
     * @param model The list of certificates to link to this customer
     * @return FetchResult<CertificateModel>
     */
    public FetchResult<CertificateModel> linkCertificatesToCustomer(Integer companyId, String customerCode, LinkCertificatesModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}/certificates/link");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        return ((RestCall<FetchResult<CertificateModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<CertificateModel>>(){})).call();
    }

    /**
     * Link certificates to a customer
     * 
     * Link one or more certificates to a customer.
     *             
     * A customer object defines information about a person or business that purchases products from your
     * company.  When you create a tax transaction in AvaTax, you can use the `customerCode` from this
     * record in your `CreateTransaction` API call.  AvaTax will search for this `customerCode` value and
     * identify any certificates linked to this `customer` object.  If any certificate applies to the transaction,
     * AvaTax will record the appropriate elements of the transaction as exempt and link it to the `certificate`.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param customerCode The unique code representing this customer
     * @param model The list of certificates to link to this customer
     * @return FetchResult<CertificateModel>
     */
    public Future<FetchResult<CertificateModel>> linkCertificatesToCustomerAsync(Integer companyId, String customerCode, LinkCertificatesModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}/certificates/link");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        return this.threadPool.submit((RestCall<FetchResult<CertificateModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<CertificateModel>>(){}));
    }

    /**
     * Link two customer records together
     * 
     * Links a Ship-To customer record with a Bill-To customer record.
     *             
     * Customer records represent businesses or individuals who can provide exemption certificates.  Some customers
     * may have certificates that are linked to their shipping address or their billing address.  To group these
     * customer records together, you may link multiple bill-to and ship-to addresses together to represent a single
     * entity that has multiple different addresses of different kinds.
     *             
     * In general, a customer will have only one primary billing address and multiple ship-to addresses, representing
     * all of the different locations where they receive goods.  To facilitate this type of customer, you can send in
     * one bill-to customer code and multiple ship-to customer codes in a single API call.
     *             
     * Note that you can only link a ship-to customer record to a bill-to customer record.  You may not link two customers
     * of the same kind together.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company defining customers.
     * @param code The code of the bill-to customer to link.
     * @param model A list of information about ship-to customers to link to this bill-to customer.
     * @return CustomerModel
     */
    public CustomerModel linkShipToCustomersToBillCustomer(Integer companyId, String code, LinkCustomersModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/billto/{code}/shipto/link");
        path.applyField("companyId", companyId);
        path.applyField("code", code);
        return ((RestCall<CustomerModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<CustomerModel>(){})).call();
    }

    /**
     * Link two customer records together
     * 
     * Links a Ship-To customer record with a Bill-To customer record.
     *             
     * Customer records represent businesses or individuals who can provide exemption certificates.  Some customers
     * may have certificates that are linked to their shipping address or their billing address.  To group these
     * customer records together, you may link multiple bill-to and ship-to addresses together to represent a single
     * entity that has multiple different addresses of different kinds.
     *             
     * In general, a customer will have only one primary billing address and multiple ship-to addresses, representing
     * all of the different locations where they receive goods.  To facilitate this type of customer, you can send in
     * one bill-to customer code and multiple ship-to customer codes in a single API call.
     *             
     * Note that you can only link a ship-to customer record to a bill-to customer record.  You may not link two customers
     * of the same kind together.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company defining customers.
     * @param code The code of the bill-to customer to link.
     * @param model A list of information about ship-to customers to link to this bill-to customer.
     * @return CustomerModel
     */
    public Future<CustomerModel> linkShipToCustomersToBillCustomerAsync(Integer companyId, String code, LinkCustomersModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/billto/{code}/shipto/link");
        path.applyField("companyId", companyId);
        path.applyField("code", code);
        return this.threadPool.submit((RestCall<CustomerModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<CustomerModel>(){}));
    }

    /**
     * List certificates linked to a customer
     * 
     * List all certificates linked to a customer.
     *             
     * A customer object defines information about a person or business that purchases products from your
     * company.  When you create a tax transaction in AvaTax, you can use the `customerCode` from this
     * record in your `CreateTransaction` API call.  AvaTax will search for this `customerCode` value and
     * identify any certificates linked to this `customer` object.  If any certificate applies to the transaction,
     * AvaTax will record the appropriate elements of the transaction as exempt and link it to the `certificate`.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param customerCode The unique code representing this customer
     * @param include OPTIONAL: A comma separated list of special fetch options. You can specify one or more of the following:      * customers - Retrieves the list of customers linked to the certificate.   * po_numbers - Retrieves all PO numbers tied to the certificate.   * attributes - Retrieves all attributes applied to the certificate.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* status, pdf, pages
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CertificateModel>
     */
    public FetchResult<CertificateModel> listCertificatesForCustomer(Integer companyId, String customerCode, String include, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}/certificates");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        path.addQuery("$include", include);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<CertificateModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CertificateModel>>(){})).call();
    }

    /**
     * List certificates linked to a customer
     * 
     * List all certificates linked to a customer.
     *             
     * A customer object defines information about a person or business that purchases products from your
     * company.  When you create a tax transaction in AvaTax, you can use the `customerCode` from this
     * record in your `CreateTransaction` API call.  AvaTax will search for this `customerCode` value and
     * identify any certificates linked to this `customer` object.  If any certificate applies to the transaction,
     * AvaTax will record the appropriate elements of the transaction as exempt and link it to the `certificate`.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param customerCode The unique code representing this customer
     * @param include OPTIONAL: A comma separated list of special fetch options. You can specify one or more of the following:      * customers - Retrieves the list of customers linked to the certificate.   * po_numbers - Retrieves all PO numbers tied to the certificate.   * attributes - Retrieves all attributes applied to the certificate.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* status, pdf, pages
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CertificateModel>
     */
    public Future<FetchResult<CertificateModel>> listCertificatesForCustomerAsync(Integer companyId, String customerCode, String include, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}/certificates");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        path.addQuery("$include", include);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<CertificateModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CertificateModel>>(){}));
    }

    /**
     * List active certificates for a location
     * 
     * List valid certificates linked to a customer in a particular country and region.
     *             
     * This API is intended to help identify whether a customer has already provided a certificate that
     * applies to a particular country and region.  This API is intended to help you remind a customer
     * when they have or have not provided copies of their exemption certificates to you during the sales
     * order process.
     *             
     * If a customer does not have a certificate on file and they wish to provide one, you should send the customer
     * a CertExpress invitation link so that the customer can upload proof of their exemption certificate.  Please
     * see the `CreateCertExpressInvitation` API to create an invitation link for this customer.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param customerCode The unique code representing this customer
     * @param country Search for certificates matching this country. Uses the ISO 3166 two character country code.
     * @param region Search for certificates matching this region. Uses the ISO 3166 two or three character state, region, or province code.
     * @return ExemptionStatusModel
     */
    public ExemptionStatusModel listValidCertificatesForCustomer(Integer companyId, String customerCode, String country, String region) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}/certificates/{country}/{region}");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        path.applyField("country", country);
        path.applyField("region", region);
        return ((RestCall<ExemptionStatusModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<ExemptionStatusModel>(){})).call();
    }

    /**
     * List active certificates for a location
     * 
     * List valid certificates linked to a customer in a particular country and region.
     *             
     * This API is intended to help identify whether a customer has already provided a certificate that
     * applies to a particular country and region.  This API is intended to help you remind a customer
     * when they have or have not provided copies of their exemption certificates to you during the sales
     * order process.
     *             
     * If a customer does not have a certificate on file and they wish to provide one, you should send the customer
     * a CertExpress invitation link so that the customer can upload proof of their exemption certificate.  Please
     * see the `CreateCertExpressInvitation` API to create an invitation link for this customer.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param customerCode The unique code representing this customer
     * @param country Search for certificates matching this country. Uses the ISO 3166 two character country code.
     * @param region Search for certificates matching this region. Uses the ISO 3166 two or three character state, region, or province code.
     * @return ExemptionStatusModel
     */
    public Future<ExemptionStatusModel> listValidCertificatesForCustomerAsync(Integer companyId, String customerCode, String country, String region) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}/certificates/{country}/{region}");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        path.applyField("country", country);
        path.applyField("region", region);
        return this.threadPool.submit((RestCall<ExemptionStatusModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<ExemptionStatusModel>(){}));
    }

    /**
     * List all customers for this company
     * 
     * List all customers recorded by this company matching the specified criteria.
     *             
     * A customer object defines information about a person or business that purchases products from your
     * company.  When you create a tax transaction in AvaTax, you can use the `customerCode` from this
     * record in your `CreateTransaction` API call.  AvaTax will search for this `customerCode` value and
     * identify any certificates linked to this `customer` object.  If any certificate applies to the transaction,
     * AvaTax will record the appropriate elements of the transaction as exempt and link it to the `certificate`.
     *             
     * You can use the `$include` parameter to fetch the following additional objects for expansion:
     *             
     * * Certificates - Fetch a list of certificates linked to this customer.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param include OPTIONAL - You can specify the value `certificates` to fetch information about certificates linked to the customer.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* shipTos
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CustomerModel>
     */
    public FetchResult<CustomerModel> queryCustomers(Integer companyId, String include, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers");
        path.applyField("companyId", companyId);
        path.addQuery("$include", include);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<CustomerModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CustomerModel>>(){})).call();
    }

    /**
     * List all customers for this company
     * 
     * List all customers recorded by this company matching the specified criteria.
     *             
     * A customer object defines information about a person or business that purchases products from your
     * company.  When you create a tax transaction in AvaTax, you can use the `customerCode` from this
     * record in your `CreateTransaction` API call.  AvaTax will search for this `customerCode` value and
     * identify any certificates linked to this `customer` object.  If any certificate applies to the transaction,
     * AvaTax will record the appropriate elements of the transaction as exempt and link it to the `certificate`.
     *             
     * You can use the `$include` parameter to fetch the following additional objects for expansion:
     *             
     * * Certificates - Fetch a list of certificates linked to this customer.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param include OPTIONAL - You can specify the value `certificates` to fetch information about certificates linked to the customer.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* shipTos
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CustomerModel>
     */
    public Future<FetchResult<CustomerModel>> queryCustomersAsync(Integer companyId, String include, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers");
        path.applyField("companyId", companyId);
        path.addQuery("$include", include);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<CustomerModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CustomerModel>>(){}));
    }

    /**
     * Unlink certificates from a customer
     * 
     * Remove one or more certificates to a customer.
     *             
     * A customer object defines information about a person or business that purchases products from your
     * company.  When you create a tax transaction in AvaTax, you can use the `customerCode` from this
     * record in your `CreateTransaction` API call.  AvaTax will search for this `customerCode` value and
     * identify any certificates linked to this `customer` object.  If any certificate applies to the transaction,
     * AvaTax will record the appropriate elements of the transaction as exempt and link it to the `certificate`.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param customerCode The unique code representing this customer
     * @param model The list of certificates to link to this customer
     * @return FetchResult<CertificateModel>
     */
    public FetchResult<CertificateModel> unlinkCertificatesFromCustomer(Integer companyId, String customerCode, LinkCertificatesModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}/certificates/unlink");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        return ((RestCall<FetchResult<CertificateModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<CertificateModel>>(){})).call();
    }

    /**
     * Unlink certificates from a customer
     * 
     * Remove one or more certificates to a customer.
     *             
     * A customer object defines information about a person or business that purchases products from your
     * company.  When you create a tax transaction in AvaTax, you can use the `customerCode` from this
     * record in your `CreateTransaction` API call.  AvaTax will search for this `customerCode` value and
     * identify any certificates linked to this `customer` object.  If any certificate applies to the transaction,
     * AvaTax will record the appropriate elements of the transaction as exempt and link it to the `certificate`.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param customerCode The unique code representing this customer
     * @param model The list of certificates to link to this customer
     * @return FetchResult<CertificateModel>
     */
    public Future<FetchResult<CertificateModel>> unlinkCertificatesFromCustomerAsync(Integer companyId, String customerCode, LinkCertificatesModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}/certificates/unlink");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        return this.threadPool.submit((RestCall<FetchResult<CertificateModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<CertificateModel>>(){}));
    }

    /**
     * Update a single customer
     * 
     * Replace the customer object at this URL with a new record.
     *             
     * A customer object defines information about a person or business that purchases products from your
     * company.  When you create a tax transaction in AvaTax, you can use the `customerCode` from this
     * record in your `CreateTransaction` API call.  AvaTax will search for this `customerCode` value and
     * identify any certificates linked to this `customer` object.  If any certificate applies to the transaction,
     * AvaTax will record the appropriate elements of the transaction as exempt and link it to the `certificate`.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param customerCode The unique code representing this customer
     * @param model The new customer model that will replace the existing record at this URL
     * @return CustomerModel
     */
    public CustomerModel updateCustomer(Integer companyId, String customerCode, CustomerModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        return ((RestCall<CustomerModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<CustomerModel>(){})).call();
    }

    /**
     * Update a single customer
     * 
     * Replace the customer object at this URL with a new record.
     *             
     * A customer object defines information about a person or business that purchases products from your
     * company.  When you create a tax transaction in AvaTax, you can use the `customerCode` from this
     * record in your `CreateTransaction` API call.  AvaTax will search for this `customerCode` value and
     * identify any certificates linked to this `customer` object.  If any certificate applies to the transaction,
     * AvaTax will record the appropriate elements of the transaction as exempt and link it to the `certificate`.
     *             
     * Using exemption certificates endpoints requires setup of an auditable document storage for each company that will use certificates.
     * Companies that do not have this storage system set up will receive the error `CertCaptureNotConfiguredError` when they call exemption
     * certificate related APIs.  To check if this company is set up, call `GetCertificateSetup`.  To request setup of the auditable document
     * storage for this company, call `RequestCertificateSetup`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The unique ID number of the company that recorded this customer
     * @param customerCode The unique code representing this customer
     * @param model The new customer model that will replace the existing record at this URL
     * @return CustomerModel
     */
    public Future<CustomerModel> updateCustomerAsync(Integer companyId, String customerCode, CustomerModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/customers/{customerCode}");
        path.applyField("companyId", companyId);
        path.applyField("customerCode", customerCode);
        return this.threadPool.submit((RestCall<CustomerModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<CustomerModel>(){}));
    }

    /**
     * Create and store new datasources for the respective companies.
     * 
     * Create one or more datasource objects.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The id of the company you which to create the datasources
     * @param model 
     * @return ArrayList<DataSourceModel>
     */
    public ArrayList<DataSourceModel> createDataSources(Integer companyId, ArrayList<DataSourceModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/datasources");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<DataSourceModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<DataSourceModel>>(){})).call();
    }

    /**
     * Create and store new datasources for the respective companies.
     * 
     * Create one or more datasource objects.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The id of the company you which to create the datasources
     * @param model 
     * @return ArrayList<DataSourceModel>
     */
    public Future<ArrayList<DataSourceModel>> createDataSourcesAsync(Integer companyId, ArrayList<DataSourceModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/datasources");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<DataSourceModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<DataSourceModel>>(){}));
    }

    /**
     * Delete a datasource by datasource id for a company.
     * 
     * Marks the existing datasource for a company as deleted.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The id of the company the datasource belongs to.
     * @param id The id of the datasource you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteDataSource(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/datasources/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a datasource by datasource id for a company.
     * 
     * Marks the existing datasource for a company as deleted.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The id of the company the datasource belongs to.
     * @param id The id of the datasource you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteDataSourceAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/datasources/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Get data source by data source id
     * 
     * Retrieve the data source by its unique ID number.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId 
     * @param id data source id
     * @return DataSourceModel
     */
    public DataSourceModel getDataSourceById(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/datasources/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<DataSourceModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<DataSourceModel>(){})).call();
    }

    /**
     * Get data source by data source id
     * 
     * Retrieve the data source by its unique ID number.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId 
     * @param id data source id
     * @return DataSourceModel
     */
    public Future<DataSourceModel> getDataSourceByIdAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/datasources/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<DataSourceModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<DataSourceModel>(){}));
    }

    /**
     * Retrieve all datasources for this company
     * 
     * Gets multiple datasource objects for a given company.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The id of the company you wish to retrieve the datasources.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* isEnabled, isSynced, isAuthorized
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<DataSourceModel>
     */
    public FetchResult<DataSourceModel> listDataSources(Integer companyId, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/datasources");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<DataSourceModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<DataSourceModel>>(){})).call();
    }

    /**
     * Retrieve all datasources for this company
     * 
     * Gets multiple datasource objects for a given company.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The id of the company you wish to retrieve the datasources.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* isEnabled, isSynced, isAuthorized
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<DataSourceModel>
     */
    public Future<FetchResult<DataSourceModel>> listDataSourcesAsync(Integer companyId, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/datasources");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<DataSourceModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<DataSourceModel>>(){}));
    }

    /**
     * Retrieve all datasources
     * 
     * Get multiple datasource objects across all companies.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* isEnabled, isSynced, isAuthorized
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<DataSourceModel>
     */
    public FetchResult<DataSourceModel> queryDataSources(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/datasources");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<DataSourceModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<DataSourceModel>>(){})).call();
    }

    /**
     * Retrieve all datasources
     * 
     * Get multiple datasource objects across all companies.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* isEnabled, isSynced, isAuthorized
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<DataSourceModel>
     */
    public Future<FetchResult<DataSourceModel>> queryDataSourcesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/datasources");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<DataSourceModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<DataSourceModel>>(){}));
    }

    /**
     * Update a datasource identified by id for a company
     * 
     * Updates a datasource for a company.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The id of the company the datasource belongs to.
     * @param id The id of the datasource you wish to delete.
     * @param model 
     * @return DataSourceModel
     */
    public DataSourceModel updateDataSource(Integer companyId, Integer id, DataSourceModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/datasources/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<DataSourceModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<DataSourceModel>(){})).call();
    }

    /**
     * Update a datasource identified by id for a company
     * 
     * Updates a datasource for a company.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param companyId The id of the company the datasource belongs to.
     * @param id The id of the datasource you wish to delete.
     * @param model 
     * @return DataSourceModel
     */
    public Future<DataSourceModel> updateDataSourceAsync(Integer companyId, Integer id, DataSourceModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/datasources/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<DataSourceModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<DataSourceModel>(){}));
    }

    /**
     * Lists all parents of an HS Code.
     * 
     * Retrieves the specified HS code and all of its parents, reflecting all sections, chapters, headings, and subheadings
     *             
     * a list of HS Codes that are the parents and information branches of the HS Code for the given
     * destination country, if lower detail is available.
     *             
     * This API will include information branches if applicable. These do not have HS Codes and cannot be referenced,
     * but can contain information relevant to deciding the correct HS Code.
     *             
     * This API is intended to be useful to review the descriptive hierarchy of an HS Code, which can be particularly helpful
     * when HS Codes can have multiple levels of generic descriptions.
     * 
     * ### Security Policies
     * 
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxGlobal.
     * 
     * @param country The name or code of the destination country.
     * @param hsCode The partial or full HS Code for which you would like to view all of the parents.
     * @return FetchResult<HsCodeModel>
     */
    public FetchResult<HsCodeModel> getCrossBorderCode(String country, String hsCode) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/crossborder/{country}/{hsCode}/hierarchy");
        path.applyField("country", country);
        path.applyField("hsCode", hsCode);
        return ((RestCall<FetchResult<HsCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<HsCodeModel>>(){})).call();
    }

    /**
     * Lists all parents of an HS Code.
     * 
     * Retrieves the specified HS code and all of its parents, reflecting all sections, chapters, headings, and subheadings
     *             
     * a list of HS Codes that are the parents and information branches of the HS Code for the given
     * destination country, if lower detail is available.
     *             
     * This API will include information branches if applicable. These do not have HS Codes and cannot be referenced,
     * but can contain information relevant to deciding the correct HS Code.
     *             
     * This API is intended to be useful to review the descriptive hierarchy of an HS Code, which can be particularly helpful
     * when HS Codes can have multiple levels of generic descriptions.
     * 
     * ### Security Policies
     * 
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxGlobal.
     * 
     * @param country The name or code of the destination country.
     * @param hsCode The partial or full HS Code for which you would like to view all of the parents.
     * @return FetchResult<HsCodeModel>
     */
    public Future<FetchResult<HsCodeModel>> getCrossBorderCodeAsync(String country, String hsCode) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/crossborder/{country}/{hsCode}/hierarchy");
        path.applyField("country", country);
        path.applyField("hsCode", hsCode);
        return this.threadPool.submit((RestCall<FetchResult<HsCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<HsCodeModel>>(){}));
    }

    /**
     * Test whether a form supports online login verification
     * 
     * This API is intended to be useful to identify whether the user should be allowed
     * 
     * @param form The name of the form you would like to verify. This is the tax form code
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* taxFormCodes, scraperType, expectedResponseTime, requiredFilingCalendarDataFields
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<SkyscraperStatusModel>
     */
    public FetchResult<SkyscraperStatusModel> getLoginVerifierByForm(String form, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/filingcalendars/loginverifiers/{form}");
        path.applyField("form", form);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<SkyscraperStatusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SkyscraperStatusModel>>(){})).call();
    }

    /**
     * Test whether a form supports online login verification
     * 
     * This API is intended to be useful to identify whether the user should be allowed
     * 
     * @param form The name of the form you would like to verify. This is the tax form code
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* taxFormCodes, scraperType, expectedResponseTime, requiredFilingCalendarDataFields
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<SkyscraperStatusModel>
     */
    public Future<FetchResult<SkyscraperStatusModel>> getLoginVerifierByFormAsync(String form, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/filingcalendars/loginverifiers/{form}");
        path.applyField("form", form);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<SkyscraperStatusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SkyscraperStatusModel>>(){}));
    }

    /**
     * Retrieve the full list of the AvaFile Forms available
     * 
     * This API is deprecated.
     *             
     * Please use the ListTaxForms API.
     *             
     * Returns the full list of Avalara-supported AvaFile Forms
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* outletTypeId
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<AvaFileFormModel>
     */
    public FetchResult<AvaFileFormModel> listAvaFileForms(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/avafileforms");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<AvaFileFormModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<AvaFileFormModel>>(){})).call();
    }

    /**
     * Retrieve the full list of the AvaFile Forms available
     * 
     * This API is deprecated.
     *             
     * Please use the ListTaxForms API.
     *             
     * Returns the full list of Avalara-supported AvaFile Forms
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* outletTypeId
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<AvaFileFormModel>
     */
    public Future<FetchResult<AvaFileFormModel>> listAvaFileFormsAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/avafileforms");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<AvaFileFormModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<AvaFileFormModel>>(){}));
    }

    /**
     * List certificate attributes used by a company
     * 
     * List the certificate attributes defined by a company.
     *             
     * A certificate may have multiple attributes that control its behavior.  You may apply or remove attributes to a
     * certificate at any time.
     *             
     * If you see the 'CertCaptureNotConfiguredError', please use CheckProvision and RequestProvision endpoints to
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CertificateAttributeModel>
     */
    public FetchResult<CertificateAttributeModel> listCertificateAttributes(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/certificateattributes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<CertificateAttributeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CertificateAttributeModel>>(){})).call();
    }

    /**
     * List certificate attributes used by a company
     * 
     * List the certificate attributes defined by a company.
     *             
     * A certificate may have multiple attributes that control its behavior.  You may apply or remove attributes to a
     * certificate at any time.
     *             
     * If you see the 'CertCaptureNotConfiguredError', please use CheckProvision and RequestProvision endpoints to
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CertificateAttributeModel>
     */
    public Future<FetchResult<CertificateAttributeModel>> listCertificateAttributesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/certificateattributes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<CertificateAttributeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CertificateAttributeModel>>(){}));
    }

    /**
     * List the certificate exempt reasons defined by a company
     * 
     * List the certificate exempt reasons defined by a company.
     *             
     * An exemption reason defines why a certificate allows a customer to be exempt
     * for purposes of tax calculation.
     *             
     * If you see the 'CertCaptureNotConfiguredError', please use CheckProvision and RequestProvision endpoints to
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ExemptionReasonModel>
     */
    public FetchResult<ExemptionReasonModel> listCertificateExemptReasons(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/certificateexemptreasons");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<ExemptionReasonModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ExemptionReasonModel>>(){})).call();
    }

    /**
     * List the certificate exempt reasons defined by a company
     * 
     * List the certificate exempt reasons defined by a company.
     *             
     * An exemption reason defines why a certificate allows a customer to be exempt
     * for purposes of tax calculation.
     *             
     * If you see the 'CertCaptureNotConfiguredError', please use CheckProvision and RequestProvision endpoints to
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ExemptionReasonModel>
     */
    public Future<FetchResult<ExemptionReasonModel>> listCertificateExemptReasonsAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/certificateexemptreasons");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<ExemptionReasonModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ExemptionReasonModel>>(){}));
    }

    /**
     * List certificate exposure zones used by a company
     * 
     * List the certificate exposure zones defined by a company.
     *             
     * An exposure zone is a location where a certificate can be valid.  Exposure zones may indicate a taxing
     * authority or other legal entity to which a certificate may apply.
     *             
     * If you see the 'CertCaptureNotConfiguredError', please use CheckProvision and RequestProvision endpoints to
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* id, companyId, name, tag, description, created, modified, region, country
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ExposureZoneModel>
     */
    public FetchResult<ExposureZoneModel> listCertificateExposureZones(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/certificateexposurezones");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<ExposureZoneModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ExposureZoneModel>>(){})).call();
    }

    /**
     * List certificate exposure zones used by a company
     * 
     * List the certificate exposure zones defined by a company.
     *             
     * An exposure zone is a location where a certificate can be valid.  Exposure zones may indicate a taxing
     * authority or other legal entity to which a certificate may apply.
     *             
     * If you see the 'CertCaptureNotConfiguredError', please use CheckProvision and RequestProvision endpoints to
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* id, companyId, name, tag, description, created, modified, region, country
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ExposureZoneModel>
     */
    public Future<FetchResult<ExposureZoneModel>> listCertificateExposureZonesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/certificateexposurezones");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<ExposureZoneModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ExposureZoneModel>>(){}));
    }

    /**
     * Retrieve the full list of communications service types
     * 
     * @param id The transaction type ID to examine
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* requiredParameters
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CommunicationsTSPairModel>
     */
    public FetchResult<CommunicationsTSPairModel> listCommunicationsServiceTypes(Integer id, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/communications/transactiontypes/{id}/servicetypes");
        path.applyField("id", id);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<CommunicationsTSPairModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CommunicationsTSPairModel>>(){})).call();
    }

    /**
     * Retrieve the full list of communications service types
     * 
     * @param id The transaction type ID to examine
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* requiredParameters
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CommunicationsTSPairModel>
     */
    public Future<FetchResult<CommunicationsTSPairModel>> listCommunicationsServiceTypesAsync(Integer id, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/communications/transactiontypes/{id}/servicetypes");
        path.applyField("id", id);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<CommunicationsTSPairModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CommunicationsTSPairModel>>(){}));
    }

    /**
     * Retrieve the full list of communications transactiontypes
     * 
     * Returns full list of communications transaction types which
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CommunicationsTransactionTypeModel>
     */
    public FetchResult<CommunicationsTransactionTypeModel> listCommunicationsTransactionTypes(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/communications/transactiontypes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<CommunicationsTransactionTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CommunicationsTransactionTypeModel>>(){})).call();
    }

    /**
     * Retrieve the full list of communications transactiontypes
     * 
     * Returns full list of communications transaction types which
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CommunicationsTransactionTypeModel>
     */
    public Future<FetchResult<CommunicationsTransactionTypeModel>> listCommunicationsTransactionTypesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/communications/transactiontypes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<CommunicationsTransactionTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CommunicationsTransactionTypeModel>>(){}));
    }

    /**
     * Retrieve the full list of communications transaction/service type pairs
     * 
     * Returns full list of communications transaction/service type pairs which
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* requiredParameters
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CommunicationsTSPairModel>
     */
    public FetchResult<CommunicationsTSPairModel> listCommunicationsTSPairs(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/communications/tspairs");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<CommunicationsTSPairModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CommunicationsTSPairModel>>(){})).call();
    }

    /**
     * Retrieve the full list of communications transaction/service type pairs
     * 
     * Returns full list of communications transaction/service type pairs which
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* requiredParameters
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CommunicationsTSPairModel>
     */
    public Future<FetchResult<CommunicationsTSPairModel>> listCommunicationsTSPairsAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/communications/tspairs");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<CommunicationsTSPairModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CommunicationsTSPairModel>>(){}));
    }

    /**
     * List all ISO 3166 countries
     * 
     * Returns a list of all ISO 3166 country codes, and their US English friendly names.
     * This API is intended to be useful when presenting a dropdown box in your website to allow customers to select a country for
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* alpha3Code, isEuropeanUnion, localizedNames, addressesRequireRegion
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<IsoCountryModel>
     */
    public FetchResult<IsoCountryModel> listCountries(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/countries");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<IsoCountryModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<IsoCountryModel>>(){})).call();
    }

    /**
     * List all ISO 3166 countries
     * 
     * Returns a list of all ISO 3166 country codes, and their US English friendly names.
     * This API is intended to be useful when presenting a dropdown box in your website to allow customers to select a country for
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* alpha3Code, isEuropeanUnion, localizedNames, addressesRequireRegion
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<IsoCountryModel>
     */
    public Future<FetchResult<IsoCountryModel>> listCountriesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/countries");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<IsoCountryModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<IsoCountryModel>>(){}));
    }

    /**
     * List certificate exposure zones used by a company
     * 
     * List available cover letters that can be used when sending invitation to use CertExpress to upload certificates.
     *             
     * The CoverLetter model represents a message sent along with an invitation to use CertExpress to
     * upload certificates.  An invitation allows customers to use CertExpress to upload their exemption
     * certificates directly; this cover letter explains why the invitation was sent.
     *             
     * If you see the 'CertCaptureNotConfiguredError', please use CheckProvision and RequestProvision endpoints to
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* id, companyId, subject, description, createdDate, modifiedDate, pageCount, templateFilename, version
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CoverLetterModel>
     */
    public FetchResult<CoverLetterModel> listCoverLetters(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/coverletters");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<CoverLetterModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CoverLetterModel>>(){})).call();
    }

    /**
     * List certificate exposure zones used by a company
     * 
     * List available cover letters that can be used when sending invitation to use CertExpress to upload certificates.
     *             
     * The CoverLetter model represents a message sent along with an invitation to use CertExpress to
     * upload certificates.  An invitation allows customers to use CertExpress to upload their exemption
     * certificates directly; this cover letter explains why the invitation was sent.
     *             
     * If you see the 'CertCaptureNotConfiguredError', please use CheckProvision and RequestProvision endpoints to
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* id, companyId, subject, description, createdDate, modifiedDate, pageCount, templateFilename, version
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CoverLetterModel>
     */
    public Future<FetchResult<CoverLetterModel>> listCoverLettersAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/coverletters");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<CoverLetterModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CoverLetterModel>>(){}));
    }

    /**
     * Lists the next level of HS Codes given a destination country and HS Code prefix.
     * 
     * Retrieves a list of HS Codes that are the children of the prefix for the given destination country, if
     * additional children are available.
     *             
     * HS Code is interchangeable with "tariff code" and definitions are generally unique to a destination country.
     * An HS Code describes an item and its eligibility/rate for tariffs. HS Codes are organized by
     * Section/Chapter/Heading/Subheading/Classification.
     *             
     * This API is intended to be useful to identify the correct HS Code to use for your item.
     * 
     * ### Security Policies
     * 
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxGlobal.
     * 
     * @param country The name or code of the destination country.
     * @param hsCode The Section or partial HS Code for which you would like to view the next level of HS Code detail, if more detail is available.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* hsCodeSource, system, destinationCountry, isDecisionNode, zeroPaddingCount, isSystemDefined, isTaxable, effDate, endDate, hsCodeSourceLength
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<HsCodeModel>
     */
    public FetchResult<HsCodeModel> listCrossBorderCodes(String country, String hsCode, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/crossborder/{country}/{hsCode}");
        path.applyField("country", country);
        path.applyField("hsCode", hsCode);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<HsCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<HsCodeModel>>(){})).call();
    }

    /**
     * Lists the next level of HS Codes given a destination country and HS Code prefix.
     * 
     * Retrieves a list of HS Codes that are the children of the prefix for the given destination country, if
     * additional children are available.
     *             
     * HS Code is interchangeable with "tariff code" and definitions are generally unique to a destination country.
     * An HS Code describes an item and its eligibility/rate for tariffs. HS Codes are organized by
     * Section/Chapter/Heading/Subheading/Classification.
     *             
     * This API is intended to be useful to identify the correct HS Code to use for your item.
     * 
     * ### Security Policies
     * 
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxGlobal.
     * 
     * @param country The name or code of the destination country.
     * @param hsCode The Section or partial HS Code for which you would like to view the next level of HS Code detail, if more detail is available.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* hsCodeSource, system, destinationCountry, isDecisionNode, zeroPaddingCount, isSystemDefined, isTaxable, effDate, endDate, hsCodeSourceLength
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<HsCodeModel>
     */
    public Future<FetchResult<HsCodeModel>> listCrossBorderCodesAsync(String country, String hsCode, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/crossborder/{country}/{hsCode}");
        path.applyField("country", country);
        path.applyField("hsCode", hsCode);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<HsCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<HsCodeModel>>(){}));
    }

    /**
     * List top level HS Code Sections.
     * 
     * Returns the full list of top level HS Code Sections. Sections are the broadest level of detail for
     * classifying tariff codes and the items to which they apply. HS Codes are organized
     * by Section/Chapter/Heading/Subheading/Classification.
     *             
     * This API is intended to be useful to identify the top level Sections for
     * further LandedCost HS Code lookups.
     * 
     * ### Security Policies
     * 
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxGlobal.
     * 
     * @return FetchResult<HsCodeModel>
     */
    public FetchResult<HsCodeModel> listCrossBorderSections() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/crossborder/sections");
        return ((RestCall<FetchResult<HsCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<HsCodeModel>>(){})).call();
    }

    /**
     * List top level HS Code Sections.
     * 
     * Returns the full list of top level HS Code Sections. Sections are the broadest level of detail for
     * classifying tariff codes and the items to which they apply. HS Codes are organized
     * by Section/Chapter/Heading/Subheading/Classification.
     *             
     * This API is intended to be useful to identify the top level Sections for
     * further LandedCost HS Code lookups.
     * 
     * ### Security Policies
     * 
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxGlobal.
     * 
     * @return FetchResult<HsCodeModel>
     */
    public Future<FetchResult<HsCodeModel>> listCrossBorderSectionsAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/crossborder/sections");
        return this.threadPool.submit((RestCall<FetchResult<HsCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<HsCodeModel>>(){}));
    }

    /**
     * List all ISO 4217 currencies supported by AvaTax.
     * 
     * Lists all ISO 4217 currencies supported by AvaTax.
     *             
     * This API produces a list of currency codes that can be used when calling AvaTax.  The values from this API can be used to fill out the
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CurrencyModel>
     */
    public FetchResult<CurrencyModel> listCurrencies(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/currencies");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<CurrencyModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CurrencyModel>>(){})).call();
    }

    /**
     * List all ISO 4217 currencies supported by AvaTax.
     * 
     * Lists all ISO 4217 currencies supported by AvaTax.
     *             
     * This API produces a list of currency codes that can be used when calling AvaTax.  The values from this API can be used to fill out the
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CurrencyModel>
     */
    public Future<FetchResult<CurrencyModel>> listCurrenciesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/currencies");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<CurrencyModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CurrencyModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported entity use codes
     * 
     * Returns the full list of Avalara-supported entity use codes.
     * Entity/Use Codes are definitions of the entity who is purchasing something, or the purpose for which the transaction
     * is occurring.  This information is generally used to determine taxability of the product.
     * In order to facilitate correct reporting of your taxes, you are encouraged to select the proper entity use codes for
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* validCountries
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<EntityUseCodeModel>
     */
    public FetchResult<EntityUseCodeModel> listEntityUseCodes(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/entityusecodes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<EntityUseCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<EntityUseCodeModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported entity use codes
     * 
     * Returns the full list of Avalara-supported entity use codes.
     * Entity/Use Codes are definitions of the entity who is purchasing something, or the purpose for which the transaction
     * is occurring.  This information is generally used to determine taxability of the product.
     * In order to facilitate correct reporting of your taxes, you are encouraged to select the proper entity use codes for
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* validCountries
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<EntityUseCodeModel>
     */
    public Future<FetchResult<EntityUseCodeModel>> listEntityUseCodesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/entityusecodes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<EntityUseCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<EntityUseCodeModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported filing frequencies.
     * 
     * Returns the full list of Avalara-supported filing frequencies.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<FilingFrequencyModel>
     */
    public FetchResult<FilingFrequencyModel> listFilingFrequencies(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/filingfrequencies");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<FilingFrequencyModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingFrequencyModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported filing frequencies.
     * 
     * Returns the full list of Avalara-supported filing frequencies.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<FilingFrequencyModel>
     */
    public Future<FetchResult<FilingFrequencyModel>> listFilingFrequenciesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/filingfrequencies");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<FilingFrequencyModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingFrequencyModel>>(){}));
    }

    /**
     * List jurisdictions based on the filter provided
     * 
     * Returns a list of all Avalara-supported taxing jurisdictions.
     *             
     * This API allows you to examine all Avalara-supported jurisdictions. You can filter your search by supplying
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* rate, salesRate, signatureCode, useRate
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<JurisdictionModel>
     */
    public FetchResult<JurisdictionModel> listJurisdictions(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/jurisdictions");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<JurisdictionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<JurisdictionModel>>(){})).call();
    }

    /**
     * List jurisdictions based on the filter provided
     * 
     * Returns a list of all Avalara-supported taxing jurisdictions.
     *             
     * This API allows you to examine all Avalara-supported jurisdictions. You can filter your search by supplying
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* rate, salesRate, signatureCode, useRate
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<JurisdictionModel>
     */
    public Future<FetchResult<JurisdictionModel>> listJurisdictionsAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/jurisdictions");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<JurisdictionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<JurisdictionModel>>(){}));
    }

    /**
     * List jurisdictions near a specific address
     * 
     * Returns a list of all Avalara-supported taxing jurisdictions that apply to this address.
     *             
     * This API allows you to identify which jurisdictions are nearby a specific address according to the best available geocoding information.
     * It is intended to allow you to create a "Jurisdiction Override", which allows an address to be configured as belonging to a nearby
     * jurisdiction in AvaTax.
     *             
     * 
     * @param line1 The first address line portion of this address.
     * @param line2 The second address line portion of this address.
     * @param line3 The third address line portion of this address.
     * @param city The city portion of this address.
     * @param region The region, state, or province code portion of this address.
     * @param postalCode The postal code or zip code portion of this address.
     * @param country The two-character ISO-3166 code of the country portion of this address.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* country, Jurisdictions
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<JurisdictionOverrideModel>
     */
    public FetchResult<JurisdictionOverrideModel> listJurisdictionsByAddress(String line1, String line2, String line3, String city, String region, String postalCode, String country, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/jurisdictionsnearaddress");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<JurisdictionOverrideModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<JurisdictionOverrideModel>>(){})).call();
    }

    /**
     * List jurisdictions near a specific address
     * 
     * Returns a list of all Avalara-supported taxing jurisdictions that apply to this address.
     *             
     * This API allows you to identify which jurisdictions are nearby a specific address according to the best available geocoding information.
     * It is intended to allow you to create a "Jurisdiction Override", which allows an address to be configured as belonging to a nearby
     * jurisdiction in AvaTax.
     *             
     * 
     * @param line1 The first address line portion of this address.
     * @param line2 The second address line portion of this address.
     * @param line3 The third address line portion of this address.
     * @param city The city portion of this address.
     * @param region The region, state, or province code portion of this address.
     * @param postalCode The postal code or zip code portion of this address.
     * @param country The two-character ISO-3166 code of the country portion of this address.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* country, Jurisdictions
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<JurisdictionOverrideModel>
     */
    public Future<FetchResult<JurisdictionOverrideModel>> listJurisdictionsByAddressAsync(String line1, String line2, String line3, String city, String region, String postalCode, String country, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/jurisdictionsnearaddress");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<JurisdictionOverrideModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<JurisdictionOverrideModel>>(){}));
    }

    /**
     * Retrieve the list of questions that are required for a tax location
     * 
     * Returns the list of additional questions you must answer when declaring a location in certain taxing jurisdictions.
     * Some tax jurisdictions require that you register or provide additional information to configure each physical place where
     * your company does business.
     * This information is not usually required in order to calculate tax correctly, but is almost always required to file your tax correctly.
     * You can call this API call for any address and obtain information about what questions must be answered in order to properly
     * 
     * @param line1 The first line of this location's address.
     * @param line2 The second line of this location's address.
     * @param line3 The third line of this location's address.
     * @param city The city part of this location's address.
     * @param region The region, state, or province part of this location's address.
     * @param postalCode The postal code of this location's address.
     * @param country The country part of this location's address.
     * @param latitude Optionally identify the location via latitude/longitude instead of via address.
     * @param longitude Optionally identify the location via latitude/longitude instead of via address.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<LocationQuestionModel>
     */
    public FetchResult<LocationQuestionModel> listLocationQuestionsByAddress(String line1, String line2, String line3, String city, String region, String postalCode, String country, BigDecimal latitude, BigDecimal longitude, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/locationquestions");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        path.addQuery("latitude", latitude);
        path.addQuery("longitude", longitude);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<LocationQuestionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<LocationQuestionModel>>(){})).call();
    }

    /**
     * Retrieve the list of questions that are required for a tax location
     * 
     * Returns the list of additional questions you must answer when declaring a location in certain taxing jurisdictions.
     * Some tax jurisdictions require that you register or provide additional information to configure each physical place where
     * your company does business.
     * This information is not usually required in order to calculate tax correctly, but is almost always required to file your tax correctly.
     * You can call this API call for any address and obtain information about what questions must be answered in order to properly
     * 
     * @param line1 The first line of this location's address.
     * @param line2 The second line of this location's address.
     * @param line3 The third line of this location's address.
     * @param city The city part of this location's address.
     * @param region The region, state, or province part of this location's address.
     * @param postalCode The postal code of this location's address.
     * @param country The country part of this location's address.
     * @param latitude Optionally identify the location via latitude/longitude instead of via address.
     * @param longitude Optionally identify the location via latitude/longitude instead of via address.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<LocationQuestionModel>
     */
    public Future<FetchResult<LocationQuestionModel>> listLocationQuestionsByAddressAsync(String line1, String line2, String line3, String city, String region, String postalCode, String country, BigDecimal latitude, BigDecimal longitude, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/locationquestions");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        path.addQuery("latitude", latitude);
        path.addQuery("longitude", longitude);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<LocationQuestionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<LocationQuestionModel>>(){}));
    }

    /**
     * List all forms where logins can be verified automatically
     * 
     * List all forms where logins can be verified automatically.
     * This API is intended to be useful to identify whether the user should be allowed
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* taxFormCodes, scraperType, expectedResponseTime, requiredFilingCalendarDataFields
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<SkyscraperStatusModel>
     */
    public FetchResult<SkyscraperStatusModel> listLoginVerifiers(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/filingcalendars/loginverifiers");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<SkyscraperStatusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SkyscraperStatusModel>>(){})).call();
    }

    /**
     * List all forms where logins can be verified automatically
     * 
     * List all forms where logins can be verified automatically.
     * This API is intended to be useful to identify whether the user should be allowed
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* taxFormCodes, scraperType, expectedResponseTime, requiredFilingCalendarDataFields
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<SkyscraperStatusModel>
     */
    public Future<FetchResult<SkyscraperStatusModel>> listLoginVerifiersAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/filingcalendars/loginverifiers");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<SkyscraperStatusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SkyscraperStatusModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported nexus for all countries and regions.
     * 
     * Returns the full list of all Avalara-supported nexus for all countries and regions.
     *             
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* streamlinedSalesTax, isSSTActive, taxAuthorityId
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NexusModel>
     */
    public FetchResult<NexusModel> listNexus(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported nexus for all countries and regions.
     * 
     * Returns the full list of all Avalara-supported nexus for all countries and regions.
     *             
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* streamlinedSalesTax, isSSTActive, taxAuthorityId
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NexusModel>
     */
    public Future<FetchResult<NexusModel>> listNexusAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){}));
    }

    /**
     * List all nexus that apply to a specific address.
     * 
     * Returns a list of all Avalara-supported taxing jurisdictions that apply to this address.
     * This API allows you to identify which tax authorities apply to a physical location, salesperson address, or point of sale.
     * In general, it is usually expected that a company will declare nexus in all the jurisdictions that apply to each physical address
     * where the company does business.
     * 
     * @param line1 The first address line portion of this address.
     * @param line2 The first address line portion of this address.
     * @param line3 The first address line portion of this address.
     * @param city The city portion of this address.
     * @param region Name or ISO 3166 code identifying the region portion of the address.      This field supports many different region identifiers:   * Two and three character ISO 3166 region codes   * Fully spelled out names of the region in ISO supported languages   * Common alternative spellings for many regions      For a full list of all supported codes and names, please see the Definitions API `ListRegions`.
     * @param postalCode The postal code or zip code portion of this address.
     * @param country Name or ISO 3166 code identifying the country portion of this address.      This field supports many different country identifiers:   * Two character ISO 3166 codes   * Three character ISO 3166 codes   * Fully spelled out names of the country in ISO supported languages   * Common alternative spellings for many countries      For a full list of all supported codes and names, please see the Definitions API `ListCountries`.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* streamlinedSalesTax, isSSTActive, taxAuthorityId
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NexusModel>
     */
    public FetchResult<NexusModel> listNexusByAddress(String line1, String line2, String line3, String city, String region, String postalCode, String country, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus/byaddress");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){})).call();
    }

    /**
     * List all nexus that apply to a specific address.
     * 
     * Returns a list of all Avalara-supported taxing jurisdictions that apply to this address.
     * This API allows you to identify which tax authorities apply to a physical location, salesperson address, or point of sale.
     * In general, it is usually expected that a company will declare nexus in all the jurisdictions that apply to each physical address
     * where the company does business.
     * 
     * @param line1 The first address line portion of this address.
     * @param line2 The first address line portion of this address.
     * @param line3 The first address line portion of this address.
     * @param city The city portion of this address.
     * @param region Name or ISO 3166 code identifying the region portion of the address.      This field supports many different region identifiers:   * Two and three character ISO 3166 region codes   * Fully spelled out names of the region in ISO supported languages   * Common alternative spellings for many regions      For a full list of all supported codes and names, please see the Definitions API `ListRegions`.
     * @param postalCode The postal code or zip code portion of this address.
     * @param country Name or ISO 3166 code identifying the country portion of this address.      This field supports many different country identifiers:   * Two character ISO 3166 codes   * Three character ISO 3166 codes   * Fully spelled out names of the country in ISO supported languages   * Common alternative spellings for many countries      For a full list of all supported codes and names, please see the Definitions API `ListCountries`.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* streamlinedSalesTax, isSSTActive, taxAuthorityId
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NexusModel>
     */
    public Future<FetchResult<NexusModel>> listNexusByAddressAsync(String line1, String line2, String line3, String city, String region, String postalCode, String country, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus/byaddress");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported nexus for a country.
     * 
     * Returns all Avalara-supported nexus for the specified country.
     *             
     * 
     * @param country The country in which you want to fetch the system nexus
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* streamlinedSalesTax, isSSTActive, taxAuthorityId
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NexusModel>
     */
    public FetchResult<NexusModel> listNexusByCountry(String country, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus/{country}");
        path.applyField("country", country);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported nexus for a country.
     * 
     * Returns all Avalara-supported nexus for the specified country.
     *             
     * 
     * @param country The country in which you want to fetch the system nexus
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* streamlinedSalesTax, isSSTActive, taxAuthorityId
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NexusModel>
     */
    public Future<FetchResult<NexusModel>> listNexusByCountryAsync(String country, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus/{country}");
        path.applyField("country", country);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported nexus for a country and region.
     * 
     * Returns all Avalara-supported nexus for the specified country and region.
     *             
     * 
     * @param country The two-character ISO-3166 code for the country.
     * @param region The two or three character region code for the region.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* streamlinedSalesTax, isSSTActive, taxAuthorityId
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NexusModel>
     */
    public FetchResult<NexusModel> listNexusByCountryAndRegion(String country, String region, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus/{country}/{region}");
        path.applyField("country", country);
        path.applyField("region", region);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported nexus for a country and region.
     * 
     * Returns all Avalara-supported nexus for the specified country and region.
     *             
     * 
     * @param country The two-character ISO-3166 code for the country.
     * @param region The two or three character region code for the region.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* streamlinedSalesTax, isSSTActive, taxAuthorityId
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NexusModel>
     */
    public Future<FetchResult<NexusModel>> listNexusByCountryAndRegionAsync(String country, String region, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus/{country}/{region}");
        path.applyField("country", country);
        path.applyField("region", region);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){}));
    }

    /**
     * List nexus related to a tax form
     * 
     * Retrieves a list of nexus related to a tax form.
     *             
     * The concept of `Nexus` indicates a place where your company has sufficient physical presence and is obligated
     * to collect and remit transaction-based taxes.
     *             
     * When defining companies in AvaTax, you must declare nexus for your company in order to correctly calculate tax
     * in all jurisdictions affected by your transactions.
     *             
     * This API is intended to provide useful information when examining a tax form.  If you are about to begin filing
     * a tax form, you may want to know whether you have declared nexus in all the jurisdictions related to that tax
     * form in order to better understand how the form will be filled out.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param formCode The form code that we are looking up the nexus for
     * @return NexusByTaxFormModel
     */
    public NexusByTaxFormModel listNexusByFormCode(String formCode) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus/byform/{formCode}");
        path.applyField("formCode", formCode);
        return ((RestCall<NexusByTaxFormModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<NexusByTaxFormModel>(){})).call();
    }

    /**
     * List nexus related to a tax form
     * 
     * Retrieves a list of nexus related to a tax form.
     *             
     * The concept of `Nexus` indicates a place where your company has sufficient physical presence and is obligated
     * to collect and remit transaction-based taxes.
     *             
     * When defining companies in AvaTax, you must declare nexus for your company in order to correctly calculate tax
     * in all jurisdictions affected by your transactions.
     *             
     * This API is intended to provide useful information when examining a tax form.  If you are about to begin filing
     * a tax form, you may want to know whether you have declared nexus in all the jurisdictions related to that tax
     * form in order to better understand how the form will be filled out.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param formCode The form code that we are looking up the nexus for
     * @return NexusByTaxFormModel
     */
    public Future<NexusByTaxFormModel> listNexusByFormCodeAsync(String formCode) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus/byform/{formCode}");
        path.applyField("formCode", formCode);
        return this.threadPool.submit((RestCall<NexusByTaxFormModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<NexusByTaxFormModel>(){}));
    }

    /**
     * Retrieve the full list of nexus tax type groups
     * 
     * Returns the full list of Avalara-supported nexus tax type groups
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* subscriptionTypeId, subscriptionDescription, tabName, showColumn
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NexusTaxTypeGroupModel>
     */
    public FetchResult<NexusTaxTypeGroupModel> listNexusTaxTypeGroups(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexustaxtypegroups");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NexusTaxTypeGroupModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusTaxTypeGroupModel>>(){})).call();
    }

    /**
     * Retrieve the full list of nexus tax type groups
     * 
     * Returns the full list of Avalara-supported nexus tax type groups
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* subscriptionTypeId, subscriptionDescription, tabName, showColumn
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NexusTaxTypeGroupModel>
     */
    public Future<FetchResult<NexusTaxTypeGroupModel>> listNexusTaxTypeGroupsAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexustaxtypegroups");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NexusTaxTypeGroupModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusTaxTypeGroupModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice customer funding options.
     * 
     * Returns the full list of Avalara-supported tax notice customer funding options.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* activeFlag, sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeCustomerFundingOptionModel>
     */
    public FetchResult<NoticeCustomerFundingOptionModel> listNoticeCustomerFundingOptions(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticecustomerfundingoptions");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NoticeCustomerFundingOptionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeCustomerFundingOptionModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice customer funding options.
     * 
     * Returns the full list of Avalara-supported tax notice customer funding options.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* activeFlag, sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeCustomerFundingOptionModel>
     */
    public Future<FetchResult<NoticeCustomerFundingOptionModel>> listNoticeCustomerFundingOptionsAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticecustomerfundingoptions");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NoticeCustomerFundingOptionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeCustomerFundingOptionModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice customer types.
     * 
     * Returns the full list of Avalara-supported tax notice customer types.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* activeFlag, sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeCustomerTypeModel>
     */
    public FetchResult<NoticeCustomerTypeModel> listNoticeCustomerTypes(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticecustomertypes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NoticeCustomerTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeCustomerTypeModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice customer types.
     * 
     * Returns the full list of Avalara-supported tax notice customer types.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* activeFlag, sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeCustomerTypeModel>
     */
    public Future<FetchResult<NoticeCustomerTypeModel>> listNoticeCustomerTypesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticecustomertypes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NoticeCustomerTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeCustomerTypeModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice filing types.
     * 
     * Returns the full list of Avalara-supported tax notice filing types.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* description, activeFlag, sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeFilingTypeModel>
     */
    public FetchResult<NoticeFilingTypeModel> listNoticeFilingtypes(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticefilingtypes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NoticeFilingTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeFilingTypeModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice filing types.
     * 
     * Returns the full list of Avalara-supported tax notice filing types.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* description, activeFlag, sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeFilingTypeModel>
     */
    public Future<FetchResult<NoticeFilingTypeModel>> listNoticeFilingtypesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticefilingtypes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NoticeFilingTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeFilingTypeModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice priorities.
     * 
     * Returns the full list of Avalara-supported tax notice priorities.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* activeFlag, sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticePriorityModel>
     */
    public FetchResult<NoticePriorityModel> listNoticePriorities(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticepriorities");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NoticePriorityModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticePriorityModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice priorities.
     * 
     * Returns the full list of Avalara-supported tax notice priorities.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* activeFlag, sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticePriorityModel>
     */
    public Future<FetchResult<NoticePriorityModel>> listNoticePrioritiesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticepriorities");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NoticePriorityModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticePriorityModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice reasons.
     * 
     * Returns the full list of Avalara-supported tax notice reasons.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* description, activeFlag, sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeReasonModel>
     */
    public FetchResult<NoticeReasonModel> listNoticeReasons(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticereasons");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NoticeReasonModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeReasonModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice reasons.
     * 
     * Returns the full list of Avalara-supported tax notice reasons.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* description, activeFlag, sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeReasonModel>
     */
    public Future<FetchResult<NoticeReasonModel>> listNoticeReasonsAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticereasons");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NoticeReasonModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeReasonModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice responsibility ids
     * 
     * Returns the full list of Avalara-supported tax notice responsibility ids
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeResponsibilityModel>
     */
    public FetchResult<NoticeResponsibilityModel> listNoticeResponsibilities(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticeresponsibilities");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NoticeResponsibilityModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeResponsibilityModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice responsibility ids
     * 
     * Returns the full list of Avalara-supported tax notice responsibility ids
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeResponsibilityModel>
     */
    public Future<FetchResult<NoticeResponsibilityModel>> listNoticeResponsibilitiesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticeresponsibilities");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NoticeResponsibilityModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeResponsibilityModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice root causes
     * 
     * Returns the full list of Avalara-supported tax notice root causes
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeRootCauseModel>
     */
    public FetchResult<NoticeRootCauseModel> listNoticeRootCauses(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticerootcauses");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NoticeRootCauseModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeRootCauseModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice root causes
     * 
     * Returns the full list of Avalara-supported tax notice root causes
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeRootCauseModel>
     */
    public Future<FetchResult<NoticeRootCauseModel>> listNoticeRootCausesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticerootcauses");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NoticeRootCauseModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeRootCauseModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice statuses.
     * 
     * Returns the full list of Avalara-supported tax notice statuses.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* isOpen, sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeStatusModel>
     */
    public FetchResult<NoticeStatusModel> listNoticeStatuses(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticestatuses");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NoticeStatusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeStatusModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice statuses.
     * 
     * Returns the full list of Avalara-supported tax notice statuses.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* isOpen, sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeStatusModel>
     */
    public Future<FetchResult<NoticeStatusModel>> listNoticeStatusesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticestatuses");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NoticeStatusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeStatusModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice types.
     * 
     * Returns the full list of Avalara-supported tax notice types.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* activeFlag, sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeTypeModel>
     */
    public FetchResult<NoticeTypeModel> listNoticeTypes(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticetypes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NoticeTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeTypeModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported tax notice types.
     * 
     * Returns the full list of Avalara-supported tax notice types.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* activeFlag, sortOrder
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeTypeModel>
     */
    public Future<FetchResult<NoticeTypeModel>> listNoticeTypesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticetypes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NoticeTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeTypeModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported extra parameters for creating transactions.
     * 
     * Returns the full list of Avalara-supported extra parameters for the 'Create Transaction' API call.
     * This list of parameters is available for use when configuring your transaction.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* serviceTypes, regularExpression, values
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ParameterModel>
     */
    public FetchResult<ParameterModel> listParameters(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/parameters");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<ParameterModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ParameterModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported extra parameters for creating transactions.
     * 
     * Returns the full list of Avalara-supported extra parameters for the 'Create Transaction' API call.
     * This list of parameters is available for use when configuring your transaction.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* serviceTypes, regularExpression, values
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ParameterModel>
     */
    public Future<FetchResult<ParameterModel>> listParametersAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/parameters");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<ParameterModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ParameterModel>>(){}));
    }

    /**
     * Retrieve the parameters by companyCode and itemCode.
     * 
     * Returns the list of parameters based on the company country and state jurisdiction and the item code.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyCode Company code.
     * @param itemCode Item code.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* serviceTypes, regularExpression, values
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ParameterModel>
     */
    public FetchResult<ParameterModel> listParametersByItem(String companyCode, String itemCode, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/parameters/byitem/{companyCode}/{itemCode}");
        path.applyField("companyCode", companyCode);
        path.applyField("itemCode", itemCode);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<ParameterModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ParameterModel>>(){})).call();
    }

    /**
     * Retrieve the parameters by companyCode and itemCode.
     * 
     * Returns the list of parameters based on the company country and state jurisdiction and the item code.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyCode Company code.
     * @param itemCode Item code.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* serviceTypes, regularExpression, values
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ParameterModel>
     */
    public Future<FetchResult<ParameterModel>> listParametersByItemAsync(String companyCode, String itemCode, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/parameters/byitem/{companyCode}/{itemCode}");
        path.applyField("companyCode", companyCode);
        path.applyField("itemCode", itemCode);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<ParameterModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ParameterModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported permissions
     * 
     * Returns the full list of Avalara-supported permission types.
     * 
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @return FetchResult<String>
     */
    public FetchResult<String> listPermissions(Integer top, Integer skip) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/permissions");
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        return ((RestCall<FetchResult<String>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<String>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported permissions
     * 
     * Returns the full list of Avalara-supported permission types.
     * 
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @return FetchResult<String>
     */
    public Future<FetchResult<String>> listPermissionsAsync(Integer top, Integer skip) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/permissions");
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        return this.threadPool.submit((RestCall<FetchResult<String>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<String>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported postal codes.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<PostalCodeModel>
     */
    public FetchResult<PostalCodeModel> listPostalCodes(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/postalcodes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<PostalCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<PostalCodeModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported postal codes.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<PostalCodeModel>
     */
    public Future<FetchResult<PostalCodeModel>> listPostalCodesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/postalcodes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<PostalCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<PostalCodeModel>>(){}));
    }

    /**
     * List all customs duty programs recognized by AvaTax
     * 
     * List all preferred customs duty programs recognized by AvaTax.
     *             
     * A customs duty program is an optional program you can use to obtain favorable treatment from customs and duty agents.
     * An example of a preferred program is NAFTA, which provides preferential rates for products being shipped from neighboring
     * countries.
     *             
     * To select a preferred program for calculating customs and duty rates, call this API to find the appropriate code for your
     * preferred program.  Next, set the parameter `AvaTax.LC.PreferredProgram` in your `CreateTransaction` call to the code of
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* effectiveDate, endDate
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<PreferredProgramModel>
     */
    public FetchResult<PreferredProgramModel> listPreferredPrograms(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/preferredprograms");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<PreferredProgramModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<PreferredProgramModel>>(){})).call();
    }

    /**
     * List all customs duty programs recognized by AvaTax
     * 
     * List all preferred customs duty programs recognized by AvaTax.
     *             
     * A customs duty program is an optional program you can use to obtain favorable treatment from customs and duty agents.
     * An example of a preferred program is NAFTA, which provides preferential rates for products being shipped from neighboring
     * countries.
     *             
     * To select a preferred program for calculating customs and duty rates, call this API to find the appropriate code for your
     * preferred program.  Next, set the parameter `AvaTax.LC.PreferredProgram` in your `CreateTransaction` call to the code of
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* effectiveDate, endDate
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<PreferredProgramModel>
     */
    public Future<FetchResult<PreferredProgramModel>> listPreferredProgramsAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/preferredprograms");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<PreferredProgramModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<PreferredProgramModel>>(){}));
    }

    /**
     * List all available product classification systems.
     * 
     * List all available product classification systems.
     *             
     * Tax authorities use product classification systems as a way to identify products and associate them with a tax rate.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* countries
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ProductClassificationSystemModel>
     */
    public FetchResult<ProductClassificationSystemModel> listProductClassificationSystems(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/productclassificationsystems");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<ProductClassificationSystemModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ProductClassificationSystemModel>>(){})).call();
    }

    /**
     * List all available product classification systems.
     * 
     * List all available product classification systems.
     *             
     * Tax authorities use product classification systems as a way to identify products and associate them with a tax rate.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* countries
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ProductClassificationSystemModel>
     */
    public Future<FetchResult<ProductClassificationSystemModel>> listProductClassificationSystemsAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/productclassificationsystems");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<ProductClassificationSystemModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ProductClassificationSystemModel>>(){}));
    }

    /**
     * List all product classification systems available to a company based on its nexus.
     * 
     * Lists all product classification systems available to a company based on its nexus.
     *             
     * Tax authorities use product classification systems as a way to identify products and associate them with a tax rate.
     * 
     * @param companyCode The company code.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* countries
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ProductClassificationSystemModel>
     */
    public FetchResult<ProductClassificationSystemModel> listProductClassificationSystemsByCompany(String companyCode, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/productclassificationsystems/bycompany/{companyCode}");
        path.applyField("companyCode", companyCode);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<ProductClassificationSystemModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ProductClassificationSystemModel>>(){})).call();
    }

    /**
     * List all product classification systems available to a company based on its nexus.
     * 
     * Lists all product classification systems available to a company based on its nexus.
     *             
     * Tax authorities use product classification systems as a way to identify products and associate them with a tax rate.
     * 
     * @param companyCode The company code.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* countries
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ProductClassificationSystemModel>
     */
    public Future<FetchResult<ProductClassificationSystemModel>> listProductClassificationSystemsByCompanyAsync(String companyCode, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/productclassificationsystems/bycompany/{companyCode}");
        path.applyField("companyCode", companyCode);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<ProductClassificationSystemModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ProductClassificationSystemModel>>(){}));
    }

    /**
     * Retrieve the full list of rate types for each country
     * 
     * Returns the full list of Avalara-supported rate type file types
     * 
     * @param country The country to examine for rate types
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<RateTypeModel>
     */
    public FetchResult<RateTypeModel> listRateTypesByCountry(String country, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/countries/{country}/ratetypes");
        path.applyField("country", country);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<RateTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<RateTypeModel>>(){})).call();
    }

    /**
     * Retrieve the full list of rate types for each country
     * 
     * Returns the full list of Avalara-supported rate type file types
     * 
     * @param country The country to examine for rate types
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<RateTypeModel>
     */
    public Future<FetchResult<RateTypeModel>> listRateTypesByCountryAsync(String country, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/countries/{country}/ratetypes");
        path.applyField("country", country);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<RateTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<RateTypeModel>>(){}));
    }

    /**
     * List all ISO 3166 regions
     * 
     * Returns a list of all ISO 3166 region codes and their US English friendly names.
     * This API is intended to be useful when presenting a dropdown box in your website to allow customers to select a region
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* localizedNames
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<IsoRegionModel>
     */
    public FetchResult<IsoRegionModel> listRegions(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/regions");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<IsoRegionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<IsoRegionModel>>(){})).call();
    }

    /**
     * List all ISO 3166 regions
     * 
     * Returns a list of all ISO 3166 region codes and their US English friendly names.
     * This API is intended to be useful when presenting a dropdown box in your website to allow customers to select a region
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* localizedNames
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<IsoRegionModel>
     */
    public Future<FetchResult<IsoRegionModel>> listRegionsAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/regions");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<IsoRegionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<IsoRegionModel>>(){}));
    }

    /**
     * List all ISO 3166 regions for a country
     * 
     * Returns a list of all ISO 3166 region codes for a specific country code, and their US English friendly names.
     * This API is intended to be useful when presenting a dropdown box in your website to allow customers to select a region
     * 
     * @param country The country of which you want to fetch ISO 3166 regions
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* localizedNames
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<IsoRegionModel>
     */
    public FetchResult<IsoRegionModel> listRegionsByCountry(String country, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/countries/{country}/regions");
        path.applyField("country", country);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<IsoRegionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<IsoRegionModel>>(){})).call();
    }

    /**
     * List all ISO 3166 regions for a country
     * 
     * Returns a list of all ISO 3166 region codes for a specific country code, and their US English friendly names.
     * This API is intended to be useful when presenting a dropdown box in your website to allow customers to select a region
     * 
     * @param country The country of which you want to fetch ISO 3166 regions
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* localizedNames
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<IsoRegionModel>
     */
    public Future<FetchResult<IsoRegionModel>> listRegionsByCountryAsync(String country, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/countries/{country}/regions");
        path.applyField("country", country);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<IsoRegionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<IsoRegionModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported resource file types
     * 
     * Returns the full list of Avalara-supported resource file types
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ResourceFileTypeModel>
     */
    public FetchResult<ResourceFileTypeModel> listResourceFileTypes(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/resourcefiletypes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<ResourceFileTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ResourceFileTypeModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported resource file types
     * 
     * Returns the full list of Avalara-supported resource file types
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ResourceFileTypeModel>
     */
    public Future<FetchResult<ResourceFileTypeModel>> listResourceFileTypesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/resourcefiletypes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<ResourceFileTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ResourceFileTypeModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported permissions
     * 
     * Returns the full list of Avalara-supported permission types.
     * This API is intended to be useful when designing a user interface for selecting the security role of a user account.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<SecurityRoleModel>
     */
    public FetchResult<SecurityRoleModel> listSecurityRoles(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/securityroles");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<SecurityRoleModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SecurityRoleModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported permissions
     * 
     * Returns the full list of Avalara-supported permission types.
     * This API is intended to be useful when designing a user interface for selecting the security role of a user account.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<SecurityRoleModel>
     */
    public Future<FetchResult<SecurityRoleModel>> listSecurityRolesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/securityroles");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<SecurityRoleModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SecurityRoleModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported subscription types
     * 
     * Returns the full list of Avalara-supported subscription types.
     * This API is intended to be useful for identifying which features you have added to your account.
     * You may always contact Avalara's sales department for information on available products or services.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<SubscriptionTypeModel>
     */
    public FetchResult<SubscriptionTypeModel> listSubscriptionTypes(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/subscriptiontypes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<SubscriptionTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SubscriptionTypeModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported subscription types
     * 
     * Returns the full list of Avalara-supported subscription types.
     * This API is intended to be useful for identifying which features you have added to your account.
     * You may always contact Avalara's sales department for information on available products or services.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<SubscriptionTypeModel>
     */
    public Future<FetchResult<SubscriptionTypeModel>> listSubscriptionTypesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/subscriptiontypes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<SubscriptionTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SubscriptionTypeModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported tax authorities.
     * 
     * Returns the full list of Avalara-supported tax authorities.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<TaxAuthorityModel>
     */
    public FetchResult<TaxAuthorityModel> listTaxAuthorities(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxauthorities");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<TaxAuthorityModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxAuthorityModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported tax authorities.
     * 
     * Returns the full list of Avalara-supported tax authorities.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<TaxAuthorityModel>
     */
    public Future<FetchResult<TaxAuthorityModel>> listTaxAuthoritiesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxauthorities");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<TaxAuthorityModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxAuthorityModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported forms for each tax authority.
     * 
     * Returns the full list of Avalara-supported forms for each tax authority.
     * This list represents tax forms that Avalara recognizes.
     * Customers who subscribe to Avalara Managed Returns Service can request these forms to be filed automatically
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<TaxAuthorityFormModel>
     */
    public FetchResult<TaxAuthorityFormModel> listTaxAuthorityForms(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxauthorityforms");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<TaxAuthorityFormModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxAuthorityFormModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported forms for each tax authority.
     * 
     * Returns the full list of Avalara-supported forms for each tax authority.
     * This list represents tax forms that Avalara recognizes.
     * Customers who subscribe to Avalara Managed Returns Service can request these forms to be filed automatically
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<TaxAuthorityFormModel>
     */
    public Future<FetchResult<TaxAuthorityFormModel>> listTaxAuthorityFormsAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxauthorityforms");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<TaxAuthorityFormModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxAuthorityFormModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported tax authority types.
     * 
     * Returns the full list of Avalara-supported tax authority types.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<TaxAuthorityTypeModel>
     */
    public FetchResult<TaxAuthorityTypeModel> listTaxAuthorityTypes(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxauthoritytypes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<TaxAuthorityTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxAuthorityTypeModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported tax authority types.
     * 
     * Returns the full list of Avalara-supported tax authority types.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<TaxAuthorityTypeModel>
     */
    public Future<FetchResult<TaxAuthorityTypeModel>> listTaxAuthorityTypesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxauthoritytypes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<TaxAuthorityTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxAuthorityTypeModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported tax codes.
     * 
     * Retrieves the list of Avalara-supported system tax codes.
     * A 'TaxCode' represents a uniquely identified type of product, good, or service.
     * Avalara supports correct tax rates and taxability rules for all TaxCodes in all supported jurisdictions.
     * If you identify your products by tax code in your 'Create Transacion' API calls, Avalara will correctly calculate tax rates and
     * taxability rules for this product in all supported jurisdictions.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<TaxCodeModel>
     */
    public FetchResult<TaxCodeModel> listTaxCodes(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxcodes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<TaxCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxCodeModel>>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported tax codes.
     * 
     * Retrieves the list of Avalara-supported system tax codes.
     * A 'TaxCode' represents a uniquely identified type of product, good, or service.
     * Avalara supports correct tax rates and taxability rules for all TaxCodes in all supported jurisdictions.
     * If you identify your products by tax code in your 'Create Transacion' API calls, Avalara will correctly calculate tax rates and
     * taxability rules for this product in all supported jurisdictions.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<TaxCodeModel>
     */
    public Future<FetchResult<TaxCodeModel>> listTaxCodesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxcodes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<TaxCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxCodeModel>>(){}));
    }

    /**
     * Retrieve the full list of Avalara-supported tax code types.
     * 
     * Returns the full list of recognized tax code types.
     * A 'Tax Code Type' represents a broad category of tax codes, and is less detailed than a single TaxCode.
     * 
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @return TaxCodeTypesModel
     */
    public TaxCodeTypesModel listTaxCodeTypes(Integer top, Integer skip) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxcodetypes");
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        return ((RestCall<TaxCodeTypesModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TaxCodeTypesModel>(){})).call();
    }

    /**
     * Retrieve the full list of Avalara-supported tax code types.
     * 
     * Returns the full list of recognized tax code types.
     * A 'Tax Code Type' represents a broad category of tax codes, and is less detailed than a single TaxCode.
     * 
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @return TaxCodeTypesModel
     */
    public Future<TaxCodeTypesModel> listTaxCodeTypesAsync(Integer top, Integer skip) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxcodetypes");
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        return this.threadPool.submit((RestCall<TaxCodeTypesModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TaxCodeTypesModel>(){}));
    }

    /**
     * Retrieve the full list of the Tax Forms available
     * 
     * Returns the full list of Avalara-supported Tax Forms
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<FormMasterModel>
     */
    public FetchResult<FormMasterModel> listTaxForms(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxforms");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<FormMasterModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FormMasterModel>>(){})).call();
    }

    /**
     * Retrieve the full list of the Tax Forms available
     * 
     * Returns the full list of Avalara-supported Tax Forms
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<FormMasterModel>
     */
    public Future<FetchResult<FormMasterModel>> listTaxFormsAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxforms");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<FormMasterModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FormMasterModel>>(){}));
    }

    /**
     * Retrieve the full list of tax sub types
     * 
     * Returns the full list of Avalara-supported tax sub-types
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<TaxSubTypeModel>
     */
    public FetchResult<TaxSubTypeModel> listTaxSubTypes(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxsubtypes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<TaxSubTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxSubTypeModel>>(){})).call();
    }

    /**
     * Retrieve the full list of tax sub types
     * 
     * Returns the full list of Avalara-supported tax sub-types
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<TaxSubTypeModel>
     */
    public Future<FetchResult<TaxSubTypeModel>> listTaxSubTypesAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxsubtypes");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<TaxSubTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxSubTypeModel>>(){}));
    }

    /**
     * Retrieve the full list of tax type groups
     * 
     * Returns the full list of Avalara-supported tax type groups
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* subscriptionTypeId, subscriptionDescription, tabName, showColumn, displaySequence
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<TaxTypeGroupModel>
     */
    public FetchResult<TaxTypeGroupModel> listTaxTypeGroups(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxtypegroups");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<TaxTypeGroupModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxTypeGroupModel>>(){})).call();
    }

    /**
     * Retrieve the full list of tax type groups
     * 
     * Returns the full list of Avalara-supported tax type groups
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* subscriptionTypeId, subscriptionDescription, tabName, showColumn, displaySequence
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<TaxTypeGroupModel>
     */
    public Future<FetchResult<TaxTypeGroupModel>> listTaxTypeGroupsAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxtypegroups");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<TaxTypeGroupModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxTypeGroupModel>>(){}));
    }

    /**
     * List all defined units of measurement
     * 
     * List all units of measurement systems defined by Avalara.
     *             
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* id
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<UomModel>
     */
    public FetchResult<UomModel> listUnitOfMeasurement(String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/unitofmeasurements");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<UomModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<UomModel>>(){})).call();
    }

    /**
     * List all defined units of measurement
     * 
     * List all units of measurement systems defined by Avalara.
     *             
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* id
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<UomModel>
     */
    public Future<FetchResult<UomModel>> listUnitOfMeasurementAsync(String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/unitofmeasurements");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<UomModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<UomModel>>(){}));
    }

    /**
     * Create one or more DistanceThreshold objects
     * 
     * Create one or more DistanceThreshold objects for this company.
     * 
     * A company-distance-threshold model indicates the distance between a company
     * and the taxing borders of various countries.  Distance thresholds are necessary
     * to correctly calculate some value-added taxes.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The unique ID number of the company that owns this DistanceThreshold
     * @param model The DistanceThreshold object or objects you wish to create.
     * @return ArrayList<CompanyDistanceThresholdModel>
     */
    public ArrayList<CompanyDistanceThresholdModel> createDistanceThreshold(Integer companyId, ArrayList<CompanyDistanceThresholdModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/distancethresholds");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<CompanyDistanceThresholdModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<CompanyDistanceThresholdModel>>(){})).call();
    }

    /**
     * Create one or more DistanceThreshold objects
     * 
     * Create one or more DistanceThreshold objects for this company.
     * 
     * A company-distance-threshold model indicates the distance between a company
     * and the taxing borders of various countries.  Distance thresholds are necessary
     * to correctly calculate some value-added taxes.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The unique ID number of the company that owns this DistanceThreshold
     * @param model The DistanceThreshold object or objects you wish to create.
     * @return ArrayList<CompanyDistanceThresholdModel>
     */
    public Future<ArrayList<CompanyDistanceThresholdModel>> createDistanceThresholdAsync(Integer companyId, ArrayList<CompanyDistanceThresholdModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/distancethresholds");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<CompanyDistanceThresholdModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<CompanyDistanceThresholdModel>>(){}));
    }

    /**
     * Delete a single DistanceThreshold object
     * 
     * Marks the DistanceThreshold object identified by this URL as deleted.
     * 
     * A company-distance-threshold model indicates the distance between a company
     * and the taxing borders of various countries.  Distance thresholds are necessary
     * to correctly calculate some value-added taxes.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The unique ID number of the company that owns this DistanceThreshold
     * @param id The unique ID number of the DistanceThreshold object you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteDistanceThreshold(Integer companyId, Long id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/distancethresholds/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a single DistanceThreshold object
     * 
     * Marks the DistanceThreshold object identified by this URL as deleted.
     * 
     * A company-distance-threshold model indicates the distance between a company
     * and the taxing borders of various countries.  Distance thresholds are necessary
     * to correctly calculate some value-added taxes.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The unique ID number of the company that owns this DistanceThreshold
     * @param id The unique ID number of the DistanceThreshold object you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteDistanceThresholdAsync(Integer companyId, Long id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/distancethresholds/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Retrieve a single DistanceThreshold
     * 
     * Retrieves a single DistanceThreshold object defined by this URL.
     * 
     * A company-distance-threshold model indicates the distance between a company
     * and the taxing borders of various countries.  Distance thresholds are necessary
     * to correctly calculate some value-added taxes.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param companyId The ID of the company that owns this DistanceThreshold object
     * @param id The unique ID number referring to this DistanceThreshold object
     * @return CompanyDistanceThresholdModel
     */
    public CompanyDistanceThresholdModel getDistanceThreshold(Integer companyId, Long id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/distancethresholds/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<CompanyDistanceThresholdModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<CompanyDistanceThresholdModel>(){})).call();
    }

    /**
     * Retrieve a single DistanceThreshold
     * 
     * Retrieves a single DistanceThreshold object defined by this URL.
     * 
     * A company-distance-threshold model indicates the distance between a company
     * and the taxing borders of various countries.  Distance thresholds are necessary
     * to correctly calculate some value-added taxes.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param companyId The ID of the company that owns this DistanceThreshold object
     * @param id The unique ID number referring to this DistanceThreshold object
     * @return CompanyDistanceThresholdModel
     */
    public Future<CompanyDistanceThresholdModel> getDistanceThresholdAsync(Integer companyId, Long id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/distancethresholds/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<CompanyDistanceThresholdModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<CompanyDistanceThresholdModel>(){}));
    }

    /**
     * Retrieve all DistanceThresholds for this company.
     * 
     * Lists all DistanceThreshold objects that belong to this company.
     * 
     * A company-distance-threshold model indicates the distance between a company
     * and the taxing borders of various countries.  Distance thresholds are necessary
     * to correctly calculate some value-added taxes.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param companyId The ID of the company whose DistanceThreshold objects you wish to list.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CompanyDistanceThresholdModel>
     */
    public FetchResult<CompanyDistanceThresholdModel> listDistanceThresholds(Integer companyId, String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/distancethresholds");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<CompanyDistanceThresholdModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CompanyDistanceThresholdModel>>(){})).call();
    }

    /**
     * Retrieve all DistanceThresholds for this company.
     * 
     * Lists all DistanceThreshold objects that belong to this company.
     * 
     * A company-distance-threshold model indicates the distance between a company
     * and the taxing borders of various countries.  Distance thresholds are necessary
     * to correctly calculate some value-added taxes.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param companyId The ID of the company whose DistanceThreshold objects you wish to list.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CompanyDistanceThresholdModel>
     */
    public Future<FetchResult<CompanyDistanceThresholdModel>> listDistanceThresholdsAsync(Integer companyId, String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/distancethresholds");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<CompanyDistanceThresholdModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CompanyDistanceThresholdModel>>(){}));
    }

    /**
     * Retrieve all DistanceThreshold objects
     * 
     * Lists all DistanceThreshold objects that belong to this account.
     * 
     * A company-distance-threshold model indicates the distance between a company
     * and the taxing borders of various countries.  Distance thresholds are necessary
     * to correctly calculate some value-added taxes.
     * 
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CompanyDistanceThresholdModel>
     */
    public FetchResult<CompanyDistanceThresholdModel> queryDistanceThresholds(String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/distancethresholds");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<CompanyDistanceThresholdModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CompanyDistanceThresholdModel>>(){})).call();
    }

    /**
     * Retrieve all DistanceThreshold objects
     * 
     * Lists all DistanceThreshold objects that belong to this account.
     * 
     * A company-distance-threshold model indicates the distance between a company
     * and the taxing borders of various countries.  Distance thresholds are necessary
     * to correctly calculate some value-added taxes.
     * 
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<CompanyDistanceThresholdModel>
     */
    public Future<FetchResult<CompanyDistanceThresholdModel>> queryDistanceThresholdsAsync(String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/distancethresholds");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<CompanyDistanceThresholdModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CompanyDistanceThresholdModel>>(){}));
    }

    /**
     * Update a DistanceThreshold object
     * 
     * Replace the existing DistanceThreshold object at this URL with an updated object.
     * 
     * A company-distance-threshold model indicates the distance between a company
     * and the taxing borders of various countries.  Distance thresholds are necessary
     * to correctly calculate some value-added taxes.
     *             
     * All data from the existing object will be replaced with data in the object you PUT.  
     * To set a field's value to null, you may either set its value to null or omit that field from the object you post.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The unique ID number of the company that owns this DistanceThreshold object.
     * @param id The unique ID number of the DistanceThreshold object to replace.
     * @param model The new DistanceThreshold object to store.
     * @return CompanyDistanceThresholdModel
     */
    public CompanyDistanceThresholdModel updateDistanceThreshold(Integer companyId, Long id, CompanyDistanceThresholdModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/distancethresholds/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<CompanyDistanceThresholdModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<CompanyDistanceThresholdModel>(){})).call();
    }

    /**
     * Update a DistanceThreshold object
     * 
     * Replace the existing DistanceThreshold object at this URL with an updated object.
     * 
     * A company-distance-threshold model indicates the distance between a company
     * and the taxing borders of various countries.  Distance thresholds are necessary
     * to correctly calculate some value-added taxes.
     *             
     * All data from the existing object will be replaced with data in the object you PUT.  
     * To set a field's value to null, you may either set its value to null or omit that field from the object you post.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The unique ID number of the company that owns this DistanceThreshold object.
     * @param id The unique ID number of the DistanceThreshold object to replace.
     * @param model The new DistanceThreshold object to store.
     * @return CompanyDistanceThresholdModel
     */
    public Future<CompanyDistanceThresholdModel> updateDistanceThresholdAsync(Integer companyId, Long id, CompanyDistanceThresholdModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/distancethresholds/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<CompanyDistanceThresholdModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<CompanyDistanceThresholdModel>(){}));
    }

    /**
     * Approve existing Filing Request
     * 
     * This API is available by invitation only.
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     * The filing request must be in the "ChangeRequest" status to be approved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing request object
     * @param id The unique ID of the filing request object
     * @return FilingRequestModel
     */
    public FilingRequestModel approveFilingRequest(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests/{id}/approve");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, null, new TypeToken<FilingRequestModel>(){})).call();
    }

    /**
     * Approve existing Filing Request
     * 
     * This API is available by invitation only.
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     * The filing request must be in the "ChangeRequest" status to be approved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing request object
     * @param id The unique ID of the filing request object
     * @return FilingRequestModel
     */
    public Future<FilingRequestModel> approveFilingRequestAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests/{id}/approve");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, null, new TypeToken<FilingRequestModel>(){}));
    }

    /**
     * Cancel existing Filing Request
     * 
     * This API is available by invitation only.
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing request object
     * @param id The unique ID of the filing request object
     * @return FilingRequestModel
     */
    public FilingRequestModel cancelFilingRequest(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests/{id}/cancel");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, null, new TypeToken<FilingRequestModel>(){})).call();
    }

    /**
     * Cancel existing Filing Request
     * 
     * This API is available by invitation only.
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing request object
     * @param id The unique ID of the filing request object
     * @return FilingRequestModel
     */
    public Future<FilingRequestModel> cancelFilingRequestAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests/{id}/cancel");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, null, new TypeToken<FilingRequestModel>(){}));
    }

    /**
     * Create a new filing request to cancel a filing calendar
     * 
     * This API is available by invitation only.
     *             
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing calendar object
     * @param id The unique ID number of the filing calendar to cancel
     * @param model The cancellation request for this filing calendar
     * @return FilingRequestModel
     */
    public FilingRequestModel cancelFilingRequests(Integer companyId, Integer id, ArrayList<FilingRequestModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}/cancel/request");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FilingRequestModel>(){})).call();
    }

    /**
     * Create a new filing request to cancel a filing calendar
     * 
     * This API is available by invitation only.
     *             
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing calendar object
     * @param id The unique ID number of the filing calendar to cancel
     * @param model The cancellation request for this filing calendar
     * @return FilingRequestModel
     */
    public Future<FilingRequestModel> cancelFilingRequestsAsync(Integer companyId, Integer id, ArrayList<FilingRequestModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}/cancel/request");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FilingRequestModel>(){}));
    }

    /**
     * Create a filing calendar
     * 
     * This API is available by invitation only and only available for users with Compliance access
     * A "filing request" represents information that compliance uses to file a return
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that will add the new filing calendar
     * @param model Filing calendars that will be added
     * @return FilingCalendarModel
     */
    public FilingCalendarModel createFilingCalendars(Integer companyId, ArrayList<FilingCalendarModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars");
        path.applyField("companyId", companyId);
        return ((RestCall<FilingCalendarModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FilingCalendarModel>(){})).call();
    }

    /**
     * Create a filing calendar
     * 
     * This API is available by invitation only and only available for users with Compliance access
     * A "filing request" represents information that compliance uses to file a return
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that will add the new filing calendar
     * @param model Filing calendars that will be added
     * @return FilingCalendarModel
     */
    public Future<FilingCalendarModel> createFilingCalendarsAsync(Integer companyId, ArrayList<FilingCalendarModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<FilingCalendarModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FilingCalendarModel>(){}));
    }

    /**
     * Create a new filing request to create a filing calendar
     * 
     * This API is available by invitation only.
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that will add the new filing calendar
     * @param model Information about the proposed new filing calendar
     * @return FilingRequestModel
     */
    public FilingRequestModel createFilingRequests(Integer companyId, ArrayList<FilingRequestModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/add/request");
        path.applyField("companyId", companyId);
        return ((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FilingRequestModel>(){})).call();
    }

    /**
     * Create a new filing request to create a filing calendar
     * 
     * This API is available by invitation only.
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that will add the new filing calendar
     * @param model Information about the proposed new filing calendar
     * @return FilingRequestModel
     */
    public Future<FilingRequestModel> createFilingRequestsAsync(Integer companyId, ArrayList<FilingRequestModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/add/request");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FilingRequestModel>(){}));
    }

    /**
     * Returns a list of options for adding the specified form.
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing calendar object
     * @param formCode The unique code of the form
     * @return ArrayList<CycleAddOptionModel>
     */
    public ArrayList<CycleAddOptionModel> cycleSafeAdd(Integer companyId, String formCode) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/add/options");
        path.applyField("companyId", companyId);
        path.addQuery("formCode", formCode);
        return ((RestCall<ArrayList<CycleAddOptionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<CycleAddOptionModel>>(){})).call();
    }

    /**
     * Returns a list of options for adding the specified form.
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing calendar object
     * @param formCode The unique code of the form
     * @return ArrayList<CycleAddOptionModel>
     */
    public Future<ArrayList<CycleAddOptionModel>> cycleSafeAddAsync(Integer companyId, String formCode) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/add/options");
        path.applyField("companyId", companyId);
        path.addQuery("formCode", formCode);
        return this.threadPool.submit((RestCall<ArrayList<CycleAddOptionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<CycleAddOptionModel>>(){}));
    }

    /**
     * Indicates when changes are allowed to be made to a filing calendar.
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing calendar object
     * @param id The unique ID of the filing calendar object
     * @param model A list of filing calendar edits to be made
     * @return CycleEditOptionModel
     */
    public CycleEditOptionModel cycleSafeEdit(Integer companyId, Integer id, ArrayList<FilingCalendarEditModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}/edit/options");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<CycleEditOptionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<CycleEditOptionModel>(){})).call();
    }

    /**
     * Indicates when changes are allowed to be made to a filing calendar.
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing calendar object
     * @param id The unique ID of the filing calendar object
     * @param model A list of filing calendar edits to be made
     * @return CycleEditOptionModel
     */
    public Future<CycleEditOptionModel> cycleSafeEditAsync(Integer companyId, Integer id, ArrayList<FilingCalendarEditModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}/edit/options");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<CycleEditOptionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<CycleEditOptionModel>(){}));
    }

    /**
     * Returns a list of options for expiring a filing calendar
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing calendar object
     * @param id The unique ID of the filing calendar object
     * @return CycleExpireModel
     */
    public CycleExpireModel cycleSafeExpiration(Integer companyId, Long id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}/cancel/options");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<CycleExpireModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<CycleExpireModel>(){})).call();
    }

    /**
     * Returns a list of options for expiring a filing calendar
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing calendar object
     * @param id The unique ID of the filing calendar object
     * @return CycleExpireModel
     */
    public Future<CycleExpireModel> cycleSafeExpirationAsync(Integer companyId, Long id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}/cancel/options");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<CycleExpireModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<CycleExpireModel>(){}));
    }

    /**
     * Delete a single filing calendar.
     * 
     * This API is available by invitation only.
     * Mark the existing notice object at this URL as deleted.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this filing calendar.
     * @param id The ID of the filing calendar you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteFilingCalendar(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a single filing calendar.
     * 
     * This API is available by invitation only.
     * Mark the existing notice object at this URL as deleted.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this filing calendar.
     * @param id The ID of the filing calendar you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteFilingCalendarAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Retrieve a single filing calendar
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this filing calendar
     * @param id The primary key of this filing calendar
     * @return FilingCalendarModel
     */
    public FilingCalendarModel getFilingCalendar(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingCalendarModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingCalendarModel>(){})).call();
    }

    /**
     * Retrieve a single filing calendar
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this filing calendar
     * @param id The primary key of this filing calendar
     * @return FilingCalendarModel
     */
    public Future<FilingCalendarModel> getFilingCalendarAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingCalendarModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingCalendarModel>(){}));
    }

    /**
     * Retrieve a single filing request
     * 
     * This API is available by invitation only.
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this filing calendar
     * @param id The primary key of this filing calendar
     * @return FilingRequestModel
     */
    public FilingRequestModel getFilingRequest(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingRequestModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingRequestModel>(){})).call();
    }

    /**
     * Retrieve a single filing request
     * 
     * This API is available by invitation only.
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this filing calendar
     * @param id The primary key of this filing calendar
     * @return FilingRequestModel
     */
    public Future<FilingRequestModel> getFilingRequestAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingRequestModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingRequestModel>(){}));
    }

    /**
     * Retrieve all filing calendars for this company
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns these batches
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* formCountry, formRegion, taxFormCode, taxAuthorityId, taxAuthorityName, taxAuthorityType, settings
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @param returnCountry A comma separated list of countries
     * @param returnRegion A comma separated list of regions
     * @return FetchResult<FilingCalendarModel>
     */
    public FetchResult<FilingCalendarModel> listFilingCalendars(Integer companyId, String filter, Integer top, Integer skip, String orderBy, String returnCountry, String returnRegion) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        path.addQuery("returnCountry", returnCountry);
        path.addQuery("returnRegion", returnRegion);
        return ((RestCall<FetchResult<FilingCalendarModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingCalendarModel>>(){})).call();
    }

    /**
     * Retrieve all filing calendars for this company
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns these batches
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* formCountry, formRegion, taxFormCode, taxAuthorityId, taxAuthorityName, taxAuthorityType, settings
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @param returnCountry A comma separated list of countries
     * @param returnRegion A comma separated list of regions
     * @return FetchResult<FilingCalendarModel>
     */
    public Future<FetchResult<FilingCalendarModel>> listFilingCalendarsAsync(Integer companyId, String filter, Integer top, Integer skip, String orderBy, String returnCountry, String returnRegion) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        path.addQuery("returnCountry", returnCountry);
        path.addQuery("returnRegion", returnRegion);
        return this.threadPool.submit((RestCall<FetchResult<FilingCalendarModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingCalendarModel>>(){}));
    }

    /**
     * Retrieve all filing requests for this company
     * 
     * This API is available by invitation only.
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns these batches
     * @param filingCalendarId Specific filing calendar id for the request
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<FilingRequestModel>
     */
    public FetchResult<FilingRequestModel> listFilingRequests(Integer companyId, Integer filingCalendarId, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests");
        path.applyField("companyId", companyId);
        path.addQuery("filingCalendarId", filingCalendarId);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<FilingRequestModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingRequestModel>>(){})).call();
    }

    /**
     * Retrieve all filing requests for this company
     * 
     * This API is available by invitation only.
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns these batches
     * @param filingCalendarId Specific filing calendar id for the request
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<FilingRequestModel>
     */
    public Future<FetchResult<FilingRequestModel>> listFilingRequestsAsync(Integer companyId, Integer filingCalendarId, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests");
        path.applyField("companyId", companyId);
        path.addQuery("filingCalendarId", filingCalendarId);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<FilingRequestModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingRequestModel>>(){}));
    }

    /**
     * New request for getting for validating customer's login credentials
     * 
     * This API is available by invitation only.
     *             
     * This API verifies that a customer has submitted correct login credentials for a tax authority's online filing system.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param model The model of the login information we are verifying
     * @return LoginVerificationOutputModel
     */
    public LoginVerificationOutputModel loginVerificationRequest(LoginVerificationInputModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingcalendars/credentials/verify");
        return ((RestCall<LoginVerificationOutputModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<LoginVerificationOutputModel>(){})).call();
    }

    /**
     * New request for getting for validating customer's login credentials
     * 
     * This API is available by invitation only.
     *             
     * This API verifies that a customer has submitted correct login credentials for a tax authority's online filing system.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param model The model of the login information we are verifying
     * @return LoginVerificationOutputModel
     */
    public Future<LoginVerificationOutputModel> loginVerificationRequestAsync(LoginVerificationInputModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingcalendars/credentials/verify");
        return this.threadPool.submit((RestCall<LoginVerificationOutputModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<LoginVerificationOutputModel>(){}));
    }

    /**
     * Gets the request status and Login Result
     * 
     * This API is available by invitation only.
     *             
     * This API checks the status of a login verification request.  It may only be called by authorized users from the account
     * that initially requested the login verification.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param jobId The unique ID number of this login request
     * @return LoginVerificationOutputModel
     */
    public LoginVerificationOutputModel loginVerificationStatus(Integer jobId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingcalendars/credentials/{jobId}");
        path.applyField("jobId", jobId);
        return ((RestCall<LoginVerificationOutputModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<LoginVerificationOutputModel>(){})).call();
    }

    /**
     * Gets the request status and Login Result
     * 
     * This API is available by invitation only.
     *             
     * This API checks the status of a login verification request.  It may only be called by authorized users from the account
     * that initially requested the login verification.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param jobId The unique ID number of this login request
     * @return LoginVerificationOutputModel
     */
    public Future<LoginVerificationOutputModel> loginVerificationStatusAsync(Integer jobId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingcalendars/credentials/{jobId}");
        path.applyField("jobId", jobId);
        return this.threadPool.submit((RestCall<LoginVerificationOutputModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<LoginVerificationOutputModel>(){}));
    }

    /**
     * Retrieve all filing calendars
     * 
     * This API is available by invitation only.
     *             
     * This API is deprecated - please use POST `/api/v2/filingrequests/query` API.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* formCountry, formRegion, taxFormCode, taxAuthorityId, taxAuthorityName, taxAuthorityType, settings
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @param returnCountry If specified, fetches only filing calendars that apply to tax filings in this specific country. Uses ISO 3166 country codes.
     * @param returnRegion If specified, fetches only filing calendars that apply to tax filings in this specific region. Uses ISO 3166 region codes.
     * @return FetchResult<FilingCalendarModel>
     */
    public FetchResult<FilingCalendarModel> queryFilingCalendars(String filter, Integer top, Integer skip, String orderBy, String returnCountry, String returnRegion) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingcalendars");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        path.addQuery("returnCountry", returnCountry);
        path.addQuery("returnRegion", returnRegion);
        return ((RestCall<FetchResult<FilingCalendarModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingCalendarModel>>(){})).call();
    }

    /**
     * Retrieve all filing calendars
     * 
     * This API is available by invitation only.
     *             
     * This API is deprecated - please use POST `/api/v2/filingrequests/query` API.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* formCountry, formRegion, taxFormCode, taxAuthorityId, taxAuthorityName, taxAuthorityType, settings
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @param returnCountry If specified, fetches only filing calendars that apply to tax filings in this specific country. Uses ISO 3166 country codes.
     * @param returnRegion If specified, fetches only filing calendars that apply to tax filings in this specific region. Uses ISO 3166 region codes.
     * @return FetchResult<FilingCalendarModel>
     */
    public Future<FetchResult<FilingCalendarModel>> queryFilingCalendarsAsync(String filter, Integer top, Integer skip, String orderBy, String returnCountry, String returnRegion) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingcalendars");
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        path.addQuery("returnCountry", returnCountry);
        path.addQuery("returnRegion", returnRegion);
        return this.threadPool.submit((RestCall<FetchResult<FilingCalendarModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingCalendarModel>>(){}));
    }

    /**
     * Retrieve all filing calendars
     * 
     * This API is available by invitation only.
     *             
     * This API is intended to replace the GET `/api/v2/filingcalendars` API. The fetch request object is posted on the body of the request instead of the URI, so it's not limited by a set number of characters.
     * The documentation of the GET API shows how filtering, sorting and pagination works.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param returnCountry If specified, fetches only filing calendars that apply to tax filings in this specific country. Uses ISO 3166 country codes.
     * @param returnRegion If specified, fetches only filing calendars that apply to tax filings in this specific region. Uses ISO 3166 region codes.
     * @param model Query object to filter, sort and paginate the filing calendars.
     * @return FetchResult<FilingCalendarModel>
     */
    public FetchResult<FilingCalendarModel> queryFilingCalendarsPost(String returnCountry, String returnRegion, QueryRequestModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingcalendars/query");
        path.addQuery("returnCountry", returnCountry);
        path.addQuery("returnRegion", returnRegion);
        return ((RestCall<FetchResult<FilingCalendarModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<FilingCalendarModel>>(){})).call();
    }

    /**
     * Retrieve all filing calendars
     * 
     * This API is available by invitation only.
     *             
     * This API is intended to replace the GET `/api/v2/filingcalendars` API. The fetch request object is posted on the body of the request instead of the URI, so it's not limited by a set number of characters.
     * The documentation of the GET API shows how filtering, sorting and pagination works.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param returnCountry If specified, fetches only filing calendars that apply to tax filings in this specific country. Uses ISO 3166 country codes.
     * @param returnRegion If specified, fetches only filing calendars that apply to tax filings in this specific region. Uses ISO 3166 region codes.
     * @param model Query object to filter, sort and paginate the filing calendars.
     * @return FetchResult<FilingCalendarModel>
     */
    public Future<FetchResult<FilingCalendarModel>> queryFilingCalendarsPostAsync(String returnCountry, String returnRegion, QueryRequestModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingcalendars/query");
        path.addQuery("returnCountry", returnCountry);
        path.addQuery("returnRegion", returnRegion);
        return this.threadPool.submit((RestCall<FetchResult<FilingCalendarModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<FilingCalendarModel>>(){}));
    }

    /**
     * Retrieve all filing requests
     * 
     * This API is available by invitation only.
     *             
     * This API is deprecated - please use POST `/api/v2/filingrequests/query` API.
     *             
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param filingCalendarId Specific filing calendar id for the request
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<FilingRequestModel>
     */
    public FetchResult<FilingRequestModel> queryFilingRequests(Integer filingCalendarId, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingrequests");
        path.addQuery("filingCalendarId", filingCalendarId);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<FilingRequestModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingRequestModel>>(){})).call();
    }

    /**
     * Retrieve all filing requests
     * 
     * This API is available by invitation only.
     *             
     * This API is deprecated - please use POST `/api/v2/filingrequests/query` API.
     *             
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param filingCalendarId Specific filing calendar id for the request
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<FilingRequestModel>
     */
    public Future<FetchResult<FilingRequestModel>> queryFilingRequestsAsync(Integer filingCalendarId, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingrequests");
        path.addQuery("filingCalendarId", filingCalendarId);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<FilingRequestModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingRequestModel>>(){}));
    }

    /**
     * Retrieve all filing requests
     * 
     * This API is available by invitation only.
     *             
     * This API is intended to replace the GET `/api/v2/filingrequests` API. The fetch request object is posted on the body of the request instead of the URI, so it's not limited by a set number of characters.
     * The documentation of the GET API shows how filtering, sorting and pagination works.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param filingCalendarId Specific filing calendar id for the request
     * @param model Query object to filter, sort and paginate the filing calendars.
     * @return FetchResult<FilingRequestModel>
     */
    public FetchResult<FilingRequestModel> queryFilingRequestsPost(Integer filingCalendarId, QueryRequestModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingrequests/query");
        path.addQuery("filingCalendarId", filingCalendarId);
        return ((RestCall<FetchResult<FilingRequestModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<FilingRequestModel>>(){})).call();
    }

    /**
     * Retrieve all filing requests
     * 
     * This API is available by invitation only.
     *             
     * This API is intended to replace the GET `/api/v2/filingrequests` API. The fetch request object is posted on the body of the request instead of the URI, so it's not limited by a set number of characters.
     * The documentation of the GET API shows how filtering, sorting and pagination works.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param filingCalendarId Specific filing calendar id for the request
     * @param model Query object to filter, sort and paginate the filing calendars.
     * @return FetchResult<FilingRequestModel>
     */
    public Future<FetchResult<FilingRequestModel>> queryFilingRequestsPostAsync(Integer filingCalendarId, QueryRequestModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingrequests/query");
        path.addQuery("filingCalendarId", filingCalendarId);
        return this.threadPool.submit((RestCall<FetchResult<FilingRequestModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<FilingRequestModel>>(){}));
    }

    /**
     * Create a new filing request to edit a filing calendar
     * 
     * This API is available by invitation only.
     *             
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     *             
     * Certain users may not update filing calendars directly.  Instead, they may submit an edit request
     * to modify the value of a filing calendar using this API.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing calendar object
     * @param id The unique ID number of the filing calendar to edit
     * @param model A list of filing calendar edits to be made
     * @return FilingRequestModel
     */
    public FilingRequestModel requestFilingCalendarUpdate(Integer companyId, Integer id, ArrayList<FilingRequestModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}/edit/request");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FilingRequestModel>(){})).call();
    }

    /**
     * Create a new filing request to edit a filing calendar
     * 
     * This API is available by invitation only.
     *             
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     *             
     * Certain users may not update filing calendars directly.  Instead, they may submit an edit request
     * to modify the value of a filing calendar using this API.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing calendar object
     * @param id The unique ID number of the filing calendar to edit
     * @param model A list of filing calendar edits to be made
     * @return FilingRequestModel
     */
    public Future<FilingRequestModel> requestFilingCalendarUpdateAsync(Integer companyId, Integer id, ArrayList<FilingRequestModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}/edit/request");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FilingRequestModel>(){}));
    }

    /**
     * Edit existing Filing Calendar
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing calendar object
     * @param id The unique ID of the filing calendar object
     * @param model The filing calendar model you are wishing to update with.
     * @return FilingCalendarModel
     */
    public FilingCalendarModel updateFilingCalendar(Integer companyId, Long id, FilingCalendarModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingCalendarModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<FilingCalendarModel>(){})).call();
    }

    /**
     * Edit existing Filing Calendar
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing calendar object
     * @param id The unique ID of the filing calendar object
     * @param model The filing calendar model you are wishing to update with.
     * @return FilingCalendarModel
     */
    public Future<FilingCalendarModel> updateFilingCalendarAsync(Integer companyId, Long id, FilingCalendarModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingCalendarModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<FilingCalendarModel>(){}));
    }

    /**
     * Edit existing Filing Request
     * 
     * This API is available by invitation only.
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing request object
     * @param id The unique ID of the filing request object
     * @param model A list of filing calendar edits to be made
     * @return FilingRequestModel
     */
    public FilingRequestModel updateFilingRequest(Integer companyId, Integer id, FilingRequestModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingRequestModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<FilingRequestModel>(){})).call();
    }

    /**
     * Edit existing Filing Request
     * 
     * This API is available by invitation only.
     * A "filing request" represents a request to change an existing filing calendar.  Filing requests
     * are reviewed and validated by Avalara Compliance before being implemented.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the filing request object
     * @param id The unique ID of the filing request object
     * @param model A list of filing calendar edits to be made
     * @return FilingRequestModel
     */
    public Future<FilingRequestModel> updateFilingRequestAsync(Integer companyId, Integer id, FilingRequestModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingRequestModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<FilingRequestModel>(){}));
    }

    /**
     * Approve all filings for the specified company in the given filing period.
     * 
     * This API is available by invitation only.
     * Approving a return means the customer is ready to let Avalara file that return.
     * Customer either approves themselves from admin console,
     * else system auto-approves the night before the filing cycle.
     * Sometimes Compliance has to manually unapprove and reapprove to modify liability or filing for the customer.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period to approve.
     * @param month The month of the filing period to approve.
     * @param model The approve request you wish to execute.
     * @return ArrayList<FilingModel>
     */
    public ArrayList<FilingModel> approveFilings(Integer companyId, Short year, Byte month, ApproveFilingsModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/approve");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return ((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){})).call();
    }

    /**
     * Approve all filings for the specified company in the given filing period.
     * 
     * This API is available by invitation only.
     * Approving a return means the customer is ready to let Avalara file that return.
     * Customer either approves themselves from admin console,
     * else system auto-approves the night before the filing cycle.
     * Sometimes Compliance has to manually unapprove and reapprove to modify liability or filing for the customer.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period to approve.
     * @param month The month of the filing period to approve.
     * @param model The approve request you wish to execute.
     * @return ArrayList<FilingModel>
     */
    public Future<ArrayList<FilingModel>> approveFilingsAsync(Integer companyId, Short year, Byte month, ApproveFilingsModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/approve");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return this.threadPool.submit((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){}));
    }

    /**
     * Approve all filings for the specified company in the given filing period and country.
     * 
     * This API is available by invitation only.
     * Approving a return means the customer is ready to let Avalara file that return.
     * Customer either approves themselves from admin console,
     * else system auto-approves the night before the filing cycle.
     * Sometimes Compliance has to manually unapprove and reapprove to modify liability or filing for the customer.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period to approve.
     * @param month The month of the filing period to approve.
     * @param country The two-character ISO-3166 code for the country.
     * @param model The approve request you wish to execute.
     * @return ArrayList<FilingModel>
     */
    public ArrayList<FilingModel> approveFilingsCountry(Integer companyId, Short year, Byte month, String country, ApproveFilingsModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/approve");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        return ((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){})).call();
    }

    /**
     * Approve all filings for the specified company in the given filing period and country.
     * 
     * This API is available by invitation only.
     * Approving a return means the customer is ready to let Avalara file that return.
     * Customer either approves themselves from admin console,
     * else system auto-approves the night before the filing cycle.
     * Sometimes Compliance has to manually unapprove and reapprove to modify liability or filing for the customer.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period to approve.
     * @param month The month of the filing period to approve.
     * @param country The two-character ISO-3166 code for the country.
     * @param model The approve request you wish to execute.
     * @return ArrayList<FilingModel>
     */
    public Future<ArrayList<FilingModel>> approveFilingsCountryAsync(Integer companyId, Short year, Byte month, String country, ApproveFilingsModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/approve");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        return this.threadPool.submit((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){}));
    }

    /**
     * Approve all filings for the specified company in the given filing period, country and region.
     * 
     * This API is available by invitation only.
     * Approving a return means the customer is ready to let Avalara file that return.
     * Customer either approves themselves from admin console,
     * else system auto-approves the night before the filing cycle
     * Sometimes Compliance has to manually unapprove and reapprove to modify liability or filing for the customer.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period to approve.
     * @param month The month of the filing period to approve.
     * @param country The two-character ISO-3166 code for the country.
     * @param region The two or three character region code for the region.
     * @param model The approve request you wish to execute.
     * @return ArrayList<FilingModel>
     */
    public ArrayList<FilingModel> approveFilingsCountryRegion(Integer companyId, Short year, Byte month, String country, String region, ApproveFilingsModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/approve");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        return ((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){})).call();
    }

    /**
     * Approve all filings for the specified company in the given filing period, country and region.
     * 
     * This API is available by invitation only.
     * Approving a return means the customer is ready to let Avalara file that return.
     * Customer either approves themselves from admin console,
     * else system auto-approves the night before the filing cycle
     * Sometimes Compliance has to manually unapprove and reapprove to modify liability or filing for the customer.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period to approve.
     * @param month The month of the filing period to approve.
     * @param country The two-character ISO-3166 code for the country.
     * @param region The two or three character region code for the region.
     * @param model The approve request you wish to execute.
     * @return ArrayList<FilingModel>
     */
    public Future<ArrayList<FilingModel>> approveFilingsCountryRegionAsync(Integer companyId, Short year, Byte month, String country, String region, ApproveFilingsModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/approve");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        return this.threadPool.submit((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){}));
    }

    /**
     * Add an adjustment to a given filing.
     * 
     * This API is available by invitation only.
     * An "Adjustment" is usually an increase or decrease to customer funding to Avalara,
     * such as early filer discount amounts that are refunded to the customer, or efile fees from websites.
     * Sometimes may be a manual change in tax liability similar to an augmentation.
     * This API creates a new adjustment for an existing tax filing.
     * This API can only be used when the filing has not yet been approved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being adjusted.
     * @param year The year of the filing's filing period being adjusted.
     * @param month The month of the filing's filing period being adjusted.
     * @param country The two-character ISO-3166 code for the country of the filing being adjusted.
     * @param region The two or three character region code for the region.
     * @param formCode The unique code of the form being adjusted.
     * @param model A list of Adjustments to be created for the specified filing.
     * @return ArrayList<FilingAdjustmentModel>
     */
    public ArrayList<FilingAdjustmentModel> createReturnAdjustment(Integer companyId, Short year, Byte month, String country, String region, String formCode, ArrayList<FilingAdjustmentModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/{formCode}/adjust");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        path.applyField("formCode", formCode);
        return ((RestCall<ArrayList<FilingAdjustmentModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingAdjustmentModel>>(){})).call();
    }

    /**
     * Add an adjustment to a given filing.
     * 
     * This API is available by invitation only.
     * An "Adjustment" is usually an increase or decrease to customer funding to Avalara,
     * such as early filer discount amounts that are refunded to the customer, or efile fees from websites.
     * Sometimes may be a manual change in tax liability similar to an augmentation.
     * This API creates a new adjustment for an existing tax filing.
     * This API can only be used when the filing has not yet been approved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being adjusted.
     * @param year The year of the filing's filing period being adjusted.
     * @param month The month of the filing's filing period being adjusted.
     * @param country The two-character ISO-3166 code for the country of the filing being adjusted.
     * @param region The two or three character region code for the region.
     * @param formCode The unique code of the form being adjusted.
     * @param model A list of Adjustments to be created for the specified filing.
     * @return ArrayList<FilingAdjustmentModel>
     */
    public Future<ArrayList<FilingAdjustmentModel>> createReturnAdjustmentAsync(Integer companyId, Short year, Byte month, String country, String region, String formCode, ArrayList<FilingAdjustmentModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/{formCode}/adjust");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        path.applyField("formCode", formCode);
        return this.threadPool.submit((RestCall<ArrayList<FilingAdjustmentModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingAdjustmentModel>>(){}));
    }

    /**
     * Add an augmentation for a given filing.
     * 
     * This API is available by invitation only.
     * An "Augmentation" is a manually added increase or decrease in tax liability, by either customer or Avalara
     * usually due to customer wanting to report tax Avatax does not support, e.g. bad debts, rental tax.
     * This API creates a new augmentation for an existing tax filing.
     * This API can only be used when the filing has not been approved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being changed.
     * @param year The month of the filing's filing period being changed.
     * @param month The month of the filing's filing period being changed.
     * @param country The two-character ISO-3166 code for the country of the filing being changed.
     * @param region The two or three character region code for the region of the filing being changed.
     * @param formCode The unique code of the form being changed.
     * @param model A list of augmentations to be created for the specified filing.
     * @return ArrayList<FilingAugmentationModel>
     */
    public ArrayList<FilingAugmentationModel> createReturnAugmentation(Integer companyId, Short year, Byte month, String country, String region, String formCode, ArrayList<FilingAugmentationModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/{formCode}/augment");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        path.applyField("formCode", formCode);
        return ((RestCall<ArrayList<FilingAugmentationModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingAugmentationModel>>(){})).call();
    }

    /**
     * Add an augmentation for a given filing.
     * 
     * This API is available by invitation only.
     * An "Augmentation" is a manually added increase or decrease in tax liability, by either customer or Avalara
     * usually due to customer wanting to report tax Avatax does not support, e.g. bad debts, rental tax.
     * This API creates a new augmentation for an existing tax filing.
     * This API can only be used when the filing has not been approved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being changed.
     * @param year The month of the filing's filing period being changed.
     * @param month The month of the filing's filing period being changed.
     * @param country The two-character ISO-3166 code for the country of the filing being changed.
     * @param region The two or three character region code for the region of the filing being changed.
     * @param formCode The unique code of the form being changed.
     * @param model A list of augmentations to be created for the specified filing.
     * @return ArrayList<FilingAugmentationModel>
     */
    public Future<ArrayList<FilingAugmentationModel>> createReturnAugmentationAsync(Integer companyId, Short year, Byte month, String country, String region, String formCode, ArrayList<FilingAugmentationModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/{formCode}/augment");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        path.applyField("formCode", formCode);
        return this.threadPool.submit((RestCall<ArrayList<FilingAugmentationModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingAugmentationModel>>(){}));
    }

    /**
     * Add an payment to a given filing.
     * 
     * This API is available by invitation only.
     * An "Payment" is usually an increase or decrease to customer funding to Avalara,
     * such as early filer discount amounts that are refunded to the customer, or efile fees from websites.
     * Sometimes may be a manual change in tax liability similar to an augmentation.
     * This API creates a new payment for an existing tax filing.
     * This API can only be used when the filing has not yet been approved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being adjusted.
     * @param year The year of the filing's filing period being adjusted.
     * @param month The month of the filing's filing period being adjusted.
     * @param country The two-character ISO-3166 code for the country of the filing being adjusted.
     * @param region The two or three character region code for the region.
     * @param formCode The unique code of the form being adjusted.
     * @param model A list of Payments to be created for the specified filing.
     * @return ArrayList<FilingPaymentModel>
     */
    public ArrayList<FilingPaymentModel> createReturnPayment(Integer companyId, Short year, Byte month, String country, String region, String formCode, ArrayList<FilingPaymentModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/{formCode}/payment");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        path.applyField("formCode", formCode);
        return ((RestCall<ArrayList<FilingPaymentModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingPaymentModel>>(){})).call();
    }

    /**
     * Add an payment to a given filing.
     * 
     * This API is available by invitation only.
     * An "Payment" is usually an increase or decrease to customer funding to Avalara,
     * such as early filer discount amounts that are refunded to the customer, or efile fees from websites.
     * Sometimes may be a manual change in tax liability similar to an augmentation.
     * This API creates a new payment for an existing tax filing.
     * This API can only be used when the filing has not yet been approved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being adjusted.
     * @param year The year of the filing's filing period being adjusted.
     * @param month The month of the filing's filing period being adjusted.
     * @param country The two-character ISO-3166 code for the country of the filing being adjusted.
     * @param region The two or three character region code for the region.
     * @param formCode The unique code of the form being adjusted.
     * @param model A list of Payments to be created for the specified filing.
     * @return ArrayList<FilingPaymentModel>
     */
    public Future<ArrayList<FilingPaymentModel>> createReturnPaymentAsync(Integer companyId, Short year, Byte month, String country, String region, String formCode, ArrayList<FilingPaymentModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/{formCode}/payment");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        path.applyField("formCode", formCode);
        return this.threadPool.submit((RestCall<ArrayList<FilingPaymentModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingPaymentModel>>(){}));
    }

    /**
     * Delete an adjustment for a given filing.
     * 
     * This API is available by invitation only.
     * An "Adjustment" is usually an increase or decrease to customer funding to Avalara,
     * such as early filer discount amounts that are refunded to the customer, or efile fees from websites.
     * Sometimes may be a manual change in tax liability similar to an augmentation.
     * This API deletes an adjustment for an existing tax filing.
     * This API can only be used when the filing has been unapproved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being adjusted.
     * @param id The ID of the adjustment being deleted.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteReturnAdjustment(Integer companyId, Long id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/adjust/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete an adjustment for a given filing.
     * 
     * This API is available by invitation only.
     * An "Adjustment" is usually an increase or decrease to customer funding to Avalara,
     * such as early filer discount amounts that are refunded to the customer, or efile fees from websites.
     * Sometimes may be a manual change in tax liability similar to an augmentation.
     * This API deletes an adjustment for an existing tax filing.
     * This API can only be used when the filing has been unapproved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being adjusted.
     * @param id The ID of the adjustment being deleted.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteReturnAdjustmentAsync(Integer companyId, Long id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/adjust/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Delete an augmentation for a given filing.
     * 
     * This API is available by invitation only.
     * An "Augmentation" is a manually added increase or decrease in tax liability, by either customer or Avalara
     * usually due to customer wanting to report tax Avatax does not support, e.g. bad debts, rental tax.
     * This API deletes an augmentation for an existing tax filing.
     * This API can only be used when the filing has been unapproved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being changed.
     * @param id The ID of the augmentation being added.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteReturnAugmentation(Integer companyId, Long id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/augment/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete an augmentation for a given filing.
     * 
     * This API is available by invitation only.
     * An "Augmentation" is a manually added increase or decrease in tax liability, by either customer or Avalara
     * usually due to customer wanting to report tax Avatax does not support, e.g. bad debts, rental tax.
     * This API deletes an augmentation for an existing tax filing.
     * This API can only be used when the filing has been unapproved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being changed.
     * @param id The ID of the augmentation being added.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteReturnAugmentationAsync(Integer companyId, Long id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/augment/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Delete an payment for a given filing.
     * 
     * This API is available by invitation only.
     * An "Payment" is usually an increase or decrease to customer funding to Avalara,
     * such as early filer discount amounts that are refunded to the customer, or efile fees from websites.
     * Sometimes may be a manual change in tax liability similar to an augmentation.
     * This API deletes an payment for an existing tax filing.
     * This API can only be used when the filing has been unapproved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being adjusted.
     * @param id The ID of the payment being deleted.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteReturnPayment(Integer companyId, Long id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/payment/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete an payment for a given filing.
     * 
     * This API is available by invitation only.
     * An "Payment" is usually an increase or decrease to customer funding to Avalara,
     * such as early filer discount amounts that are refunded to the customer, or efile fees from websites.
     * Sometimes may be a manual change in tax liability similar to an augmentation.
     * This API deletes an payment for an existing tax filing.
     * This API can only be used when the filing has been unapproved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being adjusted.
     * @param id The ID of the payment being deleted.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteReturnPaymentAsync(Integer companyId, Long id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/payment/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Retrieve worksheet checkup report for company and filing period.
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param filingsId The unique id of the worksheet.
     * @param companyId The unique ID of the company that owns the worksheet.
     * @return FilingsCheckupModel
     */
    public FilingsCheckupModel filingsCheckupReport(Integer filingsId, Integer companyId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{filingsId}/checkup");
        path.applyField("filingsId", filingsId);
        path.applyField("companyId", companyId);
        return ((RestCall<FilingsCheckupModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingsCheckupModel>(){})).call();
    }

    /**
     * Retrieve worksheet checkup report for company and filing period.
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param filingsId The unique id of the worksheet.
     * @param companyId The unique ID of the company that owns the worksheet.
     * @return FilingsCheckupModel
     */
    public Future<FilingsCheckupModel> filingsCheckupReportAsync(Integer filingsId, Integer companyId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{filingsId}/checkup");
        path.applyField("filingsId", filingsId);
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<FilingsCheckupModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingsCheckupModel>(){}));
    }

    /**
     * Retrieve worksheet checkup report for company and filing period.
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the worksheets object.
     * @param year The year of the filing period.
     * @param month The month of the filing period.
     * @return FilingsCheckupModel
     */
    public FilingsCheckupModel filingsCheckupReports(Integer companyId, Short year, Byte month) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/checkup");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return ((RestCall<FilingsCheckupModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingsCheckupModel>(){})).call();
    }

    /**
     * Retrieve worksheet checkup report for company and filing period.
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The unique ID of the company that owns the worksheets object.
     * @param year The year of the filing period.
     * @param month The month of the filing period.
     * @return FilingsCheckupModel
     */
    public Future<FilingsCheckupModel> filingsCheckupReportsAsync(Integer companyId, Short year, Byte month) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/checkup");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return this.threadPool.submit((RestCall<FilingsCheckupModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingsCheckupModel>(){}));
    }

    /**
     * Retrieve a list of filings for the specified accrual return.
     * 
     * 
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param companyId The ID of the company that owns these batches
     * @param filingReturnId The ID of the accrual return
     * @return FetchResult<FilingReturnModel>
     */
    public FetchResult<FilingReturnModel> getAccrualFillings(Integer companyId, Long filingReturnId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/accrual/{filingReturnId}");
        path.applyField("companyId", companyId);
        path.applyField("filingReturnId", filingReturnId);
        return ((RestCall<FetchResult<FilingReturnModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingReturnModel>>(){})).call();
    }

    /**
     * Retrieve a list of filings for the specified accrual return.
     * 
     * 
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param companyId The ID of the company that owns these batches
     * @param filingReturnId The ID of the accrual return
     * @return FetchResult<FilingReturnModel>
     */
    public Future<FetchResult<FilingReturnModel>> getAccrualFillingsAsync(Integer companyId, Long filingReturnId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/accrual/{filingReturnId}");
        path.applyField("companyId", companyId);
        path.applyField("filingReturnId", filingReturnId);
        return this.threadPool.submit((RestCall<FetchResult<FilingReturnModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingReturnModel>>(){}));
    }

    /**
     * Retrieve a single attachment for a filing
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param filingReturnId The unique id of the worksheet return.
     * @param fileId The unique id of the document you are downloading
     * @return String
     */
    public String getFilingAttachment(Integer companyId, Long filingReturnId, Long fileId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{filingReturnId}/attachment");
        path.applyField("companyId", companyId);
        path.applyField("filingReturnId", filingReturnId);
        path.addQuery("fileId", fileId);
        return ((RestCall<String>)restCallFactory.createRestCall("get", path, null, new TypeToken<String>(){})).call();
    }

    /**
     * Retrieve a single attachment for a filing
     * 
     * This API is available by invitation only.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param filingReturnId The unique id of the worksheet return.
     * @param fileId The unique id of the document you are downloading
     * @return String
     */
    public Future<String> getFilingAttachmentAsync(Integer companyId, Long filingReturnId, Long fileId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{filingReturnId}/attachment");
        path.applyField("companyId", companyId);
        path.applyField("filingReturnId", filingReturnId);
        path.addQuery("fileId", fileId);
        return this.threadPool.submit((RestCall<String>)restCallFactory.createRestCall("get", path, null, new TypeToken<String>(){}));
    }

    /**
     * Retrieve a list of filings for the specified company in the year and month of a given filing period.
     * 
     * This API is available by invitation only.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period.
     * @param month The two digit month of the filing period.
     * @return String
     */
    public String getFilingAttachments(Integer companyId, Short year, Byte month) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/attachments");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return ((RestCall<String>)restCallFactory.createRestCall("get", path, null, new TypeToken<String>(){})).call();
    }

    /**
     * Retrieve a list of filings for the specified company in the year and month of a given filing period.
     * 
     * This API is available by invitation only.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period.
     * @param month The two digit month of the filing period.
     * @return String
     */
    public Future<String> getFilingAttachmentsAsync(Integer companyId, Short year, Byte month) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/attachments");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return this.threadPool.submit((RestCall<String>)restCallFactory.createRestCall("get", path, null, new TypeToken<String>(){}));
    }

    /**
     * Retrieve a single trace file for a company filing period
     * 
     * This API is available by invitation only.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period.
     * @param month The two digit month of the filing period.
     * @return String
     */
    public String getFilingAttachmentsTraceFile(Integer companyId, Short year, Byte month) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/attachments/tracefile");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return ((RestCall<String>)restCallFactory.createRestCall("get", path, null, new TypeToken<String>(){})).call();
    }

    /**
     * Retrieve a single trace file for a company filing period
     * 
     * This API is available by invitation only.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period.
     * @param month The two digit month of the filing period.
     * @return String
     */
    public Future<String> getFilingAttachmentsTraceFileAsync(Integer companyId, Short year, Byte month) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/attachments/tracefile");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return this.threadPool.submit((RestCall<String>)restCallFactory.createRestCall("get", path, null, new TypeToken<String>(){}));
    }

    /**
     * Retrieve a filing for the specified company and id.
     * 
     * This API is available by invitation only.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param id The id of the filing return your retrieving
     * @param details Indicates if you would like the credit details returned
     * @return FilingReturnModel
     */
    public FilingReturnModel getFilingReturn(Integer companyId, Integer id, Boolean details) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/returns/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.addQuery("details", details);
        return ((RestCall<FilingReturnModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingReturnModel>(){})).call();
    }

    /**
     * Retrieve a filing for the specified company and id.
     * 
     * This API is available by invitation only.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param id The id of the filing return your retrieving
     * @param details Indicates if you would like the credit details returned
     * @return FilingReturnModel
     */
    public Future<FilingReturnModel> getFilingReturnAsync(Integer companyId, Integer id, Boolean details) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/returns/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.addQuery("details", details);
        return this.threadPool.submit((RestCall<FilingReturnModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingReturnModel>(){}));
    }

    /**
     * Retrieve a list of filings for the specified company in the year and month of a given filing period.
     * 
     * This API is available by invitation only.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period.
     * @param month The two digit month of the filing period.
     * @return FetchResult<FilingModel>
     */
    public FetchResult<FilingModel> getFilings(Integer companyId, Short year, Byte month) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return ((RestCall<FetchResult<FilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingModel>>(){})).call();
    }

    /**
     * Retrieve a list of filings for the specified company in the year and month of a given filing period.
     * 
     * This API is available by invitation only.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period.
     * @param month The two digit month of the filing period.
     * @return FetchResult<FilingModel>
     */
    public Future<FetchResult<FilingModel>> getFilingsAsync(Integer companyId, Short year, Byte month) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return this.threadPool.submit((RestCall<FetchResult<FilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingModel>>(){}));
    }

    /**
     * Retrieve a list of filings for the specified company in the given filing period and country.
     * 
     * This API is available by invitation only.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period.
     * @param month The two digit month of the filing period.
     * @param country The two-character ISO-3166 code for the country.
     * @return FetchResult<FilingModel>
     */
    public FetchResult<FilingModel> getFilingsByCountry(Integer companyId, Short year, Byte month, String country) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        return ((RestCall<FetchResult<FilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingModel>>(){})).call();
    }

    /**
     * Retrieve a list of filings for the specified company in the given filing period and country.
     * 
     * This API is available by invitation only.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period.
     * @param month The two digit month of the filing period.
     * @param country The two-character ISO-3166 code for the country.
     * @return FetchResult<FilingModel>
     */
    public Future<FetchResult<FilingModel>> getFilingsByCountryAsync(Integer companyId, Short year, Byte month, String country) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        return this.threadPool.submit((RestCall<FetchResult<FilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingModel>>(){}));
    }

    /**
     * Retrieve a list of filings for the specified company in the filing period, country and region.
     * 
     * This API is available by invitation only.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period.
     * @param month The two digit month of the filing period.
     * @param country The two-character ISO-3166 code for the country.
     * @param region The two or three character region code for the region.
     * @return FetchResult<FilingModel>
     */
    public FetchResult<FilingModel> getFilingsByCountryRegion(Integer companyId, Short year, Byte month, String country, String region) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        return ((RestCall<FetchResult<FilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingModel>>(){})).call();
    }

    /**
     * Retrieve a list of filings for the specified company in the filing period, country and region.
     * 
     * This API is available by invitation only.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period.
     * @param month The two digit month of the filing period.
     * @param country The two-character ISO-3166 code for the country.
     * @param region The two or three character region code for the region.
     * @return FetchResult<FilingModel>
     */
    public Future<FetchResult<FilingModel>> getFilingsByCountryRegionAsync(Integer companyId, Short year, Byte month, String country, String region) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        return this.threadPool.submit((RestCall<FetchResult<FilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingModel>>(){}));
    }

    /**
     * Retrieve a list of filings for the specified company in the given filing period, country, region and form.
     * 
     * This API is available by invitation only.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period.
     * @param month The two digit month of the filing period.
     * @param country The two-character ISO-3166 code for the country.
     * @param region The two or three character region code for the region.
     * @param formCode The unique code of the form.
     * @return FetchResult<FilingModel>
     */
    public FetchResult<FilingModel> getFilingsByReturnName(Integer companyId, Short year, Byte month, String country, String region, String formCode) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/{formCode}");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        path.applyField("formCode", formCode);
        return ((RestCall<FetchResult<FilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingModel>>(){})).call();
    }

    /**
     * Retrieve a list of filings for the specified company in the given filing period, country, region and form.
     * 
     * This API is available by invitation only.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period.
     * @param month The two digit month of the filing period.
     * @param country The two-character ISO-3166 code for the country.
     * @param region The two or three character region code for the region.
     * @param formCode The unique code of the form.
     * @return FetchResult<FilingModel>
     */
    public Future<FetchResult<FilingModel>> getFilingsByReturnNameAsync(Integer companyId, Short year, Byte month, String country, String region, String formCode) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/{formCode}");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        path.applyField("formCode", formCode);
        return this.threadPool.submit((RestCall<FetchResult<FilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingModel>>(){}));
    }

    /**
     * Retrieve a list of filings for the specified company in the year and month of a given filing period.
This gets the basic information from the filings and doesn't include anything extra.
     * 
     * 
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param companyId The ID of the company that owns these batches
     * @param endPeriodMonth The month of the period you are trying to retrieve
     * @param endPeriodYear The year of the period you are trying to retrieve
     * @param frequency The frequency of the return you are trying to retrieve (See FilingFrequencyId::* for a list of allowable values)
     * @param status The status of the return(s) you are trying to retrieve (See FilingStatusId::* for a list of allowable values)
     * @param country The country of the return(s) you are trying to retrieve
     * @param region The region of the return(s) you are trying to retrieve
     * @param filingCalendarId The filing calendar id of the return you are trying to retrieve
     * @return FetchResult<FilingReturnModelBasic>
     */
    public FetchResult<FilingReturnModelBasic> getFilingsReturns(Integer companyId, Integer endPeriodMonth, Integer endPeriodYear, FilingFrequencyId frequency, FilingStatusId status, String country, String region, Long filingCalendarId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/returns");
        path.applyField("companyId", companyId);
        path.addQuery("endPeriodMonth", endPeriodMonth);
        path.addQuery("endPeriodYear", endPeriodYear);
        path.addQuery("frequency", frequency);
        path.addQuery("status", status);
        path.addQuery("country", country);
        path.addQuery("region", region);
        path.addQuery("filingCalendarId", filingCalendarId);
        return ((RestCall<FetchResult<FilingReturnModelBasic>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingReturnModelBasic>>(){})).call();
    }

    /**
     * Retrieve a list of filings for the specified company in the year and month of a given filing period.
This gets the basic information from the filings and doesn't include anything extra.
     * 
     * 
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param companyId The ID of the company that owns these batches
     * @param endPeriodMonth The month of the period you are trying to retrieve
     * @param endPeriodYear The year of the period you are trying to retrieve
     * @param frequency The frequency of the return you are trying to retrieve (See FilingFrequencyId::* for a list of allowable values)
     * @param status The status of the return(s) you are trying to retrieve (See FilingStatusId::* for a list of allowable values)
     * @param country The country of the return(s) you are trying to retrieve
     * @param region The region of the return(s) you are trying to retrieve
     * @param filingCalendarId The filing calendar id of the return you are trying to retrieve
     * @return FetchResult<FilingReturnModelBasic>
     */
    public Future<FetchResult<FilingReturnModelBasic>> getFilingsReturnsAsync(Integer companyId, Integer endPeriodMonth, Integer endPeriodYear, FilingFrequencyId frequency, FilingStatusId status, String country, String region, Long filingCalendarId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/returns");
        path.applyField("companyId", companyId);
        path.addQuery("endPeriodMonth", endPeriodMonth);
        path.addQuery("endPeriodYear", endPeriodYear);
        path.addQuery("frequency", frequency);
        path.addQuery("status", status);
        path.addQuery("country", country);
        path.addQuery("region", region);
        path.addQuery("filingCalendarId", filingCalendarId);
        return this.threadPool.submit((RestCall<FetchResult<FilingReturnModelBasic>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingReturnModelBasic>>(){}));
    }

    /**
     * Retrieve a list of filings for the specified company in the year and month of a given filing period.
     * 
     * This API is available by invitation only.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period.
     * @param month The two digit month of the filing period.
     * @param country The two-character ISO-3166 code for the country.
     * @param region The two or three character region code for the region.
     * @param formCode The unique code of the form.
     * @return FetchResult<MultiTaxFilingModel>
     */
    public FetchResult<MultiTaxFilingModel> getTaxFilings(Integer companyId, Integer year, Integer month, String country, String region, String formCode) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings");
        path.applyField("companyId", companyId);
        path.addQuery("year", year);
        path.addQuery("month", month);
        path.addQuery("country", country);
        path.addQuery("region", region);
        path.addQuery("formCode", formCode);
        return ((RestCall<FetchResult<MultiTaxFilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<MultiTaxFilingModel>>(){})).call();
    }

    /**
     * Retrieve a list of filings for the specified company in the year and month of a given filing period.
     * 
     * This API is available by invitation only.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPTester, FirmAdmin, FirmUser, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period.
     * @param month The two digit month of the filing period.
     * @param country The two-character ISO-3166 code for the country.
     * @param region The two or three character region code for the region.
     * @param formCode The unique code of the form.
     * @return FetchResult<MultiTaxFilingModel>
     */
    public Future<FetchResult<MultiTaxFilingModel>> getTaxFilingsAsync(Integer companyId, Integer year, Integer month, String country, String region, String formCode) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings");
        path.applyField("companyId", companyId);
        path.addQuery("year", year);
        path.addQuery("month", month);
        path.addQuery("country", country);
        path.addQuery("region", region);
        path.addQuery("formCode", formCode);
        return this.threadPool.submit((RestCall<FetchResult<MultiTaxFilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<MultiTaxFilingModel>>(){}));
    }

    /**
     * Rebuild a set of filings for the specified company in the given filing period.
     * 
     * This API is available by invitation only.
     * Rebuilding a return means re-creating or updating the amounts to be filed (worksheet) for a filing.
     * Rebuilding has to be done whenever a customer adds transactions to a filing.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * This API requires filing to be unapproved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period to be rebuilt.
     * @param month The month of the filing period to be rebuilt.
     * @param model The rebuild request you wish to execute.
     * @return FetchResult<FilingModel>
     */
    public FetchResult<FilingModel> rebuildFilings(Integer companyId, Short year, Byte month, RebuildFilingsModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/rebuild");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return ((RestCall<FetchResult<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<FilingModel>>(){})).call();
    }

    /**
     * Rebuild a set of filings for the specified company in the given filing period.
     * 
     * This API is available by invitation only.
     * Rebuilding a return means re-creating or updating the amounts to be filed (worksheet) for a filing.
     * Rebuilding has to be done whenever a customer adds transactions to a filing.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * This API requires filing to be unapproved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period to be rebuilt.
     * @param month The month of the filing period to be rebuilt.
     * @param model The rebuild request you wish to execute.
     * @return FetchResult<FilingModel>
     */
    public Future<FetchResult<FilingModel>> rebuildFilingsAsync(Integer companyId, Short year, Byte month, RebuildFilingsModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/rebuild");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return this.threadPool.submit((RestCall<FetchResult<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<FilingModel>>(){}));
    }

    /**
     * Rebuild a set of filings for the specified company in the given filing period and country.
     * 
     * This API is available by invitation only.
     * Rebuilding a return means re-creating or updating the amounts to be filed (worksheet) for a filing.
     * Rebuilding has to be done whenever a customer adds transactions to a filing.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * This API requires filing to be unapproved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period to be rebuilt.
     * @param month The month of the filing period to be rebuilt.
     * @param country The two-character ISO-3166 code for the country.
     * @param model The rebuild request you wish to execute.
     * @return FetchResult<FilingModel>
     */
    public FetchResult<FilingModel> rebuildFilingsByCountry(Integer companyId, Short year, Byte month, String country, RebuildFilingsModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/rebuild");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        return ((RestCall<FetchResult<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<FilingModel>>(){})).call();
    }

    /**
     * Rebuild a set of filings for the specified company in the given filing period and country.
     * 
     * This API is available by invitation only.
     * Rebuilding a return means re-creating or updating the amounts to be filed (worksheet) for a filing.
     * Rebuilding has to be done whenever a customer adds transactions to a filing.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * This API requires filing to be unapproved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period to be rebuilt.
     * @param month The month of the filing period to be rebuilt.
     * @param country The two-character ISO-3166 code for the country.
     * @param model The rebuild request you wish to execute.
     * @return FetchResult<FilingModel>
     */
    public Future<FetchResult<FilingModel>> rebuildFilingsByCountryAsync(Integer companyId, Short year, Byte month, String country, RebuildFilingsModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/rebuild");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        return this.threadPool.submit((RestCall<FetchResult<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<FilingModel>>(){}));
    }

    /**
     * Rebuild a set of filings for the specified company in the given filing period, country and region.
     * 
     * This API is available by invitation only.audit.CheckAuthorizationReturns(null, companyId);
     * Rebuilding a return means re-creating or updating the amounts to be filed for a filing.
     * Rebuilding has to be done whenever a customer adds transactions to a filing.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * This API requires filing to be unapproved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period to be rebuilt.
     * @param month The month of the filing period to be rebuilt.
     * @param country The two-character ISO-3166 code for the country.
     * @param region The two or three character region code for the region.
     * @param model The rebuild request you wish to execute.
     * @return FetchResult<FilingModel>
     */
    public FetchResult<FilingModel> rebuildFilingsByCountryRegion(Integer companyId, Short year, Byte month, String country, String region, RebuildFilingsModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/rebuild");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        return ((RestCall<FetchResult<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<FilingModel>>(){})).call();
    }

    /**
     * Rebuild a set of filings for the specified company in the given filing period, country and region.
     * 
     * This API is available by invitation only.audit.CheckAuthorizationReturns(null, companyId);
     * Rebuilding a return means re-creating or updating the amounts to be filed for a filing.
     * Rebuilding has to be done whenever a customer adds transactions to a filing.
     * A "filing period" is the year and month of the date of the latest customer transaction allowed to be reported on a filing,
     * based on filing frequency of filing.
     * This API requires filing to be unapproved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * 
     * @param companyId The ID of the company that owns the filings.
     * @param year The year of the filing period to be rebuilt.
     * @param month The month of the filing period to be rebuilt.
     * @param country The two-character ISO-3166 code for the country.
     * @param region The two or three character region code for the region.
     * @param model The rebuild request you wish to execute.
     * @return FetchResult<FilingModel>
     */
    public Future<FetchResult<FilingModel>> rebuildFilingsByCountryRegionAsync(Integer companyId, Short year, Byte month, String country, String region, RebuildFilingsModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/rebuild");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        return this.threadPool.submit((RestCall<FetchResult<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<FilingModel>>(){}));
    }

    /**
     * Edit an adjustment for a given filing.
     * 
     * This API is available by invitation only.
     * An "Adjustment" is usually an increase or decrease to customer funding to Avalara,
     * such as early filer discount amounts that are refunded to the customer, or efile fees from websites.
     * Sometimes may be a manual change in tax liability similar to an augmentation.
     * This API modifies an adjustment for an existing tax filing.
     * This API can only be used when the filing has not yet been approved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being adjusted.
     * @param id The ID of the adjustment being edited.
     * @param model The updated Adjustment.
     * @return FilingAdjustmentModel
     */
    public FilingAdjustmentModel updateReturnAdjustment(Integer companyId, Long id, FilingAdjustmentModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/adjust/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingAdjustmentModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<FilingAdjustmentModel>(){})).call();
    }

    /**
     * Edit an adjustment for a given filing.
     * 
     * This API is available by invitation only.
     * An "Adjustment" is usually an increase or decrease to customer funding to Avalara,
     * such as early filer discount amounts that are refunded to the customer, or efile fees from websites.
     * Sometimes may be a manual change in tax liability similar to an augmentation.
     * This API modifies an adjustment for an existing tax filing.
     * This API can only be used when the filing has not yet been approved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being adjusted.
     * @param id The ID of the adjustment being edited.
     * @param model The updated Adjustment.
     * @return FilingAdjustmentModel
     */
    public Future<FilingAdjustmentModel> updateReturnAdjustmentAsync(Integer companyId, Long id, FilingAdjustmentModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/adjust/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingAdjustmentModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<FilingAdjustmentModel>(){}));
    }

    /**
     * Edit an augmentation for a given filing.
     * 
     * This API is available by invitation only.
     * An "Augmentation" is a manually added increase or decrease in tax liability, by either customer or Avalara
     * usually due to customer wanting to report tax Avatax does not support, e.g. bad debts, rental tax.
     * This API modifies an augmentation for an existing tax filing.
     * This API can only be used when the filing has not been approved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being changed.
     * @param id The ID of the augmentation being edited.
     * @param model The updated Augmentation.
     * @return FilingModel
     */
    public FilingModel updateReturnAugmentation(Integer companyId, Long id, FilingAugmentationModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/augment/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<FilingModel>(){})).call();
    }

    /**
     * Edit an augmentation for a given filing.
     * 
     * This API is available by invitation only.
     * An "Augmentation" is a manually added increase or decrease in tax liability, by either customer or Avalara
     * usually due to customer wanting to report tax Avatax does not support, e.g. bad debts, rental tax.
     * This API modifies an augmentation for an existing tax filing.
     * This API can only be used when the filing has not been approved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being changed.
     * @param id The ID of the augmentation being edited.
     * @param model The updated Augmentation.
     * @return FilingModel
     */
    public Future<FilingModel> updateReturnAugmentationAsync(Integer companyId, Long id, FilingAugmentationModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/augment/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<FilingModel>(){}));
    }

    /**
     * Edit an payment for a given filing.
     * 
     * This API is available by invitation only.
     * An "Payment" is usually an increase or decrease to customer funding to Avalara,
     * such as early filer discount amounts that are refunded to the customer, or efile fees from websites.
     * Sometimes may be a manual change in tax liability similar to an augmentation.
     * This API modifies an payment for an existing tax filing.
     * This API can only be used when the filing has not yet been approved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being adjusted.
     * @param id The ID of the payment being edited.
     * @param model The updated Payment.
     * @return FilingPaymentModel
     */
    public FilingPaymentModel updateReturnPayment(Integer companyId, Long id, FilingPaymentModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/payment/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingPaymentModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<FilingPaymentModel>(){})).call();
    }

    /**
     * Edit an payment for a given filing.
     * 
     * This API is available by invitation only.
     * An "Payment" is usually an increase or decrease to customer funding to Avalara,
     * such as early filer discount amounts that are refunded to the customer, or efile fees from websites.
     * Sometimes may be a manual change in tax liability similar to an augmentation.
     * This API modifies an payment for an existing tax filing.
     * This API can only be used when the filing has not yet been approved.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, Compliance Root User, ComplianceAdmin, CSPTester, FirmUser, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns the filing being adjusted.
     * @param id The ID of the payment being edited.
     * @param model The updated Payment.
     * @return FilingPaymentModel
     */
    public Future<FilingPaymentModel> updateReturnPaymentAsync(Integer companyId, Long id, FilingPaymentModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/payment/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingPaymentModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<FilingPaymentModel>(){}));
    }

    /**
     * FREE API - Request a free trial of AvaTax
     * 
     * Call this API to obtain a free AvaTax account.
     *             
     * This API is free to use.  No authentication credentials are required to call this API. You must read and
     * accept [Avalara's terms and conditions](https://www1.avalara.com/us/en/legal/terms.html) for the account to be
     * created.
     *             
     * If all conditions are met, this API will grant a free trial version of AvaTax.  For a list of functionality
     * available in the free trial and its limitations, please see the [AvaTax Developer Website Free Trial page](https://developer.avalara.com/avatax/signup/).
     *             
     * After your free trial concludes, you will still be able to use the [Free AvaTax API Suite](https://developer.avalara.com/api-reference/avatax/rest/v2/methods/Free/).
     * 
     * ### Security Policies
     * 
     * * This API may be called without providing authentication credentials.
     * 
     * @param model Required information to provision a free trial account.
     * @return NewAccountModel
     */
    public NewAccountModel requestFreeTrial(FreeTrialRequestModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/freetrials/request");
        return ((RestCall<NewAccountModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<NewAccountModel>(){})).call();
    }

    /**
     * FREE API - Request a free trial of AvaTax
     * 
     * Call this API to obtain a free AvaTax account.
     *             
     * This API is free to use.  No authentication credentials are required to call this API. You must read and
     * accept [Avalara's terms and conditions](https://www1.avalara.com/us/en/legal/terms.html) for the account to be
     * created.
     *             
     * If all conditions are met, this API will grant a free trial version of AvaTax.  For a list of functionality
     * available in the free trial and its limitations, please see the [AvaTax Developer Website Free Trial page](https://developer.avalara.com/avatax/signup/).
     *             
     * After your free trial concludes, you will still be able to use the [Free AvaTax API Suite](https://developer.avalara.com/api-reference/avatax/rest/v2/methods/Free/).
     * 
     * ### Security Policies
     * 
     * * This API may be called without providing authentication credentials.
     * 
     * @param model Required information to provision a free trial account.
     * @return NewAccountModel
     */
    public Future<NewAccountModel> requestFreeTrialAsync(FreeTrialRequestModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/freetrials/request");
        return this.threadPool.submit((RestCall<NewAccountModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<NewAccountModel>(){}));
    }

    /**
     * FREE API - Sales tax rates for a specified address
     * 
     * # Free-To-Use
     * 
     * The TaxRates API is a free-to-use, no cost option for estimating sales tax rates.
     * Any customer can request a free AvaTax account and make use of the TaxRates API.
     * 
     * Usage of this API is subject to rate limits.  Users who exceed the rate limit will receive HTTP
     * response code 429 - `Too Many Requests`.
     * 
     * This API assumes that you are selling general tangible personal property at a retail point-of-sale
     * location in the United States only.  
     * 
     * For more powerful tax calculation, please consider upgrading to the `CreateTransaction` API,
     * which supports features including, but not limited to:
     * 
     * * Nexus declarations
     * * Taxability based on product/service type
     * * Sourcing rules affecting origin/destination states
     * * Customers who are exempt from certain taxes
     * * States that have dollar value thresholds for tax amounts
     * * Refunds for products purchased on a different date
     * * Detailed jurisdiction names and state assigned codes
     * * And more!
     * 
     * Please see [Estimating Tax with REST v2](http://developer.avalara.com/blog/2016/11/04/estimating-tax-with-rest-v2/)
     * 
     * @param line1 The street address of the location.
     * @param line2 The street address of the location.
     * @param line3 The street address of the location.
     * @param city The city name of the location.
     * @param region Name or ISO 3166 code identifying the region within the country.    This field supports many different region identifiers:   * Two and three character ISO 3166 region codes   * Fully spelled out names of the region in ISO supported languages   * Common alternative spellings for many regions    For a full list of all supported codes and names, please see the Definitions API `ListRegions`.
     * @param postalCode The postal code of the location.
     * @param country Name or ISO 3166 code identifying the country.    This field supports many different country identifiers:   * Two character ISO 3166 codes   * Three character ISO 3166 codes   * Fully spelled out names of the country in ISO supported languages   * Common alternative spellings for many countries    For a full list of all supported codes and names, please see the Definitions API `ListCountries`.
     * @return TaxRateModel
     */
    public TaxRateModel taxRatesByAddress(String line1, String line2, String line3, String city, String region, String postalCode, String country) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/taxrates/byaddress");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        return ((RestCall<TaxRateModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TaxRateModel>(){})).call();
    }

    /**
     * FREE API - Sales tax rates for a specified address
     * 
     * # Free-To-Use
     * 
     * The TaxRates API is a free-to-use, no cost option for estimating sales tax rates.
     * Any customer can request a free AvaTax account and make use of the TaxRates API.
     * 
     * Usage of this API is subject to rate limits.  Users who exceed the rate limit will receive HTTP
     * response code 429 - `Too Many Requests`.
     * 
     * This API assumes that you are selling general tangible personal property at a retail point-of-sale
     * location in the United States only.  
     * 
     * For more powerful tax calculation, please consider upgrading to the `CreateTransaction` API,
     * which supports features including, but not limited to:
     * 
     * * Nexus declarations
     * * Taxability based on product/service type
     * * Sourcing rules affecting origin/destination states
     * * Customers who are exempt from certain taxes
     * * States that have dollar value thresholds for tax amounts
     * * Refunds for products purchased on a different date
     * * Detailed jurisdiction names and state assigned codes
     * * And more!
     * 
     * Please see [Estimating Tax with REST v2](http://developer.avalara.com/blog/2016/11/04/estimating-tax-with-rest-v2/)
     * 
     * @param line1 The street address of the location.
     * @param line2 The street address of the location.
     * @param line3 The street address of the location.
     * @param city The city name of the location.
     * @param region Name or ISO 3166 code identifying the region within the country.    This field supports many different region identifiers:   * Two and three character ISO 3166 region codes   * Fully spelled out names of the region in ISO supported languages   * Common alternative spellings for many regions    For a full list of all supported codes and names, please see the Definitions API `ListRegions`.
     * @param postalCode The postal code of the location.
     * @param country Name or ISO 3166 code identifying the country.    This field supports many different country identifiers:   * Two character ISO 3166 codes   * Three character ISO 3166 codes   * Fully spelled out names of the country in ISO supported languages   * Common alternative spellings for many countries    For a full list of all supported codes and names, please see the Definitions API `ListCountries`.
     * @return TaxRateModel
     */
    public Future<TaxRateModel> taxRatesByAddressAsync(String line1, String line2, String line3, String city, String region, String postalCode, String country) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/taxrates/byaddress");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        return this.threadPool.submit((RestCall<TaxRateModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TaxRateModel>(){}));
    }

    /**
     * FREE API - Sales tax rates for a specified country and postal code. This API is only available for US postal codes.
     * 
     * # Free-To-Use
     * 
     * This API is only available for a US postal codes.
     * 
     * The TaxRates API is a free-to-use, no cost option for estimating sales tax rates.
     * Any customer can request a free AvaTax account and make use of the TaxRates API.
     * 
     * Usage of this API is subject to rate limits.  Users who exceed the rate limit will receive HTTP
     * response code 429 - `Too Many Requests`.
     * 
     * This API assumes that you are selling general tangible personal property at a retail point-of-sale
     * location in the United States only.  
     * 
     * For more powerful tax calculation, please consider upgrading to the `CreateTransaction` API,
     * which supports features including, but not limited to:
     * 
     * * Nexus declarations
     * * Taxability based on product/service type
     * * Sourcing rules affecting origin/destination states
     * * Customers who are exempt from certain taxes
     * * States that have dollar value thresholds for tax amounts
     * * Refunds for products purchased on a different date
     * * Detailed jurisdiction names and state assigned codes
     * * And more!
     * 
     * Please see [Estimating Tax with REST v2](http://developer.avalara.com/blog/2016/11/04/estimating-tax-with-rest-v2/)
     * 
     * @param country Name or ISO 3166 code identifying the country.    This field supports many different country identifiers:   * Two character ISO 3166 codes   * Three character ISO 3166 codes   * Fully spelled out names of the country in ISO supported languages   * Common alternative spellings for many countries    For a full list of all supported codes and names, please see the Definitions API `ListCountries`.
     * @param postalCode The postal code of the location.
     * @return TaxRateModel
     */
    public TaxRateModel taxRatesByPostalCode(String country, String postalCode) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/taxrates/bypostalcode");
        path.addQuery("country", country);
        path.addQuery("postalCode", postalCode);
        return ((RestCall<TaxRateModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TaxRateModel>(){})).call();
    }

    /**
     * FREE API - Sales tax rates for a specified country and postal code. This API is only available for US postal codes.
     * 
     * # Free-To-Use
     * 
     * This API is only available for a US postal codes.
     * 
     * The TaxRates API is a free-to-use, no cost option for estimating sales tax rates.
     * Any customer can request a free AvaTax account and make use of the TaxRates API.
     * 
     * Usage of this API is subject to rate limits.  Users who exceed the rate limit will receive HTTP
     * response code 429 - `Too Many Requests`.
     * 
     * This API assumes that you are selling general tangible personal property at a retail point-of-sale
     * location in the United States only.  
     * 
     * For more powerful tax calculation, please consider upgrading to the `CreateTransaction` API,
     * which supports features including, but not limited to:
     * 
     * * Nexus declarations
     * * Taxability based on product/service type
     * * Sourcing rules affecting origin/destination states
     * * Customers who are exempt from certain taxes
     * * States that have dollar value thresholds for tax amounts
     * * Refunds for products purchased on a different date
     * * Detailed jurisdiction names and state assigned codes
     * * And more!
     * 
     * Please see [Estimating Tax with REST v2](http://developer.avalara.com/blog/2016/11/04/estimating-tax-with-rest-v2/)
     * 
     * @param country Name or ISO 3166 code identifying the country.    This field supports many different country identifiers:   * Two character ISO 3166 codes   * Three character ISO 3166 codes   * Fully spelled out names of the country in ISO supported languages   * Common alternative spellings for many countries    For a full list of all supported codes and names, please see the Definitions API `ListCountries`.
     * @param postalCode The postal code of the location.
     * @return TaxRateModel
     */
    public Future<TaxRateModel> taxRatesByPostalCodeAsync(String country, String postalCode) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/taxrates/bypostalcode");
        path.addQuery("country", country);
        path.addQuery("postalCode", postalCode);
        return this.threadPool.submit((RestCall<TaxRateModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TaxRateModel>(){}));
    }

    /**
     * Request the javascript for a funding setup widget
     * 
     * This API is available by invitation only.
     * Companies that use the Avalara Managed Returns or the SST Certified Service Provider services are
     * required to setup their funding configuration before Avalara can begin filing tax returns on their
     * behalf.
     * Funding configuration for each company is set up by submitting a funding setup request, which can
     * be sent either via email or via an embedded HTML widget.
     * When the funding configuration is submitted to Avalara, it will be reviewed by treasury team members
     * before approval.
     * This API returns back the actual javascript code to insert into your application to render the
     * JavaScript funding setup widget inline.
     * Use the 'methodReturn.javaScript' return value to insert this widget into your HTML page.
     * This API requires a subscription to Avalara Managed Returns or SST Certified Service Provider.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param id The unique ID number of this funding request
     * @return FundingStatusModel
     */
    public FundingStatusModel activateFundingRequest(Long id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/fundingrequests/{id}/widget");
        path.applyField("id", id);
        return ((RestCall<FundingStatusModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FundingStatusModel>(){})).call();
    }

    /**
     * Request the javascript for a funding setup widget
     * 
     * This API is available by invitation only.
     * Companies that use the Avalara Managed Returns or the SST Certified Service Provider services are
     * required to setup their funding configuration before Avalara can begin filing tax returns on their
     * behalf.
     * Funding configuration for each company is set up by submitting a funding setup request, which can
     * be sent either via email or via an embedded HTML widget.
     * When the funding configuration is submitted to Avalara, it will be reviewed by treasury team members
     * before approval.
     * This API returns back the actual javascript code to insert into your application to render the
     * JavaScript funding setup widget inline.
     * Use the 'methodReturn.javaScript' return value to insert this widget into your HTML page.
     * This API requires a subscription to Avalara Managed Returns or SST Certified Service Provider.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param id The unique ID number of this funding request
     * @return FundingStatusModel
     */
    public Future<FundingStatusModel> activateFundingRequestAsync(Long id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/fundingrequests/{id}/widget");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FundingStatusModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FundingStatusModel>(){}));
    }

    /**
     * Retrieve status about a funding setup request
     * 
     * This API is available by invitation only.
     * Companies that use the Avalara Managed Returns or the SST Certified Service Provider services are
     * required to setup their funding configuration before Avalara can begin filing tax returns on their
     * behalf.
     * Funding configuration for each company is set up by submitting a funding setup request, which can
     * be sent either via email or via an embedded HTML widget.
     * When the funding configuration is submitted to Avalara, it will be reviewed by treasury team members
     * before approval.
     * This API checks the status on an existing funding request.
     * This API requires a subscription to Avalara Managed Returns or SST Certified Service Provider.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param id The unique ID number of this funding request
     * @return FundingStatusModel
     */
    public FundingStatusModel fundingRequestStatus(Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/fundingrequests/{id}");
        path.applyField("id", id);
        return ((RestCall<FundingStatusModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FundingStatusModel>(){})).call();
    }

    /**
     * Retrieve status about a funding setup request
     * 
     * This API is available by invitation only.
     * Companies that use the Avalara Managed Returns or the SST Certified Service Provider services are
     * required to setup their funding configuration before Avalara can begin filing tax returns on their
     * behalf.
     * Funding configuration for each company is set up by submitting a funding setup request, which can
     * be sent either via email or via an embedded HTML widget.
     * When the funding configuration is submitted to Avalara, it will be reviewed by treasury team members
     * before approval.
     * This API checks the status on an existing funding request.
     * This API requires a subscription to Avalara Managed Returns or SST Certified Service Provider.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, FirmAdmin, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param id The unique ID number of this funding request
     * @return FundingStatusModel
     */
    public Future<FundingStatusModel> fundingRequestStatusAsync(Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/fundingrequests/{id}");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FundingStatusModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FundingStatusModel>(){}));
    }

    /**
     * Add classifications to an item.
     * 
     * Add classifications to an item.
     *             
     * A classification is the code for a product in a particular tax system. Classifications enable an item to be used in multiple tax systems which may have different tax rates for a product.
     *             
     * When an item is used in a transaction, the applicable classification will be used to determine the appropriate tax rate.
     *             
     * An item may only have one classification per tax system.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The company id.
     * @param itemId The item id.
     * @param model The item classifications you wish to create.
     * @return ArrayList<ItemClassificationOutputModel>
     */
    public ArrayList<ItemClassificationOutputModel> createItemClassifications(Integer companyId, Long itemId, ArrayList<ItemClassificationInputModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/classifications");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        return ((RestCall<ArrayList<ItemClassificationOutputModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<ItemClassificationOutputModel>>(){})).call();
    }

    /**
     * Add classifications to an item.
     * 
     * Add classifications to an item.
     *             
     * A classification is the code for a product in a particular tax system. Classifications enable an item to be used in multiple tax systems which may have different tax rates for a product.
     *             
     * When an item is used in a transaction, the applicable classification will be used to determine the appropriate tax rate.
     *             
     * An item may only have one classification per tax system.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The company id.
     * @param itemId The item id.
     * @param model The item classifications you wish to create.
     * @return ArrayList<ItemClassificationOutputModel>
     */
    public Future<ArrayList<ItemClassificationOutputModel>> createItemClassificationsAsync(Integer companyId, Long itemId, ArrayList<ItemClassificationInputModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/classifications");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        return this.threadPool.submit((RestCall<ArrayList<ItemClassificationOutputModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<ItemClassificationOutputModel>>(){}));
    }

    /**
     * Add parameters to an item.
     * 
     * Add parameters to an item.
     *             
     * Some items can be taxed differently depending on the properties of that item, such as the item grade or by a particular measurement of that item. In AvaTax, these tax-affecting properties are called "parameters".
     *             
     * A parameter added to an item will be used by default in tax calculation but will not show on the transaction line referencing the item .
     *             
     * A parameter specified on a transaction line will override an item parameter if they share the same parameter name.
     *             
     * To see available parameters for this item, call `/api/v2/definitions/parameters?$filter=attributeType eq Product`
     *             
     * Some parameters are only available for use if you have subscribed to specific AvaTax services. To see which parameters you are able to use, add the query parameter "$showSubscribed=true" to the parameter definition call above.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this item parameter.
     * @param itemId The item id.
     * @param model The item parameters you wish to create.
     * @return ArrayList<ItemParameterModel>
     */
    public ArrayList<ItemParameterModel> createItemParameters(Integer companyId, Long itemId, ArrayList<ItemParameterModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/parameters");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        return ((RestCall<ArrayList<ItemParameterModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<ItemParameterModel>>(){})).call();
    }

    /**
     * Add parameters to an item.
     * 
     * Add parameters to an item.
     *             
     * Some items can be taxed differently depending on the properties of that item, such as the item grade or by a particular measurement of that item. In AvaTax, these tax-affecting properties are called "parameters".
     *             
     * A parameter added to an item will be used by default in tax calculation but will not show on the transaction line referencing the item .
     *             
     * A parameter specified on a transaction line will override an item parameter if they share the same parameter name.
     *             
     * To see available parameters for this item, call `/api/v2/definitions/parameters?$filter=attributeType eq Product`
     *             
     * Some parameters are only available for use if you have subscribed to specific AvaTax services. To see which parameters you are able to use, add the query parameter "$showSubscribed=true" to the parameter definition call above.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this item parameter.
     * @param itemId The item id.
     * @param model The item parameters you wish to create.
     * @return ArrayList<ItemParameterModel>
     */
    public Future<ArrayList<ItemParameterModel>> createItemParametersAsync(Integer companyId, Long itemId, ArrayList<ItemParameterModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/parameters");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        return this.threadPool.submit((RestCall<ArrayList<ItemParameterModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<ItemParameterModel>>(){}));
    }

    /**
     * Create a new item
     * 
     * Creates one or more new item objects attached to this company.
     *             
     * Items are a way of separating your tax calculation process from your tax configuration details.  If you choose, you
     * can provide `itemCode` values for each `CreateTransaction()` API call rather than specifying tax codes, parameters, descriptions,
     * and other data fields.  AvaTax will automatically look up each `itemCode` and apply the correct tax codes and parameters
     * from the item table instead.  This allows your CreateTransaction call to be as simple as possible, and your tax compliance
     * team can manage your item catalog and adjust the tax behavior of items without having to modify your software.
     *             
     * The tax code takes precedence over the tax code id if both are provided.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this item.
     * @param model The item you wish to create.
     * @return ArrayList<ItemModel>
     */
    public ArrayList<ItemModel> createItems(Integer companyId, ArrayList<ItemModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<ItemModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<ItemModel>>(){})).call();
    }

    /**
     * Create a new item
     * 
     * Creates one or more new item objects attached to this company.
     *             
     * Items are a way of separating your tax calculation process from your tax configuration details.  If you choose, you
     * can provide `itemCode` values for each `CreateTransaction()` API call rather than specifying tax codes, parameters, descriptions,
     * and other data fields.  AvaTax will automatically look up each `itemCode` and apply the correct tax codes and parameters
     * from the item table instead.  This allows your CreateTransaction call to be as simple as possible, and your tax compliance
     * team can manage your item catalog and adjust the tax behavior of items without having to modify your software.
     *             
     * The tax code takes precedence over the tax code id if both are provided.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this item.
     * @param model The item you wish to create.
     * @return ArrayList<ItemModel>
     */
    public Future<ArrayList<ItemModel>> createItemsAsync(Integer companyId, ArrayList<ItemModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<ItemModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<ItemModel>>(){}));
    }

    /**
     * Delete a single item
     * 
     * Deletes the item object at this URL.
     *             
     * Items are a way of separating your tax calculation process from your tax configuration details.  If you choose, you
     * can provide `itemCode` values for each `CreateTransaction()` API call rather than specifying tax codes, parameters, descriptions,
     * and other data fields.  AvaTax will automatically look up each `itemCode` and apply the correct tax codes and parameters
     * from the item table instead.  This allows your CreateTransaction call to be as simple as possible, and your tax compliance
     * team can manage your item catalog and adjust the tax behavior of items without having to modify your software.
     *             
     * Deleting an item will also delete the parameters and classifications associated with that item.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this item.
     * @param id The ID of the item you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteItem(Integer companyId, Long id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a single item
     * 
     * Deletes the item object at this URL.
     *             
     * Items are a way of separating your tax calculation process from your tax configuration details.  If you choose, you
     * can provide `itemCode` values for each `CreateTransaction()` API call rather than specifying tax codes, parameters, descriptions,
     * and other data fields.  AvaTax will automatically look up each `itemCode` and apply the correct tax codes and parameters
     * from the item table instead.  This allows your CreateTransaction call to be as simple as possible, and your tax compliance
     * team can manage your item catalog and adjust the tax behavior of items without having to modify your software.
     *             
     * Deleting an item will also delete the parameters and classifications associated with that item.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this item.
     * @param id The ID of the item you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteItemAsync(Integer companyId, Long id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Delete a single item classification.
     * 
     * Delete a single item classification.
     *             
     * A classification is the code for a product in a particular tax system. Classifications enable an item to be used in multiple tax systems which may have different tax rates for a product.
     *             
     * When an item is used in a transaction, the applicable classification will be used to determine the appropriate tax rate.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The company id.
     * @param itemId The item id.
     * @param id The item classification id.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteItemClassification(Integer companyId, Long itemId, Long id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/classifications/{id}");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a single item classification.
     * 
     * Delete a single item classification.
     *             
     * A classification is the code for a product in a particular tax system. Classifications enable an item to be used in multiple tax systems which may have different tax rates for a product.
     *             
     * When an item is used in a transaction, the applicable classification will be used to determine the appropriate tax rate.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The company id.
     * @param itemId The item id.
     * @param id The item classification id.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteItemClassificationAsync(Integer companyId, Long itemId, Long id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/classifications/{id}");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Delete a single item parameter
     * 
     * Delete a single item parameter.
     *             
     * Some items can be taxed differently depending on the properties of that item, such as the item grade or by a particular measurement of that item. In AvaTax, these tax-affecting properties are called "parameters".
     *             
     * A parameter added to an item will be used by default in tax calculation but will not show on the transaction line referencing the item .
     *             
     * A parameter specified on a transaction line will override an item parameter if they share the same parameter name.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The company id
     * @param itemId The item id
     * @param id The parameter id
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteItemParameter(Integer companyId, Long itemId, Long id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/parameters/{id}");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a single item parameter
     * 
     * Delete a single item parameter.
     *             
     * Some items can be taxed differently depending on the properties of that item, such as the item grade or by a particular measurement of that item. In AvaTax, these tax-affecting properties are called "parameters".
     *             
     * A parameter added to an item will be used by default in tax calculation but will not show on the transaction line referencing the item .
     *             
     * A parameter specified on a transaction line will override an item parameter if they share the same parameter name.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The company id
     * @param itemId The item id
     * @param id The parameter id
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteItemParameterAsync(Integer companyId, Long itemId, Long id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/parameters/{id}");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Retrieve a single item
     * 
     * Get the `Item` object identified by this URL.
     *             
     * Items are a way of separating your tax calculation process from your tax configuration details.  If you choose, you
     * can provide `itemCode` values for each `CreateTransaction()` API call rather than specifying tax codes, parameters, descriptions,
     * and other data fields.  AvaTax will automatically look up each `itemCode` and apply the correct tax codes and parameters
     * from the item table instead.  This allows your CreateTransaction call to be as simple as possible, and your tax compliance
     * team can manage your item catalog and adjust the tax behavior of items without having to modify your software.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The ID of the company that owns this item object
     * @param id The primary key of this item
     * @param include A comma separated list of additional data to retrieve.
     * @return ItemModel
     */
    public ItemModel getItem(Integer companyId, Long id, String include) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.addQuery("$include", include);
        return ((RestCall<ItemModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<ItemModel>(){})).call();
    }

    /**
     * Retrieve a single item
     * 
     * Get the `Item` object identified by this URL.
     *             
     * Items are a way of separating your tax calculation process from your tax configuration details.  If you choose, you
     * can provide `itemCode` values for each `CreateTransaction()` API call rather than specifying tax codes, parameters, descriptions,
     * and other data fields.  AvaTax will automatically look up each `itemCode` and apply the correct tax codes and parameters
     * from the item table instead.  This allows your CreateTransaction call to be as simple as possible, and your tax compliance
     * team can manage your item catalog and adjust the tax behavior of items without having to modify your software.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The ID of the company that owns this item object
     * @param id The primary key of this item
     * @param include A comma separated list of additional data to retrieve.
     * @return ItemModel
     */
    public Future<ItemModel> getItemAsync(Integer companyId, Long id, String include) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.addQuery("$include", include);
        return this.threadPool.submit((RestCall<ItemModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<ItemModel>(){}));
    }

    /**
     * Retrieve a single item classification.
     * 
     * Retrieve a single item classification.
     *             
     * A classification is the code for a product in a particular tax system. Classifications enable an item to be used in multiple tax systems which may have different tax rates for a product.
     *             
     * When an item is used in a transaction, the applicable classification will be used to determine the appropriate tax rate.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The company id.
     * @param itemId The item id.
     * @param id The item classification id.
     * @return ItemClassificationOutputModel
     */
    public ItemClassificationOutputModel getItemClassification(Integer companyId, Long itemId, Long id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/classifications/{id}");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        path.applyField("id", id);
        return ((RestCall<ItemClassificationOutputModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<ItemClassificationOutputModel>(){})).call();
    }

    /**
     * Retrieve a single item classification.
     * 
     * Retrieve a single item classification.
     *             
     * A classification is the code for a product in a particular tax system. Classifications enable an item to be used in multiple tax systems which may have different tax rates for a product.
     *             
     * When an item is used in a transaction, the applicable classification will be used to determine the appropriate tax rate.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The company id.
     * @param itemId The item id.
     * @param id The item classification id.
     * @return ItemClassificationOutputModel
     */
    public Future<ItemClassificationOutputModel> getItemClassificationAsync(Integer companyId, Long itemId, Long id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/classifications/{id}");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ItemClassificationOutputModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<ItemClassificationOutputModel>(){}));
    }

    /**
     * Retrieve a single item parameter
     * 
     * Retrieve a single item parameter.
     *             
     * Some items can be taxed differently depending on the properties of that item, such as the item grade or by a particular measurement of that item. In AvaTax, these tax-affecting properties are called "parameters".
     *             
     * A parameter added to an item will be used by default in tax calculation but will not show on the transaction line referencing the item .
     *             
     * A parameter specified on a transaction line will override an item parameter if they share the same parameter name.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The company id
     * @param itemId The item id
     * @param id The parameter id
     * @return ItemParameterModel
     */
    public ItemParameterModel getItemParameter(Integer companyId, Long itemId, Long id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/parameters/{id}");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        path.applyField("id", id);
        return ((RestCall<ItemParameterModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<ItemParameterModel>(){})).call();
    }

    /**
     * Retrieve a single item parameter
     * 
     * Retrieve a single item parameter.
     *             
     * Some items can be taxed differently depending on the properties of that item, such as the item grade or by a particular measurement of that item. In AvaTax, these tax-affecting properties are called "parameters".
     *             
     * A parameter added to an item will be used by default in tax calculation but will not show on the transaction line referencing the item .
     *             
     * A parameter specified on a transaction line will override an item parameter if they share the same parameter name.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The company id
     * @param itemId The item id
     * @param id The parameter id
     * @return ItemParameterModel
     */
    public Future<ItemParameterModel> getItemParameterAsync(Integer companyId, Long itemId, Long id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/parameters/{id}");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ItemParameterModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<ItemParameterModel>(){}));
    }

    /**
     * Retrieve classifications for an item.
     * 
     * List classifications for an item.
     *             
     * A classification is the code for a product in a particular tax system. Classifications enable an item to be used in multiple tax systems which may have different tax rates for a product.
     *             
     * When an item is used in a transaction, the applicable classification will be used to determine the appropriate tax rate.
     *             
     * Search for specific objects using the criteria in the `$filter` classification; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` classifications.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The company id.
     * @param itemId The item id.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* productCode, systemCode
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ItemClassificationOutputModel>
     */
    public FetchResult<ItemClassificationOutputModel> listItemClassifications(Integer companyId, Long itemId, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/classifications");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<ItemClassificationOutputModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ItemClassificationOutputModel>>(){})).call();
    }

    /**
     * Retrieve classifications for an item.
     * 
     * List classifications for an item.
     *             
     * A classification is the code for a product in a particular tax system. Classifications enable an item to be used in multiple tax systems which may have different tax rates for a product.
     *             
     * When an item is used in a transaction, the applicable classification will be used to determine the appropriate tax rate.
     *             
     * Search for specific objects using the criteria in the `$filter` classification; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` classifications.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The company id.
     * @param itemId The item id.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* productCode, systemCode
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ItemClassificationOutputModel>
     */
    public Future<FetchResult<ItemClassificationOutputModel>> listItemClassificationsAsync(Integer companyId, Long itemId, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/classifications");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<ItemClassificationOutputModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ItemClassificationOutputModel>>(){}));
    }

    /**
     * Retrieve parameters for an item
     * 
     * List parameters for an item.
     *             
     * Some items can be taxed differently depending on the properties of that item, such as the item grade or by a particular measurement of that item. In AvaTax, these tax-affecting properties are called "parameters".
     *             
     * A parameter added to an item will be used by default in tax calculation but will not show on the transaction line referencing the item .
     *             
     * A parameter specified on a transaction line will override an item parameter if they share the same parameter name.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The company id
     * @param itemId The item id
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* name, unit
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ItemParameterModel>
     */
    public FetchResult<ItemParameterModel> listItemParameters(Integer companyId, Long itemId, String filter, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/parameters");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<ItemParameterModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ItemParameterModel>>(){})).call();
    }

    /**
     * Retrieve parameters for an item
     * 
     * List parameters for an item.
     *             
     * Some items can be taxed differently depending on the properties of that item, such as the item grade or by a particular measurement of that item. In AvaTax, these tax-affecting properties are called "parameters".
     *             
     * A parameter added to an item will be used by default in tax calculation but will not show on the transaction line referencing the item .
     *             
     * A parameter specified on a transaction line will override an item parameter if they share the same parameter name.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The company id
     * @param itemId The item id
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* name, unit
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ItemParameterModel>
     */
    public Future<FetchResult<ItemParameterModel>> listItemParametersAsync(Integer companyId, Long itemId, String filter, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/parameters");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        path.addQuery("$filter", filter);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<ItemParameterModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ItemParameterModel>>(){}));
    }

    /**
     * Retrieve items for this company
     * 
     * List all items defined for the current company.
     *             
     * Items are a way of separating your tax calculation process from your tax configuration details.  If you choose, you
     * can provide `itemCode` values for each `CreateTransaction()` API call rather than specifying tax codes, parameters, descriptions,
     * and other data fields.  AvaTax will automatically look up each `itemCode` and apply the correct tax codes and parameters
     * from the item table instead.  This allows your CreateTransaction call to be as simple as possible, and your tax compliance
     * team can manage your item catalog and adjust the tax behavior of items without having to modify your software.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     *             
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     *             
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * Parameters
     * * Classifications
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The ID of the company that defined these items
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* taxCode, classifications, parameters
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ItemModel>
     */
    public FetchResult<ItemModel> listItemsByCompany(Integer companyId, String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<ItemModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ItemModel>>(){})).call();
    }

    /**
     * Retrieve items for this company
     * 
     * List all items defined for the current company.
     *             
     * Items are a way of separating your tax calculation process from your tax configuration details.  If you choose, you
     * can provide `itemCode` values for each `CreateTransaction()` API call rather than specifying tax codes, parameters, descriptions,
     * and other data fields.  AvaTax will automatically look up each `itemCode` and apply the correct tax codes and parameters
     * from the item table instead.  This allows your CreateTransaction call to be as simple as possible, and your tax compliance
     * team can manage your item catalog and adjust the tax behavior of items without having to modify your software.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     *             
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     *             
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * Parameters
     * * Classifications
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The ID of the company that defined these items
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* taxCode, classifications, parameters
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ItemModel>
     */
    public Future<FetchResult<ItemModel>> listItemsByCompanyAsync(Integer companyId, String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<ItemModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ItemModel>>(){}));
    }

    /**
     * Retrieve all items
     * 
     * Get multiple item objects across all companies.
     *             
     * Items are a way of separating your tax calculation process from your tax configuration details.  If you choose, you
     * can provide `itemCode` values for each `CreateTransaction()` API call rather than specifying tax codes, parameters, descriptions,
     * and other data fields.  AvaTax will automatically look up each `itemCode` and apply the correct tax codes and parameters
     * from the item table instead.  This allows your CreateTransaction call to be as simple as possible, and your tax compliance
     * team can manage your item catalog and adjust the tax behavior of items without having to modify your software.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     *             
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* taxCode, classifications, parameters
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ItemModel>
     */
    public FetchResult<ItemModel> queryItems(String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/items");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<ItemModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ItemModel>>(){})).call();
    }

    /**
     * Retrieve all items
     * 
     * Get multiple item objects across all companies.
     *             
     * Items are a way of separating your tax calculation process from your tax configuration details.  If you choose, you
     * can provide `itemCode` values for each `CreateTransaction()` API call rather than specifying tax codes, parameters, descriptions,
     * and other data fields.  AvaTax will automatically look up each `itemCode` and apply the correct tax codes and parameters
     * from the item table instead.  This allows your CreateTransaction call to be as simple as possible, and your tax compliance
     * team can manage your item catalog and adjust the tax behavior of items without having to modify your software.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     *             
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* taxCode, classifications, parameters
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<ItemModel>
     */
    public Future<FetchResult<ItemModel>> queryItemsAsync(String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/items");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<ItemModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ItemModel>>(){}));
    }

    /**
     * Update a single item
     * 
     * Replace the existing `Item` object at this URL with an updated object.
     *             
     * Items are a way of separating your tax calculation process from your tax configuration details.  If you choose, you
     * can provide `itemCode` values for each `CreateTransaction()` API call rather than specifying tax codes, parameters, descriptions,
     * and other data fields.  AvaTax will automatically look up each `itemCode` and apply the correct tax codes and parameters
     * from the item table instead.  This allows your CreateTransaction call to be as simple as possible, and your tax compliance
     * team can manage your item catalog and adjust the tax behavior of items without having to modify your software.
     *             
     * All data from the existing object will be replaced with data in the object you PUT.  To set a field's value to null,
     * you may either set its value to null or omit that field from the object you post.
     *             
     * The tax code takes precedence over the tax code id if both are provided.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that this item belongs to.
     * @param id The ID of the item you wish to update
     * @param model The item object you wish to update.
     * @return ItemModel
     */
    public ItemModel updateItem(Integer companyId, Long id, ItemModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ItemModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<ItemModel>(){})).call();
    }

    /**
     * Update a single item
     * 
     * Replace the existing `Item` object at this URL with an updated object.
     *             
     * Items are a way of separating your tax calculation process from your tax configuration details.  If you choose, you
     * can provide `itemCode` values for each `CreateTransaction()` API call rather than specifying tax codes, parameters, descriptions,
     * and other data fields.  AvaTax will automatically look up each `itemCode` and apply the correct tax codes and parameters
     * from the item table instead.  This allows your CreateTransaction call to be as simple as possible, and your tax compliance
     * team can manage your item catalog and adjust the tax behavior of items without having to modify your software.
     *             
     * All data from the existing object will be replaced with data in the object you PUT.  To set a field's value to null,
     * you may either set its value to null or omit that field from the object you post.
     *             
     * The tax code takes precedence over the tax code id if both are provided.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that this item belongs to.
     * @param id The ID of the item you wish to update
     * @param model The item object you wish to update.
     * @return ItemModel
     */
    public Future<ItemModel> updateItemAsync(Integer companyId, Long id, ItemModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ItemModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<ItemModel>(){}));
    }

    /**
     * Update an item classification.
     * 
     * Update an item classification.
     *             
     * A classification is the code for a product in a particular tax system. Classifications enable an item to be used in multiple tax systems which may have different tax rates for a product.
     *             
     * When an item is used in a transaction, the applicable classification will be used to determine the appropriate tax rate.
     *             
     * An item may only have one classification per tax system.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The company id.
     * @param itemId The item id.
     * @param id The item classification id.
     * @param model The item object you wish to update.
     * @return ItemClassificationOutputModel
     */
    public ItemClassificationOutputModel updateItemClassification(Integer companyId, Long itemId, Long id, ItemClassificationInputModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/classifications/{id}");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        path.applyField("id", id);
        return ((RestCall<ItemClassificationOutputModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<ItemClassificationOutputModel>(){})).call();
    }

    /**
     * Update an item classification.
     * 
     * Update an item classification.
     *             
     * A classification is the code for a product in a particular tax system. Classifications enable an item to be used in multiple tax systems which may have different tax rates for a product.
     *             
     * When an item is used in a transaction, the applicable classification will be used to determine the appropriate tax rate.
     *             
     * An item may only have one classification per tax system.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The company id.
     * @param itemId The item id.
     * @param id The item classification id.
     * @param model The item object you wish to update.
     * @return ItemClassificationOutputModel
     */
    public Future<ItemClassificationOutputModel> updateItemClassificationAsync(Integer companyId, Long itemId, Long id, ItemClassificationInputModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/classifications/{id}");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ItemClassificationOutputModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<ItemClassificationOutputModel>(){}));
    }

    /**
     * Update an item parameter
     * 
     * Update an item parameter.
     *             
     * Some items can be taxed differently depending on the properties of that item, such as the item grade or by a particular measurement of that item. In AvaTax, these tax-affecting properties are called "parameters".
     *             
     * A parameter added to an item will be used by default in tax calculation but will not show on the transaction line referencing the item .
     *             
     * A parameter specified on a transaction line will override an item parameter if they share the same parameter name.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The company id.
     * @param itemId The item id
     * @param id The item parameter id
     * @param model The item object you wish to update.
     * @return ItemParameterModel
     */
    public ItemParameterModel updateItemParameter(Integer companyId, Long itemId, Long id, ItemParameterModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/parameters/{id}");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        path.applyField("id", id);
        return ((RestCall<ItemParameterModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<ItemParameterModel>(){})).call();
    }

    /**
     * Update an item parameter
     * 
     * Update an item parameter.
     *             
     * Some items can be taxed differently depending on the properties of that item, such as the item grade or by a particular measurement of that item. In AvaTax, these tax-affecting properties are called "parameters".
     *             
     * A parameter added to an item will be used by default in tax calculation but will not show on the transaction line referencing the item .
     *             
     * A parameter specified on a transaction line will override an item parameter if they share the same parameter name.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The company id.
     * @param itemId The item id
     * @param id The item parameter id
     * @param model The item object you wish to update.
     * @return ItemParameterModel
     */
    public Future<ItemParameterModel> updateItemParameterAsync(Integer companyId, Long itemId, Long id, ItemParameterModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{itemId}/parameters/{id}");
        path.applyField("companyId", companyId);
        path.applyField("itemId", itemId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ItemParameterModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<ItemParameterModel>(){}));
    }

    /**
     * Create one or more overrides
     * 
     * Creates one or more jurisdiction override objects for this account.
     *             
     * A Jurisdiction Override is a configuration setting that allows you to select the taxing
     * jurisdiction for a specific address.  If you encounter an address that is on the boundary
     * between two different jurisdictions, you can choose to set up a jurisdiction override
     * to switch this address to use different taxing jurisdictions.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param accountId The ID of the account that owns this override
     * @param model The jurisdiction override objects to create
     * @return ArrayList<JurisdictionOverrideModel>
     */
    public ArrayList<JurisdictionOverrideModel> createJurisdictionOverrides(Integer accountId, ArrayList<JurisdictionOverrideModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/jurisdictionoverrides");
        path.applyField("accountId", accountId);
        return ((RestCall<ArrayList<JurisdictionOverrideModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<JurisdictionOverrideModel>>(){})).call();
    }

    /**
     * Create one or more overrides
     * 
     * Creates one or more jurisdiction override objects for this account.
     *             
     * A Jurisdiction Override is a configuration setting that allows you to select the taxing
     * jurisdiction for a specific address.  If you encounter an address that is on the boundary
     * between two different jurisdictions, you can choose to set up a jurisdiction override
     * to switch this address to use different taxing jurisdictions.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param accountId The ID of the account that owns this override
     * @param model The jurisdiction override objects to create
     * @return ArrayList<JurisdictionOverrideModel>
     */
    public Future<ArrayList<JurisdictionOverrideModel>> createJurisdictionOverridesAsync(Integer accountId, ArrayList<JurisdictionOverrideModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/jurisdictionoverrides");
        path.applyField("accountId", accountId);
        return this.threadPool.submit((RestCall<ArrayList<JurisdictionOverrideModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<JurisdictionOverrideModel>>(){}));
    }

    /**
     * Delete a single override
     * 
     * Marks the item object at this URL as deleted.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param accountId The ID of the account that owns this override
     * @param id The ID of the override you wish to delete
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteJurisdictionOverride(Integer accountId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/jurisdictionoverrides/{id}");
        path.applyField("accountId", accountId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a single override
     * 
     * Marks the item object at this URL as deleted.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param accountId The ID of the account that owns this override
     * @param id The ID of the override you wish to delete
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteJurisdictionOverrideAsync(Integer accountId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/jurisdictionoverrides/{id}");
        path.applyField("accountId", accountId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Retrieve a single override
     * 
     * Get the item object identified by this URL.
     *             
     * A Jurisdiction Override is a configuration setting that allows you to select the taxing
     * jurisdiction for a specific address.  If you encounter an address that is on the boundary
     * between two different jurisdictions, you can choose to set up a jurisdiction override
     * to switch this address to use different taxing jurisdictions.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param accountId The ID of the account that owns this override
     * @param id The primary key of this override
     * @return JurisdictionOverrideModel
     */
    public JurisdictionOverrideModel getJurisdictionOverride(Integer accountId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/jurisdictionoverrides/{id}");
        path.applyField("accountId", accountId);
        path.applyField("id", id);
        return ((RestCall<JurisdictionOverrideModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<JurisdictionOverrideModel>(){})).call();
    }

    /**
     * Retrieve a single override
     * 
     * Get the item object identified by this URL.
     *             
     * A Jurisdiction Override is a configuration setting that allows you to select the taxing
     * jurisdiction for a specific address.  If you encounter an address that is on the boundary
     * between two different jurisdictions, you can choose to set up a jurisdiction override
     * to switch this address to use different taxing jurisdictions.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param accountId The ID of the account that owns this override
     * @param id The primary key of this override
     * @return JurisdictionOverrideModel
     */
    public Future<JurisdictionOverrideModel> getJurisdictionOverrideAsync(Integer accountId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/jurisdictionoverrides/{id}");
        path.applyField("accountId", accountId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<JurisdictionOverrideModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<JurisdictionOverrideModel>(){}));
    }

    /**
     * Retrieve overrides for this account
     * 
     * List all jurisdiction override objects defined for this account.
     *             
     * A Jurisdiction Override is a configuration setting that allows you to select the taxing
     * jurisdiction for a specific address.  If you encounter an address that is on the boundary
     * between two different jurisdictions, you can choose to set up a jurisdiction override
     * to switch this address to use different taxing jurisdictions.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param accountId The ID of the account that owns this override
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* country, Jurisdictions
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<JurisdictionOverrideModel>
     */
    public FetchResult<JurisdictionOverrideModel> listJurisdictionOverridesByAccount(Integer accountId, String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/jurisdictionoverrides");
        path.applyField("accountId", accountId);
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<JurisdictionOverrideModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<JurisdictionOverrideModel>>(){})).call();
    }

    /**
     * Retrieve overrides for this account
     * 
     * List all jurisdiction override objects defined for this account.
     *             
     * A Jurisdiction Override is a configuration setting that allows you to select the taxing
     * jurisdiction for a specific address.  If you encounter an address that is on the boundary
     * between two different jurisdictions, you can choose to set up a jurisdiction override
     * to switch this address to use different taxing jurisdictions.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param accountId The ID of the account that owns this override
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* country, Jurisdictions
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<JurisdictionOverrideModel>
     */
    public Future<FetchResult<JurisdictionOverrideModel>> listJurisdictionOverridesByAccountAsync(Integer accountId, String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/jurisdictionoverrides");
        path.applyField("accountId", accountId);
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<JurisdictionOverrideModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<JurisdictionOverrideModel>>(){}));
    }

    /**
     * Retrieve all overrides
     * 
     * Get multiple jurisdiction override objects across all companies.
     *             
     * A Jurisdiction Override is a configuration setting that allows you to select the taxing
     * jurisdiction for a specific address.  If you encounter an address that is on the boundary
     * between two different jurisdictions, you can choose to set up a jurisdiction override
     * to switch this address to use different taxing jurisdictions.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* country, Jurisdictions
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<JurisdictionOverrideModel>
     */
    public FetchResult<JurisdictionOverrideModel> queryJurisdictionOverrides(String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/jurisdictionoverrides");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<JurisdictionOverrideModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<JurisdictionOverrideModel>>(){})).call();
    }

    /**
     * Retrieve all overrides
     * 
     * Get multiple jurisdiction override objects across all companies.
     *             
     * A Jurisdiction Override is a configuration setting that allows you to select the taxing
     * jurisdiction for a specific address.  If you encounter an address that is on the boundary
     * between two different jurisdictions, you can choose to set up a jurisdiction override
     * to switch this address to use different taxing jurisdictions.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* country, Jurisdictions
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<JurisdictionOverrideModel>
     */
    public Future<FetchResult<JurisdictionOverrideModel>> queryJurisdictionOverridesAsync(String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/jurisdictionoverrides");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<JurisdictionOverrideModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<JurisdictionOverrideModel>>(){}));
    }

    /**
     * Update a single jurisdictionoverride
     * 
     * Replace the existing jurisdictionoverride object at this URL with an updated object.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param accountId The ID of the account that this jurisdictionoverride belongs to.
     * @param id The ID of the jurisdictionoverride you wish to update
     * @param model The jurisdictionoverride object you wish to update.
     * @return JurisdictionOverrideModel
     */
    public JurisdictionOverrideModel updateJurisdictionOverride(Integer accountId, Integer id, JurisdictionOverrideModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/jurisdictionoverrides/{id}");
        path.applyField("accountId", accountId);
        path.applyField("id", id);
        return ((RestCall<JurisdictionOverrideModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<JurisdictionOverrideModel>(){})).call();
    }

    /**
     * Update a single jurisdictionoverride
     * 
     * Replace the existing jurisdictionoverride object at this URL with an updated object.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param accountId The ID of the account that this jurisdictionoverride belongs to.
     * @param id The ID of the jurisdictionoverride you wish to update
     * @param model The jurisdictionoverride object you wish to update.
     * @return JurisdictionOverrideModel
     */
    public Future<JurisdictionOverrideModel> updateJurisdictionOverrideAsync(Integer accountId, Integer id, JurisdictionOverrideModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/jurisdictionoverrides/{id}");
        path.applyField("accountId", accountId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<JurisdictionOverrideModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<JurisdictionOverrideModel>(){}));
    }

    /**
     * Create a new location
     * 
     * Create one or more new location objects attached to this company.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this location.
     * @param model The location you wish to create.
     * @return ArrayList<LocationModel>
     */
    public ArrayList<LocationModel> createLocations(Integer companyId, ArrayList<LocationModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<LocationModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<LocationModel>>(){})).call();
    }

    /**
     * Create a new location
     * 
     * Create one or more new location objects attached to this company.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this location.
     * @param model The location you wish to create.
     * @return ArrayList<LocationModel>
     */
    public Future<ArrayList<LocationModel>> createLocationsAsync(Integer companyId, ArrayList<LocationModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<LocationModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<LocationModel>>(){}));
    }

    /**
     * Delete a single location
     * 
     * Mark the location object at this URL as deleted.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this location.
     * @param id The ID of the location you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteLocation(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a single location
     * 
     * Mark the location object at this URL as deleted.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this location.
     * @param id The ID of the location you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteLocationAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Retrieve a single location
     * 
     * Get the location object identified by this URL.
     * An 'Location' represents a physical address where a company does business.
     * Many taxing authorities require that you define a list of all locations where your company does business.
     * These locations may require additional custom configuration or tax registration with these authorities.
     * For more information on metadata requirements, see the '/api/v2/definitions/locationquestions' API.
     * 
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * LocationSettings
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param companyId The ID of the company that owns this location
     * @param id The primary key of this location
     * @param include A comma separated list of additional data to retrieve. You may specify `LocationSettings` to retrieve location settings.
     * @return LocationModel
     */
    public LocationModel getLocation(Integer companyId, Integer id, String include) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.addQuery("$include", include);
        return ((RestCall<LocationModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<LocationModel>(){})).call();
    }

    /**
     * Retrieve a single location
     * 
     * Get the location object identified by this URL.
     * An 'Location' represents a physical address where a company does business.
     * Many taxing authorities require that you define a list of all locations where your company does business.
     * These locations may require additional custom configuration or tax registration with these authorities.
     * For more information on metadata requirements, see the '/api/v2/definitions/locationquestions' API.
     * 
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * LocationSettings
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param companyId The ID of the company that owns this location
     * @param id The primary key of this location
     * @param include A comma separated list of additional data to retrieve. You may specify `LocationSettings` to retrieve location settings.
     * @return LocationModel
     */
    public Future<LocationModel> getLocationAsync(Integer companyId, Integer id, String include) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.addQuery("$include", include);
        return this.threadPool.submit((RestCall<LocationModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<LocationModel>(){}));
    }

    /**
     * Retrieve locations for this company
     * 
     * List all location objects defined for this company.
     * An 'Location' represents a physical address where a company does business.
     * Many taxing authorities require that you define a list of all locations where your company does business.
     * These locations may require additional custom configuration or tax registration with these authorities.
     * For more information on metadata requirements, see the '/api/v2/definitions/locationquestions' API.
     * 
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * LocationSettings
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param companyId The ID of the company that owns these locations
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* settings
     * @param include A comma separated list of additional data to retrieve. You may specify `LocationSettings` to retrieve location settings.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<LocationModel>
     */
    public FetchResult<LocationModel> listLocationsByCompany(Integer companyId, String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<LocationModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<LocationModel>>(){})).call();
    }

    /**
     * Retrieve locations for this company
     * 
     * List all location objects defined for this company.
     * An 'Location' represents a physical address where a company does business.
     * Many taxing authorities require that you define a list of all locations where your company does business.
     * These locations may require additional custom configuration or tax registration with these authorities.
     * For more information on metadata requirements, see the '/api/v2/definitions/locationquestions' API.
     * 
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * LocationSettings
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param companyId The ID of the company that owns these locations
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* settings
     * @param include A comma separated list of additional data to retrieve. You may specify `LocationSettings` to retrieve location settings.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<LocationModel>
     */
    public Future<FetchResult<LocationModel>> listLocationsByCompanyAsync(Integer companyId, String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<LocationModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<LocationModel>>(){}));
    }

    /**
     * Retrieve all locations
     * 
     * Get multiple location objects across all companies.
     * An 'Location' represents a physical address where a company does business.
     * Many taxing authorities require that you define a list of all locations where your company does business.
     * These locations may require additional custom configuration or tax registration with these authorities.
     * For more information on metadata requirements, see the '/api/v2/definitions/locationquestions' API.
     * 
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * LocationSettings
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* settings
     * @param include A comma separated list of additional data to retrieve. You may specify `LocationSettings` to retrieve location settings.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<LocationModel>
     */
    public FetchResult<LocationModel> queryLocations(String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/locations");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<LocationModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<LocationModel>>(){})).call();
    }

    /**
     * Retrieve all locations
     * 
     * Get multiple location objects across all companies.
     * An 'Location' represents a physical address where a company does business.
     * Many taxing authorities require that you define a list of all locations where your company does business.
     * These locations may require additional custom configuration or tax registration with these authorities.
     * For more information on metadata requirements, see the '/api/v2/definitions/locationquestions' API.
     * 
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * LocationSettings
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* settings
     * @param include A comma separated list of additional data to retrieve. You may specify `LocationSettings` to retrieve location settings.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<LocationModel>
     */
    public Future<FetchResult<LocationModel>> queryLocationsAsync(String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/locations");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<LocationModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<LocationModel>>(){}));
    }

    /**
     * Update a single location
     * 
     * Replace the existing location object at this URL with an updated object.
     * All data from the existing object will be replaced with data in the object you PUT.  
     * To set a field's value to null, you may either set its value to null or omit that field from the object you post.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that this location belongs to.
     * @param id The ID of the location you wish to update
     * @param model The location you wish to update.
     * @return LocationModel
     */
    public LocationModel updateLocation(Integer companyId, Integer id, LocationModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<LocationModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<LocationModel>(){})).call();
    }

    /**
     * Update a single location
     * 
     * Replace the existing location object at this URL with an updated object.
     * All data from the existing object will be replaced with data in the object you PUT.  
     * To set a field's value to null, you may either set its value to null or omit that field from the object you post.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that this location belongs to.
     * @param id The ID of the location you wish to update
     * @param model The location you wish to update.
     * @return LocationModel
     */
    public Future<LocationModel> updateLocationAsync(Integer companyId, Integer id, LocationModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<LocationModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<LocationModel>(){}));
    }

    /**
     * Validate the location against local requirements
     * 
     * Returns validation information for this location.
     * This API call is intended to compare this location against the currently known taxing authority rules and regulations,
     * and provide information about what additional work is required to completely setup this location.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param companyId The ID of the company that owns this location
     * @param id The primary key of this location
     * @return LocationValidationModel
     */
    public LocationValidationModel validateLocation(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}/validate");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<LocationValidationModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<LocationValidationModel>(){})).call();
    }

    /**
     * Validate the location against local requirements
     * 
     * Returns validation information for this location.
     * This API call is intended to compare this location against the currently known taxing authority rules and regulations,
     * and provide information about what additional work is required to completely setup this location.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, FirmAdmin, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * 
     * @param companyId The ID of the company that owns this location
     * @param id The primary key of this location
     * @return LocationValidationModel
     */
    public Future<LocationValidationModel> validateLocationAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}/validate");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<LocationValidationModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<LocationValidationModel>(){}));
    }

    /**
     * Adjust a MultiDocument transaction
     * 
     * Adjusts the current MultiDocument transaction uniquely identified by this URL.
     *             
     * A transaction represents a unique potentially taxable action that your company has recorded, and transactions include actions like
     * sales, purchases, inventory transfer, and returns (also called refunds).
     *             
     * When you adjust a transaction, that transaction's status is recorded as `Adjusted`.
     *             
     * Both the revisions will be available for retrieval based on their code and ID numbers. Only transactions in Committed status can be reported on a tax filing by Avalara's Managed Returns Service.
     *             
     * Transactions that have been previously reported to a tax authority by Avalara Managed Returns are considered locked and are no longer available for adjustments.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param code The transaction code for this MultiDocument transaction
     * @param type The transaction type for this MultiDocument transaction (See DocumentType::* for a list of allowable values)
     * @param include Specifies objects to include in this fetch call
     * @param model The adjust request you wish to execute
     * @return MultiDocumentModel
     */
    public MultiDocumentModel adjustMultiDocumentTransaction(String code, DocumentType type, String include, AdjustMultiDocumentModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument/{code}/type/{type}/adjust");
        path.applyField("code", code);
        path.applyField("type", type);
        path.addQuery("include", include);
        return ((RestCall<MultiDocumentModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<MultiDocumentModel>(){})).call();
    }

    /**
     * Adjust a MultiDocument transaction
     * 
     * Adjusts the current MultiDocument transaction uniquely identified by this URL.
     *             
     * A transaction represents a unique potentially taxable action that your company has recorded, and transactions include actions like
     * sales, purchases, inventory transfer, and returns (also called refunds).
     *             
     * When you adjust a transaction, that transaction's status is recorded as `Adjusted`.
     *             
     * Both the revisions will be available for retrieval based on their code and ID numbers. Only transactions in Committed status can be reported on a tax filing by Avalara's Managed Returns Service.
     *             
     * Transactions that have been previously reported to a tax authority by Avalara Managed Returns are considered locked and are no longer available for adjustments.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param code The transaction code for this MultiDocument transaction
     * @param type The transaction type for this MultiDocument transaction (See DocumentType::* for a list of allowable values)
     * @param include Specifies objects to include in this fetch call
     * @param model The adjust request you wish to execute
     * @return MultiDocumentModel
     */
    public Future<MultiDocumentModel> adjustMultiDocumentTransactionAsync(String code, DocumentType type, String include, AdjustMultiDocumentModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument/{code}/type/{type}/adjust");
        path.applyField("code", code);
        path.applyField("type", type);
        path.addQuery("include", include);
        return this.threadPool.submit((RestCall<MultiDocumentModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<MultiDocumentModel>(){}));
    }

    /**
     * Get audit information about a MultiDocument transaction
     * 
     * Retrieve audit information about a MultiDocument transaction stored in AvaTax.
     *             
     * The audit API retrieves audit information related to a specific MultiDocument transaction.  This audit
     * information includes the following:
     *             
     * * The `code` of the MultiDocument transaction
     * * The `type` of the MultiDocument transaction
     * * The server timestamp representing the exact server time when the transaction was created
     * * The server duration - how long it took to process this transaction
     * * Whether exact API call details were logged
     * * A reconstructed API call showing what the original create call looked like
     *             
     * A transaction represents a unique potentially taxable action that your company has recorded, and transactions include actions like
     * sales, purchases, inventory transfer, and returns (also called refunds).
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, ProStoresOperator, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param code The transaction code for this MultiDocument transaction
     * @param type The transaction type for this MultiDocument transaction (See DocumentType::* for a list of allowable values)
     * @return AuditMultiDocumentModel
     */
    public AuditMultiDocumentModel auditMultiDocumentTransaction(String code, DocumentType type) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument/{code}/type/{type}/audit");
        path.applyField("code", code);
        path.applyField("type", type);
        return ((RestCall<AuditMultiDocumentModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<AuditMultiDocumentModel>(){})).call();
    }

    /**
     * Get audit information about a MultiDocument transaction
     * 
     * Retrieve audit information about a MultiDocument transaction stored in AvaTax.
     *             
     * The audit API retrieves audit information related to a specific MultiDocument transaction.  This audit
     * information includes the following:
     *             
     * * The `code` of the MultiDocument transaction
     * * The `type` of the MultiDocument transaction
     * * The server timestamp representing the exact server time when the transaction was created
     * * The server duration - how long it took to process this transaction
     * * Whether exact API call details were logged
     * * A reconstructed API call showing what the original create call looked like
     *             
     * A transaction represents a unique potentially taxable action that your company has recorded, and transactions include actions like
     * sales, purchases, inventory transfer, and returns (also called refunds).
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, ProStoresOperator, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param code The transaction code for this MultiDocument transaction
     * @param type The transaction type for this MultiDocument transaction (See DocumentType::* for a list of allowable values)
     * @return AuditMultiDocumentModel
     */
    public Future<AuditMultiDocumentModel> auditMultiDocumentTransactionAsync(String code, DocumentType type) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument/{code}/type/{type}/audit");
        path.applyField("code", code);
        path.applyField("type", type);
        return this.threadPool.submit((RestCall<AuditMultiDocumentModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<AuditMultiDocumentModel>(){}));
    }

    /**
     * Commit a MultiDocument transaction
     * 
     * Marks a list of transactions by changing its status to `Committed`.
     *             
     * Transactions that are committed are available to be reported to a tax authority by Avalara Managed Returns.
     *             
     * A transaction represents a unique potentially taxable action that your company has recorded, and transactions include actions like
     * sales, purchases, inventory transfer, and returns (also called refunds).
     *             
     * Any changes made to a committed transaction will generate a transaction history.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, ProStoresOperator, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param model The commit request you wish to execute
     * @return MultiDocumentModel
     */
    public MultiDocumentModel commitMultiDocumentTransaction(CommitMultiDocumentModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument/commit");
        return ((RestCall<MultiDocumentModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<MultiDocumentModel>(){})).call();
    }

    /**
     * Commit a MultiDocument transaction
     * 
     * Marks a list of transactions by changing its status to `Committed`.
     *             
     * Transactions that are committed are available to be reported to a tax authority by Avalara Managed Returns.
     *             
     * A transaction represents a unique potentially taxable action that your company has recorded, and transactions include actions like
     * sales, purchases, inventory transfer, and returns (also called refunds).
     *             
     * Any changes made to a committed transaction will generate a transaction history.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, ProStoresOperator, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param model The commit request you wish to execute
     * @return MultiDocumentModel
     */
    public Future<MultiDocumentModel> commitMultiDocumentTransactionAsync(CommitMultiDocumentModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument/commit");
        return this.threadPool.submit((RestCall<MultiDocumentModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<MultiDocumentModel>(){}));
    }

    /**
     * Create a new MultiDocument transaction
     * 
     * Records a new MultiDocument transaction in AvaTax.
     *             
     * A traditional transaction requires exactly two parties: a seller and a buyer.  MultiDocument transactions can
     * involve a marketplace of vendors, each of which contributes some portion of the final transaction.  Within
     * a MultiDocument transaction, each individual buyer and seller pair are matched up and converted to a separate
     * document.  This separation of documents allows each seller to file their taxes separately.
     *             
     * This API will report an error if you attempt to create a transaction when one already exists with the specified `code`.
     * If you would like the API to automatically update the transaction when it already exists, please set the `allowAdjust`
     * value to `true`.
     *             
     * To generate a refund for a transaction, use the `RefundTransaction` API.
     *             
     * The field `type` identifies the kind of transaction - for example, a sale, purchase, or refund.  If you do not specify
     * a `type` value, you will receive an estimate of type `SalesOrder`, which will not be recorded.
     *             
     * The origin and destination locations for a transaction must be identified by either address or geocode.  For address-based transactions, please
     * provide addresses in the fields `line`, `city`, `region`, `country` and `postalCode`.  For geocode-based transactions, please provide the geocode
     * information in the fields `latitude` and `longitude`.  If either `latitude` or `longitude` or both are null, the transaction will be calculated
     * using the best available address location information.
     *             
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * Lines
     * * Details (implies lines)
     * * Summary (implies details)
     * * Addresses
     * * SummaryOnly (omit lines and details - reduces API response size)
     * * LinesOnly (omit details - reduces API response size)
     * * ForceTimeout - Simulates a timeout.  This adds a 30 second delay and error to your API call.  This can be used to test your code to ensure it can respond correctly in the case of a dropped connection.
     *             
     * If you omit the `$include` parameter, the API will assume you want `Summary,Addresses`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param include Specifies objects to include in the response after transaction is created
     * @param model the multi document transaction model
     * @return MultiDocumentModel
     */
    public MultiDocumentModel createMultiDocumentTransaction(String include, CreateMultiDocumentModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument");
        path.addQuery("$include", include);
        return ((RestCall<MultiDocumentModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<MultiDocumentModel>(){})).call();
    }

    /**
     * Create a new MultiDocument transaction
     * 
     * Records a new MultiDocument transaction in AvaTax.
     *             
     * A traditional transaction requires exactly two parties: a seller and a buyer.  MultiDocument transactions can
     * involve a marketplace of vendors, each of which contributes some portion of the final transaction.  Within
     * a MultiDocument transaction, each individual buyer and seller pair are matched up and converted to a separate
     * document.  This separation of documents allows each seller to file their taxes separately.
     *             
     * This API will report an error if you attempt to create a transaction when one already exists with the specified `code`.
     * If you would like the API to automatically update the transaction when it already exists, please set the `allowAdjust`
     * value to `true`.
     *             
     * To generate a refund for a transaction, use the `RefundTransaction` API.
     *             
     * The field `type` identifies the kind of transaction - for example, a sale, purchase, or refund.  If you do not specify
     * a `type` value, you will receive an estimate of type `SalesOrder`, which will not be recorded.
     *             
     * The origin and destination locations for a transaction must be identified by either address or geocode.  For address-based transactions, please
     * provide addresses in the fields `line`, `city`, `region`, `country` and `postalCode`.  For geocode-based transactions, please provide the geocode
     * information in the fields `latitude` and `longitude`.  If either `latitude` or `longitude` or both are null, the transaction will be calculated
     * using the best available address location information.
     *             
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * Lines
     * * Details (implies lines)
     * * Summary (implies details)
     * * Addresses
     * * SummaryOnly (omit lines and details - reduces API response size)
     * * LinesOnly (omit details - reduces API response size)
     * * ForceTimeout - Simulates a timeout.  This adds a 30 second delay and error to your API call.  This can be used to test your code to ensure it can respond correctly in the case of a dropped connection.
     *             
     * If you omit the `$include` parameter, the API will assume you want `Summary,Addresses`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param include Specifies objects to include in the response after transaction is created
     * @param model the multi document transaction model
     * @return MultiDocumentModel
     */
    public Future<MultiDocumentModel> createMultiDocumentTransactionAsync(String include, CreateMultiDocumentModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument");
        path.addQuery("$include", include);
        return this.threadPool.submit((RestCall<MultiDocumentModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<MultiDocumentModel>(){}));
    }

    /**
     * Retrieve a MultiDocument transaction
     * 
     * Get the current MultiDocument transaction identified by this URL.
     *             
     * If this transaction was adjusted, the return value of this API will be the current transaction with this code.
     *             
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * Lines
     * * Details (implies lines)
     * * Summary (implies details)
     * * Addresses
     * * SummaryOnly (omit lines and details - reduces API response size)
     * * LinesOnly (omit details - reduces API response size)
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, ProStoresOperator, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param code 
     * @param type  (See DocumentType::* for a list of allowable values)
     * @param include Specifies objects to include in the response after transaction is created
     * @return MultiDocumentModel
     */
    public MultiDocumentModel getMultiDocumentTransactionByCodeAndType(String code, DocumentType type, String include) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument/{code}/type/{type}");
        path.applyField("code", code);
        path.applyField("type", type);
        path.addQuery("$include", include);
        return ((RestCall<MultiDocumentModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<MultiDocumentModel>(){})).call();
    }

    /**
     * Retrieve a MultiDocument transaction
     * 
     * Get the current MultiDocument transaction identified by this URL.
     *             
     * If this transaction was adjusted, the return value of this API will be the current transaction with this code.
     *             
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * Lines
     * * Details (implies lines)
     * * Summary (implies details)
     * * Addresses
     * * SummaryOnly (omit lines and details - reduces API response size)
     * * LinesOnly (omit details - reduces API response size)
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, ProStoresOperator, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param code 
     * @param type  (See DocumentType::* for a list of allowable values)
     * @param include Specifies objects to include in the response after transaction is created
     * @return MultiDocumentModel
     */
    public Future<MultiDocumentModel> getMultiDocumentTransactionByCodeAndTypeAsync(String code, DocumentType type, String include) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument/{code}/type/{type}");
        path.applyField("code", code);
        path.applyField("type", type);
        path.addQuery("$include", include);
        return this.threadPool.submit((RestCall<MultiDocumentModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<MultiDocumentModel>(){}));
    }

    /**
     * Retrieve a MultiDocument transaction by ID
     * 
     * Get the unique MultiDocument transaction identified by this URL.
     *             
     * A traditional transaction requires exactly two parties: a seller and a buyer.  MultiDocument transactions can
     * involve a marketplace of vendors, each of which contributes some portion of the final transaction.  Within
     * a MultiDocument transaction, each individual buyer and seller pair are matched up and converted to a separate
     * document.  This separation of documents allows each seller to file their taxes separately.
     *             
     * This endpoint retrieves the exact transaction identified by this ID number even if that transaction was later adjusted
     * by using the `AdjustTransaction` endpoint.
     *             
     * A transaction represents a unique potentially taxable action that your company has recorded, and transactions include actions like
     * sales, purchases, inventory transfer, and returns (also called refunds).
     *             
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * Lines
     * * Details (implies lines)
     * * Summary (implies details)
     * * Addresses
     * * SummaryOnly (omit lines and details - reduces API response size)
     * * LinesOnly (omit details - reduces API response size)
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, ProStoresOperator, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param id The unique ID number of the MultiDocument transaction to retrieve
     * @param include Specifies objects to include in the response after transaction is created
     * @return MultiDocumentModel
     */
    public MultiDocumentModel getMultiDocumentTransactionById(Long id, String include) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument/{id}");
        path.applyField("id", id);
        path.addQuery("$include", include);
        return ((RestCall<MultiDocumentModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<MultiDocumentModel>(){})).call();
    }

    /**
     * Retrieve a MultiDocument transaction by ID
     * 
     * Get the unique MultiDocument transaction identified by this URL.
     *             
     * A traditional transaction requires exactly two parties: a seller and a buyer.  MultiDocument transactions can
     * involve a marketplace of vendors, each of which contributes some portion of the final transaction.  Within
     * a MultiDocument transaction, each individual buyer and seller pair are matched up and converted to a separate
     * document.  This separation of documents allows each seller to file their taxes separately.
     *             
     * This endpoint retrieves the exact transaction identified by this ID number even if that transaction was later adjusted
     * by using the `AdjustTransaction` endpoint.
     *             
     * A transaction represents a unique potentially taxable action that your company has recorded, and transactions include actions like
     * sales, purchases, inventory transfer, and returns (also called refunds).
     *             
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * Lines
     * * Details (implies lines)
     * * Summary (implies details)
     * * Addresses
     * * SummaryOnly (omit lines and details - reduces API response size)
     * * LinesOnly (omit details - reduces API response size)
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, ProStoresOperator, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param id The unique ID number of the MultiDocument transaction to retrieve
     * @param include Specifies objects to include in the response after transaction is created
     * @return MultiDocumentModel
     */
    public Future<MultiDocumentModel> getMultiDocumentTransactionByIdAsync(Long id, String include) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument/{id}");
        path.applyField("id", id);
        path.addQuery("$include", include);
        return this.threadPool.submit((RestCall<MultiDocumentModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<MultiDocumentModel>(){}));
    }

    /**
     * Retrieve all MultiDocument transactions
     * 
     * List all MultiDocument transactions within this account.
     *             
     * This endpoint is limited to returning 1,000 MultiDocument transactions at a time.  To retrieve more than 1,000 MultiDocument
     * transactions, please use the pagination features of the API.
     *             
     * A transaction represents a unique potentially taxable action that your company has recorded, and transactions include actions like
     * sales, purchases, inventory transfer, and returns (also called refunds).
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     *             
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * Lines
     * * Details (implies lines)
     * * Summary (implies details)
     * * Addresses
     * * SummaryOnly (omit lines and details - reduces API response size)
     * * LinesOnly (omit details - reduces API response size)
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, ProStoresOperator, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* documents
     * @param include Specifies objects to include in the response after transaction is created
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<MultiDocumentModel>
     */
    public FetchResult<MultiDocumentModel> listMultiDocumentTransactions(String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<MultiDocumentModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<MultiDocumentModel>>(){})).call();
    }

    /**
     * Retrieve all MultiDocument transactions
     * 
     * List all MultiDocument transactions within this account.
     *             
     * This endpoint is limited to returning 1,000 MultiDocument transactions at a time.  To retrieve more than 1,000 MultiDocument
     * transactions, please use the pagination features of the API.
     *             
     * A transaction represents a unique potentially taxable action that your company has recorded, and transactions include actions like
     * sales, purchases, inventory transfer, and returns (also called refunds).
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     *             
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * Lines
     * * Details (implies lines)
     * * Summary (implies details)
     * * Addresses
     * * SummaryOnly (omit lines and details - reduces API response size)
     * * LinesOnly (omit details - reduces API response size)
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, ProStoresOperator, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* documents
     * @param include Specifies objects to include in the response after transaction is created
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<MultiDocumentModel>
     */
    public Future<FetchResult<MultiDocumentModel>> listMultiDocumentTransactionsAsync(String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<MultiDocumentModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<MultiDocumentModel>>(){}));
    }

    /**
     * Create a refund for a MultiDocument transaction
     * 
     * Create a refund for a MultiDocument transaction.
     *             
     * A traditional transaction requires exactly two parties: a seller and a buyer.  MultiDocument transactions can
     * involve a marketplace of vendors, each of which contributes some portion of the final transaction.  Within
     * a MultiDocument transaction, each individual buyer and seller pair are matched up and converted to a separate
     * document.  This separation of documents allows each seller to file their taxes separately.
     *             
     * The `RefundTransaction` API allows you to quickly and easily create a `ReturnInvoice` representing a refund
     * for a previously created `SalesInvoice` transaction.  You can choose to create a full or partial refund, and
     * specify individual line items from the original sale for refund.
     *             
     * The `RefundTransaction` API ensures that the tax amount you refund to the customer exactly matches the tax that
     * was calculated during the original transaction, regardless of any changes to your company's configuration, rules,
     * nexus, or any other setting.
     *             
     * This API is intended to be a shortcut to allow you to quickly and accurately generate a refund for the following
     * common refund scenarios:
     *             
     * * A full refund of a previous sale
     * * Refunding the tax that was charged on a previous sale, when the customer provides an exemption certificate after the purchase
     * * Refunding one or more items (lines) from a previous sale
     * * Granting a customer a percentage refund of a previous sale
     *             
     * For more complex scenarios than the ones above, please use `CreateTransaction` with document type `ReturnInvoice` to
     * create a custom refund transaction.
     *             
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * Lines
     * * Details (implies lines)
     * * Summary (implies details)
     * * Addresses
     * * SummaryOnly (omit lines and details - reduces API response size)
     * * LinesOnly (omit details - reduces API response size)
     *             
     * If you omit the `$include` parameter, the API will assume you want `Summary,Addresses`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param code The code of this MultiDocument transaction
     * @param type The type of this MultiDocument transaction (See DocumentType::* for a list of allowable values)
     * @param include Specifies objects to include in the response after transaction is created
     * @param model Information about the refund to create
     * @return MultiDocumentModel
     */
    public MultiDocumentModel refundMultiDocumentTransaction(String code, DocumentType type, String include, RefundTransactionModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument/{code}/type/{type}/refund");
        path.applyField("code", code);
        path.applyField("type", type);
        path.addQuery("$include", include);
        return ((RestCall<MultiDocumentModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<MultiDocumentModel>(){})).call();
    }

    /**
     * Create a refund for a MultiDocument transaction
     * 
     * Create a refund for a MultiDocument transaction.
     *             
     * A traditional transaction requires exactly two parties: a seller and a buyer.  MultiDocument transactions can
     * involve a marketplace of vendors, each of which contributes some portion of the final transaction.  Within
     * a MultiDocument transaction, each individual buyer and seller pair are matched up and converted to a separate
     * document.  This separation of documents allows each seller to file their taxes separately.
     *             
     * The `RefundTransaction` API allows you to quickly and easily create a `ReturnInvoice` representing a refund
     * for a previously created `SalesInvoice` transaction.  You can choose to create a full or partial refund, and
     * specify individual line items from the original sale for refund.
     *             
     * The `RefundTransaction` API ensures that the tax amount you refund to the customer exactly matches the tax that
     * was calculated during the original transaction, regardless of any changes to your company's configuration, rules,
     * nexus, or any other setting.
     *             
     * This API is intended to be a shortcut to allow you to quickly and accurately generate a refund for the following
     * common refund scenarios:
     *             
     * * A full refund of a previous sale
     * * Refunding the tax that was charged on a previous sale, when the customer provides an exemption certificate after the purchase
     * * Refunding one or more items (lines) from a previous sale
     * * Granting a customer a percentage refund of a previous sale
     *             
     * For more complex scenarios than the ones above, please use `CreateTransaction` with document type `ReturnInvoice` to
     * create a custom refund transaction.
     *             
     * You may specify one or more of the following values in the `$include` parameter to fetch additional nested data, using commas to separate multiple values:
     *             
     * * Lines
     * * Details (implies lines)
     * * Summary (implies details)
     * * Addresses
     * * SummaryOnly (omit lines and details - reduces API response size)
     * * LinesOnly (omit details - reduces API response size)
     *             
     * If you omit the `$include` parameter, the API will assume you want `Summary,Addresses`.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, CSPTester, SSTAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param code The code of this MultiDocument transaction
     * @param type The type of this MultiDocument transaction (See DocumentType::* for a list of allowable values)
     * @param include Specifies objects to include in the response after transaction is created
     * @param model Information about the refund to create
     * @return MultiDocumentModel
     */
    public Future<MultiDocumentModel> refundMultiDocumentTransactionAsync(String code, DocumentType type, String include, RefundTransactionModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument/{code}/type/{type}/refund");
        path.applyField("code", code);
        path.applyField("type", type);
        path.addQuery("$include", include);
        return this.threadPool.submit((RestCall<MultiDocumentModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<MultiDocumentModel>(){}));
    }

    /**
     * Verify a MultiDocument transaction
     * 
     * Verifies that the MultiDocument transaction uniquely identified by this URL matches certain expected values.
     *             
     * If the transaction does not match these expected values, this API will return an error code indicating which value did not match.
     *             
     * A transaction represents a unique potentially taxable action that your company has recorded, and transactions include actions like
     * sales, purchases, inventory transfer, and returns (also called refunds).
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, ProStoresOperator, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param model Information from your accounting system to verify against this MultiDocument transaction as it is stored in AvaTax
     * @return MultiDocumentModel
     */
    public MultiDocumentModel verifyMultiDocumentTransaction(VerifyMultiDocumentModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument/verify");
        return ((RestCall<MultiDocumentModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<MultiDocumentModel>(){})).call();
    }

    /**
     * Verify a MultiDocument transaction
     * 
     * Verifies that the MultiDocument transaction uniquely identified by this URL matches certain expected values.
     *             
     * If the transaction does not match these expected values, this API will return an error code indicating which value did not match.
     *             
     * A transaction represents a unique potentially taxable action that your company has recorded, and transactions include actions like
     * sales, purchases, inventory transfer, and returns (also called refunds).
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, CSPAdmin, CSPTester, ProStoresOperator, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param model Information from your accounting system to verify against this MultiDocument transaction as it is stored in AvaTax
     * @return MultiDocumentModel
     */
    public Future<MultiDocumentModel> verifyMultiDocumentTransactionAsync(VerifyMultiDocumentModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument/verify");
        return this.threadPool.submit((RestCall<MultiDocumentModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<MultiDocumentModel>(){}));
    }

    /**
     * Void a MultiDocument transaction
     * 
     * Voids the current transaction uniquely identified by this URL.
     *             
     * A transaction represents a unique potentially taxable action that your company has recorded, and transactions include actions like
     * sales, purchases, inventory transfer, and returns (also called refunds).
     *             
     * When you void a transaction, that transaction's status is recorded as `DocVoided`.
     *             
     * Transactions that have been previously reported to a tax authority by Avalara Managed Returns Service are considered `locked`,
     * and they are no longer available to be voided.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, CompanyAdmin, CSPTester, ProStoresOperator, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param code The transaction code for this MultiDocument transaction
     * @param type The transaction type for this MultiDocument transaction (See DocumentType::* for a list of allowable values)
     * @param model The void request you wish to execute
     * @return MultiDocumentModel
     */
    public MultiDocumentModel voidMultiDocumentTransaction(String code, DocumentType type, VoidTransactionModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument/{code}/type/{type}/void");
        path.applyField("code", code);
        path.applyField("type", type);
        return ((RestCall<MultiDocumentModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<MultiDocumentModel>(){})).call();
    }

    /**
     * Void a MultiDocument transaction
     * 
     * Voids the current transaction uniquely identified by this URL.
     *             
     * A transaction represents a unique potentially taxable action that your company has recorded, and transactions include actions like
     * sales, purchases, inventory transfer, and returns (also called refunds).
     *             
     * When you void a transaction, that transaction's status is recorded as `DocVoided`.
     *             
     * Transactions that have been previously reported to a tax authority by Avalara Managed Returns Service are considered `locked`,
     * and they are no longer available to be voided.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, CompanyAdmin, CSPTester, ProStoresOperator, SSTAdmin, TechnicalSupportAdmin.
     * * This API depends on the following active services<br />*Required* (all):  AvaTaxPro.
     * 
     * @param code The transaction code for this MultiDocument transaction
     * @param type The transaction type for this MultiDocument transaction (See DocumentType::* for a list of allowable values)
     * @param model The void request you wish to execute
     * @return MultiDocumentModel
     */
    public Future<MultiDocumentModel> voidMultiDocumentTransactionAsync(String code, DocumentType type, VoidTransactionModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/multidocument/{code}/type/{type}/void");
        path.applyField("code", code);
        path.applyField("type", type);
        return this.threadPool.submit((RestCall<MultiDocumentModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<MultiDocumentModel>(){}));
    }

    /**
     * Create a new nexus
     * 
     * Creates one or more new nexus declarations attached to this company.
     *             
     * The concept of Nexus indicates a place where your company is legally obligated to collect and remit transactional
     * taxes.  The legal requirements for nexus may vary per country and per jurisdiction; please seek advice from your
     * accountant or lawyer prior to declaring nexus.
     *             
     * To create a nexus declaration for your company, you must first call the Definitions API `ListNexus` to obtain a
     * list of Avalara-defined nexus.  Once you have determined which nexus you wish to declare, you should customize
     * only the user-selectable fields in this object.
     *             
     * The user selectable fields for the nexus object are `companyId`, `effectiveDate`, `endDate`, `localNexusTypeId`,
     * `taxId`, `nexusTypeId`, `hasPermanentEstablishment`, and `isSellerImporterOfRecord`.
     *             
     * When calling `CreateNexus` or `UpdateNexus`, all values in your nexus object except for the user-selectable fields
     * must match an Avalara-defined system nexus object.  You can retrieve a list of Avalara-defined system nexus objects
     * by calling `ListNexus`.  If any data does not match, AvaTax may not recognize your nexus declaration.
     *             
     * Please note that nexus changes may not take effect immediately and you should plan to update your nexus settings in advance
     * of calculating tax for a location.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this nexus.
     * @param model The nexus you wish to create.
     * @return ArrayList<NexusModel>
     */
    public ArrayList<NexusModel> createNexus(Integer companyId, ArrayList<NexusModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<NexusModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NexusModel>>(){})).call();
    }

    /**
     * Create a new nexus
     * 
     * Creates one or more new nexus declarations attached to this company.
     *             
     * The concept of Nexus indicates a place where your company is legally obligated to collect and remit transactional
     * taxes.  The legal requirements for nexus may vary per country and per jurisdiction; please seek advice from your
     * accountant or lawyer prior to declaring nexus.
     *             
     * To create a nexus declaration for your company, you must first call the Definitions API `ListNexus` to obtain a
     * list of Avalara-defined nexus.  Once you have determined which nexus you wish to declare, you should customize
     * only the user-selectable fields in this object.
     *             
     * The user selectable fields for the nexus object are `companyId`, `effectiveDate`, `endDate`, `localNexusTypeId`,
     * `taxId`, `nexusTypeId`, `hasPermanentEstablishment`, and `isSellerImporterOfRecord`.
     *             
     * When calling `CreateNexus` or `UpdateNexus`, all values in your nexus object except for the user-selectable fields
     * must match an Avalara-defined system nexus object.  You can retrieve a list of Avalara-defined system nexus objects
     * by calling `ListNexus`.  If any data does not match, AvaTax may not recognize your nexus declaration.
     *             
     * Please note that nexus changes may not take effect immediately and you should plan to update your nexus settings in advance
     * of calculating tax for a location.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this nexus.
     * @param model The nexus you wish to create.
     * @return ArrayList<NexusModel>
     */
    public Future<ArrayList<NexusModel>> createNexusAsync(Integer companyId, ArrayList<NexusModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<NexusModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NexusModel>>(){}));
    }

    /**
     * Creates nexus for a list of addresses.
     * 
     * This call is intended to simplify adding all applicable nexus to a company, for an address or addresses. Calling this
     * API declares nexus for this company, for the list of addresses provided,
     * for the date range provided. You may also use this API to extend effective date on an already-declared nexus.
     *             
     * The concept of Nexus indicates a place where your company is legally obligated to collect and remit transactional
     * taxes.  The legal requirements for nexus may vary per country and per jurisdiction; please seek advice from your
     * accountant or lawyer prior to declaring nexus.
     *             
     * Note that not all fields within a nexus can be updated; Avalara publishes a list of all defined nexus at the
     * '/api/v2/definitions/nexus' endpoint.
     *             
     * You may only define nexus matching the official list of declared nexus.
     *             
     * Please note that nexus changes may not take effect immediately and you should plan to update your nexus settings in advance
     * of calculating tax for a location.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that will own this nexus.
     * @param model The nexus you wish to create.
     * @return ArrayList<NexusByAddressModel>
     */
    public ArrayList<NexusByAddressModel> declareNexusByAddress(Integer companyId, ArrayList<DeclareNexusByAddressModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus/byaddress");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<NexusByAddressModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NexusByAddressModel>>(){})).call();
    }

    /**
     * Creates nexus for a list of addresses.
     * 
     * This call is intended to simplify adding all applicable nexus to a company, for an address or addresses. Calling this
     * API declares nexus for this company, for the list of addresses provided,
     * for the date range provided. You may also use this API to extend effective date on an already-declared nexus.
     *             
     * The concept of Nexus indicates a place where your company is legally obligated to collect and remit transactional
     * taxes.  The legal requirements for nexus may vary per country and per jurisdiction; please seek advice from your
     * accountant or lawyer prior to declaring nexus.
     *             
     * Note that not all fields within a nexus can be updated; Avalara publishes a list of all defined nexus at the
     * '/api/v2/definitions/nexus' endpoint.
     *             
     * You may only define nexus matching the official list of declared nexus.
     *             
     * Please note that nexus changes may not take effect immediately and you should plan to update your nexus settings in advance
     * of calculating tax for a location.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that will own this nexus.
     * @param model The nexus you wish to create.
     * @return ArrayList<NexusByAddressModel>
     */
    public Future<ArrayList<NexusByAddressModel>> declareNexusByAddressAsync(Integer companyId, ArrayList<DeclareNexusByAddressModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus/byaddress");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<NexusByAddressModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NexusByAddressModel>>(){}));
    }

    /**
     * Delete a single nexus
     * 
     * Marks the existing nexus object at this URL as deleted.
     *             
     * The concept of Nexus indicates a place where your company is legally obligated to collect and remit transactional
     * taxes.  The legal requirements for nexus may vary per country and per jurisdiction; please seek advice from your
     * accountant or lawyer prior to declaring nexus.
     *             
     * Please note that nexus changes may not take effect immediately and you should plan to update your nexus settings in advance
     * of calculating tax for a location.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this nexus.
     * @param id The ID of the nexus you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteNexus(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a single nexus
     * 
     * Marks the existing nexus object at this URL as deleted.
     *             
     * The concept of Nexus indicates a place where your company is legally obligated to collect and remit transactional
     * taxes.  The legal requirements for nexus may vary per country and per jurisdiction; please seek advice from your
     * accountant or lawyer prior to declaring nexus.
     *             
     * Please note that nexus changes may not take effect immediately and you should plan to update your nexus settings in advance
     * of calculating tax for a location.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that owns this nexus.
     * @param id The ID of the nexus you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteNexusAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Retrieve a single nexus
     * 
     * Get the nexus object identified by this URL.
     *             
     * The concept of Nexus indicates a place where your company is legally obligated to collect and remit transactional
     * taxes.  The legal requirements for nexus may vary per country and per jurisdiction; please seek advice from your
     * accountant or lawyer prior to declaring nexus.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The ID of the company that owns this nexus object
     * @param id The primary key of this nexus
     * @return NexusModel
     */
    public NexusModel getNexus(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<NexusModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<NexusModel>(){})).call();
    }

    /**
     * Retrieve a single nexus
     * 
     * Get the nexus object identified by this URL.
     *             
     * The concept of Nexus indicates a place where your company is legally obligated to collect and remit transactional
     * taxes.  The legal requirements for nexus may vary per country and per jurisdiction; please seek advice from your
     * accountant or lawyer prior to declaring nexus.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The ID of the company that owns this nexus object
     * @param id The primary key of this nexus
     * @return NexusModel
     */
    public Future<NexusModel> getNexusAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<NexusModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<NexusModel>(){}));
    }

    /**
     * List company nexus related to a tax form
     * 
     * Retrieves a list of nexus related to a tax form.
     *             
     * The concept of Nexus indicates a place where your company is legally obligated to collect and remit transactional
     * taxes.  The legal requirements for nexus may vary per country and per jurisdiction; please seek advice from your
     * accountant or lawyer prior to declaring nexus.
     *             
     * This API is intended to provide useful information when examining a tax form.  If you are about to begin filing
     * a tax form, you may want to know whether you have declared nexus in all the jurisdictions related to that tax
     * form in order to better understand how the form will be filled out.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The ID of the company that owns this nexus object
     * @param formCode The form code that we are looking up the nexus for
     * @return NexusByTaxFormModel
     */
    public NexusByTaxFormModel getNexusByFormCode(Integer companyId, String formCode) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus/byform/{formCode}");
        path.applyField("companyId", companyId);
        path.applyField("formCode", formCode);
        return ((RestCall<NexusByTaxFormModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<NexusByTaxFormModel>(){})).call();
    }

    /**
     * List company nexus related to a tax form
     * 
     * Retrieves a list of nexus related to a tax form.
     *             
     * The concept of Nexus indicates a place where your company is legally obligated to collect and remit transactional
     * taxes.  The legal requirements for nexus may vary per country and per jurisdiction; please seek advice from your
     * accountant or lawyer prior to declaring nexus.
     *             
     * This API is intended to provide useful information when examining a tax form.  If you are about to begin filing
     * a tax form, you may want to know whether you have declared nexus in all the jurisdictions related to that tax
     * form in order to better understand how the form will be filled out.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The ID of the company that owns this nexus object
     * @param formCode The form code that we are looking up the nexus for
     * @return NexusByTaxFormModel
     */
    public Future<NexusByTaxFormModel> getNexusByFormCodeAsync(Integer companyId, String formCode) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus/byform/{formCode}");
        path.applyField("companyId", companyId);
        path.applyField("formCode", formCode);
        return this.threadPool.submit((RestCall<NexusByTaxFormModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<NexusByTaxFormModel>(){}));
    }

    /**
     * Retrieve nexus for this company
     * 
     * List all nexus objects defined for this company.
     *             
     * The concept of Nexus indicates a place where your company is legally obligated to collect and remit transactional
     * taxes.  The legal requirements for nexus may vary per country and per jurisdiction; please seek advice from your
     * accountant or lawyer prior to declaring nexus.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The ID of the company that owns these nexus objects
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* streamlinedSalesTax, isSSTActive, taxAuthorityId
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NexusModel>
     */
    public FetchResult<NexusModel> listNexusByCompany(Integer companyId, String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){})).call();
    }

    /**
     * Retrieve nexus for this company
     * 
     * List all nexus objects defined for this company.
     *             
     * The concept of Nexus indicates a place where your company is legally obligated to collect and remit transactional
     * taxes.  The legal requirements for nexus may vary per country and per jurisdiction; please seek advice from your
     * accountant or lawyer prior to declaring nexus.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param companyId The ID of the company that owns these nexus objects
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* streamlinedSalesTax, isSSTActive, taxAuthorityId
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NexusModel>
     */
    public Future<FetchResult<NexusModel>> listNexusByCompanyAsync(Integer companyId, String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){}));
    }

    /**
     * Retrieve all nexus
     * 
     * Get multiple nexus objects across all companies.
     *             
     * The concept of Nexus indicates a place where your company is legally obligated to collect and remit transactional
     * taxes.  The legal requirements for nexus may vary per country and per jurisdiction; please seek advice from your
     * accountant or lawyer prior to declaring nexus.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* streamlinedSalesTax, isSSTActive, taxAuthorityId
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NexusModel>
     */
    public FetchResult<NexusModel> queryNexus(String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/nexus");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){})).call();
    }

    /**
     * Retrieve all nexus
     * 
     * Get multiple nexus objects across all companies.
     *             
     * The concept of Nexus indicates a place where your company is legally obligated to collect and remit transactional
     * taxes.  The legal requirements for nexus may vary per country and per jurisdiction; please seek advice from your
     * accountant or lawyer prior to declaring nexus.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountUser, CompanyAdmin, CompanyUser, Compliance Root User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, SiteAdmin, SSTAdmin, SystemAdmin, TechnicalSupportAdmin, TechnicalSupportUser.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* streamlinedSalesTax, isSSTActive, taxAuthorityId
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NexusModel>
     */
    public Future<FetchResult<NexusModel>> queryNexusAsync(String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/nexus");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){}));
    }

    /**
     * Update a single nexus
     * 
     * Replace the existing nexus declaration object at this URL with an updated object.
     *             
     * The concept of Nexus indicates a place where your company is legally obligated to collect and remit transactional
     * taxes.  The legal requirements for nexus may vary per country and per jurisdiction; please seek advice from your
     * accountant or lawyer prior to declaring nexus.
     *             
     * To create a nexus declaration for your company, you must first call the Definitions API `ListNexus` to obtain a
     * list of Avalara-defined nexus.  Once you have determined which nexus you wish to declare, you should customize
     * only the user-selectable fields in this object.
     *             
     * The user selectable fields for the nexus object are `companyId`, `effectiveDate`, `endDate`, `localNexusTypeId`,
     * `taxId`, `nexusTypeId`, `hasPermanentEstablishment`, and `isSellerImporterOfRecord`.
     *             
     * When calling `CreateNexus` or `UpdateNexus`, all values in your nexus object except for the user-selectable fields
     * must match an Avalara-defined system nexus object.  You can retrieve a list of Avalara-defined system nexus objects
     * by calling `ListNexus`.  If any data does not match, AvaTax may not recognize your nexus declaration.
     *             
     * Please note that nexus changes may not take effect immediately and you should plan to update your nexus settings in advance
     * of calculating tax for a location.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that this nexus belongs to.
     * @param id The ID of the nexus you wish to update
     * @param model The nexus object you wish to update.
     * @return NexusModel
     */
    public NexusModel updateNexus(Integer companyId, Integer id, NexusModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<NexusModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<NexusModel>(){})).call();
    }

    /**
     * Update a single nexus
     * 
     * Replace the existing nexus declaration object at this URL with an updated object.
     *             
     * The concept of Nexus indicates a place where your company is legally obligated to collect and remit transactional
     * taxes.  The legal requirements for nexus may vary per country and per jurisdiction; please seek advice from your
     * accountant or lawyer prior to declaring nexus.
     *             
     * To create a nexus declaration for your company, you must first call the Definitions API `ListNexus` to obtain a
     * list of Avalara-defined nexus.  Once you have determined which nexus you wish to declare, you should customize
     * only the user-selectable fields in this object.
     *             
     * The user selectable fields for the nexus object are `companyId`, `effectiveDate`, `endDate`, `localNexusTypeId`,
     * `taxId`, `nexusTypeId`, `hasPermanentEstablishment`, and `isSellerImporterOfRecord`.
     *             
     * When calling `CreateNexus` or `UpdateNexus`, all values in your nexus object except for the user-selectable fields
     * must match an Avalara-defined system nexus object.  You can retrieve a list of Avalara-defined system nexus objects
     * by calling `ListNexus`.  If any data does not match, AvaTax may not recognize your nexus declaration.
     *             
     * Please note that nexus changes may not take effect immediately and you should plan to update your nexus settings in advance
     * of calculating tax for a location.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, CompanyAdmin, CSPTester, SSTAdmin, TechnicalSupportAdmin.
     * 
     * @param companyId The ID of the company that this nexus belongs to.
     * @param id The ID of the nexus you wish to update
     * @param model The nexus object you wish to update.
     * @return NexusModel
     */
    public Future<NexusModel> updateNexusAsync(Integer companyId, Integer id, NexusModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<NexusModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<NexusModel>(){}));
    }

    /**
     * Create a new notice comment.
     * 
     * This API is available by invitation only.
     * 'Notice comments' are updates by the notice team on the work to be done and that has been done so far on a notice.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param id The ID of the tax notice we are adding the comment for.
     * @param model The notice comments you wish to create.
     * @return ArrayList<NoticeCommentModel>
     */
    public ArrayList<NoticeCommentModel> createNoticeComment(Integer companyId, Integer id, ArrayList<NoticeCommentModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/comments");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<NoticeCommentModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NoticeCommentModel>>(){})).call();
    }

    /**
     * Create a new notice comment.
     * 
     * This API is available by invitation only.
     * 'Notice comments' are updates by the notice team on the work to be done and that has been done so far on a notice.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param id The ID of the tax notice we are adding the comment for.
     * @param model The notice comments you wish to create.
     * @return ArrayList<NoticeCommentModel>
     */
    public Future<ArrayList<NoticeCommentModel>> createNoticeCommentAsync(Integer companyId, Integer id, ArrayList<NoticeCommentModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/comments");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<NoticeCommentModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NoticeCommentModel>>(){}));
    }

    /**
     * Create a new notice finance details.
     * 
     * This API is available by invitation only.
     * 'Notice finance details' is the categorical breakdown of the total charge levied by the tax authority on our customer,
     * as broken down in our "notice log" found in Workflow. Main examples of the categories are 'Tax Due', 'Interest', 'Penalty', 'Total Abated'.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param id The ID of the notice added to the finance details.
     * @param model The notice finance details you wish to create.
     * @return ArrayList<NoticeFinanceModel>
     */
    public ArrayList<NoticeFinanceModel> createNoticeFinanceDetails(Integer companyId, Integer id, ArrayList<NoticeFinanceModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/financedetails");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<NoticeFinanceModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NoticeFinanceModel>>(){})).call();
    }

    /**
     * Create a new notice finance details.
     * 
     * This API is available by invitation only.
     * 'Notice finance details' is the categorical breakdown of the total charge levied by the tax authority on our customer,
     * as broken down in our "notice log" found in Workflow. Main examples of the categories are 'Tax Due', 'Interest', 'Penalty', 'Total Abated'.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param id The ID of the notice added to the finance details.
     * @param model The notice finance details you wish to create.
     * @return ArrayList<NoticeFinanceModel>
     */
    public Future<ArrayList<NoticeFinanceModel>> createNoticeFinanceDetailsAsync(Integer companyId, Integer id, ArrayList<NoticeFinanceModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/financedetails");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<NoticeFinanceModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NoticeFinanceModel>>(){}));
    }

    /**
     * Create a new notice responsibility.
     * 
     * This API is available by invitation only.
     * 'Notice comments' are updates by the notice team on the work to be done and that has been done so far on a notice.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param id The ID of the tax notice we are adding the responsibility for.
     * @param model The notice responsibilities you wish to create.
     * @return ArrayList<NoticeResponsibilityDetailModel>
     */
    public ArrayList<NoticeResponsibilityDetailModel> createNoticeResponsibilities(Integer companyId, Integer id, ArrayList<NoticeResponsibilityDetailModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/responsibilities");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<NoticeResponsibilityDetailModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NoticeResponsibilityDetailModel>>(){})).call();
    }

    /**
     * Create a new notice responsibility.
     * 
     * This API is available by invitation only.
     * 'Notice comments' are updates by the notice team on the work to be done and that has been done so far on a notice.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param id The ID of the tax notice we are adding the responsibility for.
     * @param model The notice responsibilities you wish to create.
     * @return ArrayList<NoticeResponsibilityDetailModel>
     */
    public Future<ArrayList<NoticeResponsibilityDetailModel>> createNoticeResponsibilitiesAsync(Integer companyId, Integer id, ArrayList<NoticeResponsibilityDetailModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/responsibilities");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<NoticeResponsibilityDetailModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NoticeResponsibilityDetailModel>>(){}));
    }

    /**
     * Create a new notice root cause.
     * 
     * This API is available by invitation only.
     * 'Notice root causes' are are those who are responsible for the notice.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param id The ID of the tax notice we are adding the responsibility for.
     * @param model The notice root causes you wish to create.
     * @return ArrayList<NoticeRootCauseDetailModel>
     */
    public ArrayList<NoticeRootCauseDetailModel> createNoticeRootCauses(Integer companyId, Integer id, ArrayList<NoticeRootCauseDetailModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/rootcauses");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<NoticeRootCauseDetailModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NoticeRootCauseDetailModel>>(){})).call();
    }

    /**
     * Create a new notice root cause.
     * 
     * This API is available by invitation only.
     * 'Notice root causes' are are those who are responsible for the notice.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param id The ID of the tax notice we are adding the responsibility for.
     * @param model The notice root causes you wish to create.
     * @return ArrayList<NoticeRootCauseDetailModel>
     */
    public Future<ArrayList<NoticeRootCauseDetailModel>> createNoticeRootCausesAsync(Integer companyId, Integer id, ArrayList<NoticeRootCauseDetailModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/rootcauses");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<NoticeRootCauseDetailModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NoticeRootCauseDetailModel>>(){}));
    }

    /**
     * Create a new notice.
     * 
     * This API is available by invitation only.
     * Create one or more new notice objects.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param model The notice object you wish to create.
     * @return ArrayList<NoticeModel>
     */
    public ArrayList<NoticeModel> createNotices(Integer companyId, ArrayList<NoticeModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<NoticeModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NoticeModel>>(){})).call();
    }

    /**
     * Create a new notice.
     * 
     * This API is available by invitation only.
     * Create one or more new notice objects.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param model The notice object you wish to create.
     * @return ArrayList<NoticeModel>
     */
    public Future<ArrayList<NoticeModel>> createNoticesAsync(Integer companyId, ArrayList<NoticeModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<NoticeModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NoticeModel>>(){}));
    }

    /**
     * Delete a single notice.
     * 
     * This API is available by invitation only.
     * 'Notice comments' are updates by the notice team on the work to be done and that has been done so far on a notice.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param id The ID of the notice you wish to delete the finance detail from.
     * @param commentDetailsId The ID of the comment you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteCommentDetails(Integer companyId, Integer id, Integer commentDetailsId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/commentdetails/{commentdetailsid}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.applyField("commentDetailsId", commentDetailsId);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a single notice.
     * 
     * This API is available by invitation only.
     * 'Notice comments' are updates by the notice team on the work to be done and that has been done so far on a notice.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param id The ID of the notice you wish to delete the finance detail from.
     * @param commentDetailsId The ID of the comment you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteCommentDetailsAsync(Integer companyId, Integer id, Integer commentDetailsId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/commentdetails/{commentdetailsid}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.applyField("commentDetailsId", commentDetailsId);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Delete a single notice.
     * 
     * This API is available by invitation only.
     * 'Notice finance details' is the categorical breakdown of the total charge levied by the tax authority on our customer,
     * as broken down in our "notice log" found in Workflow. Main examples of the categories are 'Tax Due', 'Interest', 'Penalty', 'Total Abated'.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param id The ID of the notice you wish to delete the finance detail from.
     * @param financeDetailsId The ID of the finance detail you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteFinanceDetails(Integer companyId, Integer id, Integer financeDetailsId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/financedetails/{financedetailsid}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.applyField("financeDetailsId", financeDetailsId);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a single notice.
     * 
     * This API is available by invitation only.
     * 'Notice finance details' is the categorical breakdown of the total charge levied by the tax authority on our customer,
     * as broken down in our "notice log" found in Workflow. Main examples of the categories are 'Tax Due', 'Interest', 'Penalty', 'Total Abated'.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param id The ID of the notice you wish to delete the finance detail from.
     * @param financeDetailsId The ID of the finance detail you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteFinanceDetailsAsync(Integer companyId, Integer id, Integer financeDetailsId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/financedetails/{financedetailsid}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.applyField("financeDetailsId", financeDetailsId);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Delete a single notice.
     * 
     * This API is available by invitation only.
     * Mark the existing notice object at this URL as deleted.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param id The ID of the notice you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteNotice(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a single notice.
     * 
     * This API is available by invitation only.
     * Mark the existing notice object at this URL as deleted.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param id The ID of the notice you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteNoticeAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Delete a single responsibility
     * 
     * This API is available by invitation only.
     * Mark the existing notice object at this URL as deleted.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param noticeId The ID of the notice you wish to delete.
     * @param id The ID of the responsibility you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteResponsibilities(Integer companyId, Integer noticeId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{noticeId}/responsibilities/{id}");
        path.applyField("companyId", companyId);
        path.applyField("noticeId", noticeId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a single responsibility
     * 
     * This API is available by invitation only.
     * Mark the existing notice object at this URL as deleted.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param noticeId The ID of the notice you wish to delete.
     * @param id The ID of the responsibility you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteResponsibilitiesAsync(Integer companyId, Integer noticeId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{noticeId}/responsibilities/{id}");
        path.applyField("companyId", companyId);
        path.applyField("noticeId", noticeId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Delete a single root cause.
     * 
     * This API is available by invitation only.
     * Mark the existing notice object at this URL as deleted.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param noticeId The ID of the notice you wish to delete.
     * @param id The ID of the root cause you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public ArrayList<ErrorDetail> deleteRootCauses(Integer companyId, Integer noticeId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{noticeId}/rootcauses/{id}");
        path.applyField("companyId", companyId);
        path.applyField("noticeId", noticeId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){})).call();
    }

    /**
     * Delete a single root cause.
     * 
     * This API is available by invitation only.
     * Mark the existing notice object at this URL as deleted.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns this notice.
     * @param noticeId The ID of the notice you wish to delete.
     * @param id The ID of the root cause you wish to delete.
     * @return ArrayList<ErrorDetail>
     */
    public Future<ArrayList<ErrorDetail>> deleteRootCausesAsync(Integer companyId, Integer noticeId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{noticeId}/rootcauses/{id}");
        path.applyField("companyId", companyId);
        path.applyField("noticeId", noticeId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<ErrorDetail>>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ArrayList<ErrorDetail>>(){}));
    }

    /**
     * Retrieve a single attachment
     * 
     * This API is available by invitation only.
     * Get the file attachment identified by this URL.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company for this attachment.
     * @param id The ResourceFileId of the attachment to download.
     * @return String
     */
    public String downloadNoticeAttachment(Integer companyId, Long id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/files/{id}/attachment");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<String>)restCallFactory.createRestCall("get", path, null, new TypeToken<String>(){})).call();
    }

    /**
     * Retrieve a single attachment
     * 
     * This API is available by invitation only.
     * Get the file attachment identified by this URL.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company for this attachment.
     * @param id The ResourceFileId of the attachment to download.
     * @return String
     */
    public Future<String> downloadNoticeAttachmentAsync(Integer companyId, Long id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/files/{id}/attachment");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<String>)restCallFactory.createRestCall("get", path, null, new TypeToken<String>(){}));
    }

    /**
     * Retrieve a single notice.
     * 
     * This API is available by invitation only.
     * Get the tax notice object identified by this URL.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company for this notice.
     * @param id The ID of this notice.
     * @return NoticeModel
     */
    public NoticeModel getNotice(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<NoticeModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<NoticeModel>(){})).call();
    }

    /**
     * Retrieve a single notice.
     * 
     * This API is available by invitation only.
     * Get the tax notice object identified by this URL.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company for this notice.
     * @param id The ID of this notice.
     * @return NoticeModel
     */
    public Future<NoticeModel> getNoticeAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<NoticeModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<NoticeModel>(){}));
    }

    /**
     * Retrieve notice comments for a specific notice.
     * 
     * This API is available by invitation only.
     * 'Notice comments' are updates by the notice team on the work to be done and that has been done so far on a notice.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param id The ID of the notice.
     * @param companyId The ID of the company that owns these notices.
     * @return FetchResult<NoticeCommentModel>
     */
    public FetchResult<NoticeCommentModel> getNoticeComments(Integer id, Integer companyId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/comments");
        path.applyField("id", id);
        path.applyField("companyId", companyId);
        return ((RestCall<FetchResult<NoticeCommentModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeCommentModel>>(){})).call();
    }

    /**
     * Retrieve notice comments for a specific notice.
     * 
     * This API is available by invitation only.
     * 'Notice comments' are updates by the notice team on the work to be done and that has been done so far on a notice.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param id The ID of the notice.
     * @param companyId The ID of the company that owns these notices.
     * @return FetchResult<NoticeCommentModel>
     */
    public Future<FetchResult<NoticeCommentModel>> getNoticeCommentsAsync(Integer id, Integer companyId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/comments");
        path.applyField("id", id);
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<FetchResult<NoticeCommentModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeCommentModel>>(){}));
    }

    /**
     * Retrieve notice finance details for a specific notice.
     * 
     * This API is available by invitation only.
     * 'Notice finance details' is the categorical breakdown of the total charge levied by the tax authority on our customer,
     * as broken down in our "notice log" found in Workflow. Main examples of the categories are 'Tax Due', 'Interest', 'Penalty', 'Total Abated'.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param id The ID of the company that owns these notices.
     * @param companyId The ID of the company that owns these notices.
     * @return FetchResult<NoticeFinanceModel>
     */
    public FetchResult<NoticeFinanceModel> getNoticeFinanceDetails(Integer id, Integer companyId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/financedetails");
        path.applyField("id", id);
        path.applyField("companyId", companyId);
        return ((RestCall<FetchResult<NoticeFinanceModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeFinanceModel>>(){})).call();
    }

    /**
     * Retrieve notice finance details for a specific notice.
     * 
     * This API is available by invitation only.
     * 'Notice finance details' is the categorical breakdown of the total charge levied by the tax authority on our customer,
     * as broken down in our "notice log" found in Workflow. Main examples of the categories are 'Tax Due', 'Interest', 'Penalty', 'Total Abated'.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param id The ID of the company that owns these notices.
     * @param companyId The ID of the company that owns these notices.
     * @return FetchResult<NoticeFinanceModel>
     */
    public Future<FetchResult<NoticeFinanceModel>> getNoticeFinanceDetailsAsync(Integer id, Integer companyId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/financedetails");
        path.applyField("id", id);
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<FetchResult<NoticeFinanceModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeFinanceModel>>(){}));
    }

    /**
     * Retrieve notice responsibilities for a specific notice.
     * 
     * This API is available by invitation only.
     * 'Notice responsibilities' are are those who are responsible for the notice.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param id The ID of the notice.
     * @param companyId The ID of the company that owns these notices.
     * @return FetchResult<NoticeResponsibilityDetailModel>
     */
    public FetchResult<NoticeResponsibilityDetailModel> getNoticeResponsibilities(Integer id, Integer companyId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/responsibilities");
        path.applyField("id", id);
        path.applyField("companyId", companyId);
        return ((RestCall<FetchResult<NoticeResponsibilityDetailModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeResponsibilityDetailModel>>(){})).call();
    }

    /**
     * Retrieve notice responsibilities for a specific notice.
     * 
     * This API is available by invitation only.
     * 'Notice responsibilities' are are those who are responsible for the notice.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param id The ID of the notice.
     * @param companyId The ID of the company that owns these notices.
     * @return FetchResult<NoticeResponsibilityDetailModel>
     */
    public Future<FetchResult<NoticeResponsibilityDetailModel>> getNoticeResponsibilitiesAsync(Integer id, Integer companyId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/responsibilities");
        path.applyField("id", id);
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<FetchResult<NoticeResponsibilityDetailModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeResponsibilityDetailModel>>(){}));
    }

    /**
     * Retrieve notice root causes for a specific notice.
     * 
     * This API is available by invitation only.
     * 'Notice root causes' are are those who are responsible for the notice.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param id The ID of the notice.
     * @param companyId The ID of the company that owns these notices.
     * @return FetchResult<NoticeRootCauseDetailModel>
     */
    public FetchResult<NoticeRootCauseDetailModel> getNoticeRootCauses(Integer id, Integer companyId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/rootcauses");
        path.applyField("id", id);
        path.applyField("companyId", companyId);
        return ((RestCall<FetchResult<NoticeRootCauseDetailModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeRootCauseDetailModel>>(){})).call();
    }

    /**
     * Retrieve notice root causes for a specific notice.
     * 
     * This API is available by invitation only.
     * 'Notice root causes' are are those who are responsible for the notice.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param id The ID of the notice.
     * @param companyId The ID of the company that owns these notices.
     * @return FetchResult<NoticeRootCauseDetailModel>
     */
    public Future<FetchResult<NoticeRootCauseDetailModel>> getNoticeRootCausesAsync(Integer id, Integer companyId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/rootcauses");
        path.applyField("id", id);
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<FetchResult<NoticeRootCauseDetailModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeRootCauseDetailModel>>(){}));
    }

    /**
     * Retrieve notices for a company.
     * 
     * This API is available by invitation only.
     * List all tax notice objects assigned to this company.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns these notices.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* status, totalRemit, ticketReferenceNo, ticketReferenceUrl, reason, type, createdByUserName, documentReference, jurisdictionName, jurisdictionType, comments, finances, responsibility, rootCause
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeModel>
     */
    public FetchResult<NoticeModel> listNoticesByCompany(Integer companyId, String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NoticeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeModel>>(){})).call();
    }

    /**
     * Retrieve notices for a company.
     * 
     * This API is available by invitation only.
     * List all tax notice objects assigned to this company.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param companyId The ID of the company that owns these notices.
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* status, totalRemit, ticketReferenceNo, ticketReferenceUrl, reason, type, createdByUserName, documentReference, jurisdictionName, jurisdictionType, comments, finances, responsibility, rootCause
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeModel>
     */
    public Future<FetchResult<NoticeModel>> listNoticesByCompanyAsync(Integer companyId, String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NoticeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeModel>>(){}));
    }

    /**
     * Retrieve all notices.
     * 
     * This API is available by invitation only.
     *             
     * This API is deprecated - please use POST `/api/v2/notices/query` API.
     *             
     * Get multiple notice objects across all companies.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* status, totalRemit, ticketReferenceNo, ticketReferenceUrl, reason, type, createdByUserName, documentReference, jurisdictionName, jurisdictionType, comments, finances, responsibility, rootCause
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeModel>
     */
    public FetchResult<NoticeModel> queryNotices(String filter, String include, Integer top, Integer skip, String orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/notices");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return ((RestCall<FetchResult<NoticeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeModel>>(){})).call();
    }

    /**
     * Retrieve all notices.
     * 
     * This API is available by invitation only.
     *             
     * This API is deprecated - please use POST `/api/v2/notices/query` API.
     *             
     * Get multiple notice objects across all companies.
     * A 'notice' represents a letter sent to a business by a tax authority regarding tax filing issues.  Avalara
     * Returns customers often receive support and assistance from the Compliance Notices team in handling notices received by taxing authorities.
     *             
     * Search for specific objects using the criteria in the `$filter` parameter; full documentation is available on [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/) .
     * Paginate your results using the `$top`, `$skip`, and `$orderby` parameters.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param filter A filter statement to identify specific records to retrieve. For more information on filtering, see [Filtering in REST](http://developer.avalara.com/avatax/filtering-in-rest/).<br />*Not filterable:* status, totalRemit, ticketReferenceNo, ticketReferenceUrl, reason, type, createdByUserName, documentReference, jurisdictionName, jurisdictionType, comments, finances, responsibility, rootCause
     * @param include A comma separated list of additional data to retrieve.
     * @param top If nonzero, return no more than this number of results. Used with `$skip` to provide pagination for large datasets. Unless otherwise specified, the maximum number of records that can be returned from an API call is 1,000 records.
     * @param skip If nonzero, skip this number of results before returning data. Used with `$top` to provide pagination for large datasets.
     * @param orderBy A comma separated list of sort statements in the format `(fieldname) [ASC|DESC]`, for example `id ASC`.
     * @return FetchResult<NoticeModel>
     */
    public Future<FetchResult<NoticeModel>> queryNoticesAsync(String filter, String include, Integer top, Integer skip, String orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/notices");
        path.addQuery("$filter", filter);
        path.addQuery("$include", include);
        path.addQuery("$top", top);
        path.addQuery("$skip", skip);
        path.addQuery("$orderBy", orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NoticeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeModel>>(){}));
    }

    /**
     * Retrieve all notices.
     * 
     * This API is available by invitation only.
     *             
     * This API is intended to replace the GET `/api/v2/notices` API. The fetch request object is posted on the body of the request instead of the URI, so it's not limited by a set number of characters.
     * The documentation of the GET API shows how filtering, sorting and pagination works.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: AccountAdmin, AccountOperator, AccountUser, CompanyAdmin, CompanyUser, Compliance Operator, Compliance Root User, Compliance Temp User, ComplianceAdmin, ComplianceUser, CSPAdmin, CSPTester, FirmAdmin, FirmUser, ProStoresOperator, Registrar, SiteAdmin, SSTAdmin, SystemAdmin, SystemOperator, TechnicalSupportAdmin, TechnicalSupportUser, TreasuryAdmin, TreasuryUser.
     * * This API depends on the following active services<br />*Returns* (at least one of):  Mrs, MRSComplianceManager, AvaTaxCsp.
     * * This API is available by invitation only.  To request access to this feature, please speak to a business development manager and request host address whitelisting for [Returns] for your servers.
     * 
     * @param model Query object to filter, sort and paginate the filing calendars.
     * @return FetchResult<NoticeModel>
     */
    public FetchResult<NoticeModel> queryNoticesPost(QueryRequestModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/notices/query");
        return ((RestCall<FetchResult<NoticeModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<FetchResult<NoticeModel>>(){})).call();
    }

    /**
     * Retrieve all notices.
     * 
     * This API is available by invitation only.
     *             
     * This API is intended to replace the GET `/api/v2/notices` API. The fetch request object is posted on the body of the request instead of the URI, so it's not limited by a set number of characters.
     * The documentation of the GET API shows how filtering, sorting and pagination works.
     * 
     * ### Security Policies
     * 
     * * This API requires one of the following user roles: 