package net.avalara.avatax.rest.client;

import com.google.gson.reflect.TypeToken;
import net.avalara.avatax.rest.client.models.*;
import net.avalara.avatax.rest.client.enums.*;

import java.math.BigDecimal;
import java.util.Base64;
import java.util.Date;
import java.util.HashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.ArrayList;

public class AvaTaxClient {

    private final ExecutorService threadPool;
    private RestCallFactory restCallFactory;

    private AvaTaxClient() {
        this(null);
    }

    private AvaTaxClient(ExecutorService threadPool) {
        if (threadPool != null) {
            this.threadPool = threadPool;
        } else {
            this.threadPool = Executors.newFixedThreadPool(3);
        }
    }

    public AvaTaxClient(String appName, String appVersion, String machineName, AvaTaxEnvironment environment) {
        this(appName, appVersion, machineName, environment == AvaTaxEnvironment.Production ? AvaTaxConstants.Production_Url : AvaTaxConstants.Sandbox_Url);
    }

    public AvaTaxClient(String appName, String appVersion, String machineName, String environmentUrl) {
        this();
        this.restCallFactory = new RestCallFactory(appName, appVersion, machineName, environmentUrl);
    }

    public AvaTaxClient withSecurity(String securityHeader) {
        this.restCallFactory.addSecurityHeader(securityHeader);

        return this;
    }

    public AvaTaxClient withSecurity(String username, String password) {
        String header = null;

        try {
            header = Base64.getEncoder().encodeToString((username + ":" + password).getBytes("utf-8"));
        } catch (java.io.UnsupportedEncodingException ex) {
            System.out.println("Could not find encoding for UTF-8.");
            ex.printStackTrace();
        }

        return withSecurity(header);
    }


//region Accounts
    /**
     * Reset this account's license key
     */
    public Future<LicenseKeyModel> accountResetLicenseKeyAsync(Integer id, ResetLicenseKeyModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}/resetlicensekey");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<LicenseKeyModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<LicenseKeyModel>(){}));
    }

    public LicenseKeyModel accountResetLicenseKey(Integer id, ResetLicenseKeyModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}/resetlicensekey");
        path.applyField("id", id);
        return ((RestCall<LicenseKeyModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<LicenseKeyModel>(){})).call();
    }
    
    /**
     * Retrieve a single account
     */
    public Future<AccountModel> getAccountAsync(Integer id, String $include) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}");
        path.applyField("id", id);
        path.addQuery("$include", $include);
        return this.threadPool.submit((RestCall<AccountModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<AccountModel>(){}));
    }

    public AccountModel getAccount(Integer id, String $include) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}");
        path.applyField("id", id);
        path.addQuery("$include", $include);
        return ((RestCall<AccountModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<AccountModel>(){})).call();
    }
    
//endregion

//region Addresses
    /**
     * Retrieve geolocation information for a specified address
     */
    public Future<AddressResolutionModel> resolveAddressAsync(String line1, String line2, String line3, String city, String region, String postalCode, String country, BigDecimal latitude, BigDecimal longitude) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/addresses/resolve");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        path.addQuery("latitude", latitude);
        path.addQuery("longitude", longitude);
        return this.threadPool.submit((RestCall<AddressResolutionModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<AddressResolutionModel>(){}));
    }

    public AddressResolutionModel resolveAddress(String line1, String line2, String line3, String city, String region, String postalCode, String country, BigDecimal latitude, BigDecimal longitude) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/addresses/resolve");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        path.addQuery("latitude", latitude);
        path.addQuery("longitude", longitude);
        return ((RestCall<AddressResolutionModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<AddressResolutionModel>(){})).call();
    }
    
    /**
     * Retrieve geolocation information for a specified address
     */
    public Future<AddressResolutionModel> resolveAddressPostAsync(AddressInfo model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/addresses/resolve");
        return this.threadPool.submit((RestCall<AddressResolutionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<AddressResolutionModel>(){}));
    }

    public AddressResolutionModel resolveAddressPost(AddressInfo model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/addresses/resolve");
        return ((RestCall<AddressResolutionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<AddressResolutionModel>(){})).call();
    }
    
//endregion

//region Batches
    /**
     * Create a new batch
     */
    public Future<ArrayList<BatchModel>> createBatchesAsync(Integer companyId, ArrayList<BatchModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<BatchModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<BatchModel>>(){}));
    }

    public ArrayList<BatchModel> createBatches(Integer companyId, ArrayList<BatchModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<BatchModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<BatchModel>>(){})).call();
    }
    
    /**
     * Delete a single batch
     */
    public Future<ErrorResult> deleteBatchAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){}));
    }

    public ErrorResult deleteBatch(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){})).call();
    }
    
    /**
     * Retrieve a single batch
     */
    public Future<BatchModel> getBatchAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<BatchModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<BatchModel>(){}));
    }

    public BatchModel getBatch(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<BatchModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<BatchModel>(){})).call();
    }
    
    /**
     * Retrieve all batches for this company
     */
    public Future<FetchResult<BatchModel>> listBatchesByCompanyAsync(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<BatchModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<BatchModel>>(){}));
    }

    public FetchResult<BatchModel> listBatchesByCompany(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<BatchModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<BatchModel>>(){})).call();
    }
    
    /**
     * Retrieve all batches
     */
    public Future<FetchResult<BatchModel>> queryBatchesAsync(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/batches");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<BatchModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<BatchModel>>(){}));
    }

    public FetchResult<BatchModel> queryBatches(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/batches");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<BatchModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<BatchModel>>(){})).call();
    }
    
    /**
     * Update a single batch
     */
    public Future<BatchModel> updateBatchAsync(Integer companyId, Integer id, BatchModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<BatchModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<BatchModel>(){}));
    }

    public BatchModel updateBatch(Integer companyId, Integer id, BatchModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/batches/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<BatchModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<BatchModel>(){})).call();
    }
    
//endregion

//region Companies
    /**
     * Quick setup for a company with a single physical address
     */
    public Future<CompanyModel> companyInitializeAsync(CompanyInitializationModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/initialize");
        return this.threadPool.submit((RestCall<CompanyModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<CompanyModel>(){}));
    }

    public CompanyModel companyInitialize(CompanyInitializationModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/initialize");
        return ((RestCall<CompanyModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<CompanyModel>(){})).call();
    }
    
    /**
     * Create new companies
     */
    public Future<ArrayList<CompanyModel>> createCompaniesAsync(ArrayList<CompanyModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies");
        return this.threadPool.submit((RestCall<ArrayList<CompanyModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<CompanyModel>>(){}));
    }

    public ArrayList<CompanyModel> createCompanies(ArrayList<CompanyModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies");
        return ((RestCall<ArrayList<CompanyModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<CompanyModel>>(){})).call();
    }
    
    /**
     * Request managed returns funding setup for a company
     */
    public Future<FundingStatusModel> createFundingRequestAsync(Integer id, FundingInitiateModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}/funding/setup");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FundingStatusModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FundingStatusModel>(){}));
    }

    public FundingStatusModel createFundingRequest(Integer id, FundingInitiateModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}/funding/setup");
        path.applyField("id", id);
        return ((RestCall<FundingStatusModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FundingStatusModel>(){})).call();
    }
    
    /**
     * Delete a single company
     */
    public Future<ErrorResult> deleteCompaniesAsync(Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){}));
    }

    public ErrorResult deleteCompanies(Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}");
        path.applyField("id", id);
        return ((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){})).call();
    }
    
    /**
     * Retrieve a single company
     */
    public Future<CompanyModel> getCompanyAsync(Integer id, String $include) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}");
        path.applyField("id", id);
        path.addQuery("$include", $include);
        return this.threadPool.submit((RestCall<CompanyModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<CompanyModel>(){}));
    }

    public CompanyModel getCompany(Integer id, String $include) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}");
        path.applyField("id", id);
        path.addQuery("$include", $include);
        return ((RestCall<CompanyModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<CompanyModel>(){})).call();
    }
    
    /**
     * Check managed returns funding configuration for a company
     */
    public Future<ArrayList<FundingStatusModel>> listFundingRequestsByCompanyAsync(Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}/funding");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<FundingStatusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<FundingStatusModel>>(){}));
    }

    public ArrayList<FundingStatusModel> listFundingRequestsByCompany(Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}/funding");
        path.applyField("id", id);
        return ((RestCall<ArrayList<FundingStatusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<FundingStatusModel>>(){})).call();
    }
    
    /**
     * Retrieve all companies
     */
    public Future<FetchResult<CompanyModel>> queryCompaniesAsync(String $include, String $filter, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies");
        path.addQuery("$include", $include);
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<CompanyModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CompanyModel>>(){}));
    }

    public FetchResult<CompanyModel> queryCompanies(String $include, String $filter, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies");
        path.addQuery("$include", $include);
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<CompanyModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<CompanyModel>>(){})).call();
    }
    
    /**
     * Update a single company
     */
    public Future<CompanyModel> updateCompanyAsync(Integer id, CompanyModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<CompanyModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<CompanyModel>(){}));
    }

    public CompanyModel updateCompany(Integer id, CompanyModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{id}");
        path.applyField("id", id);
        return ((RestCall<CompanyModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<CompanyModel>(){})).call();
    }
    
//endregion

//region Contacts
    /**
     * Create a new contact
     */
    public Future<ArrayList<ContactModel>> createContactsAsync(Integer companyId, ArrayList<ContactModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<ContactModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<ContactModel>>(){}));
    }

    public ArrayList<ContactModel> createContacts(Integer companyId, ArrayList<ContactModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<ContactModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<ContactModel>>(){})).call();
    }
    
    /**
     * Delete a single contact
     */
    public Future<ErrorResult> deleteContactAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){}));
    }

    public ErrorResult deleteContact(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){})).call();
    }
    
    /**
     * Retrieve a single contact
     */
    public Future<ContactModel> getContactAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ContactModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<ContactModel>(){}));
    }

    public ContactModel getContact(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ContactModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<ContactModel>(){})).call();
    }
    
    /**
     * Retrieve contacts for this company
     */
    public Future<FetchResult<ContactModel>> listContactsByCompanyAsync(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<ContactModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ContactModel>>(){}));
    }

    public FetchResult<ContactModel> listContactsByCompany(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<ContactModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ContactModel>>(){})).call();
    }
    
    /**
     * Retrieve all contacts
     */
    public Future<FetchResult<ContactModel>> queryContactsAsync(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/contacts");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<ContactModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ContactModel>>(){}));
    }

    public FetchResult<ContactModel> queryContacts(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/contacts");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<ContactModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ContactModel>>(){})).call();
    }
    
    /**
     * Update a single contact
     */
    public Future<ContactModel> updateContactAsync(Integer companyId, Integer id, ContactModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ContactModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<ContactModel>(){}));
    }

    public ContactModel updateContact(Integer companyId, Integer id, ContactModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/contacts/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ContactModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<ContactModel>(){})).call();
    }
    
//endregion

//region Definitions
    /**
     * Retrieve the full list of Avalara-supported nexus for a country and region.
     */
    public Future<FetchResult<NexusModel>> definitionsNexusByCountryByRegionGetAsync(String country, String region) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus/{country}/{region}");
        path.applyField("country", country);
        path.applyField("region", region);
        return this.threadPool.submit((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){}));
    }

    public FetchResult<NexusModel> definitionsNexusByCountryByRegionGet(String country, String region) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus/{country}/{region}");
        path.applyField("country", country);
        path.applyField("region", region);
        return ((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported nexus for a country.
     */
    public Future<FetchResult<NexusModel>> definitionsNexusByCountryGetAsync(String country) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus/{country}");
        path.applyField("country", country);
        return this.threadPool.submit((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){}));
    }

    public FetchResult<NexusModel> definitionsNexusByCountryGet(String country) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus/{country}");
        path.applyField("country", country);
        return ((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported nexus for all countries and regions.
     */
    public Future<FetchResult<NexusModel>> definitionsNexusGetAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus");
        return this.threadPool.submit((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){}));
    }

    public FetchResult<NexusModel> definitionsNexusGet() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus");
        return ((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){})).call();
    }
    
    /**
     * List all ISO 3166 countries
     */
    public Future<FetchResult<IsoCountryModel>> listCountriesAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/countries");
        return this.threadPool.submit((RestCall<FetchResult<IsoCountryModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<IsoCountryModel>>(){}));
    }

    public FetchResult<IsoCountryModel> listCountries() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/countries");
        return ((RestCall<FetchResult<IsoCountryModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<IsoCountryModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported entity use codes
     */
    public Future<FetchResult<EntityUseCodeModel>> listEntityUseCodesAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/entityusecodes");
        return this.threadPool.submit((RestCall<FetchResult<EntityUseCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<EntityUseCodeModel>>(){}));
    }

    public FetchResult<EntityUseCodeModel> listEntityUseCodes() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/entityusecodes");
        return ((RestCall<FetchResult<EntityUseCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<EntityUseCodeModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported filing frequencies.
     */
    public Future<FetchResult<FilingFrequencyModel>> listFilingFrequenciesAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/filingfrequencies");
        return this.threadPool.submit((RestCall<FetchResult<FilingFrequencyModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingFrequencyModel>>(){}));
    }

    public FetchResult<FilingFrequencyModel> listFilingFrequencies() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/filingfrequencies");
        return ((RestCall<FetchResult<FilingFrequencyModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingFrequencyModel>>(){})).call();
    }
    
    /**
     * Retrieve the list of questions that are required for a tax location
     */
    public Future<FetchResult<LocationQuestionModel>> listLocationQuestionsByAddressAsync(String line1, String line2, String line3, String city, String region, String postalCode, String country, BigDecimal latitude, BigDecimal longitude) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/locationquestions");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        path.addQuery("latitude", latitude);
        path.addQuery("longitude", longitude);
        return this.threadPool.submit((RestCall<FetchResult<LocationQuestionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<LocationQuestionModel>>(){}));
    }

    public FetchResult<LocationQuestionModel> listLocationQuestionsByAddress(String line1, String line2, String line3, String city, String region, String postalCode, String country, BigDecimal latitude, BigDecimal longitude) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/locationquestions");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        path.addQuery("latitude", latitude);
        path.addQuery("longitude", longitude);
        return ((RestCall<FetchResult<LocationQuestionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<LocationQuestionModel>>(){})).call();
    }
    
    /**
     * List all nexus that apply to a specific address.
     */
    public Future<FetchResult<NexusModel>> listNexusByAddressAsync(String line1, String line2, String line3, String city, String region, String postalCode, String country) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus/byaddress");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        return this.threadPool.submit((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){}));
    }

    public FetchResult<NexusModel> listNexusByAddress(String line1, String line2, String line3, String city, String region, String postalCode, String country) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/nexus/byaddress");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        return ((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported tax notice customer funding options.
     */
    public Future<FetchResult<NoticeCustomerFundingOptionModel>> listNoticeCustomerFundingOptionsAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticecustomerfundingoptions");
        return this.threadPool.submit((RestCall<FetchResult<NoticeCustomerFundingOptionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeCustomerFundingOptionModel>>(){}));
    }

    public FetchResult<NoticeCustomerFundingOptionModel> listNoticeCustomerFundingOptions() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticecustomerfundingoptions");
        return ((RestCall<FetchResult<NoticeCustomerFundingOptionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeCustomerFundingOptionModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported tax notice customer types.
     */
    public Future<FetchResult<NoticeCustomerTypeModel>> listNoticeCustomerTypesAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticecustomertypes");
        return this.threadPool.submit((RestCall<FetchResult<NoticeCustomerTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeCustomerTypeModel>>(){}));
    }

    public FetchResult<NoticeCustomerTypeModel> listNoticeCustomerTypes() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticecustomertypes");
        return ((RestCall<FetchResult<NoticeCustomerTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeCustomerTypeModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported tax notice filing types.
     */
    public Future<FetchResult<NoticeFilingTypeModel>> listNoticeFilingtypesAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticefilingtypes");
        return this.threadPool.submit((RestCall<FetchResult<NoticeFilingTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeFilingTypeModel>>(){}));
    }

    public FetchResult<NoticeFilingTypeModel> listNoticeFilingtypes() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticefilingtypes");
        return ((RestCall<FetchResult<NoticeFilingTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeFilingTypeModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported tax notice priorities.
     */
    public Future<FetchResult<NoticePriorityModel>> listNoticePrioritiesAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticepriorities");
        return this.threadPool.submit((RestCall<FetchResult<NoticePriorityModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticePriorityModel>>(){}));
    }

    public FetchResult<NoticePriorityModel> listNoticePriorities() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticepriorities");
        return ((RestCall<FetchResult<NoticePriorityModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticePriorityModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported tax notice reasons.
     */
    public Future<FetchResult<NoticeReasonModel>> listNoticeReasonsAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticereasons");
        return this.threadPool.submit((RestCall<FetchResult<NoticeReasonModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeReasonModel>>(){}));
    }

    public FetchResult<NoticeReasonModel> listNoticeReasons() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticereasons");
        return ((RestCall<FetchResult<NoticeReasonModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeReasonModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported tax notice statuses.
     */
    public Future<FetchResult<NoticeStatusModel>> listNoticeStatusesAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticestatuses");
        return this.threadPool.submit((RestCall<FetchResult<NoticeStatusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeStatusModel>>(){}));
    }

    public FetchResult<NoticeStatusModel> listNoticeStatuses() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticestatuses");
        return ((RestCall<FetchResult<NoticeStatusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeStatusModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported tax notice types.
     */
    public Future<FetchResult<NoticeTypeModel>> listNoticeTypesAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticetypes");
        return this.threadPool.submit((RestCall<FetchResult<NoticeTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeTypeModel>>(){}));
    }

    public FetchResult<NoticeTypeModel> listNoticeTypes() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/noticetypes");
        return ((RestCall<FetchResult<NoticeTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeTypeModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported extra parameters for creating transactions.
     */
    public Future<FetchResult<ParameterModel>> listParametersAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/parameters");
        return this.threadPool.submit((RestCall<FetchResult<ParameterModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ParameterModel>>(){}));
    }

    public FetchResult<ParameterModel> listParameters() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/parameters");
        return ((RestCall<FetchResult<ParameterModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ParameterModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported permissions
     */
    public Future<FetchResult<String>> listPermissionsAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/permissions");
        return this.threadPool.submit((RestCall<FetchResult<String>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<String>>(){}));
    }

    public FetchResult<String> listPermissions() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/permissions");
        return ((RestCall<FetchResult<String>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<String>>(){})).call();
    }
    
    /**
     * List all ISO 3166 regions
     */
    public Future<FetchResult<IsoRegionModel>> listRegionsAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/regions");
        return this.threadPool.submit((RestCall<FetchResult<IsoRegionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<IsoRegionModel>>(){}));
    }

    public FetchResult<IsoRegionModel> listRegions() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/regions");
        return ((RestCall<FetchResult<IsoRegionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<IsoRegionModel>>(){})).call();
    }
    
    /**
     * List all ISO 3166 regions for a country
     */
    public Future<FetchResult<IsoRegionModel>> listRegionsByCountryAsync(String country) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/countries/{country}/regions");
        path.applyField("country", country);
        return this.threadPool.submit((RestCall<FetchResult<IsoRegionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<IsoRegionModel>>(){}));
    }

    public FetchResult<IsoRegionModel> listRegionsByCountry(String country) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/countries/{country}/regions");
        path.applyField("country", country);
        return ((RestCall<FetchResult<IsoRegionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<IsoRegionModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported permissions
     */
    public Future<FetchResult<SecurityRoleModel>> listSecurityRolesAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/securityroles");
        return this.threadPool.submit((RestCall<FetchResult<SecurityRoleModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SecurityRoleModel>>(){}));
    }

    public FetchResult<SecurityRoleModel> listSecurityRoles() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/securityroles");
        return ((RestCall<FetchResult<SecurityRoleModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SecurityRoleModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported subscription types
     */
    public Future<FetchResult<SubscriptionTypeModel>> listSubscriptionTypesAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/subscriptiontypes");
        return this.threadPool.submit((RestCall<FetchResult<SubscriptionTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SubscriptionTypeModel>>(){}));
    }

    public FetchResult<SubscriptionTypeModel> listSubscriptionTypes() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/subscriptiontypes");
        return ((RestCall<FetchResult<SubscriptionTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SubscriptionTypeModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported tax authorities.
     */
    public Future<FetchResult<TaxAuthorityModel>> listTaxAuthoritiesAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxauthorities");
        return this.threadPool.submit((RestCall<FetchResult<TaxAuthorityModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxAuthorityModel>>(){}));
    }

    public FetchResult<TaxAuthorityModel> listTaxAuthorities() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxauthorities");
        return ((RestCall<FetchResult<TaxAuthorityModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxAuthorityModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported forms for each tax authority.
     */
    public Future<FetchResult<TaxAuthorityFormModel>> listTaxAuthorityFormsAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxauthorityforms");
        return this.threadPool.submit((RestCall<FetchResult<TaxAuthorityFormModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxAuthorityFormModel>>(){}));
    }

    public FetchResult<TaxAuthorityFormModel> listTaxAuthorityForms() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxauthorityforms");
        return ((RestCall<FetchResult<TaxAuthorityFormModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxAuthorityFormModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported tax authority types.
     */
    public Future<FetchResult<TaxAuthorityTypeModel>> listTaxAuthorityTypesAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxauthoritytypes");
        return this.threadPool.submit((RestCall<FetchResult<TaxAuthorityTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxAuthorityTypeModel>>(){}));
    }

    public FetchResult<TaxAuthorityTypeModel> listTaxAuthorityTypes() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxauthoritytypes");
        return ((RestCall<FetchResult<TaxAuthorityTypeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxAuthorityTypeModel>>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported tax code types.
     */
    public Future<TaxCodeTypesModel> listTaxCodeTypesAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxcodetypes");
        return this.threadPool.submit((RestCall<TaxCodeTypesModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TaxCodeTypesModel>(){}));
    }

    public TaxCodeTypesModel listTaxCodeTypes() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxcodetypes");
        return ((RestCall<TaxCodeTypesModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TaxCodeTypesModel>(){})).call();
    }
    
    /**
     * Retrieve the full list of Avalara-supported tax codes.
     */
    public Future<FetchResult<TaxCodeModel>> listTaxCodesAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxcodes");
        return this.threadPool.submit((RestCall<FetchResult<TaxCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxCodeModel>>(){}));
    }

    public FetchResult<TaxCodeModel> listTaxCodes() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/definitions/taxcodes");
        return ((RestCall<FetchResult<TaxCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxCodeModel>>(){})).call();
    }
    
//endregion

//region FilingCalendars
    /**
     * Retrieve a single filing calendar
     */
    public Future<FilingCalendarModel> companiesByCompanyIdFilingcalendarsByIdGetAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingCalendarModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingCalendarModel>(){}));
    }

    public FilingCalendarModel companiesByCompanyIdFilingcalendarsByIdGet(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingCalendarModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingCalendarModel>(){})).call();
    }
    
    /**
     * Retrieve all filing calendars for this company
     */
    public Future<FetchResult<FilingCalendarModel>> companiesByCompanyIdFilingcalendarsGetAsync(Integer companyId, String $filter, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<FilingCalendarModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingCalendarModel>>(){}));
    }

    public FetchResult<FilingCalendarModel> companiesByCompanyIdFilingcalendarsGet(Integer companyId, String $filter, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<FilingCalendarModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingCalendarModel>>(){})).call();
    }
    
    /**
     * Retrieve all filing requests for this company
     */
    public Future<FetchResult<FilingRequestModel>> companiesByCompanyIdFilingrequestsGetAsync(Integer companyId, String $filter, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<FilingRequestModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingRequestModel>>(){}));
    }

    public FetchResult<FilingRequestModel> companiesByCompanyIdFilingrequestsGet(Integer companyId, String $filter, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<FilingRequestModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingRequestModel>>(){})).call();
    }
    
    /**
     * Returns a list of options for adding the specified form.
     */
    public Future<ArrayList<CycleAddOptionModel>> cycleSafeAddAsync(Integer companyId, String formCode) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/add/options");
        path.applyField("companyId", companyId);
        path.addQuery("formCode", formCode);
        return this.threadPool.submit((RestCall<ArrayList<CycleAddOptionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<CycleAddOptionModel>>(){}));
    }

    public ArrayList<CycleAddOptionModel> cycleSafeAdd(Integer companyId, String formCode) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/add/options");
        path.applyField("companyId", companyId);
        path.addQuery("formCode", formCode);
        return ((RestCall<ArrayList<CycleAddOptionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<CycleAddOptionModel>>(){})).call();
    }
    
    /**
     * Indicates when changes are allowed to be made to a filing calendar.
     */
    public Future<CycleEditOptionModel> cycleSafeEditAsync(Integer companyId, Integer id, ArrayList<FilingCalendarEditModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}/edit/options");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<CycleEditOptionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<CycleEditOptionModel>(){}));
    }

    public CycleEditOptionModel cycleSafeEdit(Integer companyId, Integer id, ArrayList<FilingCalendarEditModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}/edit/options");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<CycleEditOptionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<CycleEditOptionModel>(){})).call();
    }
    
    /**
     * Returns a list of options for expiring a filing calendar
     */
    public Future<CycleExpireModel> cycleSafeExpirationAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}/cancel/options");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<CycleExpireModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<CycleExpireModel>(){}));
    }

    public CycleExpireModel cycleSafeExpiration(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}/cancel/options");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<CycleExpireModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<CycleExpireModel>(){})).call();
    }
    
    /**
     * Create a new filing request to create a filing calendar
     */
    public Future<FilingRequestModel> filingRequestsAddAsync(Integer companyId, FilingRequestModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/add/request");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FilingRequestModel>(){}));
    }

    public FilingRequestModel filingRequestsAdd(Integer companyId, FilingRequestModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/add/request");
        path.applyField("companyId", companyId);
        return ((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FilingRequestModel>(){})).call();
    }
    
    /**
     * Approve existing Filing Request
     */
    public Future<FilingRequestModel> filingRequestsApproveAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests/{id}/approve");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, null, new TypeToken<FilingRequestModel>(){}));
    }

    public FilingRequestModel filingRequestsApprove(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests/{id}/approve");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, null, new TypeToken<FilingRequestModel>(){})).call();
    }
    
    /**
     * Cancel existing Filing Request
     */
    public Future<FilingRequestModel> filingRequestsCancelAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests/{id}/cancel");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, null, new TypeToken<FilingRequestModel>(){}));
    }

    public FilingRequestModel filingRequestsCancel(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests/{id}/cancel");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, null, new TypeToken<FilingRequestModel>(){})).call();
    }
    
    /**
     * Create a new filing request to cancel a filing calendar
     */
    public Future<FilingRequestModel> filingRequestsNewCancelAsync(Integer companyId, Integer id, FilingRequestModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}/cancel/request");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FilingRequestModel>(){}));
    }

    public FilingRequestModel filingRequestsNewCancel(Integer companyId, Integer id, FilingRequestModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}/cancel/request");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FilingRequestModel>(){})).call();
    }
    
    /**
     * Create a new filing request to edit a filing calendar
     */
    public Future<FilingRequestModel> filingRequestsNewEditAsync(Integer companyId, Integer id, FilingRequestModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}/edit/request");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FilingRequestModel>(){}));
    }

    public FilingRequestModel filingRequestsNewEdit(Integer companyId, Integer id, FilingRequestModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingcalendars/{id}/edit/request");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingRequestModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<FilingRequestModel>(){})).call();
    }
    
    /**
     * Edit existing Filing Request
     */
    public Future<FilingRequestModel> filingRequestsUpdateAsync(Integer companyId, Integer id, FilingRequestModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingRequestModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<FilingRequestModel>(){}));
    }

    public FilingRequestModel filingRequestsUpdate(Integer companyId, Integer id, FilingRequestModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingRequestModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<FilingRequestModel>(){})).call();
    }
    
    /**
     * Retrieve all filing calendars
     */
    public Future<FetchResult<FilingCalendarModel>> queryFilingCalendarsAsync(String $filter, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingcalendars");
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<FilingCalendarModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingCalendarModel>>(){}));
    }

    public FetchResult<FilingCalendarModel> queryFilingCalendars(String $filter, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingcalendars");
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<FilingCalendarModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingCalendarModel>>(){})).call();
    }
    
    /**
     * Retrieve all filing requests
     */
    public Future<FetchResult<FilingRequestModel>> queryFilingRequestsAsync(String $filter, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingrequests");
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<FilingRequestModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingRequestModel>>(){}));
    }

    public FetchResult<FilingRequestModel> queryFilingRequests(String $filter, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingrequests");
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<FilingRequestModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<FilingRequestModel>>(){})).call();
    }
    
    /**
     * Retrieve a single filing request
     */
    public Future<FilingRequestModel> filingRequestsAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingRequestModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingRequestModel>(){}));
    }

    public FilingRequestModel filingRequests(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filingrequests/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingRequestModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingRequestModel>(){})).call();
    }
    
    /**
     * Gets the request status and Login Result
     */
    public Future<LoginVerificationOutputModel> loginVerificationGetAsync(Integer jobId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingcalendars/credentials/{jobId}");
        path.applyField("jobId", jobId);
        return this.threadPool.submit((RestCall<LoginVerificationOutputModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<LoginVerificationOutputModel>(){}));
    }

    public LoginVerificationOutputModel loginVerificationGet(Integer jobId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingcalendars/credentials/{jobId}");
        path.applyField("jobId", jobId);
        return ((RestCall<LoginVerificationOutputModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<LoginVerificationOutputModel>(){})).call();
    }
    
    /**
     * New request for getting for validating customer's login credentials
     */
    public Future<LoginVerificationOutputModel> loginVerificationPostAsync(LoginVerificationInputModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingcalendars/credentials/verify");
        return this.threadPool.submit((RestCall<LoginVerificationOutputModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<LoginVerificationOutputModel>(){}));
    }

    public LoginVerificationOutputModel loginVerificationPost(LoginVerificationInputModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/filingcalendars/credentials/verify");
        return ((RestCall<LoginVerificationOutputModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<LoginVerificationOutputModel>(){})).call();
    }
    
//endregion

//region Filings
    /**
     * Approve all filings for the specified company in the given filing period.
     */
    public Future<ArrayList<FilingModel>> approveFilingsAsync(Integer companyId, Integer year, Integer month, ApproveFilingsModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/approve");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return this.threadPool.submit((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){}));
    }

    public ArrayList<FilingModel> approveFilings(Integer companyId, Integer year, Integer month, ApproveFilingsModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/approve");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return ((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){})).call();
    }
    
    /**
     * Approve all filings for the specified company in the given filing period and country.
     */
    public Future<ArrayList<FilingModel>> approveFilingsCountryAsync(Integer companyId, Integer year, Integer month, String country, ApproveFilingsModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/approve");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        return this.threadPool.submit((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){}));
    }

    public ArrayList<FilingModel> approveFilingsCountry(Integer companyId, Integer year, Integer month, String country, ApproveFilingsModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/approve");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        return ((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){})).call();
    }
    
    /**
     * Approve all filings for the specified company in the given filing period, country and region.
     */
    public Future<ArrayList<FilingModel>> approveFilingsCountryRegionAsync(Integer companyId, Integer year, Integer month, String country, String region, ApproveFilingsModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/approve");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        return this.threadPool.submit((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){}));
    }

    public ArrayList<FilingModel> approveFilingsCountryRegion(Integer companyId, Integer year, Integer month, String country, String region, ApproveFilingsModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/approve");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        return ((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){})).call();
    }
    
    /**
     * Add an adjustment to a given filing.
     */
    public Future<ArrayList<FilingAdjustmentModel>> createReturnAdjustmentAsync(Integer companyId, Integer year, Integer month, String country, String region, String formCode, ArrayList<FilingAdjustmentModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/{formCode}/adjust");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        path.applyField("formCode", formCode);
        return this.threadPool.submit((RestCall<ArrayList<FilingAdjustmentModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingAdjustmentModel>>(){}));
    }

    public ArrayList<FilingAdjustmentModel> createReturnAdjustment(Integer companyId, Integer year, Integer month, String country, String region, String formCode, ArrayList<FilingAdjustmentModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/{formCode}/adjust");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        path.applyField("formCode", formCode);
        return ((RestCall<ArrayList<FilingAdjustmentModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingAdjustmentModel>>(){})).call();
    }
    
    /**
     * Add an augmentation for a given filing.
     */
    public Future<ArrayList<FilingAugmentationModel>> createReturnAugmentationAsync(Integer companyId, Integer year, Integer month, String country, String region, String formCode, ArrayList<FilingAugmentationModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/{formCode}/augment");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        path.applyField("formCode", formCode);
        return this.threadPool.submit((RestCall<ArrayList<FilingAugmentationModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingAugmentationModel>>(){}));
    }

    public ArrayList<FilingAugmentationModel> createReturnAugmentation(Integer companyId, Integer year, Integer month, String country, String region, String formCode, ArrayList<FilingAugmentationModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/{formCode}/augment");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        path.applyField("formCode", formCode);
        return ((RestCall<ArrayList<FilingAugmentationModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingAugmentationModel>>(){})).call();
    }
    
    /**
     * Delete an adjustment for a given filing.
     */
    public Future<ErrorResult> deleteReturnAdjustmentAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/adjust/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){}));
    }

    public ErrorResult deleteReturnAdjustment(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/adjust/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){})).call();
    }
    
    /**
     * Delete an augmentation for a given filing.
     */
    public Future<ErrorResult> deleteReturnAugmentationAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/augment/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){}));
    }

    public ErrorResult deleteReturnAugmentation(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/augment/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){})).call();
    }
    
    /**
     * Retrieve worksheet checkup report for company and filing period.
     */
    public Future<FilingsCheckupModel> filingsCheckupReportAsync(Integer worksheetId, Integer companyId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{worksheetid}/checkup");
        path.applyField("worksheetId", worksheetId);
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<FilingsCheckupModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingsCheckupModel>(){}));
    }

    public FilingsCheckupModel filingsCheckupReport(Integer worksheetId, Integer companyId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{worksheetid}/checkup");
        path.applyField("worksheetId", worksheetId);
        path.applyField("companyId", companyId);
        return ((RestCall<FilingsCheckupModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingsCheckupModel>(){})).call();
    }
    
    /**
     * Retrieve worksheet checkup report for company and filing period.
     */
    public Future<FilingsCheckupModel> filingsCheckupReportsAsync(Integer companyId, Integer year, Integer month) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/checkup");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return this.threadPool.submit((RestCall<FilingsCheckupModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingsCheckupModel>(){}));
    }

    public FilingsCheckupModel filingsCheckupReports(Integer companyId, Integer year, Integer month) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/checkup");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return ((RestCall<FilingsCheckupModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FilingsCheckupModel>(){})).call();
    }
    
    /**
     * Retrieve a list of filings for the specified company in the year and month of a given filing period.
     */
    public Future<ArrayList<FilingModel>> getFilingsAsync(Integer companyId, Integer year, Integer month) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return this.threadPool.submit((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<FilingModel>>(){}));
    }

    public ArrayList<FilingModel> getFilings(Integer companyId, Integer year, Integer month) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return ((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<FilingModel>>(){})).call();
    }
    
    /**
     * Retrieve a list of filings for the specified company in the given filing period and country.
     */
    public Future<ArrayList<FilingModel>> getFilingsByCountryAsync(Integer companyId, Integer year, Integer month, String country) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        return this.threadPool.submit((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<FilingModel>>(){}));
    }

    public ArrayList<FilingModel> getFilingsByCountry(Integer companyId, Integer year, Integer month, String country) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        return ((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<FilingModel>>(){})).call();
    }
    
    /**
     * Retrieve a list of filings for the specified company in the filing period, country and region.
     */
    public Future<ArrayList<FilingModel>> getFilingsByCountryRegionAsync(Integer companyId, Integer year, Integer month, String country, String region) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        return this.threadPool.submit((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<FilingModel>>(){}));
    }

    public ArrayList<FilingModel> getFilingsByCountryRegion(Integer companyId, Integer year, Integer month, String country, String region) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        return ((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<FilingModel>>(){})).call();
    }
    
    /**
     * Retrieve a list of filings for the specified company in the given filing period, country, region and form.
     */
    public Future<ArrayList<FilingModel>> getFilingsByReturnNameAsync(Integer companyId, Integer year, Integer month, String country, String region, String formCode) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/{formCode}");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        path.applyField("formCode", formCode);
        return this.threadPool.submit((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<FilingModel>>(){}));
    }

    public ArrayList<FilingModel> getFilingsByReturnName(Integer companyId, Integer year, Integer month, String country, String region, String formCode) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/{formCode}");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        path.applyField("formCode", formCode);
        return ((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<ArrayList<FilingModel>>(){})).call();
    }
    
    /**
     * Rebuild a set of filings for the specified company in the given filing period.
     */
    public Future<ArrayList<FilingModel>> rebuildFilingsAsync(Integer companyId, Integer year, Integer month, RebuildFilingsModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/rebuild");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return this.threadPool.submit((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){}));
    }

    public ArrayList<FilingModel> rebuildFilings(Integer companyId, Integer year, Integer month, RebuildFilingsModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/rebuild");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        return ((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){})).call();
    }
    
    /**
     * Rebuild a set of filings for the specified company in the given filing period and country.
     */
    public Future<ArrayList<FilingModel>> rebuildFilingsByCountryAsync(Integer companyId, Integer year, Integer month, String country, RebuildFilingsModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/rebuild");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        return this.threadPool.submit((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){}));
    }

    public ArrayList<FilingModel> rebuildFilingsByCountry(Integer companyId, Integer year, Integer month, String country, RebuildFilingsModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/rebuild");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        return ((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){})).call();
    }
    
    /**
     * Rebuild a set of filings for the specified company in the given filing period, country and region.
     */
    public Future<ArrayList<FilingModel>> rebuildFilingsByCountryRegionAsync(Integer companyId, Integer year, Integer month, String country, String region, RebuildFilingsModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/rebuild");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        return this.threadPool.submit((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){}));
    }

    public ArrayList<FilingModel> rebuildFilingsByCountryRegion(Integer companyId, Integer year, Integer month, String country, String region, RebuildFilingsModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/{year}/{month}/{country}/{region}/rebuild");
        path.applyField("companyId", companyId);
        path.applyField("year", year);
        path.applyField("month", month);
        path.applyField("country", country);
        path.applyField("region", region);
        return ((RestCall<ArrayList<FilingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<FilingModel>>(){})).call();
    }
    
    /**
     * Edit an adjustment for a given filing.
     */
    public Future<FilingAdjustmentModel> updateReturnAdjustmentAsync(Integer companyId, Long id, FilingAdjustmentModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/adjust/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingAdjustmentModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<FilingAdjustmentModel>(){}));
    }

    public FilingAdjustmentModel updateReturnAdjustment(Integer companyId, Long id, FilingAdjustmentModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/adjust/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingAdjustmentModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<FilingAdjustmentModel>(){})).call();
    }
    
    /**
     * Edit an augmentation for a given filing.
     */
    public Future<FilingAugmentationModel> updateReturnAugmentationAsync(Integer companyId, Long id, FilingAugmentationModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/augment/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FilingAugmentationModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<FilingAugmentationModel>(){}));
    }

    public FilingAugmentationModel updateReturnAugmentation(Integer companyId, Long id, FilingAugmentationModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/filings/augment/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<FilingAugmentationModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<FilingAugmentationModel>(){})).call();
    }
    
//endregion

//region FundingRequests
    /**
     * Request the javascript for a funding setup widget
     */
    public Future<FundingStatusModel> activateFundingRequestAsync(Long id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/fundingrequests/{id}/widget");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FundingStatusModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FundingStatusModel>(){}));
    }

    public FundingStatusModel activateFundingRequest(Long id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/fundingrequests/{id}/widget");
        path.applyField("id", id);
        return ((RestCall<FundingStatusModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FundingStatusModel>(){})).call();
    }
    
    /**
     * Retrieve status about a funding setup request
     */
    public Future<FundingStatusModel> fundingRequestStatusAsync(Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/fundingrequests/{id}");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<FundingStatusModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FundingStatusModel>(){}));
    }

    public FundingStatusModel fundingRequestStatus(Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/fundingrequests/{id}");
        path.applyField("id", id);
        return ((RestCall<FundingStatusModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<FundingStatusModel>(){})).call();
    }
    
//endregion

//region Items
    /**
     * Create a new item
     */
    public Future<ArrayList<ItemModel>> createItemsAsync(Integer companyId, ArrayList<ItemModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<ItemModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<ItemModel>>(){}));
    }

    public ArrayList<ItemModel> createItems(Integer companyId, ArrayList<ItemModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<ItemModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<ItemModel>>(){})).call();
    }
    
    /**
     * Delete a single item
     */
    public Future<ErrorResult> deleteItemAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){}));
    }

    public ErrorResult deleteItem(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){})).call();
    }
    
    /**
     * Retrieve a single item
     */
    public Future<ItemModel> getItemAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ItemModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<ItemModel>(){}));
    }

    public ItemModel getItem(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ItemModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<ItemModel>(){})).call();
    }
    
    /**
     * Retrieve items for this company
     */
    public Future<FetchResult<ItemModel>> listItemsByCompanyAsync(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<ItemModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ItemModel>>(){}));
    }

    public FetchResult<ItemModel> listItemsByCompany(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<ItemModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ItemModel>>(){})).call();
    }
    
    /**
     * Retrieve all items
     */
    public Future<FetchResult<ItemModel>> queryItemsAsync(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/items");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<ItemModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ItemModel>>(){}));
    }

    public FetchResult<ItemModel> queryItems(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/items");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<ItemModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<ItemModel>>(){})).call();
    }
    
    /**
     * Update a single item
     */
    public Future<ItemModel> updateItemAsync(Integer companyId, Integer id, ItemModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ItemModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<ItemModel>(){}));
    }

    public ItemModel updateItem(Integer companyId, Integer id, ItemModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/items/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ItemModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<ItemModel>(){})).call();
    }
    
//endregion

//region Locations
    /**
     * Point of sale data file generation
     */
    public Future<String> buildPointOfSaleDataForLocationAsync(Integer companyId, Integer id, Date date, String format, Integer partnerId, Boolean includeJurisCodes) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}/pointofsaledata");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.addQuery("date", date);
        path.addQuery("format", format);
        path.addQuery("partnerId", partnerId);
        path.addQuery("includeJurisCodes", includeJurisCodes);
        return this.threadPool.submit((RestCall<String>)restCallFactory.createRestCall("get", path, null, new TypeToken<String>(){}));
    }

    public String buildPointOfSaleDataForLocation(Integer companyId, Integer id, Date date, String format, Integer partnerId, Boolean includeJurisCodes) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}/pointofsaledata");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        path.addQuery("date", date);
        path.addQuery("format", format);
        path.addQuery("partnerId", partnerId);
        path.addQuery("includeJurisCodes", includeJurisCodes);
        return ((RestCall<String>)restCallFactory.createRestCall("get", path, null, new TypeToken<String>(){})).call();
    }
    
    /**
     * Create a new location
     */
    public Future<ArrayList<LocationModel>> createLocationsAsync(Integer companyId, ArrayList<LocationModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<LocationModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<LocationModel>>(){}));
    }

    public ArrayList<LocationModel> createLocations(Integer companyId, ArrayList<LocationModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<LocationModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<LocationModel>>(){})).call();
    }
    
    /**
     * Delete a single location
     */
    public Future<ErrorResult> deleteLocationAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){}));
    }

    public ErrorResult deleteLocation(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){})).call();
    }
    
    /**
     * Retrieve a single location
     */
    public Future<LocationModel> getLocationAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<LocationModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<LocationModel>(){}));
    }

    public LocationModel getLocation(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<LocationModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<LocationModel>(){})).call();
    }
    
    /**
     * Retrieve locations for this company
     */
    public Future<FetchResult<LocationModel>> listLocationsByCompanyAsync(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<LocationModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<LocationModel>>(){}));
    }

    public FetchResult<LocationModel> listLocationsByCompany(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<LocationModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<LocationModel>>(){})).call();
    }
    
    /**
     * Retrieve all locations
     */
    public Future<FetchResult<LocationModel>> queryLocationsAsync(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/locations");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<LocationModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<LocationModel>>(){}));
    }

    public FetchResult<LocationModel> queryLocations(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/locations");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<LocationModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<LocationModel>>(){})).call();
    }
    
    /**
     * Update a single location
     */
    public Future<LocationModel> updateLocationAsync(Integer companyId, Integer id, LocationModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<LocationModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<LocationModel>(){}));
    }

    public LocationModel updateLocation(Integer companyId, Integer id, LocationModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<LocationModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<LocationModel>(){})).call();
    }
    
    /**
     * Validate the location against local requirements
     */
    public Future<LocationValidationModel> validateLocationAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}/validate");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<LocationValidationModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<LocationValidationModel>(){}));
    }

    public LocationValidationModel validateLocation(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/locations/{id}/validate");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<LocationValidationModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<LocationValidationModel>(){})).call();
    }
    
//endregion

//region Nexus
    /**
     * Create a new nexus
     */
    public Future<ArrayList<NexusModel>> createNexusAsync(Integer companyId, ArrayList<NexusModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<NexusModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NexusModel>>(){}));
    }

    public ArrayList<NexusModel> createNexus(Integer companyId, ArrayList<NexusModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<NexusModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NexusModel>>(){})).call();
    }
    
    /**
     * Delete a single nexus
     */
    public Future<ErrorResult> deleteNexusAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){}));
    }

    public ErrorResult deleteNexus(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){})).call();
    }
    
    /**
     * Retrieve a single nexus
     */
    public Future<NexusModel> getNexusAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<NexusModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<NexusModel>(){}));
    }

    public NexusModel getNexus(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<NexusModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<NexusModel>(){})).call();
    }
    
    /**
     * Retrieve nexus for this company
     */
    public Future<FetchResult<NexusModel>> listNexusByCompanyAsync(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){}));
    }

    public FetchResult<NexusModel> listNexusByCompany(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){})).call();
    }
    
    /**
     * Retrieve all nexus
     */
    public Future<FetchResult<NexusModel>> queryNexusAsync(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/nexus");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){}));
    }

    public FetchResult<NexusModel> queryNexus(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/nexus");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<NexusModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NexusModel>>(){})).call();
    }
    
    /**
     * Update a single nexus
     */
    public Future<NexusModel> updateNexusAsync(Integer companyId, Integer id, NexusModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<NexusModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<NexusModel>(){}));
    }

    public NexusModel updateNexus(Integer companyId, Integer id, NexusModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/nexus/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<NexusModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<NexusModel>(){})).call();
    }
    
//endregion

//region Notices
    /**
     * Create a new notice comment.
     */
    public Future<ArrayList<NoticeCommentModel>> createNoticeCommentAsync(Integer companyId, Integer id, ArrayList<NoticeCommentModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/comments");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<NoticeCommentModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NoticeCommentModel>>(){}));
    }

    public ArrayList<NoticeCommentModel> createNoticeComment(Integer companyId, Integer id, ArrayList<NoticeCommentModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/comments");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<NoticeCommentModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NoticeCommentModel>>(){})).call();
    }
    
    /**
     * Create a new notice finance details.
     */
    public Future<ArrayList<NoticeFinanceModel>> createNoticeFinanceDetailsAsync(Integer companyId, Integer id, ArrayList<NoticeFinanceModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/financedetails");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ArrayList<NoticeFinanceModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NoticeFinanceModel>>(){}));
    }

    public ArrayList<NoticeFinanceModel> createNoticeFinanceDetails(Integer companyId, Integer id, ArrayList<NoticeFinanceModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/financedetails");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ArrayList<NoticeFinanceModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NoticeFinanceModel>>(){})).call();
    }
    
    /**
     * Create a new notice.
     */
    public Future<ArrayList<NoticeModel>> createNoticesAsync(Integer companyId, ArrayList<NoticeModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<NoticeModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NoticeModel>>(){}));
    }

    public ArrayList<NoticeModel> createNotices(Integer companyId, ArrayList<NoticeModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<NoticeModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<NoticeModel>>(){})).call();
    }
    
    /**
     * Delete a single notice.
     */
    public Future<ErrorResult> deleteNoticeAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){}));
    }

    public ErrorResult deleteNotice(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){})).call();
    }
    
    /**
     * Retrieve a single notice.
     */
    public Future<NoticeModel> getNoticeAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<NoticeModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<NoticeModel>(){}));
    }

    public NoticeModel getNotice(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<NoticeModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<NoticeModel>(){})).call();
    }
    
    /**
     * Retrieve notice comments for a specific notice.
     */
    public Future<FetchResult<NoticeCommentModel>> getNoticeCommentsAsync(Integer id, Integer companyId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/comments");
        path.applyField("id", id);
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<FetchResult<NoticeCommentModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeCommentModel>>(){}));
    }

    public FetchResult<NoticeCommentModel> getNoticeComments(Integer id, Integer companyId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/comments");
        path.applyField("id", id);
        path.applyField("companyId", companyId);
        return ((RestCall<FetchResult<NoticeCommentModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeCommentModel>>(){})).call();
    }
    
    /**
     * Retrieve notice finance details for a specific notice.
     */
    public Future<FetchResult<NoticeFinanceModel>> getNoticeFinanceDetailsAsync(Integer id, Integer companyId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/financedetails");
        path.applyField("id", id);
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<FetchResult<NoticeFinanceModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeFinanceModel>>(){}));
    }

    public FetchResult<NoticeFinanceModel> getNoticeFinanceDetails(Integer id, Integer companyId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}/financedetails");
        path.applyField("id", id);
        path.applyField("companyId", companyId);
        return ((RestCall<FetchResult<NoticeFinanceModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeFinanceModel>>(){})).call();
    }
    
    /**
     * Retrieve notices for a company.
     */
    public Future<FetchResult<NoticeModel>> listNoticesByCompanyAsync(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NoticeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeModel>>(){}));
    }

    public FetchResult<NoticeModel> listNoticesByCompany(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<NoticeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeModel>>(){})).call();
    }
    
    /**
     * Retrieve all notices.
     */
    public Future<FetchResult<NoticeModel>> queryNoticesAsync(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/notices");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<NoticeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeModel>>(){}));
    }

    public FetchResult<NoticeModel> queryNotices(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/notices");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<NoticeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<NoticeModel>>(){})).call();
    }
    
    /**
     * Update a single notice.
     */
    public Future<NoticeModel> updateNoticeAsync(Integer companyId, Integer id, NoticeModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<NoticeModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<NoticeModel>(){}));
    }

    public NoticeModel updateNotice(Integer companyId, Integer id, NoticeModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/notices/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<NoticeModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<NoticeModel>(){})).call();
    }
    
//endregion

//region PointOfSale
    /**
     * Point of sale data file generation
     */
    public Future<String> buildPointOfSaleDataFileAsync(PointOfSaleDataRequestModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/pointofsaledata/build");
        return this.threadPool.submit((RestCall<String>)restCallFactory.createRestCall("post", path, model, new TypeToken<String>(){}));
    }

    public String buildPointOfSaleDataFile(PointOfSaleDataRequestModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/pointofsaledata/build");
        return ((RestCall<String>)restCallFactory.createRestCall("post", path, model, new TypeToken<String>(){})).call();
    }
    
//endregion

//region Registrar
    /**
     * Change Password
     */
    public Future<String> changePasswordAsync(PasswordChangeModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/passwords");
        return this.threadPool.submit((RestCall<String>)restCallFactory.createRestCall("put", path, model, new TypeToken<String>(){}));
    }

    public String changePassword(PasswordChangeModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/passwords");
        return ((RestCall<String>)restCallFactory.createRestCall("put", path, model, new TypeToken<String>(){})).call();
    }
    
    /**
     * Create a new account
     */
    public Future<AccountModel> createAccountAsync(AccountModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts");
        return this.threadPool.submit((RestCall<AccountModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<AccountModel>(){}));
    }

    public AccountModel createAccount(AccountModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts");
        return ((RestCall<AccountModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<AccountModel>(){})).call();
    }
    
    /**
     * Create a new subscription
     */
    public Future<ArrayList<SubscriptionModel>> createSubscriptionsAsync(Integer accountId, ArrayList<SubscriptionModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/subscriptions");
        path.applyField("accountId", accountId);
        return this.threadPool.submit((RestCall<ArrayList<SubscriptionModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<SubscriptionModel>>(){}));
    }

    public ArrayList<SubscriptionModel> createSubscriptions(Integer accountId, ArrayList<SubscriptionModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/subscriptions");
        path.applyField("accountId", accountId);
        return ((RestCall<ArrayList<SubscriptionModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<SubscriptionModel>>(){})).call();
    }
    
    /**
     * Create new users
     */
    public Future<ArrayList<UserModel>> createUsersAsync(Integer accountId, ArrayList<UserModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/users");
        path.applyField("accountId", accountId);
        return this.threadPool.submit((RestCall<ArrayList<UserModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<UserModel>>(){}));
    }

    public ArrayList<UserModel> createUsers(Integer accountId, ArrayList<UserModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/users");
        path.applyField("accountId", accountId);
        return ((RestCall<ArrayList<UserModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<UserModel>>(){})).call();
    }
    
    /**
     * Delete a single account
     */
    public Future<ErrorResult> deleteAccountAsync(Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){}));
    }

    public ErrorResult deleteAccount(Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}");
        path.applyField("id", id);
        return ((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){})).call();
    }
    
    /**
     * Delete a single subscription
     */
    public Future<ErrorResult> deleteSubscriptionAsync(Integer accountId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/subscriptions/{id}");
        path.applyField("accountId", accountId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){}));
    }

    public ErrorResult deleteSubscription(Integer accountId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/subscriptions/{id}");
        path.applyField("accountId", accountId);
        path.applyField("id", id);
        return ((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){})).call();
    }
    
    /**
     * Delete a single user
     */
    public Future<ErrorResult> deleteUserAsync(Integer id, Integer accountId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/users/{id}");
        path.applyField("id", id);
        path.applyField("accountId", accountId);
        return this.threadPool.submit((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){}));
    }

    public ErrorResult deleteUser(Integer id, Integer accountId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/users/{id}");
        path.applyField("id", id);
        path.applyField("accountId", accountId);
        return ((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){})).call();
    }
    
    /**
     * Retrieve all accounts
     */
    public Future<FetchResult<AccountModel>> queryAccountsAsync(String $include, String $filter, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts");
        path.addQuery("$include", $include);
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<AccountModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<AccountModel>>(){}));
    }

    public FetchResult<AccountModel> queryAccounts(String $include, String $filter, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts");
        path.addQuery("$include", $include);
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<AccountModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<AccountModel>>(){})).call();
    }
    
    /**
     * Reset a user's password programmatically
     */
    public Future<String> resetPasswordAsync(Integer userId, SetPasswordModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/passwords/{userId}/reset");
        path.applyField("userId", userId);
        return this.threadPool.submit((RestCall<String>)restCallFactory.createRestCall("post", path, model, new TypeToken<String>(){}));
    }

    public String resetPassword(Integer userId, SetPasswordModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/passwords/{userId}/reset");
        path.applyField("userId", userId);
        return ((RestCall<String>)restCallFactory.createRestCall("post", path, model, new TypeToken<String>(){})).call();
    }
    
    /**
     * Update a single account
     */
    public Future<AccountModel> updateAccountAsync(Integer id, AccountModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}");
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<AccountModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<AccountModel>(){}));
    }

    public AccountModel updateAccount(Integer id, AccountModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{id}");
        path.applyField("id", id);
        return ((RestCall<AccountModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<AccountModel>(){})).call();
    }
    
    /**
     * Update a single subscription
     */
    public Future<SubscriptionModel> updateSubscriptionAsync(Integer accountId, Integer id, SubscriptionModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/subscriptions/{id}");
        path.applyField("accountId", accountId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<SubscriptionModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<SubscriptionModel>(){}));
    }

    public SubscriptionModel updateSubscription(Integer accountId, Integer id, SubscriptionModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/subscriptions/{id}");
        path.applyField("accountId", accountId);
        path.applyField("id", id);
        return ((RestCall<SubscriptionModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<SubscriptionModel>(){})).call();
    }
    
//endregion

//region Settings
    /**
     * Create a new setting
     */
    public Future<ArrayList<SettingModel>> createSettingsAsync(Integer companyId, ArrayList<SettingModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/settings");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<SettingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<SettingModel>>(){}));
    }

    public ArrayList<SettingModel> createSettings(Integer companyId, ArrayList<SettingModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/settings");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<SettingModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<SettingModel>>(){})).call();
    }
    
    /**
     * Delete a single setting
     */
    public Future<ErrorResult> deleteSettingAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/settings/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){}));
    }

    public ErrorResult deleteSetting(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/settings/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){})).call();
    }
    
    /**
     * Retrieve a single setting
     */
    public Future<SettingModel> getSettingAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/settings/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<SettingModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<SettingModel>(){}));
    }

    public SettingModel getSetting(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/settings/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<SettingModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<SettingModel>(){})).call();
    }
    
    /**
     * Retrieve all settings for this company
     */
    public Future<FetchResult<SettingModel>> listSettingsByCompanyAsync(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/settings");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<SettingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SettingModel>>(){}));
    }

    public FetchResult<SettingModel> listSettingsByCompany(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/settings");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<SettingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SettingModel>>(){})).call();
    }
    
    /**
     * Retrieve all settings
     */
    public Future<FetchResult<SettingModel>> querySettingsAsync(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/settings");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<SettingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SettingModel>>(){}));
    }

    public FetchResult<SettingModel> querySettings(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/settings");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<SettingModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SettingModel>>(){})).call();
    }
    
    /**
     * Update a single setting
     */
    public Future<SettingModel> updateSettingAsync(Integer companyId, Integer id, SettingModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/settings/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<SettingModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<SettingModel>(){}));
    }

    public SettingModel updateSetting(Integer companyId, Integer id, SettingModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/settings/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<SettingModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<SettingModel>(){})).call();
    }
    
//endregion

//region Subscriptions
    /**
     * Retrieve a single subscription
     */
    public Future<SubscriptionModel> getSubscriptionAsync(Integer accountId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/subscriptions/{id}");
        path.applyField("accountId", accountId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<SubscriptionModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<SubscriptionModel>(){}));
    }

    public SubscriptionModel getSubscription(Integer accountId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/subscriptions/{id}");
        path.applyField("accountId", accountId);
        path.applyField("id", id);
        return ((RestCall<SubscriptionModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<SubscriptionModel>(){})).call();
    }
    
    /**
     * Retrieve subscriptions for this account
     */
    public Future<FetchResult<SubscriptionModel>> listSubscriptionsByAccountAsync(Integer accountId, String $filter, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/subscriptions");
        path.applyField("accountId", accountId);
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<SubscriptionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SubscriptionModel>>(){}));
    }

    public FetchResult<SubscriptionModel> listSubscriptionsByAccount(Integer accountId, String $filter, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/subscriptions");
        path.applyField("accountId", accountId);
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<SubscriptionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SubscriptionModel>>(){})).call();
    }
    
    /**
     * Retrieve all subscriptions
     */
    public Future<FetchResult<SubscriptionModel>> querySubscriptionsAsync(String $filter, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/subscriptions");
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<SubscriptionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SubscriptionModel>>(){}));
    }

    public FetchResult<SubscriptionModel> querySubscriptions(String $filter, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/subscriptions");
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<SubscriptionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<SubscriptionModel>>(){})).call();
    }
    
//endregion

//region TaxCodes
    /**
     * Create a new tax code
     */
    public Future<ArrayList<TaxCodeModel>> createTaxCodesAsync(Integer companyId, ArrayList<TaxCodeModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxcodes");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<TaxCodeModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<TaxCodeModel>>(){}));
    }

    public ArrayList<TaxCodeModel> createTaxCodes(Integer companyId, ArrayList<TaxCodeModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxcodes");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<TaxCodeModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<TaxCodeModel>>(){})).call();
    }
    
    /**
     * Delete a single tax code
     */
    public Future<ErrorResult> deleteTaxCodeAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxcodes/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){}));
    }

    public ErrorResult deleteTaxCode(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxcodes/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){})).call();
    }
    
    /**
     * Retrieve a single tax code
     */
    public Future<TaxCodeModel> getTaxCodeAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxcodes/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<TaxCodeModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TaxCodeModel>(){}));
    }

    public TaxCodeModel getTaxCode(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxcodes/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<TaxCodeModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TaxCodeModel>(){})).call();
    }
    
    /**
     * Retrieve tax codes for this company
     */
    public Future<FetchResult<TaxCodeModel>> listTaxCodesByCompanyAsync(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxcodes");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<TaxCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxCodeModel>>(){}));
    }

    public FetchResult<TaxCodeModel> listTaxCodesByCompany(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxcodes");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<TaxCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxCodeModel>>(){})).call();
    }
    
    /**
     * Retrieve all tax codes
     */
    public Future<FetchResult<TaxCodeModel>> queryTaxCodesAsync(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/taxcodes");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<TaxCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxCodeModel>>(){}));
    }

    public FetchResult<TaxCodeModel> queryTaxCodes(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/taxcodes");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<TaxCodeModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxCodeModel>>(){})).call();
    }
    
    /**
     * Update a single tax code
     */
    public Future<TaxCodeModel> updateTaxCodeAsync(Integer companyId, Integer id, TaxCodeModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxcodes/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<TaxCodeModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<TaxCodeModel>(){}));
    }

    public TaxCodeModel updateTaxCode(Integer companyId, Integer id, TaxCodeModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxcodes/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<TaxCodeModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<TaxCodeModel>(){})).call();
    }
    
//endregion

//region TaxRates
    /**
     * Retrieve tax rates for a specified address
     */
    public Future<TaxRateModel> taxRatesByAddressAsync(String line1, String line2, String line3, String city, String region, String postalCode, String country) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/taxrates/byaddress");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        return this.threadPool.submit((RestCall<TaxRateModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TaxRateModel>(){}));
    }

    public TaxRateModel taxRatesByAddress(String line1, String line2, String line3, String city, String region, String postalCode, String country) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/taxrates/byaddress");
        path.addQuery("line1", line1);
        path.addQuery("line2", line2);
        path.addQuery("line3", line3);
        path.addQuery("city", city);
        path.addQuery("region", region);
        path.addQuery("postalCode", postalCode);
        path.addQuery("country", country);
        return ((RestCall<TaxRateModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TaxRateModel>(){})).call();
    }
    
    /**
     * Retrieve tax rates for a specified country and postal code
     */
    public Future<TaxRateModel> taxRatesByPostalCodeAsync(String country, String postalCode) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/taxrates/bypostalcode");
        path.addQuery("country", country);
        path.addQuery("postalCode", postalCode);
        return this.threadPool.submit((RestCall<TaxRateModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TaxRateModel>(){}));
    }

    public TaxRateModel taxRatesByPostalCode(String country, String postalCode) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/taxrates/bypostalcode");
        path.addQuery("country", country);
        path.addQuery("postalCode", postalCode);
        return ((RestCall<TaxRateModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TaxRateModel>(){})).call();
    }
    
//endregion

//region TaxRules
    /**
     * Create a new tax rule
     */
    public Future<ArrayList<TaxRuleModel>> createTaxRulesAsync(Integer companyId, ArrayList<TaxRuleModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxrules");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<TaxRuleModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<TaxRuleModel>>(){}));
    }

    public ArrayList<TaxRuleModel> createTaxRules(Integer companyId, ArrayList<TaxRuleModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxrules");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<TaxRuleModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<TaxRuleModel>>(){})).call();
    }
    
    /**
     * Delete a single tax rule
     */
    public Future<ErrorResult> deleteTaxRuleAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxrules/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){}));
    }

    public ErrorResult deleteTaxRule(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxrules/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){})).call();
    }
    
    /**
     * Retrieve a single tax rule
     */
    public Future<TaxRuleModel> getTaxRuleAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxrules/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<TaxRuleModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TaxRuleModel>(){}));
    }

    public TaxRuleModel getTaxRule(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxrules/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<TaxRuleModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TaxRuleModel>(){})).call();
    }
    
    /**
     * Retrieve tax rules for this company
     */
    public Future<FetchResult<TaxRuleModel>> listTaxRulesAsync(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxrules");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<TaxRuleModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxRuleModel>>(){}));
    }

    public FetchResult<TaxRuleModel> listTaxRules(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxrules");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<TaxRuleModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxRuleModel>>(){})).call();
    }
    
    /**
     * Retrieve all tax rules
     */
    public Future<FetchResult<TaxRuleModel>> queryTaxRulesAsync(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/taxrules");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<TaxRuleModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxRuleModel>>(){}));
    }

    public FetchResult<TaxRuleModel> queryTaxRules(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/taxrules");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<TaxRuleModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TaxRuleModel>>(){})).call();
    }
    
    /**
     * Update a single tax rule
     */
    public Future<TaxRuleModel> updateTaxRuleAsync(Integer companyId, Integer id, TaxRuleModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxrules/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<TaxRuleModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<TaxRuleModel>(){}));
    }

    public TaxRuleModel updateTaxRule(Integer companyId, Integer id, TaxRuleModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/taxrules/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<TaxRuleModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<TaxRuleModel>(){})).call();
    }
    
//endregion

//region Transactions
    /**
     * Correct a previously created transaction
     */
    public Future<TransactionModel> adjustTransactionAsync(String companyCode, String transactionCode, AdjustTransactionModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyCode}/transactions/{transactionCode}/adjust");
        path.applyField("companyCode", companyCode);
        path.applyField("transactionCode", transactionCode);
        return this.threadPool.submit((RestCall<TransactionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<TransactionModel>(){}));
    }

    public TransactionModel adjustTransaction(String companyCode, String transactionCode, AdjustTransactionModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyCode}/transactions/{transactionCode}/adjust");
        path.applyField("companyCode", companyCode);
        path.applyField("transactionCode", transactionCode);
        return ((RestCall<TransactionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<TransactionModel>(){})).call();
    }
    
    /**
     * Change a transaction's code
     */
    public Future<TransactionModel> changeTransactionCodeAsync(String companyCode, String transactionCode, ChangeTransactionCodeModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyCode}/transactions/{transactionCode}/changecode");
        path.applyField("companyCode", companyCode);
        path.applyField("transactionCode", transactionCode);
        return this.threadPool.submit((RestCall<TransactionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<TransactionModel>(){}));
    }

    public TransactionModel changeTransactionCode(String companyCode, String transactionCode, ChangeTransactionCodeModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyCode}/transactions/{transactionCode}/changecode");
        path.applyField("companyCode", companyCode);
        path.applyField("transactionCode", transactionCode);
        return ((RestCall<TransactionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<TransactionModel>(){})).call();
    }
    
    /**
     * Commit a transaction for reporting
     */
    public Future<TransactionModel> commitTransactionAsync(String companyCode, String transactionCode, CommitTransactionModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyCode}/transactions/{transactionCode}/commit");
        path.applyField("companyCode", companyCode);
        path.applyField("transactionCode", transactionCode);
        return this.threadPool.submit((RestCall<TransactionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<TransactionModel>(){}));
    }

    public TransactionModel commitTransaction(String companyCode, String transactionCode, CommitTransactionModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyCode}/transactions/{transactionCode}/commit");
        path.applyField("companyCode", companyCode);
        path.applyField("transactionCode", transactionCode);
        return ((RestCall<TransactionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<TransactionModel>(){})).call();
    }
    
    /**
     * Create a new transaction
     */
    public Future<TransactionModel> createTransactionAsync(CreateTransactionModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/create");
        return this.threadPool.submit((RestCall<TransactionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<TransactionModel>(){}));
    }

    public TransactionModel createTransaction(CreateTransactionModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/create");
        return ((RestCall<TransactionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<TransactionModel>(){})).call();
    }
    
    /**
     * Retrieve a single transaction by code
     */
    public Future<TransactionModel> getTransactionByCodeAsync(String companyCode, String transactionCode, String $include) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyCode}/transactions/{transactionCode}");
        path.applyField("companyCode", companyCode);
        path.applyField("transactionCode", transactionCode);
        path.addQuery("$include", $include);
        return this.threadPool.submit((RestCall<TransactionModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TransactionModel>(){}));
    }

    public TransactionModel getTransactionByCode(String companyCode, String transactionCode, String $include) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyCode}/transactions/{transactionCode}");
        path.applyField("companyCode", companyCode);
        path.applyField("transactionCode", transactionCode);
        path.addQuery("$include", $include);
        return ((RestCall<TransactionModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TransactionModel>(){})).call();
    }
    
    /**
     * Retrieve a single transaction by ID
     */
    public Future<TransactionModel> getTransactionByIdAsync(Long id, String $include) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/{id}");
        path.applyField("id", id);
        path.addQuery("$include", $include);
        return this.threadPool.submit((RestCall<TransactionModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TransactionModel>(){}));
    }

    public TransactionModel getTransactionById(Long id, String $include) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/transactions/{id}");
        path.applyField("id", id);
        path.addQuery("$include", $include);
        return ((RestCall<TransactionModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<TransactionModel>(){})).call();
    }
    
    /**
     * Retrieve all transactions
     */
    public Future<FetchResult<TransactionModel>> listTransactionsByCompanyAsync(String companyCode, String $include, String $filter, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyCode}/transactions");
        path.applyField("companyCode", companyCode);
        path.addQuery("$include", $include);
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<TransactionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TransactionModel>>(){}));
    }

    public FetchResult<TransactionModel> listTransactionsByCompany(String companyCode, String $include, String $filter, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyCode}/transactions");
        path.applyField("companyCode", companyCode);
        path.addQuery("$include", $include);
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<TransactionModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<TransactionModel>>(){})).call();
    }
    
    /**
     * Perform multiple actions on a transaction
     */
    public Future<TransactionModel> settleTransactionAsync(String companyCode, String transactionCode, SettleTransactionModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyCode}/transactions/{transactionCode}/settle");
        path.applyField("companyCode", companyCode);
        path.applyField("transactionCode", transactionCode);
        return this.threadPool.submit((RestCall<TransactionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<TransactionModel>(){}));
    }

    public TransactionModel settleTransaction(String companyCode, String transactionCode, SettleTransactionModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyCode}/transactions/{transactionCode}/settle");
        path.applyField("companyCode", companyCode);
        path.applyField("transactionCode", transactionCode);
        return ((RestCall<TransactionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<TransactionModel>(){})).call();
    }
    
    /**
     * Verify a transaction
     */
    public Future<TransactionModel> verifyTransactionAsync(String companyCode, String transactionCode, VerifyTransactionModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyCode}/transactions/{transactionCode}/verify");
        path.applyField("companyCode", companyCode);
        path.applyField("transactionCode", transactionCode);
        return this.threadPool.submit((RestCall<TransactionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<TransactionModel>(){}));
    }

    public TransactionModel verifyTransaction(String companyCode, String transactionCode, VerifyTransactionModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyCode}/transactions/{transactionCode}/verify");
        path.applyField("companyCode", companyCode);
        path.applyField("transactionCode", transactionCode);
        return ((RestCall<TransactionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<TransactionModel>(){})).call();
    }
    
    /**
     * Void a transaction
     */
    public Future<TransactionModel> voidTransactionAsync(String companyCode, String transactionCode, VoidTransactionModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyCode}/transactions/{transactionCode}/void");
        path.applyField("companyCode", companyCode);
        path.applyField("transactionCode", transactionCode);
        return this.threadPool.submit((RestCall<TransactionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<TransactionModel>(){}));
    }

    public TransactionModel voidTransaction(String companyCode, String transactionCode, VoidTransactionModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyCode}/transactions/{transactionCode}/void");
        path.applyField("companyCode", companyCode);
        path.applyField("transactionCode", transactionCode);
        return ((RestCall<TransactionModel>)restCallFactory.createRestCall("post", path, model, new TypeToken<TransactionModel>(){})).call();
    }
    
//endregion

//region Upcs
    /**
     * Create a new UPC
     */
    public Future<ArrayList<UPCModel>> createUPCsAsync(Integer companyId, ArrayList<UPCModel> model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/upcs");
        path.applyField("companyId", companyId);
        return this.threadPool.submit((RestCall<ArrayList<UPCModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<UPCModel>>(){}));
    }

    public ArrayList<UPCModel> createUPCs(Integer companyId, ArrayList<UPCModel> model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/upcs");
        path.applyField("companyId", companyId);
        return ((RestCall<ArrayList<UPCModel>>)restCallFactory.createRestCall("post", path, model, new TypeToken<ArrayList<UPCModel>>(){})).call();
    }
    
    /**
     * Delete a single UPC
     */
    public Future<ErrorResult> deleteUPCAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/upcs/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){}));
    }

    public ErrorResult deleteUPC(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/upcs/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<ErrorResult>)restCallFactory.createRestCall("delete", path, null, new TypeToken<ErrorResult>(){})).call();
    }
    
    /**
     * Retrieve a single UPC
     */
    public Future<UPCModel> getUPCAsync(Integer companyId, Integer id) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/upcs/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<UPCModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<UPCModel>(){}));
    }

    public UPCModel getUPC(Integer companyId, Integer id) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/upcs/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<UPCModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<UPCModel>(){})).call();
    }
    
    /**
     * Retrieve UPCs for this company
     */
    public Future<FetchResult<UPCModel>> listUPCsByCompanyAsync(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/upcs");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<UPCModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<UPCModel>>(){}));
    }

    public FetchResult<UPCModel> listUPCsByCompany(Integer companyId, String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/upcs");
        path.applyField("companyId", companyId);
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<UPCModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<UPCModel>>(){})).call();
    }
    
    /**
     * Retrieve all UPCs
     */
    public Future<FetchResult<UPCModel>> queryUPCsAsync(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/upcs");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<UPCModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<UPCModel>>(){}));
    }

    public FetchResult<UPCModel> queryUPCs(String $filter, String $include, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/upcs");
        path.addQuery("$filter", $filter);
        path.addQuery("$include", $include);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<UPCModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<UPCModel>>(){})).call();
    }
    
    /**
     * Update a single UPC
     */
    public Future<UPCModel> updateUPCAsync(Integer companyId, Integer id, UPCModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/upcs/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return this.threadPool.submit((RestCall<UPCModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<UPCModel>(){}));
    }

    public UPCModel updateUPC(Integer companyId, Integer id, UPCModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/companies/{companyId}/upcs/{id}");
        path.applyField("companyId", companyId);
        path.applyField("id", id);
        return ((RestCall<UPCModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<UPCModel>(){})).call();
    }
    
//endregion

//region Users
    /**
     * Retrieve a single user
     */
    public Future<UserModel> getUserAsync(Integer id, Integer accountId, String $include) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/users/{id}");
        path.applyField("id", id);
        path.applyField("accountId", accountId);
        path.addQuery("$include", $include);
        return this.threadPool.submit((RestCall<UserModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<UserModel>(){}));
    }

    public UserModel getUser(Integer id, Integer accountId, String $include) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/users/{id}");
        path.applyField("id", id);
        path.applyField("accountId", accountId);
        path.addQuery("$include", $include);
        return ((RestCall<UserModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<UserModel>(){})).call();
    }
    
    /**
     * Retrieve all entitlements for a single user
     */
    public Future<UserEntitlementModel> getUserEntitlementsAsync(Integer id, Integer accountId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/users/{id}/entitlements");
        path.applyField("id", id);
        path.applyField("accountId", accountId);
        return this.threadPool.submit((RestCall<UserEntitlementModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<UserEntitlementModel>(){}));
    }

    public UserEntitlementModel getUserEntitlements(Integer id, Integer accountId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/users/{id}/entitlements");
        path.applyField("id", id);
        path.applyField("accountId", accountId);
        return ((RestCall<UserEntitlementModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<UserEntitlementModel>(){})).call();
    }
    
    /**
     * Retrieve users for this account
     */
    public Future<FetchResult<UserModel>> listUsersByAccountAsync(Integer accountId, String $include, String $filter, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/users");
        path.applyField("accountId", accountId);
        path.addQuery("$include", $include);
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<UserModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<UserModel>>(){}));
    }

    public FetchResult<UserModel> listUsersByAccount(Integer accountId, String $include, String $filter, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/users");
        path.applyField("accountId", accountId);
        path.addQuery("$include", $include);
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<UserModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<UserModel>>(){})).call();
    }
    
    /**
     * Retrieve all users
     */
    public Future<FetchResult<UserModel>> queryUsersAsync(String $include, String $filter, Integer $top, Integer $skip, String $orderBy) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/users");
        path.addQuery("$include", $include);
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return this.threadPool.submit((RestCall<FetchResult<UserModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<UserModel>>(){}));
    }

    public FetchResult<UserModel> queryUsers(String $include, String $filter, Integer $top, Integer $skip, String $orderBy) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/users");
        path.addQuery("$include", $include);
        path.addQuery("$filter", $filter);
        path.addQuery("$top", $top);
        path.addQuery("$skip", $skip);
        path.addQuery("$orderBy", $orderBy);
        return ((RestCall<FetchResult<UserModel>>)restCallFactory.createRestCall("get", path, null, new TypeToken<FetchResult<UserModel>>(){})).call();
    }
    
    /**
     * Update a single user
     */
    public Future<UserModel> updateUserAsync(Integer id, Integer accountId, UserModel model) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/users/{id}");
        path.applyField("id", id);
        path.applyField("accountId", accountId);
        return this.threadPool.submit((RestCall<UserModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<UserModel>(){}));
    }

    public UserModel updateUser(Integer id, Integer accountId, UserModel model) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/accounts/{accountId}/users/{id}");
        path.applyField("id", id);
        path.applyField("accountId", accountId);
        return ((RestCall<UserModel>)restCallFactory.createRestCall("put", path, model, new TypeToken<UserModel>(){})).call();
    }
    
//endregion

//region Utilities
    /**
     * Checks if the current user is subscribed to a specific service
     */
    public Future<SubscriptionModel> getMySubscriptionAsync(String serviceTypeId) {
        AvaTaxPath path = new AvaTaxPath("/api/v2/utilities/subscriptions/{serviceTypeId}");
        path.applyField("serviceTypeId", serviceTypeId);
        return this.threadPool.submit((RestCall<SubscriptionModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<SubscriptionModel>(){}));
    }

    public SubscriptionModel getMySubscription(String serviceTypeId) throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/utilities/subscriptions/{serviceTypeId}");
        path.applyField("serviceTypeId", serviceTypeId);
        return ((RestCall<SubscriptionModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<SubscriptionModel>(){})).call();
    }
    
    /**
     * List all services to which the current user is subscribed
     */
    public Future<SubscriptionModel> listMySubscriptionsAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/utilities/subscriptions");
        return this.threadPool.submit((RestCall<SubscriptionModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<SubscriptionModel>(){}));
    }

    public SubscriptionModel listMySubscriptions() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/utilities/subscriptions");
        return ((RestCall<SubscriptionModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<SubscriptionModel>(){})).call();
    }
    
    /**
     * Tests connectivity and version of the service
     */
    public Future<PingResultModel> pingAsync() {
        AvaTaxPath path = new AvaTaxPath("/api/v2/utilities/ping");
        return this.threadPool.submit((RestCall<PingResultModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<PingResultModel>(){}));
    }

    public PingResultModel ping() throws Exception {
        AvaTaxPath path = new AvaTaxPath("/api/v2/utilities/ping");
        return ((RestCall<PingResultModel>)restCallFactory.createRestCall("get", path, null, new TypeToken<PingResultModel>(){})).call();
    }
    
//endregion

}
    