/*
 * Copyright (c) 2019, BookRain Ltd.
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of BookRain Ltd. nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY BookRain Ltd. AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.bookrain.codegen.controller;

import com.alibaba.fastjson.JSONObject;
import com.bookrain.codegen.config.CommonConfig;
import com.bookrain.codegen.config.DataSourceConfig;
import com.bookrain.codegen.config.StrategyConfig;
import com.bookrain.codegen.config.TemplateConfig;
import com.bookrain.codegen.constants.CodeGenConstants;
import com.bookrain.codegen.dto.TableField;
import com.bookrain.codegen.dto.TableInfo;
import com.bookrain.codegen.dto.in.AddField;
import com.bookrain.codegen.dto.in.QueryField;
import com.bookrain.codegen.dto.in.UpdateField;
import com.bookrain.codegen.dto.out.ListField;
import com.bookrain.codegen.dto.out.ObjectField;
import com.bookrain.codegen.dto.out.PageField;
import com.bookrain.codegen.enums.AddType;
import com.bookrain.codegen.enums.CodeGenType;
import com.bookrain.codegen.enums.FieldType;
import com.bookrain.codegen.enums.IdType;
import com.bookrain.codegen.enums.QueryType;
import com.bookrain.codegen.enums.ShowType;
import com.bookrain.codegen.enums.UpdateType;
import com.bookrain.codegen.service.CodeGenService;
import com.bookrain.core.api.ApiResponse;
import com.bookrain.core.controller.BaseController;
import com.bookrain.core.utils.BeanUtils;
import com.bookrain.core.utils.StringUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import io.swagger.annotations.ApiOperation;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import lombok.Data;
import lombok.experimental.Accessors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

/**
 * .
 * <p>
 *
 * @author Bookrain Chu
 * @version 1.0
 * @date 2019-06-03 22:22:24
 */
@Api(tags = "代码生成")
@RestController
@RequestMapping("/codegen")
public class CodeGenController extends BaseController {

    @Autowired
    private DataSourceConfig dataSourceConfig;

    @Autowired
    private CodeGenService codeGenService;

    @ApiOperation(value = "获取新增测试初始化数据")
    @GetMapping(value = "/initCodeGen", produces = "application/json")
    public ApiResponse initCodeGen() throws SQLException {
        InitCodeGenResponse initCodeGenResponse = new InitCodeGenResponse();
        initCodeGenResponse.setAllTableNames(codeGenService.getTableNames(dataSourceConfig));

        // 主键生成策略
        JSONObject idTypes = new JSONObject();
        for (IdType value : IdType.values()) {
            idTypes.put(value.name(), value.getDesc());
        }

        // 生成代码类型
        JSONObject codeGenTypes = new JSONObject();
        for (CodeGenType value : CodeGenType.values()) {
            codeGenTypes.put(value.name(), value.getDesc());
        }

        // 生成代码类型
        JSONObject fieldTypes = new JSONObject();
        for (FieldType value : FieldType.values()) {
            fieldTypes.put(value.name(), value.getPkg() == null ? value.getType() : value.getPkg());
        }

        // 生成代码类型
        JSONObject showTypes = new JSONObject();
        for (ShowType value : ShowType.values()) {
            showTypes.put(value.name(), value.getDesc());
        }

        // 生成代码类型
        JSONObject queryTypes = new JSONObject();
        for (QueryType value : QueryType.values()) {
            queryTypes.put(value.name(), value.getDesc());
        }

        // 新增类型
        JSONObject addTypes = new JSONObject();
        for (AddType value : AddType.values()) {
            addTypes.put(value.name(), value.getDesc());
        }

        // 更新类型
        JSONObject updateTypes = new JSONObject();
        for (UpdateType value : UpdateType.values()) {
            updateTypes.put(value.name(), value.getDesc());
        }
        initCodeGenResponse.setIdTypes(idTypes)
            .setShowTypes(showTypes)
            .setQueryTypes(queryTypes)
            .setCodeGenTypes(codeGenTypes)
            .setFieldTypes(fieldTypes)
            .setAddTypes(addTypes)
            .setUpdateTypes(updateTypes);
        return ok(initCodeGenResponse);
    }

    @ApiOperation(value = "获取新增测试初始化数据")
    @GetMapping(value = "/getGenerateInfo", produces = "application/json")
    @ApiImplicitParam(name = "tableName", value = "表名")
    public ApiResponse getGenerateInfo(String tableName) throws Exception {
        GetCodeGenInfoResponse getCodeGenInfoResponse = new GetCodeGenInfoResponse();
        String[] tableNameSplits = tableName.split("_");
        String pkg;
        String dir;
        String namespace;
        StringBuilder mapping = new StringBuilder();
        StringBuilder module = new StringBuilder();
        if (tableNameSplits.length > 1) {
            pkg = tableNameSplits[1];
            dir = tableNameSplits[1];
            namespace = tableNameSplits[1];
            for (int i = 1; i < tableNameSplits.length; i++) {
                module.append(StringUtils.capitalFirst(tableNameSplits[i]));
                mapping.append("/" + StringUtils.capitalFirst(tableNameSplits[i]).toLowerCase());
            }
        } else {
            pkg = tableNameSplits[0];
            module = new StringBuilder(StringUtils.capitalFirst(tableNameSplits[0]));
            mapping.append("/" + StringUtils.capitalFirst(tableNameSplits[0]).toLowerCase());
            dir = tableNameSplits[0];
            namespace = tableNameSplits[0];
        }
        TableInfo table = codeGenService.getTableInfo(dataSourceConfig, tableName);

        TableInfoDto tableInfoDto = new TableInfoDto()
            .setName(table.getName())
            .setIdType(table.getIdType())
            .setComment(table.getComment());
        List<TableFieldDto> tableInfoDtos = new ArrayList<>();
        for (int i = 0; i < table.getFields().size(); i++) {
            TableField tableField = table.getFields().get(i);
            TableFieldDto tableFieldDto = new TableFieldDto()
                .setId(i)
                .setPk(tableField.getPk())
                .setComment(tableField.getComment())
                .setColumnName(tableField.getColumnName())
                .setColumnType(tableField.getColumnType())
                .setType(tableField.getType())
                .setName(tableField.getName());
            tableInfoDtos.add(tableFieldDto);
        }
        tableInfoDto.setFields(tableInfoDtos);
        getCodeGenInfoResponse
            .setTableInfo(tableInfoDto)
            .setModule(module.toString())
            .setJavaOutputDir(
                (System.getProperty("java.io.tmpdir") + "/" + dir.toLowerCase() + "/java").replace("//", "/"))
            .setJsOutputDir(
                (System.getProperty("java.io.tmpdir") + "/" + dir.toLowerCase() + "/react").replace("//", "/"))
            .setDate(new SimpleDateFormat("yyyy-MM-dd").format(new Date()))
            .setNamespace(namespace.toLowerCase())
            .setDescription(table.getComment())
            .setPkg(pkg.toLowerCase())
            .setBaseMapping(mapping.toString().toLowerCase());
        return ok(getCodeGenInfoResponse);
    }

    @ApiOperation(value = "获取新增测试初始化数据")
    @PostMapping(value = "/generate", produces = "application/json")
    public ApiResponse generate(@RequestBody GenerateRequest generateRequest) throws Exception {
        String module = StringUtils.capitalFirst(generateRequest.getModule());
        CommonConfig commonConfig = new CommonConfig();
        commonConfig.setJavaOutputDir(generateRequest.getJavaOutputDir());
        commonConfig.setJsOutputDir(generateRequest.getJsOutputDir());
        commonConfig.setDescription(generateRequest.getDescription());
        commonConfig.setAuthor(generateRequest.getAuthor());
        commonConfig.setModule(generateRequest.getModule());
        commonConfig.setDate(generateRequest.getDate());

        StrategyConfig strategyConfig = new StrategyConfig();
        strategyConfig.setPkg(generateRequest.getPkg());
        strategyConfig.setBasePkg(generateRequest.getBasePkg());
        strategyConfig.setBaseMapping(generateRequest.getBaseMapping());
        strategyConfig.setDtoName(module + "Dto");
        strategyConfig.setEntityName(module);
        strategyConfig.setMapperName(module + "Mapper");
        strategyConfig.setServiceName(module + "Service");
        strategyConfig.setServiceImplName(module + "ServiceImpl");
        strategyConfig.setControllerName(module + "Controller");
        strategyConfig.setMapperXmlName(module + "Mapper");
        strategyConfig.setNamespace(generateRequest.getNamespace());
        strategyConfig.setBaseMapping(generateRequest.getBaseMapping());

        // table
        GenerateTableInfoDto generateTableInfoDto = generateRequest.getTableInfo();
        TableInfo tableInfo = new TableInfo();
        tableInfo.setIdType(generateTableInfoDto.getIdType());
        tableInfo.setName(generateTableInfoDto.getName());
        tableInfo.setComment(generateTableInfoDto.getComment());
        tableInfo.setFields(BeanUtils.copyToList(generateTableInfoDto.getFields(), TableField.class,
            "queryType", "addType", "updateType"));

        // 输入类型
        // 新增字段
        List<AddField> addFields = new ArrayList<>();
        generateTableInfoDto.getFields().forEach(generateTableFieldInfoDto -> {
            AddType addType = generateTableFieldInfoDto.getAddType();
            if (addType != AddType.NONE) {
                AddField addField = new AddField();
                addField.setName(generateTableFieldInfoDto.getName());
                addField.setComment(generateTableFieldInfoDto.getComment());
                addField.setRequired(addType == AddType.REQUIRED);
                addField.setType(generateTableFieldInfoDto.getType());
                addField.setShowType(generateTableFieldInfoDto.getShowType());
                addFields.add(addField);
            }
        });

        // 查询字段
        List<QueryField> queryFields = new ArrayList<>();
        generateTableInfoDto.getFields().forEach(generateTableFieldInfoDto -> {
            QueryType queryType = generateTableFieldInfoDto.getQueryType();
            if (queryType != QueryType.NONE) {
                QueryField queryField = new QueryField();
                queryField.setQueryType(queryType);
                queryField.setColumnName(generateTableFieldInfoDto.getColumnName());
                queryField.setComment(generateTableFieldInfoDto.getComment());
                queryField.setName(generateTableFieldInfoDto.getName());
                queryField.setType(generateTableFieldInfoDto.getType());
                queryField.setShowType(generateTableFieldInfoDto.getShowType());
                queryFields.add(queryField);
            }
        });

        // 更新字段
        List<UpdateField> updateFields = new ArrayList<>();
        generateTableInfoDto.getFields().forEach(generateTableFieldInfoDto -> {
            UpdateType updateType = generateTableFieldInfoDto.getUpdateType();
            if (updateType != UpdateType.NONE) {
                UpdateField updateField = new UpdateField();
                updateField.setName(generateTableFieldInfoDto.getName());
                updateField.setComment(generateTableFieldInfoDto.getComment());
                updateField.setRequired(updateType == UpdateType.REQUIRED);
                updateField.setType(generateTableFieldInfoDto.getType());
                updateField.setShowType(generateTableFieldInfoDto.getShowType());
                updateFields.add(updateField);
            }
        });

        // 输出类型
        // 单条字段
        List<ObjectField> objectFields = new ArrayList<>();
        generateTableInfoDto.getFields().forEach(generateTableFieldInfoDto -> {
            if (generateTableFieldInfoDto.getObjField()) {
                ObjectField objectField = new ObjectField();
                objectField.setName(generateTableFieldInfoDto.getName());
                objectField.setComment(generateTableFieldInfoDto.getComment());
                objectField.setType(generateTableFieldInfoDto.getType());
                objectFields.add(objectField);
            }
        });

        // 列表字段
        List<ListField> listFields = new ArrayList<>();
        generateTableInfoDto.getFields().forEach(generateTableFieldInfoDto -> {
            if (generateTableFieldInfoDto.getListField()) {
                ListField listField = new ListField();
                listField.setName(generateTableFieldInfoDto.getName());
                listField.setComment(generateTableFieldInfoDto.getComment());
                listField.setType(generateTableFieldInfoDto.getType());
                listFields.add(listField);
            }
        });

        // 分页字段
        List<PageField> pageFields = new ArrayList<>();
        generateTableInfoDto.getFields().forEach(generateTableFieldInfoDto -> {
            if (generateTableFieldInfoDto.getPageField()) {
                PageField pageField = new PageField();
                pageField.setName(generateTableFieldInfoDto.getName());
                pageField.setComment(generateTableFieldInfoDto.getComment());
                pageField.setType(generateTableFieldInfoDto.getType());
                pageFields.add(pageField);
            }
        });

        strategyConfig.setTable(tableInfo)
            .setAddFields(addFields)
            .setQueryFields(queryFields)
            .setUpdateFields(updateFields)
            .setObjFields(objectFields)
            .setListFields(listFields)
            .setPageFields(pageFields);

        codeGenService.generate(dataSourceConfig, commonConfig, new TemplateConfig(), strategyConfig);
        return ok();
    }


    /*******************************************************实体对象****************************************************/
    @Data
    @Accessors(chain = true)
    @ApiModel(description = "获取代码生成初始化数据")
    public static class InitCodeGenResponse {

        @ApiModelProperty(value = "数据库所有表名称")
        private List<String> allTableNames;
        @ApiModelProperty(value = "主键生成策略")
        private JSONObject idTypes;
        @ApiModelProperty(value = "查询类型")
        private JSONObject queryTypes;
        @ApiModelProperty(value = "显示类型，页面文本框的类型")
        private JSONObject showTypes;
        @ApiModelProperty(value = "生成代码类型")
        private JSONObject codeGenTypes;
        @ApiModelProperty(value = "属性类型")
        private JSONObject fieldTypes;
        @ApiModelProperty(value = "新增类型")
        private JSONObject addTypes;
        @ApiModelProperty(value = "更新类型")
        private JSONObject updateTypes;
    }

    @Data
    @Accessors(chain = true)
    @ApiModel(description = "获取新增测试初始化数据响应")
    public static class GetCodeGenInfoResponse {

        private TableInfoDto tableInfo;
        @ApiModelProperty(value = "基础拦截路径")
        private String baseMapping;
        @ApiModelProperty(value = "作者")
        private String author = CodeGenConstants.DEFAULT_AUTHOR;
        @ApiModelProperty(value = "模块名")
        private String module;
        @ApiModelProperty(value = "日期")
        private String date;
        @ApiModelProperty(value = "后台输出路径")
        private String javaOutputDir;
        @ApiModelProperty(value = "react输出路径")
        private String jsOutputDir;
        @ApiModelProperty(value = "模块包")
        private String pkg;
        @ApiModelProperty(value = "基础包路径")
        private String basePkg = CodeGenConstants.DEFAULT_BASE_PKG;
        @ApiModelProperty(value = "模块描述")
        private String description;
        @ApiModelProperty(value = "react命名空间")
        private String namespace;
    }

    @Data
    @Accessors(chain = true)
    @ApiModel(description = "表信息")
    public static class TableInfoDto {

        @ApiModelProperty(value = "主键生成类型")
        private IdType idType;
        @ApiModelProperty(value = "表名")
        private String name;
        @ApiModelProperty(value = "表注释")
        private String comment;
        private List<TableFieldDto> fields;
    }

    @Data
    @Accessors(chain = true)
    @ApiModel(description = "表字段信息")
    public static class TableFieldDto {

        @ApiModelProperty(value = "字段序号")
        private int id;
        @ApiModelProperty(value = "是否是主键")
        private Boolean pk;
        @ApiModelProperty(value = "数据库列名")
        private String columnName;
        @ApiModelProperty(value = "数据库列类型")
        private String columnType;
        @ApiModelProperty(value = "属性名称")
        private String name;
        @ApiModelProperty(value = "数据库列注释")
        private String comment;
        @ApiModelProperty(value = "属性类型")
        private FieldType type;
    }

    @Data
    @Accessors(chain = true)
    @ApiModel(description = "生成代码请求")
    public static class GenerateRequest {

        @ApiModelProperty(value = "基础包路径")
        private String baseMapping;
        @ApiModelProperty(value = "基础包路径")
        private String basePkg;
        @ApiModelProperty(value = "模块包")
        private String pkg;
        @ApiModelProperty(value = "生成的模块名,文件前缀，默认会首字母大写")
        private String module;
        @ApiModelProperty(value = "代码生成类型")
        private CodeGenType codeGenType;
        @ApiModelProperty(value = "模块描述")
        private String description;
        @ApiModelProperty(value = "作者")
        private String author;
        @ApiModelProperty(value = "时间")
        private String date;
        @ApiModelProperty(value = "生成java后端代码路径")
        private String javaOutputDir;
        @ApiModelProperty(value = "生成react前端代码路径")
        private String jsOutputDir;
        @ApiModelProperty(value = "react命名空间")
        private String namespace;
        private GenerateTableInfoDto tableInfo;
    }

    @Data
    @Accessors(chain = true)
    @ApiModel(description = "生成代码请求数据库内容")
    public static class GenerateTableInfoDto {

        @ApiModelProperty(value = "表名")
        private String name;
        @ApiModelProperty(value = "主键生成策略")
        private IdType idType;
        @ApiModelProperty(value = "表注释")
        private String comment;
        private List<GenerateTableFieldInfoDto> fields;
    }

    @Data
    @Accessors(chain = true)
    @ApiModel(description = "生成代码请求数据库字段内容")
    public static class GenerateTableFieldInfoDto {

        @ApiModelProperty(value = "是否主键")
        private Boolean pk;
        @ApiModelProperty(value = "列名")
        private String columnName;
        @ApiModelProperty(value = "列类型")
        private String columnType;
        @ApiModelProperty(value = "属性名")
        private String name;
        @ApiModelProperty(value = "属性类型")
        private FieldType type;
        @ApiModelProperty(value = "字段注释")
        private String comment;
        @ApiModelProperty(value = "显示类型")
        private ShowType showType;
        @ApiModelProperty(value = "查询类型")
        private QueryType queryType;
        @ApiModelProperty(value = "新增类型")
        private AddType addType;
        @ApiModelProperty(value = "更新类型")
        private UpdateType updateType;
        @ApiModelProperty(value = "单条字段返回")
        private Boolean objField;
        @ApiModelProperty(value = "列表字段返回")
        private Boolean listField;
        @ApiModelProperty(value = "分页字段")
        private Boolean pageField;
    }
}

