/*
 * Copyright (c) 2019, BookRain Ltd.
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of BookRain Ltd. nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY BookRain Ltd. AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.bookrain.codegen.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SimplePropertyPreFilter;
import com.bookrain.codegen.config.CommonConfig;
import com.bookrain.codegen.config.DataSourceConfig;
import com.bookrain.codegen.config.StrategyConfig;
import com.bookrain.codegen.config.TemplateConfig;
import com.bookrain.codegen.database.ColumnTypeConverter;
import com.bookrain.codegen.database.TableQuery;
import com.bookrain.codegen.dto.TableField;
import com.bookrain.codegen.dto.TableInfo;
import com.bookrain.codegen.service.CodeGenService;
import com.bookrain.core.utils.StringUtils;
import freemarker.cache.URLTemplateLoader;
import freemarker.template.Configuration;
import freemarker.template.Template;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.net.URL;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Service;

/**
 * 生成代码服务实现类.
 * <p>
 *
 * @author Bookrain Chu
 * @version 1.0
 * @date 2019-06-01 16:16:44
 */
@Slf4j
@Service
public class CodeGenServiceImpl implements CodeGenService {

    @Autowired
    private DataSourceConfig dataSourceConfig;

    /**
     * 生成模板代码.<p>
     *
     * @param dataSourceConfig 数据库配置
     * @param commonConfig 通用配置
     * @param templateConfig 模板配置
     * @param strategyConfig 生成策略配置
     */
    @Override
    public void generate(DataSourceConfig dataSourceConfig, CommonConfig commonConfig, TemplateConfig templateConfig, StrategyConfig strategyConfig) throws Exception {
        // 创建freeMarker配置实例
        Configuration configuration = new Configuration(Configuration.VERSION_2_3_28);

        // 获取模版路径
        // configuration.setDirectoryForTemplateLoading(new ClassPathResource("templates").getFile());
        configuration.setTemplateLoader(new URLTemplateLoader() {
            @Override
            protected URL getURL(String name) {
                try {
                    return new ClassPathResource("templates/" + name).getURL();
                } catch (IOException e) {
                    log.error("获取freemarker模板失败", e.getLocalizedMessage());
                    return null;
                }
            }
        });

        // 创建数据模型
        JSONObject data = new JSONObject();
        data.putAll(JSONObject.parseObject(JSON.toJSON(commonConfig).toString()));

        SimplePropertyPreFilter strategyConfigPPF = new SimplePropertyPreFilter();
        strategyConfigPPF.getExcludes().add("table");
        strategyConfigPPF.getExcludes().add("queryFields");
        strategyConfigPPF.getExcludes().add("addFields");
        strategyConfigPPF.getExcludes().add("updateFields");
        strategyConfigPPF.getExcludes().add("pageFields");
        strategyConfigPPF.getExcludes().add("listFields");
        strategyConfigPPF.getExcludes().add("objFields");
        data.putAll(JSONObject.parseObject(JSON.toJSONString(strategyConfig, strategyConfigPPF)));

        List<String> controllerImports = new ArrayList<>();
        data.put("controllerImports", controllerImports);

        // 输入类型
        List<JSONObject> queryFields = new ArrayList<>();
        strategyConfig.getQueryFields().forEach(queryField -> {
            JSONObject field = new JSONObject();
            field.put("columnName", queryField.getColumnName());
            field.put("name", queryField.getName());
            field.put("type", queryField.getType().getType());
            field.put("comment", queryField.getComment());
            field.put("queryType", queryField.getQueryType());
            field.put("showType", queryField.getShowType());
            if (queryField.getType().getPkg() != null && !controllerImports.contains(queryField.getType().getPkg())) {
                controllerImports.add(queryField.getType().getPkg());
            }
            queryFields.add(field);
        });
        data.put("queryFields", queryFields);

        List<JSONObject> getFields = new ArrayList<>();
        strategyConfig.getAddFields().forEach(addField -> {
            JSONObject field = new JSONObject();
            field.put("name", addField.getName());
            field.put("type", addField.getType().getType());
            field.put("comment", addField.getComment());
            if (addField.getType().getPkg() != null && !controllerImports.contains(addField.getType().getPkg())) {
                controllerImports.add(addField.getType().getPkg());
            }
            getFields.add(field);
        });
        data.put("getFields", getFields);

        List<JSONObject> addFields = new ArrayList<>();
        strategyConfig.getAddFields().forEach(addField -> {
            JSONObject field = new JSONObject();
            field.put("name", addField.getName());
            field.put("type", addField.getType().getType());
            field.put("comment", addField.getComment());
            field.put("required", addField.getRequired());
            if (addField.getType().getPkg() != null && !controllerImports.contains(addField.getType().getPkg())) {
                controllerImports.add(addField.getType().getPkg());
            }
            addFields.add(field);
        });
        data.put("addFields", addFields);

        List<JSONObject> updateFields = new ArrayList<>();
        strategyConfig.getUpdateFields().forEach(updateField -> {
            JSONObject field = new JSONObject();
            field.put("name", updateField.getName());
            field.put("type", updateField.getType().getType());
            field.put("comment", updateField.getComment());
            field.put("required", updateField.getRequired());
            if (updateField.getType().getPkg() != null && !controllerImports.contains(updateField.getType().getPkg())) {
                controllerImports.add(updateField.getType().getPkg());
            }
            updateFields.add(field);
        });
        data.put("updateFields", updateFields);

        // 输出类型
        List<JSONObject> listFields = new ArrayList<>();
        strategyConfig.getListFields().forEach(listField -> {
            JSONObject field = new JSONObject();
            field.put("name", listField.getName());
            field.put("type", listField.getType().getType());
            field.put("comment", listField.getComment());
            if (listField.getType().getPkg() != null && !controllerImports.contains(listField.getType().getPkg())) {
                controllerImports.add(listField.getType().getPkg());
            }
            listFields.add(field);
        });
        data.put("listFields", listFields);

        List<JSONObject> objFields = new ArrayList<>();
        strategyConfig.getObjFields().forEach(objField -> {
            JSONObject field = new JSONObject();
            field.put("name", objField.getName());
            field.put("type", objField.getType().getType());
            field.put("comment", objField.getComment());
            if (objField.getType().getPkg() != null && !controllerImports.contains(objField.getType().getPkg())) {
                controllerImports.add(objField.getType().getPkg());
            }
            objFields.add(field);
        });
        data.put("objFields", objFields);

        List<JSONObject> pageFields = new ArrayList<>();
        strategyConfig.getPageFields().forEach(pageField -> {
            JSONObject field = new JSONObject();
            field.put("name", pageField.getName());
            field.put("type", pageField.getType().getType());
            field.put("comment", pageField.getComment());
            if (pageField.getType().getPkg() != null && !controllerImports.contains(pageField.getType().getPkg())) {
                controllerImports.add(pageField.getType().getPkg());
            }
            pageFields.add(field);
        });
        data.put("pageFields", pageFields);

        JSONObject table = new JSONObject();
        SimplePropertyPreFilter tablePPF = new SimplePropertyPreFilter();
        tablePPF.getExcludes().add("fields");
        table.putAll(JSONObject.parseObject(JSON.toJSONString(strategyConfig.getTable(), tablePPF)));
        JSONArray fields = new JSONArray();
        table.put("fields", fields);
        for (TableField f : strategyConfig.getTable().getFields()) {
            JSONObject field = JSONObject.parseObject(JSON.toJSON(f).toString());
            field.put("type", f.getType().getType());
            field.put("typePkg", f.getType().getPkg());
            fields.add(field);
        }
        data.put("table", table);

        // java
        String javaBaseDir = commonConfig.getJavaOutputDir() + "/src/main/java/" + strategyConfig.getBasePkg().replace('.', '/');
        if (StringUtils.isNotBlank(strategyConfig.getPkg())) {
            javaBaseDir = javaBaseDir + "/" + strategyConfig.getPkg().replace('.', '/');
        }

        // 生成entity
        generateFile(configuration,
            strategyConfig.getOverride(),
            "java/entity.java.ftl",
            javaBaseDir + "/" + strategyConfig.getEntityPkg().replace('.', '/'),
            strategyConfig.getEntityName() + ".java",
            data);

        // 生成dto
        generateFile(configuration,
            strategyConfig.getOverride(),
            "java/dto.java.ftl",
            javaBaseDir + "/" + strategyConfig.getDtoPkg().replace('.', '/'),
            strategyConfig.getDtoName() + ".java",
            data);

        // 生成mapper
        generateFile(configuration,
            strategyConfig.getOverride(),
            "java/mapper.java.ftl",
            javaBaseDir + "/" + strategyConfig.getMapperPkg().replace('.', '/'),
            strategyConfig.getMapperName() + ".java",
            data);

        // 生成mapper xml
        String mapperXmlDir =
            commonConfig.getJavaOutputDir()
                + "/src/main/resources/"
                + strategyConfig.getMapperXmlDir();
        if (StringUtils.isNotBlank(strategyConfig.getPkg())) {
            mapperXmlDir = mapperXmlDir + "/" + strategyConfig.getPkg().replace('.', '/');
        }
        generateFile(configuration,
            strategyConfig.getOverride(),
            "java/mapper.xml.ftl",
            mapperXmlDir,
            strategyConfig.getMapperName() + ".xml",
            data);

        // 生成service
        generateFile(configuration,
            strategyConfig.getOverride(),
            "java/service.java.ftl",
            javaBaseDir + "/" + strategyConfig.getServicePkg().replace('.', '/'),
            strategyConfig.getServiceName() + ".java",
            data);

        // 生成service impl
        generateFile(configuration,
            strategyConfig.getOverride(),
            "java/serviceImpl.java.ftl",
            javaBaseDir + "/" + strategyConfig.getServiceImplPkg().replace('.', '/'),
            strategyConfig.getServiceImplName() + ".java",
            data);

        // 生成controller
        generateFile(configuration,
            strategyConfig.getOverride(),
            "java/controller.java.ftl",
            javaBaseDir + "/" + strategyConfig.getControllerPkg().replace('.', '/'),
            strategyConfig.getControllerName() + ".java",
            data);
    }


    /**
     * 获取表信息.<p>
     *
     * @param tableName 表名
     * @return 表信息
     */
    @Override
    public TableInfo getTableInfo(DataSourceConfig dataSourceConfig, String tableName) throws SQLException {
        Connection connection = dataSourceConfig.getConnection();
        TableQuery tableQuery = dataSourceConfig.getTableQuery();
        ColumnTypeConverter columnTypeConverter = dataSourceConfig.getColumnTypeConverter();
        TableInfo tableInfo = null;
        String tableSql = String.format(tableQuery.tableSql(), tableName);
        try (PreparedStatement preparedStatement = connection.prepareStatement(tableSql);
            ResultSet results = preparedStatement.executeQuery()) {
            while (results.next()) {
                tableName = results.getString(tableQuery.tableName());
                if (StringUtils.isNotEmpty(tableName)) {
                    String tableComment = results.getString(tableQuery.tableComment());
                    tableInfo = new TableInfo();
                    tableInfo.setName(tableName);
                    tableInfo.setComment(tableComment);
                    List<TableField> fields = new ArrayList<>();
                    tableInfo.setFields(fields);
                    // TODO 支持其他数据库
                    String tableFieldsSql = String.format(tableQuery.tableFieldsSql(), tableName);
                    try (
                        PreparedStatement fieldsPreparedStatement = connection.prepareStatement(tableFieldsSql);
                        ResultSet fieldsResults = fieldsPreparedStatement.executeQuery()) {
                        while (fieldsResults.next()) {
                            TableField field = new TableField();
                            // 判断是否是主键
                            // TODO 支持其他数据库
                            String key = fieldsResults.getString(tableQuery.key());
                            field.setPk(StringUtils.isNotEmpty(tableQuery.key()) && "PRI".equals(key.toUpperCase()));
                            String fieldName = fieldsResults.getString(tableQuery.fieldName());
                            field.setColumnName(fieldName);
                            field.setColumnType(fieldsResults.getString(tableQuery.fieldType()));
                            StringBuilder propertyNameBuilder = new StringBuilder();
                            for (String split : fieldName.toLowerCase().split("_")) {
                                propertyNameBuilder.append(StringUtils.capitalFirst(split));
                            }
                            field.setName(StringUtils.lowerFirst(propertyNameBuilder.toString()));
                            field.setType(columnTypeConverter.convert(tableQuery.fieldType()));
                            field.setComment(fieldsResults.getString(tableQuery.fieldComment()));
                            fields.add(field);
                        }
                    }
                } else {
                    log.error("table[" + tableName + "]不存在");
                }
            }
        }
        return tableInfo;
    }

    /**
     * 获取数据库所有表名.<p>
     *
     * @param dataSourceConfig 数据库配置Ò
     * @return 数据库所有表名
     */
    @Override
    public List<String> getTableNames(DataSourceConfig dataSourceConfig) throws SQLException {
        List<String> tables = new ArrayList<>();
        Connection connection = dataSourceConfig.getConnection();
        TableQuery tableQuery = dataSourceConfig.getTableQuery();
        try (PreparedStatement preparedStatement = connection.prepareStatement(tableQuery.allTableSql());
            ResultSet results = preparedStatement.executeQuery()) {
            while (results.next()) {
                tables.add(results.getString(tableQuery.tableName()));
            }
        }
        return tables;
    }

    private void generateFile(Configuration configuration, Boolean override, String templateFilepath, String outputDir,
        String filename, JSONObject data) throws Exception {
        Template template = configuration.getTemplate(templateFilepath);
        new File(outputDir).mkdirs();
        File file = new File(outputDir + "/" + filename);
        if (override || !file.exists()) {
            try (Writer out = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file)))) {
                template.process(data, out);
            }
        }
    }
}
