/*
 * Copyright (c) 2019, BookRain Ltd.
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of BookRain Ltd. nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY BookRain Ltd. AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.bookrain.file.csv.builder;

import com.bookrain.core.utils.StringUtils;
import com.bookrain.file.common.annotation.DateTimeFormat;
import com.bookrain.file.csv.annotation.CsvProperty;
import com.bookrain.file.csv.context.CsvReaderContext;
import com.bookrain.file.csv.handler.CsvReaderHandler;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;
import org.apache.commons.csv.QuoteMode;

import java.io.Reader;
import java.lang.reflect.Field;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * .
 *
 * <p>
 *
 * @author Bookrain Chu
 * @version 1.0
 * @date 2019-10-16 17:15
 */
@Slf4j
public class CsvReaderBuilder {

    private CSVFormat csvFormat = CSVFormat.DEFAULT;
    private CsvReaderHandler handler;
    private Reader reader;
    private Class clazz;
    private boolean autoClose = true;

    public CsvReaderBuilder reader(Reader reader) {
        this.reader = reader;
        return this;
    }

    public CsvReaderBuilder autoClose(boolean autoClose) {
        this.autoClose = autoClose;
        return this;
    }

    public CsvReaderBuilder handler(CsvReaderHandler handler) {
        this.handler = handler;
        return this;
    }

    public CsvReaderBuilder allowDuplicateHeaderNames() {
        csvFormat = csvFormat.withAllowDuplicateHeaderNames();
        return this;
    }

    public CsvReaderBuilder allowMissingColumnNames() {
        csvFormat = csvFormat.withAllowMissingColumnNames();
        return this;
    }

    public CsvReaderBuilder autoFlush(boolean autoFlush) {
        csvFormat = csvFormat.withAutoFlush(autoFlush);
        return this;
    }

    public CsvReaderBuilder commentMarker(Character commentMarker) {
        csvFormat = csvFormat.withCommentMarker(commentMarker);
        return this;
    }

    public CsvReaderBuilder delimiter(char delimiter) {
        csvFormat = csvFormat.withDelimiter(delimiter);
        return this;
    }

    public CsvReaderBuilder escape(Character escapeCharacter) {
        csvFormat = csvFormat.withEscape(escapeCharacter);
        return this;
    }

    public CsvReaderBuilder header(String[] headers) {
        csvFormat = csvFormat.withHeader(headers);
        return this;
    }

    public CsvReaderBuilder header(Class clazz) {
        this.clazz = clazz;
        return this;
    }

    public CsvReaderBuilder ignoreEmptyLines() {
        csvFormat = csvFormat.withIgnoreEmptyLines();
        return this;
    }

    public CsvReaderBuilder ignoreHeaderCase() {
        csvFormat = csvFormat.withIgnoreHeaderCase();
        return this;
    }

    public CsvReaderBuilder ignoreSurroundingSpaces() {
        csvFormat = csvFormat.withIgnoreSurroundingSpaces();
        return this;
    }

    public CsvReaderBuilder nullString(String nullString) {
        csvFormat = csvFormat.withNullString(nullString);
        return this;
    }

    public CsvReaderBuilder quote(Character quoteCharacter) {
        csvFormat = csvFormat.withQuote(quoteCharacter);
        return this;
    }

    public CsvReaderBuilder quoteMode(QuoteMode quoteMode) {
        csvFormat = csvFormat.withQuoteMode(quoteMode);
        return this;
    }

    public CsvReaderBuilder recordSeparator(char recordSeparator) {
        csvFormat = csvFormat.withRecordSeparator(recordSeparator);
        return this;
    }

    public CsvReaderBuilder skipHeaderRecord() {
        csvFormat = csvFormat.withSkipHeaderRecord();
        if (csvFormat.getHeader() == null) {
            csvFormat.withHeader(new String[]{});
        }
        return this;
    }

    public CsvReaderBuilder trailingDelimiter() {
        csvFormat = csvFormat.withTrailingDelimiter();
        return this;
    }

    public CsvReaderBuilder trim() {
        csvFormat = csvFormat.withTrim();
        return this;
    }

    public void doRead() throws Exception {
        doRead(handler);
    }

    @SuppressWarnings("unchecked")
    public void doRead(CsvReaderHandler handler) throws Exception {
        try {
            CSVParser csvParser = new CSVParser(reader, csvFormat);
            for (CSVRecord record : csvParser.getRecords()) {
                CsvReaderContext context = new CsvReaderContext(csvFormat, reader);
                context.setComment(record.getComment());
                context.setRowIndex(record.getRecordNumber() - 1);
                Object row;
                // 有实体类
                if (this.clazz != null) {
                    row = this.clazz.newInstance();
                    Field[] fields = this.clazz.getDeclaredFields();
                    for (Field field : fields) {
                        field.setAccessible(true);
                        CsvProperty csvProperty = field.getAnnotation(CsvProperty.class);
                        if (csvProperty != null) {
                            // 索引溢出
                            if (csvProperty.index() >= record.size()) {
                                throw new ArrayIndexOutOfBoundsException("index is out of range [" + record.size() + "]");
                            } else {
                                // 进行数据转换
                                Object val = csvProperty.converter().newInstance().convertToJavaData(record.get(csvProperty.index()));
                                DateTimeFormat dtf = field.getAnnotation(DateTimeFormat.class);

                                // 判断是否需要进行时间转换
                                if (dtf != null && StringUtils.isNotBlank(dtf.value())
                                        && field.getType() == Date.class
                                        && StringUtils.isNotBlank(String.valueOf(val))) {
                                    field.set(row, new SimpleDateFormat(dtf.value()).parse(String.valueOf(val)));
                                } else {
                                    field.set(row, val);
                                }
                            }
                        }
                    }
                } else {
                    row = new HashMap<>();
                    for (int i = 0; i < record.size(); i++) {
                        ((Map) row).put(i, record.get(i));
                    }
                }
                context.setRow(row);
                handler.before(context);
                try {
                    handler.handle(row, context);
                } catch (Exception e) {
                    handler.excetpion(e, context);
                }
                handler.after(context);
            }
        } finally {
            if (autoClose) {
                reader.close();
            }
        }
    }
}
