/*
 * Copyright (c) 2019, BookRain Ltd.
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of BookRain Ltd. nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY BookRain Ltd. AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.bookrain.file.csv.builder;

import com.bookrain.core.utils.StringUtils;
import com.bookrain.file.csv.annotation.CsvProperty;
import com.bookrain.file.common.annotation.DateTimeFormat;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVPrinter;
import org.apache.commons.csv.QuoteMode;
import org.ehcache.spi.serialization.UnsupportedTypeException;

import java.io.Writer;
import java.lang.reflect.Field;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * .
 *
 * <p>
 *
 * @author Bookrain Chu
 * @version 1.0
 * @date 2019-10-17 09:05
 */
@Slf4j
public class CsvWriterBuilder {
    private CSVFormat csvFormat = CSVFormat.DEFAULT;
    private Class clazz;
    private Writer writer;
    private boolean autoClose = true;

    public CsvWriterBuilder writer(Writer writer) {
        this.writer = writer;
        return this;
    }

    public CsvWriterBuilder autoClose(boolean autoClose) {
        this.autoClose = autoClose;
        return this;
    }

    public CsvWriterBuilder allowDuplicateHeaderNames() {
        csvFormat = csvFormat.withAllowDuplicateHeaderNames();
        return this;
    }

    public CsvWriterBuilder allowMissingColumnNames() {
        csvFormat = csvFormat.withAllowMissingColumnNames();
        return this;
    }

    public CsvWriterBuilder autoFlush(boolean autoFlush) {
        csvFormat = csvFormat.withAutoFlush(autoFlush);
        return this;
    }

    public CsvWriterBuilder commentMarker(Character commentMarker) {
        csvFormat = csvFormat.withCommentMarker(commentMarker);
        return this;
    }

    public CsvWriterBuilder delimiter(char delimiter) {
        csvFormat = csvFormat.withDelimiter(delimiter);
        return this;
    }

    public CsvWriterBuilder escape(Character escapeCharacter) {
        csvFormat = csvFormat.withEscape(escapeCharacter);
        return this;
    }

    public CsvWriterBuilder header(String[] headers) {
        csvFormat = csvFormat.withHeader(headers);
        return this;
    }

    public CsvWriterBuilder header(Class clazz) {
        this.clazz = clazz;
        return this;
    }

    public CsvWriterBuilder ignoreEmptyLines() {
        csvFormat = csvFormat.withIgnoreEmptyLines();
        return this;
    }

    public CsvWriterBuilder ignoreHeaderCase() {
        csvFormat = csvFormat.withIgnoreHeaderCase();
        return this;
    }

    public CsvWriterBuilder ignoreSurroundingSpaces() {
        csvFormat = csvFormat.withIgnoreSurroundingSpaces();
        return this;
    }

    public CsvWriterBuilder nullString(String nullString) {
        csvFormat = csvFormat.withNullString(nullString);
        return this;
    }

    public CsvWriterBuilder quote(Character quoteCharacter) {
        csvFormat = csvFormat.withQuote(quoteCharacter);
        return this;
    }

    public CsvWriterBuilder quoteMode(QuoteMode quoteMode) {
        csvFormat = csvFormat.withQuoteMode(quoteMode);
        return this;
    }

    public CsvWriterBuilder recordSeparator(char recordSeparator) {
        csvFormat = csvFormat.withRecordSeparator(recordSeparator);
        return this;
    }

    public CsvWriterBuilder skipHeaderRecord() {
        csvFormat = csvFormat.withSkipHeaderRecord();
        if (csvFormat.getHeader() == null) {
            csvFormat.withHeader(new String[]{});
        }
        return this;
    }

    public CsvWriterBuilder trailingDelimiter() {
        csvFormat = csvFormat.withTrailingDelimiter();
        return this;
    }

    public CsvWriterBuilder trim() {
        csvFormat = csvFormat.withTrim();
        return this;
    }

    @SuppressWarnings("unchecked")
    public void doWrite(List data) throws Exception {
        try {
            CSVPrinter printer = new CSVPrinter(writer, csvFormat);
            for (Object record : data) {
                List<String> row = new ArrayList<>();
                if (this.clazz != null && record.getClass() == this.clazz) {
                    Field[] fields = this.clazz.getDeclaredFields();
                    for (Field field : fields) {
                        field.setAccessible(true);
                        CsvProperty csvProperty = field.getAnnotation(CsvProperty.class);
                        if (csvProperty != null) {
                            int index = csvProperty.index();
                            Object val = csvProperty.converter().newInstance().convertToFileData(field.get(record));
                            DateTimeFormat dtf = field.getAnnotation(DateTimeFormat.class);
                            if (dtf != null && StringUtils.isNotBlank(dtf.value())) {
                                if (field.getType() == Date.class) {
                                    row.add(index, new SimpleDateFormat(dtf.value()).format(val));
                                }
                            } else {
                                row.add(index, String.valueOf(val));
                            }
                        }
                    }
                    printer.printRecord(row);
                } else if (record instanceof List) {
                    printer.printRecord((List) record);
                } else {
                    throw new UnsupportedTypeException(record.getClass().getCanonicalName());
                }
            }
        } finally {
            if (autoClose) {
                writer.close();
            }
        }
    }
}
