/*
 * Copyright (c) 2019, BookRain Ltd.
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of BookRain Ltd. nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY BookRain Ltd. AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.bookrain.file.txt.builder;

import com.bookrain.core.utils.StringUtils;
import com.bookrain.file.common.annotation.DateTimeFormat;
import com.bookrain.file.txt.annotation.TxtProperty;
import com.bookrain.file.txt.context.TxtReaderContext;
import com.bookrain.file.txt.handler.TxtReaderHandler;

import java.io.BufferedReader;
import java.io.Reader;
import java.lang.reflect.Field;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * .
 *
 * <p>
 *
 * @author Bookrain Chu
 * @version 1.0
 * @date 2019-10-18 09:08
 */
public class TxtReaderBuilder {
    private String seperator;
    private Reader reader;
    private boolean autoClose;
    private Class clazz;
    private TxtReaderHandler handler;
    private boolean isHeader;
    private List<String> headers = new ArrayList<>();

    public TxtReaderBuilder seperator(String seperator) {
        this.seperator = seperator;
        return this;
    }

    public TxtReaderBuilder reader(Reader reader) {
        this.reader = reader;
        return this;
    }

    public TxtReaderBuilder handler(TxtReaderHandler handler) {
        this.handler = handler;
        return this;
    }

    public TxtReaderBuilder header(Class clazz) {
        this.clazz = clazz;
        return this;
    }

    public TxtReaderBuilder header(List<String> headers) {
        this.headers = headers;
        return this;
    }

    public TxtReaderBuilder isHeader(boolean header) {
        this.isHeader = isHeader;
        return this;
    }

    public TxtReaderBuilder autoClose(boolean autoClose) {
        this.autoClose = autoClose;
        return this;
    }


    @SuppressWarnings("unchecked")
    public void doRead(TxtReaderHandler handler) throws Exception {
        if (this.reader != null) {
            try {
                BufferedReader br = new BufferedReader(reader);
                String line;
                int rowIndex = -1;
                while ((line = br.readLine()) != null) {
                    TxtReaderContext context = new TxtReaderContext();
                    // 判断是否是第一行，且是否有头部
                    if (isHeader && rowIndex < 0) {
                        String[] headers = line.split(seperator);
                        if (headers.length > 0) {
                            context.getHeaders().addAll(Arrays.asList(headers));
                            this.headers = Arrays.asList(headers);
                        } else if (this.headers.size() > 0) {
                            context.getHeaders().addAll(this.headers);
                        }
                    } else {
                        rowIndex++;
                        context.setRawData(line);
                        context.setRowIndex(rowIndex);
                        String[] cols = line.split(seperator);
                        Object row;
                        // 如果没有设置实体对象，数据封装到map中
                        if (this.clazz == null) {
                            row = new HashMap<>();
                            for (int i = 0; i < cols.length; i++) {
                                ((Map) row).put(i, cols[i]);
                            }
                            context.setRow(row);
                        } else {
                            row = this.clazz.newInstance();
                            Field[] fields = this.clazz.getDeclaredFields();
                            for (Field field : fields) {
                                field.setAccessible(true);
                                TxtProperty txtProperty = field.getAnnotation(TxtProperty.class);
                                if (txtProperty != null) {
                                    int index = txtProperty.index();
                                    // 列索引溢出
                                    if (index >= cols.length) {
                                        throw new ArrayIndexOutOfBoundsException("index is out of range [" + cols.length + "]");
                                    } else {
                                        // 进行数据转换
                                        Object val = txtProperty.converter().newInstance().convertToJavaData(cols[index]);
                                        // 判断是否需要进行时间类型转换
                                        DateTimeFormat dtf = field.getAnnotation(DateTimeFormat.class);
                                        if (dtf != null && StringUtils.isNotBlank(dtf.value())
                                                && field.getType() == Date.class
                                                && StringUtils.isNotBlank(String.valueOf(val))) {
                                            field.set(row, new SimpleDateFormat(dtf.value()).parse(String.valueOf(val)));
                                        } else {
                                            field.set(row, val);
                                        }
                                    }
                                }
                            }
                            context.setRow(row);
                        }

                        // 处理数据
                        handler.before(context);
                        try {
                            handler.handle(row, context);
                        } catch (Exception e) {
                            handler.excetpion(e, context);
                        }
                        handler.after(context);
                    }
                }
            } catch (Exception e) {
                throw e;
            } finally {
                if (autoClose) {
                    this.reader.close();
                }
            }
        } else {
            throw new NullPointerException("reader is null");
        }
    }
}
