/*
 * Copyright (c) 2019, BookRain Ltd.
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of BookRain Ltd. nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY BookRain Ltd. AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.bookrain.file.txt.builder;

import com.bookrain.core.utils.StringUtils;
import com.bookrain.file.common.annotation.DateTimeFormat;
import com.bookrain.file.txt.annotation.TxtProperty;
import com.bookrain.file.txt.handler.TxtReaderHandler;
import com.google.common.base.Joiner;
import org.ehcache.spi.serialization.UnsupportedTypeException;

import java.io.Writer;
import java.lang.reflect.Field;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * .
 *
 * <p>
 *
 * @author Bookrain Chu
 * @version 1.0
 * @date 2019-10-18 09:08
 */
public class TxtWriterBuilder {
    private String seperator = ",";
    private Writer writer;
    private boolean autoClose;
    private Class clazz;
    private TxtReaderHandler handler;
    private boolean isHeader;
    private List<String> headers = new ArrayList<>();

    public TxtWriterBuilder seperator(String seperator) {
        this.seperator = seperator;
        return this;
    }

    public TxtWriterBuilder writer(Writer writer) {
        this.writer = writer;
        return this;
    }

    public TxtWriterBuilder handler(TxtReaderHandler handler) {
        this.handler = handler;
        return this;
    }

    public TxtWriterBuilder header(Class clazz) {
        this.clazz = clazz;
        return this;
    }

    public TxtWriterBuilder header(List<String> headers) {
        this.headers = headers;
        return this;
    }

    public TxtWriterBuilder isHeader(boolean header) {
        this.isHeader = isHeader;
        return this;
    }

    public TxtWriterBuilder autoClose(boolean autoClose) {
        this.autoClose = autoClose;
        return this;
    }


    @SuppressWarnings("unchecked")
    public void doWrite(List data) throws Exception {
        if (this.writer != null) {
            try {
//                BufferedWriter bw = new BufferedWriter(writer);
                if (isHeader && headers.size() > 0) {
                    writer.write(Joiner.on(seperator).useForNull("").join(this.headers));
                } else {
                    for (int i = 0; i < data.size(); i++) {
                        if (this.clazz != null && data.get(i).getClass() == clazz) {
                            List row = new ArrayList<>();
                            Field[] fields = this.clazz.getDeclaredFields();
                            for (Field field : fields) {
                                TxtProperty txtProperty = field.getAnnotation(TxtProperty.class);
                                if (txtProperty != null) {
                                    int index = txtProperty.index();
                                    Object val = txtProperty.converter().newInstance().convertToFileData(field.get(data.get(i)));
                                    DateTimeFormat dtf = field.getAnnotation(DateTimeFormat.class);
                                    if (dtf != null && StringUtils.isNotBlank(dtf.value())
                                            && field.getType() == Date.class && val != null) {
                                        row.add(index, new SimpleDateFormat(dtf.value()).format(val));
                                    } else {
                                        row.add(index, val);
                                    }
                                }
                            }
                            writer.write(Joiner.on(seperator).useForNull("").join(row));
                        } else if (data.get(i) instanceof List) {
                            writer.write(Joiner.on(seperator).useForNull("").join((List) data.get(i)));
                            if (i < data.size() - 1) {
                                writer.write(System.getProperty("line.separator"));
                            }
                        } else {
                            throw new UnsupportedTypeException("data type is not supported");
                        }
                    }
                }
            } catch (Exception e) {
                throw e;
            } finally {
                if (autoClose) {
                    this.writer.close();
                }
            }
        } else {
            throw new NullPointerException();
        }
    }
}
