/*
 * Copyright (c) 2019, BookRain Ltd.
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of BookRain Ltd. nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY BookRain Ltd. AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.bookrain.qrcode.generator;

import static java.awt.SystemColor.info;

import com.bookrain.qrcode.properties.QrCodeProperties;
import com.bookrain.qrcode.properties.QrCodeProperties.Desc;
import com.bookrain.qrcode.properties.QrCodeProperties.Logo;
import com.bookrain.qrcode.properties.QrCodeProperties.Point;
import com.google.zxing.BarcodeFormat;
import com.google.zxing.EncodeHintType;
import com.google.zxing.MultiFormatWriter;
import com.google.zxing.WriterException;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.qrcode.decoder.ErrorCorrectionLevel;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.imageio.ImageIO;

/**
 * .
 *
 * <p>
 *
 * @author Bookrain Chu
 * @version 1.0
 * @date 2019-07-08 00:42
 */
public class QrCodeGenerator {

    /**
     * 仅生成二维码图片.<p>
     * @param qrCodeProperties 属性
     * @param output 输出流
     * @throws Exception 异常
     */
    public static void drawQrCodeImage(QrCodeProperties qrCodeProperties, OutputStream output) throws Exception {
        BufferedImage qrCodeImage = drawRawQrCodeImage(qrCodeProperties);
        int width = qrCodeImage.getWidth();
        int height = qrCodeImage.getHeight();
        int widthWithoutMargin = qrCodeProperties.getBottomRight().getX() - qrCodeProperties.getTopLeft().getX();
        if (qrCodeProperties.getShowLogo()) {
            Logo logo = qrCodeProperties.getLogo();
            int logoWidth;
            int logoHeight;
            if (qrCodeProperties.getLogo().getSize() == null) {
                logoWidth = (int) (qrCodeProperties.getSize().getWidth() * logo.getRatio());
                logoHeight = (int) (qrCodeProperties.getSize().getHeight() * logo.getRatio());
            } else {
                logoWidth = logo.getSize().getWidth();
                logoHeight = logo.getSize().getHeight();
            }

            int x = (width - logoWidth) / 2;
            int y = (height - logoHeight) / 2;
            qrCodeImage.getGraphics().drawImage(logo.getLogo(), x, y, logoWidth, logoHeight, null);
        }

        if (qrCodeProperties.getShowDesc()) {
            // 原始二维码的底部留白
            Desc desc = qrCodeProperties.getDesc();
            int fontHeight = qrCodeImage.getGraphics().getFontMetrics(desc.getFont()).getHeight();
            List<String> lines = new ArrayList<>();
            int lineNo = 1;
            int lineWidth = 0;
            String line = "";
            for (int i = 0; i < desc.getDesc().length(); i++) {
                char c = desc.getDesc().charAt(i);
                int charWidth = qrCodeImage.getGraphics().getFontMetrics(desc.getFont()).charWidth(c);
                if (lineWidth + charWidth > widthWithoutMargin) {
                    lineNo++;
                    lines.add(line);
                    line = "";
                    lineWidth = 0;
                    continue;
                }
                lineWidth += charWidth;
                line += c;
            }
            if (!"".equals(line)) {
                lines.add(line);
            }
            int descHeight = fontHeight * lineNo;
            int descBufferedImageHeight = height + descHeight + +desc.getMargin() - (height - qrCodeProperties.getBottomRight().getY());
            BufferedImage descBufferedImage = new BufferedImage(width, descBufferedImageHeight, BufferedImage.TYPE_INT_RGB);
            Graphics graphics = descBufferedImage.getGraphics();
            int finalDescHeight = descHeight + desc.getMargin() - (height - qrCodeProperties.getBottomRight().getY());
            if (finalDescHeight > 0) {
                graphics.setColor(Color.WHITE);
                graphics.fillRect(0, height, width, finalDescHeight);
            }
            graphics.setColor(Color.black);
            graphics.setFont(desc.getFont());
            graphics.drawImage(qrCodeImage, 0, 0, null);

            for (int i = 0; i < lines.size(); i++) {
                lineWidth = graphics.getFontMetrics(desc.getFont()).stringWidth(lines.get(i));
                int startX = qrCodeProperties.getTopLeft().getX();
                if (desc.getCenter()) {
                    startX = (width - lineWidth) / 2;
                }
                graphics.drawString(lines.get(i), startX, qrCodeProperties.getBottomRight().getY() + 1 + graphics.getFontMetrics().getAscent() + fontHeight * i + desc.getMargin());
            }
            graphics.dispose();
            qrCodeImage = descBufferedImage;
        }
        ImageIO.write(qrCodeImage, qrCodeProperties.getFileFormat(), output);
    }


    /**
     * 仅生成二维码图片.<p>
     * @param qrCodeProperties 属性
     * @param qrCodeFile 文件
     * @throws Exception 异常
     */
    public static void drawQrCodeImage(QrCodeProperties qrCodeProperties, File qrCodeFile) throws Exception {
        drawQrCodeImage(qrCodeProperties, new FileOutputStream(qrCodeFile));
    }

    /**
     * 仅生成二维码图片.<p>
     * @param qrCodeProperties 属性
     * @param filePath 文件路径
     * @throws Exception 异常
     */
    public static void drawQrCodeImage(QrCodeProperties qrCodeProperties, String filePath) throws Exception {
        drawQrCodeImage(qrCodeProperties, new File(filePath));
    }

    /**
     * 仅生成二维码图片.<p>
     * @param qrCodeProperties 属性
     * @throws Exception 异常
     */
    private static BufferedImage drawRawQrCodeImage(QrCodeProperties qrCodeProperties) throws Exception {
        String contents = qrCodeProperties.getContent();
        Map<EncodeHintType, Object> hint = new HashMap<>();
        hint.put(EncodeHintType.ERROR_CORRECTION, ErrorCorrectionLevel.H);
        hint.put(EncodeHintType.CHARACTER_SET, qrCodeProperties.getCharSet());
        hint.put(EncodeHintType.MARGIN, qrCodeProperties.getMargin());
        MultiFormatWriter writer = new MultiFormatWriter();
        BitMatrix bm = writer.encode(contents, BarcodeFormat.QR_CODE, qrCodeProperties.getSize().getWidth(), qrCodeProperties.getSize().getHeight(), hint);
        int[] topLeft = bm.getTopLeftOnBit();
        int[] bottomRight = bm.getBottomRightOnBit();
        qrCodeProperties.setTopLeft(new Point().setX(topLeft[0]).setY(topLeft[1]));
        qrCodeProperties.setBottomRight(new Point().setX(bottomRight[0]).setY(bottomRight[1]));
        BufferedImage img = new BufferedImage(bm.getWidth(), bm.getHeight(), BufferedImage.TYPE_INT_RGB);
        for (int x = 0; x < bm.getWidth(); x++) {
            for (int y = 0; y < bm.getHeight(); y++) {
                img.setRGB(x, y, bm.get(x, y) ? Color.BLACK.getRGB() : Color.WHITE.getRGB());
            }
        }
        return img;
    }
}
