/*
 * Copyright (c) 2019, BookRain Ltd.
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of BookRain Ltd. nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY BookRain Ltd. AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.bookrain.wechat.pay.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.bookrain.wechat.pay.constants.WxPayConstants;
import com.bookrain.wechat.pay.constants.WxPayConstants.SignType;
import com.bookrain.wechat.pay.model.WXPayOrderQueryRequest;
import com.bookrain.wechat.pay.model.WxPayCloseOrderRequest;
import com.bookrain.wechat.pay.model.WxPayDownloadBillRequest;
import com.bookrain.wechat.pay.model.WxPayMicroPayRequest;
import com.bookrain.wechat.pay.model.WxPayRefundQueryRequest;
import com.bookrain.wechat.pay.model.WxPayRefundRequest;
import com.bookrain.wechat.pay.model.WxPayReverseRequest;
import com.bookrain.wechat.pay.model.WxPayShortUrlRequest;
import com.bookrain.wechat.pay.model.WxPayUnifiedOrderRequest;
import com.bookrain.wechat.properties.WeChatProperties;
import com.bookrain.wechat.pay.rest.WxPayRestTemplateBuilder;
import com.bookrain.wechat.pay.utils.WxPayUtil;
import com.bookrain.wechat.pay.model.WxPayAuthCodeToOpenIdRequest;
import com.bookrain.wechat.pay.model.WxPayReportRequest;
import com.bookrain.wechat.pay.service.WxPayService;
import java.util.HashMap;
import java.util.Map;
import org.springframework.stereotype.Service;

/**
 * .
 *
 * <p>
 *
 * @author Bookrain Chu
 * @version 1.0
 * @date 2019-07-25 14:37
 */
@Service
public class WxPayServiceImpl implements WxPayService {

    private static final Integer DEFAULT_TIMEOUT = 60000;
    private Boolean useSandbox;
    private String appId;
    private String appSecret;
    private String mchID;
    private SignType signType;
    private String key;
    private final WxPayRestTemplateBuilder wxPayRestTemplateBuilder;

    public WxPayServiceImpl(WeChatProperties weChatProperties, WxPayRestTemplateBuilder wxPayRestTemplateBuilder) {
        this.useSandbox = weChatProperties.getUseSandbox();
        this.appId = weChatProperties.getAppID();
        this.appSecret = weChatProperties.getAppSecret();
        this.mchID = weChatProperties.getMchID();
        this.key = weChatProperties.getKey();
        try {
            this.signType = SignType.valueOf(weChatProperties.getSignType());
        } catch (Exception e) {
            this.signType = SignType.MD5;
        }

        this.wxPayRestTemplateBuilder = wxPayRestTemplateBuilder;
    }

    /**
     * 作用：提交刷卡支付<br> 场景：刷卡支付
     *
     * @param microPayRequest 向wxpay post的请求数据
     * @return API返回数据
     */
    @Override
    public Map<String, String> microPay(WxPayMicroPayRequest microPayRequest) throws Exception {
        return this.microPay(microPayRequest, DEFAULT_TIMEOUT);
    }

    /**
     * 作用：提交刷卡支付<br> 场景：刷卡支付
     *
     * @param microPayRequest 向wxpay post的请求数据
     * @param timeout 超时时间
     * @return API返回数据
     */
    @Override
    public Map<String, String> microPay(WxPayMicroPayRequest microPayRequest, int timeout) throws Exception {
        String url;
        if (this.useSandbox) {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.SANDBOX_MICROPAY_URL_SUFFIX;
        } else {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.MICROPAY_URL_SUFFIX;
        }
        String request = WxPayUtil.mapToXml(this.fillRequestData((Map<String, String>) JSONObject.toJSON(microPayRequest)));
        String respXml = wxPayRestTemplateBuilder.createRestTemplate(timeout).postForObject(url, request, String.class);
        return this.processResponseXml(respXml);
    }

    /**
     * 作用：统一下单<br> 场景：公共号支付、扫码支付、APP支付
     *
     * @param unifiedOrderRequest 向wxpay post的请求数据
     * @return API返回数据
     */
    @Override
    public Map<String, String> unifiedOrder(WxPayUnifiedOrderRequest unifiedOrderRequest) throws Exception {
        return this.unifiedOrder(unifiedOrderRequest, DEFAULT_TIMEOUT);
    }

    /**
     * 作用：统一下单<br> 场景：公共号支付、扫码支付、APP支付
     *
     * @param unifiedOrderRequest 向wxpay post的请求数据
     * @param timeout 超时时间
     * @return API返回数据
     */
    @Override
    public Map<String, String> unifiedOrder(WxPayUnifiedOrderRequest unifiedOrderRequest, int timeout) throws Exception {
        String url;
        if (this.useSandbox) {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.SANDBOX_UNIFIEDORDER_URL_SUFFIX;
        } else {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.UNIFIEDORDER_URL_SUFFIX;
        }
        String request = WxPayUtil.mapToXml(this.fillRequestData((Map<String, String>) JSONObject.toJSON(unifiedOrderRequest)));
        String respXml = wxPayRestTemplateBuilder.createRestTemplate(timeout).postForObject(url, request, String.class);
        return this.processResponseXml(respXml);
    }

    /**
     * 作用：查询订单<br> 场景：刷卡支付、公共号支付、扫码支付、APP支付
     *
     * @param orderQueryRequest 向wxpay post的请求数据
     * @return API返回数据
     */
    @Override
    public Map<String, String> orderQuery(WXPayOrderQueryRequest orderQueryRequest) throws Exception {
        return this.orderQuery(orderQueryRequest, DEFAULT_TIMEOUT);
    }

    /**
     * 作用：查询订单<br> 场景：刷卡支付、公共号支付、扫码支付、APP支付
     *
     * @param orderQueryRequest 向wxpay post的请求数据 int
     * @param timeout 超时时间
     * @return API返回数据
     */
    @Override
    public Map<String, String> orderQuery(WXPayOrderQueryRequest orderQueryRequest, int timeout) throws Exception {
        String url;
        if (this.useSandbox) {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.SANDBOX_ORDERQUERY_URL_SUFFIX;
        } else {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.ORDERQUERY_URL_SUFFIX;
        }
        String request = WxPayUtil.mapToXml(this.fillRequestData((Map<String, String>) JSONObject.toJSON(orderQueryRequest)));
        String respXml = wxPayRestTemplateBuilder.createRestTemplate(timeout).postForObject(url, request, String.class);
        return this.processResponseXml(respXml);
    }

    /**
     * 作用：撤销订单<br> 场景：刷卡支付
     *
     * @param payReverseRequest 向wxpay post的请求数据
     * @return API返回数据
     */
    @Override
    public Map<String, String> reverse(WxPayReverseRequest payReverseRequest) throws Exception {
        return this.reverse(payReverseRequest, DEFAULT_TIMEOUT);
    }

    /**
     * 作用：撤销订单<br> 场景：刷卡支付<br> 其他：需要证书
     *
     * @param payReverseRequest 向wxpay post的请求数据
     * @param timeout 超时时间
     * @return API返回数据
     */
    @Override
    public Map<String, String> reverse(WxPayReverseRequest payReverseRequest, int timeout) throws Exception {
        String url;
        if (this.useSandbox) {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.SANDBOX_REVERSE_URL_SUFFIX;
        } else {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.REVERSE_URL_SUFFIX;
        }
        String request = WxPayUtil.mapToXml(this.fillRequestData((Map<String, String>) JSONObject.toJSON(payReverseRequest)));
        String respXml = wxPayRestTemplateBuilder.createRestTemplate(timeout).postForObject(url, request, String.class);
        return this.processResponseXml(respXml);
    }

    /**
     * 作用：关闭订单<br> 场景：公共号支付、扫码支付、APP支付
     *
     * @param closeOrderRequest 向wxpay post的请求数据
     * @return API返回数据
     */
    @Override
    public Map<String, String> closeOrder(WxPayCloseOrderRequest closeOrderRequest) throws Exception {
        return this.closeOrder(closeOrderRequest, DEFAULT_TIMEOUT);
    }

    /**
     * 作用：关闭订单<br> 场景：公共号支付、扫码支付、APP支付
     *
     * @param closeOrderRequest 向wxpay post的请求数据
     * @param timeout 超时时间
     * @return API返回数据
     */
    @Override
    public Map<String, String> closeOrder(WxPayCloseOrderRequest closeOrderRequest, int timeout) throws Exception {
        String url;
        if (this.useSandbox) {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.SANDBOX_CLOSEORDER_URL_SUFFIX;
        } else {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.CLOSEORDER_URL_SUFFIX;
        }
        String request = WxPayUtil.mapToXml(this.fillRequestData((Map<String, String>) JSONObject.toJSON(closeOrderRequest)));
        String respXml = wxPayRestTemplateBuilder.createRestTemplate(timeout).postForObject(url, request, String.class);
        return this.processResponseXml(respXml);
    }

    /**
     * 作用：申请退款<br> 场景：刷卡支付、公共号支付、扫码支付、APP支付
     *
     * @param refundRequest 向wxpay post的请求数据
     * @return API返回数据
     */
    @Override
    public Map<String, String> refund(WxPayRefundRequest refundRequest) throws Exception {
        return this.refund(refundRequest, DEFAULT_TIMEOUT);
    }

    /**
     * 作用：申请退款<br> 场景：刷卡支付、公共号支付、扫码支付、APP支付<br> 其他：需要证书
     *
     * @param refundRequest 向wxpay post的请求数据
     * @param timeout 超时时间
     * @return API返回数据
     */
    @Override
    public Map<String, String> refund(WxPayRefundRequest refundRequest, int timeout) throws Exception {
        String url;
        if (this.useSandbox) {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.SANDBOX_REFUND_URL_SUFFIX;
        } else {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.REFUND_URL_SUFFIX;
        }
        String request = WxPayUtil.mapToXml(this.fillRequestData((Map<String, String>) JSONObject.toJSON(refundRequest)));
        String respXml = wxPayRestTemplateBuilder.createRestTemplate(true, timeout).postForObject(url, request, String.class);
        return this.processResponseXml(respXml);
    }

    /**
     * 作用：退款查询<br> 场景：刷卡支付、公共号支付、扫码支付、APP支付
     *
     * @param refundQueryRequest 向wxpay post的请求数据
     * @return API返回数据
     */
    @Override
    public Map<String, String> refundQuery(WxPayRefundQueryRequest refundQueryRequest) throws Exception {
        return this.refundQuery(refundQueryRequest, DEFAULT_TIMEOUT);
    }

    /**
     * 作用：退款查询<br> 场景：刷卡支付、公共号支付、扫码支付、APP支付
     *
     * @param refundQueryRequest 向wxpay post的请求数据
     * @param timeout 超时时间
     * @return API返回数据
     */
    @Override
    public Map<String, String> refundQuery(WxPayRefundQueryRequest refundQueryRequest, int timeout) throws Exception {
        String url;
        if (this.useSandbox) {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.SANDBOX_REFUNDQUERY_URL_SUFFIX;
        } else {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.REFUNDQUERY_URL_SUFFIX;
        }
        String request = WxPayUtil.mapToXml(this.fillRequestData((Map<String, String>) JSONObject.toJSON(refundQueryRequest)));
        String respXml = wxPayRestTemplateBuilder.createRestTemplate(timeout).postForObject(url, request, String.class);
        return this.processResponseXml(respXml);
    }

    /**
     * 作用：对账单下载（成功时返回对账单数据，失败时返回XML格式数据）<br> 场景：刷卡支付、公共号支付、扫码支付、APP支付
     *
     * @param downloadBillRequest 向wxpay post的请求数据
     * @return API返回数据
     */
    @Override
    public Map<String, String> downloadBill(WxPayDownloadBillRequest downloadBillRequest) throws Exception {
        return this.downloadBill(downloadBillRequest, DEFAULT_TIMEOUT);
    }

    /**
     * 作用：对账单下载<br> 场景：刷卡支付、公共号支付、扫码支付、APP支付<br> 其他：无论是否成功都返回Map。若成功，返回的Map中含有return_code、return_msg、data， 其中return_code为`SUCCESS`，data为对账单数据。
     *
     * @param downloadBillRequest 向wxpay post的请求数据
     * @param timeout 超时时间
     * @return 经过封装的API返回数据
     */
    @Override
    public Map<String, String> downloadBill(WxPayDownloadBillRequest downloadBillRequest, int timeout) throws Exception {
        String url;
        if (this.useSandbox) {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.SANDBOX_DOWNLOADBILL_URL_SUFFIX;
        } else {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.DOWNLOADBILL_URL_SUFFIX;
        }
        String request = WxPayUtil.mapToXml(this.fillRequestData((Map<String, String>) JSONObject.toJSON(downloadBillRequest)));
        String respStr = wxPayRestTemplateBuilder.createRestTemplate(timeout).postForObject(url, request, String.class);
        Map<String, String> ret;
        // 出现错误，返回XML数据
        if (respStr.indexOf("<") == 0) {
            ret = WxPayUtil.xmlToMap(respStr);
        } else {
            // 正常返回csv数据
            ret = new HashMap<String, String>();
            ret.put("return_code", WxPayConstants.SUCCESS);
            ret.put("return_msg", "ok");
            ret.put("data", respStr);
        }
        return ret;
    }

    /**
     * 作用：交易保障<br> 场景：刷卡支付、公共号支付、扫码支付、APP支付
     *
     * @param reportRequest 向wxpay post的请求数据
     * @return API返回数据
     */
    @Override
    public Map<String, String> report(WxPayReportRequest reportRequest) throws Exception {
        return this.report(reportRequest, DEFAULT_TIMEOUT);
    }

    /**
     * 作用：交易保障<br> 场景：刷卡支付、公共号支付、扫码支付、APP支付
     *
     * @param reportRequest 向wxpay post的请求数据
     * @param timeout 超时时间
     * @return API返回数据
     */
    @Override
    public Map<String, String> report(WxPayReportRequest reportRequest, int timeout) throws Exception {
        String url;
        if (this.useSandbox) {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.SANDBOX_REPORT_URL_SUFFIX;
        } else {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.REPORT_URL_SUFFIX;
        }
        String request = WxPayUtil.mapToXml(this.fillRequestData((Map<String, String>) JSONObject.toJSON(reportRequest)));
        String respXml = wxPayRestTemplateBuilder.createRestTemplate(timeout).postForObject(url, request, String.class);
        return WxPayUtil.xmlToMap(respXml);
    }

    /**
     * 作用：转换短链接<br> 场景：刷卡支付、扫码支付
     *
     * @param shortUrlRequest 向wxpay post的请求数据
     * @return API返回数据
     */
    @Override
    public Map<String, String> shortUrl(WxPayShortUrlRequest shortUrlRequest) throws Exception {
        return this.shortUrl(shortUrlRequest, DEFAULT_TIMEOUT);
    }

    /**
     * 作用：转换短链接<br> 场景：刷卡支付、扫码支付
     *
     * @param shortUrlRequest 向wxpay post的请求数据
     * @param timeout 超时时间
     * @return API返回数据
     */
    @Override
    public Map<String, String> shortUrl(WxPayShortUrlRequest shortUrlRequest, int timeout) throws Exception {
        String url;
        if (this.useSandbox) {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.SANDBOX_SHORTURL_URL_SUFFIX;
        } else {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.SHORTURL_URL_SUFFIX;
        }
        String request = WxPayUtil.mapToXml(this.fillRequestData((Map<String, String>) JSONObject.toJSON(shortUrlRequest)));
        String respXml = wxPayRestTemplateBuilder.createRestTemplate(timeout).postForObject(url, request, String.class);
        return this.processResponseXml(respXml);
    }

    /**
     * 作用：授权码查询OPENID接口<br> 场景：刷卡支付
     *
     * @param authCodeToOpenIdRequest 向wxpay post的请求数据
     * @return API返回数据
     */
    @Override
    public Map<String, String> authCodeToOpenid(WxPayAuthCodeToOpenIdRequest authCodeToOpenIdRequest) throws Exception {
        return this.authCodeToOpenid(authCodeToOpenIdRequest, DEFAULT_TIMEOUT);
    }

    /**
     * 作用：授权码查询OPENID接口<br> 场景：刷卡支付
     *
     * @param authCodeToOpenIdRequest 向wxpay post的请求数据
     * @param timeout 超时时间
     * @return API返回数据
     */
    @Override
    public Map<String, String> authCodeToOpenid(WxPayAuthCodeToOpenIdRequest authCodeToOpenIdRequest, int timeout) throws Exception {
        String url;
        if (this.useSandbox) {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.SANDBOX_AUTHCODETOOPENID_URL_SUFFIX;
        } else {
            url = "https://" + WxPayConstants.DOMAIN_API + WxPayConstants.AUTHCODETOOPENID_URL_SUFFIX;
        }
        String request = WxPayUtil.mapToXml(this.fillRequestData((Map<String, String>) JSONObject.toJSON(authCodeToOpenIdRequest)));
        String respXml = wxPayRestTemplateBuilder.createRestTemplate(timeout).postForObject(url, request, String.class);
        return this.processResponseXml(respXml);
    }

    /**
     * 向 Map 中添加 appid、mch_id、nonce_str、sign_type、sign <br> 该函数适用于商户适用于统一下单等接口，不适用于红包、代金券接口
     *
     * @param reqData 请求数据
     * @return 转换后的数据
     * @throws Exception 异常
     */
    public Map<String, String> fillRequestData(Map<String, String> reqData) throws Exception {
        reqData.put("appid", this.appId);
        reqData.put("mch_id", this.appSecret);
        reqData.put("nonce_str", WxPayUtil.generateNonceStr());
        if (SignType.MD5.equals(this.signType)) {
            reqData.put("sign_type", WxPayConstants.MD5);
        } else if (SignType.HMACSHA256.equals(this.signType)) {
            reqData.put("sign_type", WxPayConstants.HMACSHA256);
        }
        reqData.put("sign", WxPayUtil.generateSignature(reqData, this.key, this.signType));
        return reqData;
    }

    /**
     * 处理 HTTPS API返回数据，转换成Map对象。return_code为SUCCESS时，验证签名。
     *
     * @param xmlStr API返回的XML格式数据
     * @return Map类型数据
     * @throws Exception 异常
     */
    public Map<String, String> processResponseXml(String xmlStr) throws Exception {
        String RETURN_CODE = "return_code";
        String return_code;
        Map<String, String> respData = WxPayUtil.xmlToMap(xmlStr);
        if (respData.containsKey(RETURN_CODE)) {
            return_code = respData.get(RETURN_CODE);
        } else {
            throw new Exception(String.format("No `return_code` in XML: %s", xmlStr));
        }

        if (return_code.equals(WxPayConstants.FAIL)) {
            return respData;
        } else if (return_code.equals(WxPayConstants.SUCCESS)) {
            if (this.isResponseSignatureValid(respData)) {
                return respData;
            } else {
                throw new Exception(String.format("Invalid sign value in XML: %s", xmlStr));
            }
        } else {
            throw new Exception(String.format("return_code value %s is invalid in XML: %s", return_code, xmlStr));
        }
    }

    /**
     * 判断xml数据的sign是否有效，必须包含sign字段，否则返回false。
     *
     * @param reqData 向wxpay post的请求数据
     * @return 签名是否有效
     * @throws Exception 异常
     */
    public boolean isResponseSignatureValid(Map<String, String> reqData) throws Exception {
        // 返回数据的签名方式和请求中给定的签名方式是一致的
        return WxPayUtil.isSignatureValid(reqData, this.key, this.signType);
    }
}
