/*
 * Decompiled with CFR 0.152.
 */
package net.consensys.cava.crypto.sodium;

import com.google.common.base.Preconditions;
import java.util.Objects;
import javax.annotation.Nullable;
import javax.security.auth.Destroyable;
import jnr.ffi.Pointer;
import net.consensys.cava.bytes.Bytes;
import net.consensys.cava.crypto.sodium.DefaultDetachedEncryptionResult;
import net.consensys.cava.crypto.sodium.DetachedEncryptionResult;
import net.consensys.cava.crypto.sodium.Sodium;
import net.consensys.cava.crypto.sodium.SodiumException;

public final class Box
implements AutoCloseable {
    private Pointer ctx;

    private Box(PublicKey publicKey, SecretKey secretKey) {
        Preconditions.checkArgument((secretKey.ptr != null ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
        this.ctx = Sodium.malloc(Sodium.crypto_box_beforenmbytes());
        try {
            int rc = Sodium.crypto_box_beforenm(this.ctx, publicKey.ptr, secretKey.ptr);
            if (rc != 0) {
                throw new SodiumException("crypto_box_beforenm: failed with result " + rc);
            }
        }
        catch (Throwable e) {
            Sodium.sodium_free(this.ctx);
            this.ctx = null;
            throw e;
        }
    }

    public static Box forKeys(PublicKey receiver, SecretKey sender) {
        return new Box(receiver, sender);
    }

    public static Bytes encrypt(Bytes message, PublicKey receiver, SecretKey sender, Nonce nonce) {
        return Bytes.wrap((byte[])Box.encrypt(message.toArrayUnsafe(), receiver, sender, nonce));
    }

    public static byte[] encrypt(byte[] message, PublicKey receiver, SecretKey sender, Nonce nonce) {
        Preconditions.checkArgument((sender.ptr != null ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
        byte[] cipherText = new byte[Box.combinedCypherTextLength(message)];
        int rc = Sodium.crypto_box_easy(cipherText, message, message.length, nonce.ptr, receiver.ptr, sender.ptr);
        if (rc != 0) {
            throw new SodiumException("crypto_box_easy: failed with result " + rc);
        }
        return cipherText;
    }

    public Bytes encrypt(Bytes message, Nonce nonce) {
        return Bytes.wrap((byte[])this.encrypt(message.toArrayUnsafe(), nonce));
    }

    public byte[] encrypt(byte[] message, Nonce nonce) {
        this.assertOpen();
        byte[] cipherText = new byte[Box.combinedCypherTextLength(message)];
        int rc = Sodium.crypto_box_easy_afternm(cipherText, message, message.length, nonce.ptr, this.ctx);
        if (rc != 0) {
            throw new SodiumException("crypto_box_easy_afternm: failed with result " + rc);
        }
        return cipherText;
    }

    public static Bytes encryptSealed(Bytes message, PublicKey receiver) {
        return Bytes.wrap((byte[])Box.encryptSealed(message.toArrayUnsafe(), receiver));
    }

    public static byte[] encryptSealed(byte[] message, PublicKey receiver) {
        long sealbytes = Sodium.crypto_box_sealbytes();
        if (sealbytes > Integer.MAX_VALUE) {
            throw new IllegalStateException("crypto_box_sealbytes: " + sealbytes + " is too large");
        }
        byte[] cipherText = new byte[(int)sealbytes + message.length];
        int rc = Sodium.crypto_box_seal(cipherText, message, message.length, receiver.ptr);
        if (rc != 0) {
            throw new SodiumException("crypto_box_seal: failed with result " + rc);
        }
        return cipherText;
    }

    private static int combinedCypherTextLength(byte[] message) {
        long macbytes = Sodium.crypto_box_macbytes();
        if (macbytes > Integer.MAX_VALUE) {
            throw new IllegalStateException("crypto_box_macbytes: " + macbytes + " is too large");
        }
        return (int)macbytes + message.length;
    }

    public static DetachedEncryptionResult encryptDetached(Bytes message, PublicKey receiver, SecretKey sender, Nonce nonce) {
        return Box.encryptDetached(message.toArrayUnsafe(), receiver, sender, nonce);
    }

    public static DetachedEncryptionResult encryptDetached(byte[] message, PublicKey receiver, SecretKey sender, Nonce nonce) {
        Preconditions.checkArgument((sender.ptr != null ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
        byte[] cipherText = new byte[message.length];
        long macbytes = Sodium.crypto_box_macbytes();
        if (macbytes > Integer.MAX_VALUE) {
            throw new IllegalStateException("crypto_box_macbytes: " + macbytes + " is too large");
        }
        byte[] mac = new byte[(int)macbytes];
        int rc = Sodium.crypto_box_detached(cipherText, mac, message, message.length, nonce.ptr, receiver.ptr, sender.ptr);
        if (rc != 0) {
            throw new SodiumException("crypto_box_detached: failed with result " + rc);
        }
        return new DefaultDetachedEncryptionResult(cipherText, mac);
    }

    public DetachedEncryptionResult encryptDetached(Bytes message, Nonce nonce) {
        return this.encryptDetached(message.toArrayUnsafe(), nonce);
    }

    public DetachedEncryptionResult encryptDetached(byte[] message, Nonce nonce) {
        this.assertOpen();
        byte[] cipherText = new byte[message.length];
        long macbytes = Sodium.crypto_box_macbytes();
        if (macbytes > Integer.MAX_VALUE) {
            throw new IllegalStateException("crypto_box_macbytes: " + macbytes + " is too large");
        }
        byte[] mac = new byte[(int)macbytes];
        int rc = Sodium.crypto_box_detached_afternm(cipherText, mac, message, message.length, nonce.ptr, this.ctx);
        if (rc != 0) {
            throw new SodiumException("crypto_box_detached_afternm: failed with result " + rc);
        }
        return new DefaultDetachedEncryptionResult(cipherText, mac);
    }

    @Nullable
    public static Bytes decrypt(Bytes cipherText, PublicKey sender, SecretKey receiver, Nonce nonce) {
        byte[] bytes = Box.decrypt(cipherText.toArrayUnsafe(), sender, receiver, nonce);
        return bytes != null ? Bytes.wrap((byte[])bytes) : null;
    }

    @Nullable
    public static byte[] decrypt(byte[] cipherText, PublicKey sender, SecretKey receiver, Nonce nonce) {
        Preconditions.checkArgument((receiver.ptr != null ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
        byte[] clearText = new byte[Box.clearTextLength(cipherText)];
        int rc = Sodium.crypto_box_open_easy(clearText, cipherText, cipherText.length, nonce.ptr, sender.ptr, receiver.ptr);
        if (rc == -1) {
            return null;
        }
        if (rc != 0) {
            throw new SodiumException("crypto_box_open_easy: failed with result " + rc);
        }
        return clearText;
    }

    @Nullable
    public Bytes decrypt(Bytes cipherText, Nonce nonce) {
        byte[] bytes = this.decrypt(cipherText.toArrayUnsafe(), nonce);
        return bytes != null ? Bytes.wrap((byte[])bytes) : null;
    }

    @Nullable
    public byte[] decrypt(byte[] cipherText, Nonce nonce) {
        this.assertOpen();
        byte[] clearText = new byte[Box.clearTextLength(cipherText)];
        int rc = Sodium.crypto_box_open_easy_afternm(clearText, cipherText, cipherText.length, nonce.ptr, this.ctx);
        if (rc == -1) {
            return null;
        }
        if (rc != 0) {
            throw new SodiumException("crypto_box_open_easy_afternm: failed with result " + rc);
        }
        return clearText;
    }

    private static int clearTextLength(byte[] cipherText) {
        long macbytes = Sodium.crypto_box_macbytes();
        if (macbytes > Integer.MAX_VALUE) {
            throw new IllegalStateException("crypto_box_macbytes: " + macbytes + " is too large");
        }
        if (macbytes > (long)cipherText.length) {
            throw new IllegalArgumentException("cipherText is too short");
        }
        return cipherText.length - (int)macbytes;
    }

    @Nullable
    public static Bytes decryptSealed(Bytes cipherText, PublicKey sender, SecretKey receiver) {
        byte[] bytes = Box.decryptSealed(cipherText.toArrayUnsafe(), sender, receiver);
        return bytes != null ? Bytes.wrap((byte[])bytes) : null;
    }

    @Nullable
    public static byte[] decryptSealed(byte[] cipherText, PublicKey sender, SecretKey receiver) {
        Preconditions.checkArgument((receiver.ptr != null ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
        long sealbytes = Sodium.crypto_box_sealbytes();
        if (sealbytes > Integer.MAX_VALUE) {
            throw new IllegalStateException("crypto_box_sealbytes: " + sealbytes + " is too large");
        }
        if (sealbytes > (long)cipherText.length) {
            throw new IllegalArgumentException("cipherText is too short");
        }
        byte[] clearText = new byte[cipherText.length - (int)sealbytes];
        int rc = Sodium.crypto_box_seal_open(clearText, cipherText, cipherText.length, sender.ptr, receiver.ptr);
        if (rc == -1) {
            return null;
        }
        if (rc != 0) {
            throw new SodiumException("crypto_box_seal_open: failed with result " + rc);
        }
        return clearText;
    }

    @Nullable
    public static Bytes decryptDetached(Bytes cipherText, Bytes mac, PublicKey sender, SecretKey receiver, Nonce nonce) {
        byte[] bytes = Box.decryptDetached(cipherText.toArrayUnsafe(), mac.toArrayUnsafe(), sender, receiver, nonce);
        return bytes != null ? Bytes.wrap((byte[])bytes) : null;
    }

    @Nullable
    public static byte[] decryptDetached(byte[] cipherText, byte[] mac, PublicKey sender, SecretKey receiver, Nonce nonce) {
        Preconditions.checkArgument((receiver.ptr != null ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
        long macbytes = Sodium.crypto_box_macbytes();
        if (macbytes > Integer.MAX_VALUE) {
            throw new IllegalStateException("crypto_box_macbytes: " + macbytes + " is too large");
        }
        if ((long)mac.length != macbytes) {
            throw new IllegalArgumentException("mac must be " + macbytes + " bytes, got " + mac.length);
        }
        byte[] clearText = new byte[cipherText.length];
        int rc = Sodium.crypto_box_open_detached(clearText, cipherText, mac, cipherText.length, nonce.ptr, sender.ptr, receiver.ptr);
        if (rc == -1) {
            return null;
        }
        if (rc != 0) {
            throw new SodiumException("crypto_box_open_detached: failed with result " + rc);
        }
        return clearText;
    }

    @Nullable
    public Bytes decryptDetached(Bytes cipherText, Bytes mac, Nonce nonce) {
        byte[] bytes = this.decryptDetached(cipherText.toArrayUnsafe(), mac.toArrayUnsafe(), nonce);
        return bytes != null ? Bytes.wrap((byte[])bytes) : null;
    }

    @Nullable
    public byte[] decryptDetached(byte[] cipherText, byte[] mac, Nonce nonce) {
        long macbytes = Sodium.crypto_box_macbytes();
        if (macbytes > Integer.MAX_VALUE) {
            throw new IllegalStateException("crypto_box_macbytes: " + macbytes + " is too large");
        }
        if ((long)mac.length != macbytes) {
            throw new IllegalArgumentException("mac must be " + macbytes + " bytes, got " + mac.length);
        }
        byte[] clearText = new byte[cipherText.length];
        int rc = Sodium.crypto_box_open_detached_afternm(clearText, cipherText, mac, cipherText.length, nonce.ptr, this.ctx);
        if (rc == -1) {
            return null;
        }
        if (rc != 0) {
            throw new SodiumException("crypto_box_open_detached_afternm: failed with result " + rc);
        }
        return clearText;
    }

    private void assertOpen() {
        if (this.ctx == null) {
            throw new IllegalStateException(this.getClass().getName() + ": already closed");
        }
    }

    @Override
    public void close() {
        if (this.ctx != null) {
            Sodium.sodium_free(this.ctx);
            this.ctx = null;
        }
    }

    protected void finalize() {
        this.close();
    }

    public static final class Nonce {
        private final Pointer ptr;
        private final int length;

        private Nonce(Pointer ptr, int length) {
            this.ptr = ptr;
            this.length = length;
        }

        protected void finalize() {
            Sodium.sodium_free(this.ptr);
        }

        public static Nonce fromBytes(Bytes bytes) {
            return Nonce.fromBytes(bytes.toArrayUnsafe());
        }

        public static Nonce fromBytes(byte[] bytes) {
            if ((long)bytes.length != Sodium.crypto_box_noncebytes()) {
                throw new IllegalArgumentException("nonce must be " + Sodium.crypto_box_noncebytes() + " bytes, got " + bytes.length);
            }
            return Sodium.dup(bytes, Nonce::new);
        }

        public static int length() {
            long npubbytes = Sodium.crypto_box_noncebytes();
            if (npubbytes > Integer.MAX_VALUE) {
                throw new SodiumException("crypto_box_noncebytes: " + npubbytes + " is too large");
            }
            return (int)npubbytes;
        }

        public static Nonce random() {
            return Sodium.randomBytes(Nonce.length(), Nonce::new);
        }

        public Nonce increment() {
            return Sodium.dupAndIncrement(this.ptr, this.length, Nonce::new);
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof Nonce)) {
                return false;
            }
            Nonce other = (Nonce)obj;
            return Sodium.sodium_memcmp(this.ptr, other.ptr, this.length) == 0;
        }

        public int hashCode() {
            return Sodium.hashCode(this.ptr, this.length);
        }

        public Bytes bytes() {
            return Bytes.wrap((byte[])this.bytesArray());
        }

        public byte[] bytesArray() {
            return Sodium.reify(this.ptr, this.length);
        }
    }

    public static final class KeyPair {
        private final PublicKey publicKey;
        private final SecretKey secretKey;

        public KeyPair(PublicKey publicKey, SecretKey secretKey) {
            this.publicKey = publicKey;
            this.secretKey = secretKey;
        }

        public static KeyPair forSecretKey(SecretKey secretKey) {
            Preconditions.checkArgument((secretKey.ptr != null ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
            return Sodium.scalarMultBase(secretKey.ptr, SecretKey.length(), (ptr, len) -> {
                int publicKeyLength = PublicKey.length();
                if (len != (long)publicKeyLength) {
                    throw new IllegalStateException("Public key length " + publicKeyLength + " is not same as generated key length " + len);
                }
                return new KeyPair(new PublicKey((Pointer)ptr, publicKeyLength), secretKey);
            });
        }

        public static KeyPair random() {
            int publicKeyLength = PublicKey.length();
            Pointer publicKey = Sodium.malloc(publicKeyLength);
            Pointer secretKey = null;
            try {
                int secretKeyLength = SecretKey.length();
                secretKey = Sodium.malloc(secretKeyLength);
                int rc = Sodium.crypto_box_keypair(publicKey, secretKey);
                if (rc != 0) {
                    throw new SodiumException("crypto_box_keypair: failed with result " + rc);
                }
                PublicKey pk = new PublicKey(publicKey, publicKeyLength);
                publicKey = null;
                SecretKey sk = new SecretKey(secretKey, secretKeyLength);
                secretKey = null;
                return new KeyPair(pk, sk);
            }
            catch (Throwable e) {
                if (publicKey != null) {
                    Sodium.sodium_free(publicKey);
                }
                if (secretKey != null) {
                    Sodium.sodium_free(secretKey);
                }
                throw e;
            }
        }

        public static KeyPair fromSeed(Seed seed) {
            int publicKeyLength = PublicKey.length();
            Pointer publicKey = Sodium.malloc(publicKeyLength);
            Pointer secretKey = null;
            try {
                int secretKeyLength = SecretKey.length();
                secretKey = Sodium.malloc(secretKeyLength);
                int rc = Sodium.crypto_box_seed_keypair(publicKey, secretKey, seed.ptr);
                if (rc != 0) {
                    throw new SodiumException("crypto_box_keypair: failed with result " + rc);
                }
                PublicKey pk = new PublicKey(publicKey, publicKeyLength);
                publicKey = null;
                SecretKey sk = new SecretKey(secretKey, secretKeyLength);
                secretKey = null;
                return new KeyPair(pk, sk);
            }
            catch (Throwable e) {
                if (publicKey != null) {
                    Sodium.sodium_free(publicKey);
                }
                if (secretKey != null) {
                    Sodium.sodium_free(secretKey);
                }
                throw e;
            }
        }

        public PublicKey publicKey() {
            return this.publicKey;
        }

        public SecretKey secretKey() {
            return this.secretKey;
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof KeyPair)) {
                return false;
            }
            KeyPair other = (KeyPair)obj;
            return this.publicKey.equals(other.publicKey) && this.secretKey.equals(other.secretKey);
        }

        public int hashCode() {
            return Objects.hash(this.publicKey, this.secretKey);
        }
    }

    public static final class Seed {
        private final Pointer ptr;
        private final int length;

        private Seed(Pointer ptr, int length) {
            this.ptr = ptr;
            this.length = length;
        }

        protected void finalize() {
            Sodium.sodium_free(this.ptr);
        }

        public static Seed fromBytes(Bytes bytes) {
            return Seed.fromBytes(bytes.toArrayUnsafe());
        }

        public static Seed fromBytes(byte[] bytes) {
            if ((long)bytes.length != Sodium.crypto_box_seedbytes()) {
                throw new IllegalArgumentException("key must be " + Sodium.crypto_box_seedbytes() + " bytes, got " + bytes.length);
            }
            return Sodium.dup(bytes, Seed::new);
        }

        public static int length() {
            long seedbytes = Sodium.crypto_box_seedbytes();
            if (seedbytes > Integer.MAX_VALUE) {
                throw new SodiumException("crypto_box_seedbytes: " + seedbytes + " is too large");
            }
            return (int)seedbytes;
        }

        public static Seed random() {
            return Sodium.randomBytes(Seed.length(), Seed::new);
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof Seed)) {
                return false;
            }
            Seed other = (Seed)obj;
            return Sodium.sodium_memcmp(this.ptr, other.ptr, this.length) == 0;
        }

        public int hashCode() {
            return Sodium.hashCode(this.ptr, this.length);
        }

        public Bytes bytes() {
            return Bytes.wrap((byte[])this.bytesArray());
        }

        public byte[] bytesArray() {
            return Sodium.reify(this.ptr, this.length);
        }
    }

    public static final class SecretKey
    implements Destroyable {
        @Nullable
        private Pointer ptr;
        private final int length;

        private SecretKey(Pointer ptr, int length) {
            this.ptr = ptr;
            this.length = length;
        }

        protected void finalize() {
            this.destroy();
        }

        @Override
        public void destroy() {
            if (this.ptr != null) {
                Pointer p = this.ptr;
                this.ptr = null;
                Sodium.sodium_free(p);
            }
        }

        @Override
        public boolean isDestroyed() {
            return this.ptr == null;
        }

        public static SecretKey fromBytes(Bytes bytes) {
            return SecretKey.fromBytes(bytes.toArrayUnsafe());
        }

        public static SecretKey fromBytes(byte[] bytes) {
            if ((long)bytes.length != Sodium.crypto_box_secretkeybytes()) {
                throw new IllegalArgumentException("key must be " + Sodium.crypto_box_secretkeybytes() + " bytes, got " + bytes.length);
            }
            return Sodium.dup(bytes, SecretKey::new);
        }

        public static int length() {
            long keybytes = Sodium.crypto_box_secretkeybytes();
            if (keybytes > Integer.MAX_VALUE) {
                throw new SodiumException("crypto_box_secretkeybytes: " + keybytes + " is too large");
            }
            return (int)keybytes;
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof SecretKey)) {
                return false;
            }
            Preconditions.checkState((this.ptr != null ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
            SecretKey other = (SecretKey)obj;
            return other.ptr != null && Sodium.sodium_memcmp(this.ptr, other.ptr, this.length) == 0;
        }

        public int hashCode() {
            Preconditions.checkState((this.ptr != null ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
            return Sodium.hashCode(this.ptr, this.length);
        }

        public Bytes bytes() {
            return Bytes.wrap((byte[])this.bytesArray());
        }

        public byte[] bytesArray() {
            Preconditions.checkState((this.ptr != null ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
            return Sodium.reify(this.ptr, this.length);
        }
    }

    public static final class PublicKey {
        private final Pointer ptr;
        private final int length;

        private PublicKey(Pointer ptr, int length) {
            this.ptr = ptr;
            this.length = length;
        }

        protected void finalize() {
            Sodium.sodium_free(this.ptr);
        }

        public static PublicKey fromBytes(Bytes bytes) {
            return PublicKey.fromBytes(bytes.toArrayUnsafe());
        }

        public static PublicKey fromBytes(byte[] bytes) {
            if ((long)bytes.length != Sodium.crypto_box_publickeybytes()) {
                throw new IllegalArgumentException("key must be " + Sodium.crypto_box_publickeybytes() + " bytes, got " + bytes.length);
            }
            return Sodium.dup(bytes, PublicKey::new);
        }

        public static int length() {
            long keybytes = Sodium.crypto_box_publickeybytes();
            if (keybytes > Integer.MAX_VALUE) {
                throw new SodiumException("crypto_box_publickeybytes: " + keybytes + " is too large");
            }
            return (int)keybytes;
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof PublicKey)) {
                return false;
            }
            PublicKey other = (PublicKey)obj;
            return Sodium.sodium_memcmp(this.ptr, other.ptr, this.length) == 0;
        }

        public int hashCode() {
            return Sodium.hashCode(this.ptr, this.length);
        }

        public Bytes bytes() {
            return Bytes.wrap((byte[])this.bytesArray());
        }

        public byte[] bytesArray() {
            return Sodium.reify(this.ptr, this.length);
        }
    }
}

