/*
 * Decompiled with CFR 0.152.
 */
package net.consensys.cava.crypto.sodium;

import com.google.common.base.Preconditions;
import java.util.Objects;
import javax.annotation.Nullable;
import javax.security.auth.Destroyable;
import jnr.ffi.Pointer;
import net.consensys.cava.bytes.Bytes;
import net.consensys.cava.crypto.sodium.Sodium;
import net.consensys.cava.crypto.sodium.SodiumException;

public final class KeyExchange {
    public static SessionKeyPair client(KeyPair clientKeys, PublicKey serverKey) {
        Preconditions.checkArgument((clientKeys.secretKey.ptr != null ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
        long sessionkeybytes = Sodium.crypto_kx_sessionkeybytes();
        if (sessionkeybytes > Integer.MAX_VALUE) {
            throw new SodiumException("crypto_kx_sessionkeybytes: " + sessionkeybytes + " is too large");
        }
        Pointer rxPtr = null;
        Pointer txPtr = null;
        try {
            rxPtr = Sodium.malloc(sessionkeybytes);
            txPtr = Sodium.malloc(sessionkeybytes);
            int rc = Sodium.crypto_kx_client_session_keys(rxPtr, txPtr, clientKeys.publicKey.ptr, clientKeys.secretKey.ptr, serverKey.ptr);
            if (rc != 0) {
                throw new SodiumException("crypto_kx_client_session_keys: failed with result " + rc);
            }
            SessionKey rxKey = new SessionKey(rxPtr, (int)sessionkeybytes);
            rxPtr = null;
            SessionKey txKey = new SessionKey(txPtr, (int)sessionkeybytes);
            txPtr = null;
            return new SessionKeyPair(rxKey, txKey);
        }
        catch (Throwable e) {
            if (rxPtr != null) {
                Sodium.sodium_free(rxPtr);
            }
            if (txPtr != null) {
                Sodium.sodium_free(txPtr);
            }
            throw e;
        }
    }

    public static SessionKeyPair server(KeyPair serverKeys, PublicKey clientKey) {
        Preconditions.checkArgument((serverKeys.secretKey.ptr != null ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
        long sessionkeybytes = Sodium.crypto_kx_sessionkeybytes();
        if (sessionkeybytes > Integer.MAX_VALUE) {
            throw new SodiumException("crypto_kx_sessionkeybytes: " + sessionkeybytes + " is too large");
        }
        Pointer rxPtr = null;
        Pointer txPtr = null;
        try {
            rxPtr = Sodium.malloc(sessionkeybytes);
            txPtr = Sodium.malloc(sessionkeybytes);
            int rc = Sodium.crypto_kx_server_session_keys(rxPtr, txPtr, serverKeys.publicKey.ptr, serverKeys.secretKey.ptr, clientKey.ptr);
            if (rc != 0) {
                throw new SodiumException("crypto_kx_client_session_keys: failed with result " + rc);
            }
            SessionKey rxKey = new SessionKey(rxPtr, (int)sessionkeybytes);
            rxPtr = null;
            SessionKey txKey = new SessionKey(txPtr, (int)sessionkeybytes);
            txPtr = null;
            return new SessionKeyPair(rxKey, txKey);
        }
        catch (Throwable e) {
            if (rxPtr != null) {
                Sodium.sodium_free(rxPtr);
            }
            if (txPtr != null) {
                Sodium.sodium_free(txPtr);
            }
            throw e;
        }
    }

    public static final class SessionKeyPair {
        private final SessionKey rxKey;
        private final SessionKey txKey;

        public SessionKeyPair(SessionKey rxKey, SessionKey txKey) {
            this.rxKey = rxKey;
            this.txKey = txKey;
        }

        public SessionKey rx() {
            return this.rxKey;
        }

        public SessionKey tx() {
            return this.txKey;
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof SessionKeyPair)) {
                return false;
            }
            SessionKeyPair other = (SessionKeyPair)obj;
            return this.rxKey.equals(other.rxKey) && this.txKey.equals(other.txKey);
        }

        public int hashCode() {
            return Objects.hash(this.rxKey, this.txKey);
        }
    }

    public static final class SessionKey {
        private final Pointer ptr;
        private final int length;

        private SessionKey(Pointer ptr, int length) {
            this.ptr = ptr;
            this.length = length;
        }

        protected void finalize() {
            Sodium.sodium_free(this.ptr);
        }

        public static SessionKey fromBytes(Bytes bytes) {
            return SessionKey.fromBytes(bytes.toArrayUnsafe());
        }

        public static SessionKey fromBytes(byte[] bytes) {
            if ((long)bytes.length != Sodium.crypto_kx_sessionkeybytes()) {
                throw new IllegalArgumentException("key must be " + Sodium.crypto_kx_sessionkeybytes() + " bytes, got " + bytes.length);
            }
            return Sodium.dup(bytes, SessionKey::new);
        }

        public static int length() {
            long keybytes = Sodium.crypto_kx_sessionkeybytes();
            if (keybytes > Integer.MAX_VALUE) {
                throw new SodiumException("crypto_kx_sessionkeybytes: " + keybytes + " is too large");
            }
            return (int)keybytes;
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof SessionKey)) {
                return false;
            }
            SessionKey other = (SessionKey)obj;
            return Sodium.sodium_memcmp(this.ptr, other.ptr, this.length) == 0;
        }

        public int hashCode() {
            return Sodium.hashCode(this.ptr, this.length);
        }

        public Bytes bytes() {
            return Bytes.wrap((byte[])this.bytesArray());
        }

        public byte[] bytesArray() {
            return Sodium.reify(this.ptr, this.length);
        }
    }

    public static final class KeyPair {
        private final PublicKey publicKey;
        private final SecretKey secretKey;

        public KeyPair(PublicKey publicKey, SecretKey secretKey) {
            this.publicKey = publicKey;
            this.secretKey = secretKey;
        }

        public static KeyPair forSecretKey(SecretKey secretKey) {
            Preconditions.checkArgument((secretKey.ptr != null ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
            return Sodium.scalarMultBase(secretKey.ptr, SecretKey.length(), (ptr, len) -> {
                int publicKeyLength = PublicKey.length();
                if (len != (long)publicKeyLength) {
                    throw new IllegalStateException("Public key length " + publicKeyLength + " is not same as generated key length " + len);
                }
                return new KeyPair(new PublicKey((Pointer)ptr, publicKeyLength), secretKey);
            });
        }

        public static KeyPair random() {
            int publicKeyLength = PublicKey.length();
            Pointer publicKey = Sodium.malloc(publicKeyLength);
            Pointer secretKey = null;
            try {
                int secretKeyLength = SecretKey.length();
                secretKey = Sodium.malloc(secretKeyLength);
                int rc = Sodium.crypto_kx_keypair(publicKey, secretKey);
                if (rc != 0) {
                    throw new SodiumException("crypto_kx_keypair: failed with result " + rc);
                }
                PublicKey pk = new PublicKey(publicKey, publicKeyLength);
                publicKey = null;
                SecretKey sk = new SecretKey(secretKey, secretKeyLength);
                secretKey = null;
                return new KeyPair(pk, sk);
            }
            catch (Throwable e) {
                if (publicKey != null) {
                    Sodium.sodium_free(publicKey);
                }
                if (secretKey != null) {
                    Sodium.sodium_free(secretKey);
                }
                throw e;
            }
        }

        public static KeyPair fromSeed(Seed seed) {
            int publicKeyLength = PublicKey.length();
            Pointer publicKey = Sodium.malloc(publicKeyLength);
            Pointer secretKey = null;
            try {
                int secretKeyLength = SecretKey.length();
                secretKey = Sodium.malloc(secretKeyLength);
                int rc = Sodium.crypto_kx_seed_keypair(publicKey, secretKey, seed.ptr);
                if (rc != 0) {
                    throw new SodiumException("crypto_kx_seed_keypair: failed with result " + rc);
                }
                PublicKey pk = new PublicKey(publicKey, publicKeyLength);
                publicKey = null;
                SecretKey sk = new SecretKey(secretKey, secretKeyLength);
                secretKey = null;
                return new KeyPair(pk, sk);
            }
            catch (Throwable e) {
                if (publicKey != null) {
                    Sodium.sodium_free(publicKey);
                }
                if (secretKey != null) {
                    Sodium.sodium_free(secretKey);
                }
                throw e;
            }
        }

        public PublicKey publicKey() {
            return this.publicKey;
        }

        public SecretKey secretKey() {
            return this.secretKey;
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof KeyPair)) {
                return false;
            }
            KeyPair other = (KeyPair)obj;
            return this.publicKey.equals(other.publicKey) && this.secretKey.equals(other.secretKey);
        }

        public int hashCode() {
            return Objects.hash(this.publicKey, this.secretKey);
        }
    }

    public static final class Seed {
        private final Pointer ptr;
        private final int length;

        private Seed(Pointer ptr, int length) {
            this.ptr = ptr;
            this.length = length;
        }

        protected void finalize() {
            Sodium.sodium_free(this.ptr);
        }

        public static Seed fromBytes(Bytes bytes) {
            return Seed.fromBytes(bytes.toArrayUnsafe());
        }

        public static Seed fromBytes(byte[] bytes) {
            if ((long)bytes.length != Sodium.crypto_kx_seedbytes()) {
                throw new IllegalArgumentException("key must be " + Sodium.crypto_kx_seedbytes() + " bytes, got " + bytes.length);
            }
            return Sodium.dup(bytes, Seed::new);
        }

        public static int length() {
            long seedbytes = Sodium.crypto_kx_seedbytes();
            if (seedbytes > Integer.MAX_VALUE) {
                throw new SodiumException("crypto_kx_seedbytes: " + seedbytes + " is too large");
            }
            return (int)seedbytes;
        }

        public static Seed random() {
            return Sodium.randomBytes(Seed.length(), Seed::new);
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof Seed)) {
                return false;
            }
            Seed other = (Seed)obj;
            return Sodium.sodium_memcmp(this.ptr, other.ptr, this.length) == 0;
        }

        public int hashCode() {
            return Sodium.hashCode(this.ptr, this.length);
        }

        public Bytes bytes() {
            return Bytes.wrap((byte[])this.bytesArray());
        }

        public byte[] bytesArray() {
            return Sodium.reify(this.ptr, this.length);
        }
    }

    public static final class SecretKey
    implements Destroyable {
        @Nullable
        private Pointer ptr;
        private final int length;

        private SecretKey(Pointer ptr, int length) {
            this.ptr = ptr;
            this.length = length;
        }

        protected void finalize() {
            this.destroy();
        }

        @Override
        public void destroy() {
            if (this.ptr != null) {
                Pointer p = this.ptr;
                this.ptr = null;
                Sodium.sodium_free(p);
            }
        }

        @Override
        public boolean isDestroyed() {
            return this.ptr == null;
        }

        public static SecretKey fromBytes(Bytes bytes) {
            return SecretKey.fromBytes(bytes.toArrayUnsafe());
        }

        public static SecretKey fromBytes(byte[] bytes) {
            if ((long)bytes.length != Sodium.crypto_kx_secretkeybytes()) {
                throw new IllegalArgumentException("key must be " + Sodium.crypto_kx_secretkeybytes() + " bytes, got " + bytes.length);
            }
            return Sodium.dup(bytes, SecretKey::new);
        }

        public static int length() {
            long keybytes = Sodium.crypto_kx_secretkeybytes();
            if (keybytes > Integer.MAX_VALUE) {
                throw new SodiumException("crypto_kx_secretkeybytes: " + keybytes + " is too large");
            }
            return (int)keybytes;
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof SecretKey)) {
                return false;
            }
            Preconditions.checkState((this.ptr != null ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
            SecretKey other = (SecretKey)obj;
            return other.ptr != null && Sodium.sodium_memcmp(this.ptr, other.ptr, this.length) == 0;
        }

        public int hashCode() {
            Preconditions.checkState((this.ptr != null ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
            return Sodium.hashCode(this.ptr, this.length);
        }

        public Bytes bytes() {
            return Bytes.wrap((byte[])this.bytesArray());
        }

        public byte[] bytesArray() {
            Preconditions.checkState((this.ptr != null ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
            return Sodium.reify(this.ptr, this.length);
        }
    }

    public static final class PublicKey {
        private final Pointer ptr;
        private final int length;

        private PublicKey(Pointer ptr, int length) {
            this.ptr = ptr;
            this.length = length;
        }

        protected void finalize() {
            Sodium.sodium_free(this.ptr);
        }

        public static PublicKey fromBytes(Bytes bytes) {
            return PublicKey.fromBytes(bytes.toArrayUnsafe());
        }

        public static PublicKey fromBytes(byte[] bytes) {
            if ((long)bytes.length != Sodium.crypto_kx_publickeybytes()) {
                throw new IllegalArgumentException("key must be " + Sodium.crypto_kx_publickeybytes() + " bytes, got " + bytes.length);
            }
            return Sodium.dup(bytes, PublicKey::new);
        }

        public static int length() {
            long keybytes = Sodium.crypto_kx_publickeybytes();
            if (keybytes > Integer.MAX_VALUE) {
                throw new SodiumException("crypto_kx_publickeybytes: " + keybytes + " is too large");
            }
            return (int)keybytes;
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof PublicKey)) {
                return false;
            }
            PublicKey other = (PublicKey)obj;
            return Sodium.sodium_memcmp(this.ptr, other.ptr, this.length) == 0;
        }

        public int hashCode() {
            return Sodium.hashCode(this.ptr, this.length);
        }

        public Bytes bytes() {
            return Bytes.wrap((byte[])this.bytesArray());
        }

        public byte[] bytesArray() {
            return Sodium.reify(this.ptr, this.length);
        }
    }
}

