package net.consentmanager.cm_sdk_android_v3

import androidx.annotation.Keep
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

interface Retryable {
    fun performRetry(completion: (Boolean) -> Unit)
}

@Keep
class RetryManager(
    private val maxRetryCount: Int = 3,
    private val baseDelay: Long = 2000,
    private val retryable: WebViewManager,
    private val coroutineScope: CoroutineScope = CoroutineScope(Dispatchers.Default)
) {
    private var retryCount = 0

    fun reset() {
        retryCount = 0
    }

    private fun canRetry(): Boolean = retryCount < maxRetryCount

    fun retry(completion: (Boolean) -> Unit) {
        if (!canRetry()) {
            println("Max retry count reached.")
            completion(false)
            return
        }

        val delay = baseDelay * (1 shl retryCount)
        retryCount++

        coroutineScope.launch {
            delay(delay)
            retryable.performRetry { success ->
                if (success) {
                    reset()
                }
                completion(success)
            }
        }
    }
}