package net.consentmanager.cm_sdk_android_v3

import android.annotation.SuppressLint
import androidx.annotation.Keep
import java.net.URLEncoder
import java.text.SimpleDateFormat
import java.util.Date
import java.util.Locale

@Keep
enum class UseCase {
    VERIFY_CONSENT_ON_INITIALIZE,
    OPEN_CONSENT,
    CHECK_CONSENT,
    PERFORM_DRY_CHECK_CONSENT,
    PERFORM_DRY_CHECK_WITH_CACHE,
    IMPORT_CONSENT,
    ENABLE_CONSENT_PURPOSES,
    DISABLE_CONSENT_PURPOSES,
    ENABLE_CONSENT_VENDORS,
    DISABLE_CONSENT_VENDORS,
    ACCEPT_ALL_CONSENT,
    REJECT_ALL_CONSENT,
    RESET_CONSENT_SETTINGS
}

@Keep
data class UrlConfig(
    val id: String,
    val domain: String,
    val language: String,
    val appName: String
)

@Keep
data class CMPUrlParams(
    val id: String,
    val domain: String,
    val useCase: UseCase,
    var consent: String? = null,
    var language: String? = null,
    var appName: String? = null,
    var packageName: String? = null,
    var bundleID: String? = null,
    var idfa: String? = null,
    var isDebugMode: Boolean = false,
    var designId: Int? = null,
    var isTV: Boolean = false,
    var jumpToSettingsPage: Boolean = false,
    var forceOpen: Boolean = false,
    var acceptAll: Boolean = false,
    var rejectAll: Boolean = false,
    var addPurposes: List<String> = emptyList(),
    var addVendors: List<String> = emptyList(),
    var updateVendors: Boolean = false,
    var attStatus: Int = 0,
    var skipCookies: Boolean = false
) {
    @Keep
    fun apply(params: Map<String, Any>) {
        params.forEach { (key, value) ->
            when (key) {
                "language" -> language = value as? String
                "appName" -> appName = value as? String
                "packageName" -> packageName = value as? String
                "isDebugMode" -> isDebugMode = value as? Boolean ?: false
                "designId" -> designId = value as? Int
                "isTV" -> isTV = value as? Boolean ?: false
                "jumpToSettingsPage" -> jumpToSettingsPage = value as? Boolean ?: false
                "forceOpen" -> forceOpen = value as? Boolean ?: false
                "acceptAll" -> acceptAll = value as? Boolean ?: false
                "rejectAll" -> rejectAll = value as? Boolean ?: false
                "addPurposes" -> addPurposes = value as? List<String> ?: emptyList()
                "addVendors" -> addVendors = value as? List<String> ?: emptyList()
                "updateVendors" -> updateVendors = value as? Boolean ?: false
            }
        }
    }
}

@Keep
object CMPUrlBuilder {
    private const val URL_V5 = "https://%s/delivery/appcmp_v5.php?cdid=%s"
    private const val SDK_VERSION = "3.0.0"

    fun build(params: CMPUrlParams): String {
        val baseUrl = String.format(URL_V5, params.domain, params.id)
        val queryParams = mutableListOf<Pair<String, String>>()

        appendCommonParams(queryParams, params)

        when (params.useCase) {
            UseCase.VERIFY_CONSENT_ON_INITIALIZE, UseCase.CHECK_CONSENT -> appendCheckConsentParams(queryParams, params)
            UseCase.OPEN_CONSENT -> appendOpenConsentParams(queryParams, params)
            UseCase.PERFORM_DRY_CHECK_CONSENT, UseCase.PERFORM_DRY_CHECK_WITH_CACHE -> appendDryParams(
                queryParams
            )
            UseCase.IMPORT_CONSENT -> appendImportParams(queryParams)
            UseCase.ENABLE_CONSENT_PURPOSES -> appendEnablePurposesParams(queryParams, params)
            UseCase.DISABLE_CONSENT_PURPOSES -> appendDisablePurposesParams(queryParams, params)
            UseCase.ENABLE_CONSENT_VENDORS -> appendEnableVendorsParams(queryParams, params)
            UseCase.DISABLE_CONSENT_VENDORS -> appendDisableVendorsParams(queryParams, params)
            UseCase.ACCEPT_ALL_CONSENT -> appendAcceptParams(queryParams)
            UseCase.REJECT_ALL_CONSENT -> appendRejectParams(queryParams)
            UseCase.RESET_CONSENT_SETTINGS -> {} // No additional parameters needed
        }

        val queryString = queryParams.joinToString("&") { (key, value) ->
            "$key=${URLEncoder.encode(value, "UTF-8")}"
        }

        var finalUrl = "$baseUrl&$queryString"

        if (params.consent != null) {
            val stamp = dateAndRandomNumberAsString()
            finalUrl += "#cmpimport=${URLEncoder.encode(params.consent, "UTF-8")}&zt=$stamp"
        }

        return finalUrl
    }

    private fun appendCommonParams(queryParams: MutableList<Pair<String, String>>, params: CMPUrlParams) {
        queryParams.add("sdkv" to SDK_VERSION)
        params.appName?.let { queryParams.add("appname" to it.replace(" ", "-")) }
        params.language?.let { queryParams.add("cmplang" to it) }
        params.bundleID?.let { queryParams.add("appid" to it) }
        params.idfa?.let { queryParams.add("idfa" to it) }
        queryParams.add("cmpatt" to params.attStatus.toString())
        if (params.isDebugMode) queryParams.add("cmpdebug" to "")
    }

    private fun appendCheckConsentParams(queryParams: MutableList<Pair<String, String>>, params: CMPUrlParams) {
        params.designId?.let { queryParams.add("usedesign" to it.toString()) }
        if (params.jumpToSettingsPage) queryParams.add("cmpscreencustom" to "")
    }

    private fun appendOpenConsentParams(queryParams: MutableList<Pair<String, String>>, params: CMPUrlParams) {
        appendCheckConsentParams(queryParams, params)
        queryParams.add("cmpscreen" to "")
        if (params.isTV) queryParams.add("tvsdk" to "1")
    }

    private fun appendDryParams(queryParams: MutableList<Pair<String, String>>) {
        queryParams.add("cmpskipcookies" to "1")
    }

    private fun appendImportParams(queryParams: MutableList<Pair<String, String>>) {
        queryParams.add("cmpskipcookies" to "1")
    }

    private fun appendEnablePurposesParams(queryParams: MutableList<Pair<String, String>>, params: CMPUrlParams) {
        queryParams.add("cmpsetpurposes" to params.addPurposes.joinToString("_"))
        if (!params.updateVendors) queryParams.add("cmpdontfixpurposes" to "")
        queryParams.add("cmpautoaccept" to "1")
        queryParams.add("cmpscreen" to "")
    }

    private fun appendDisablePurposesParams(queryParams: MutableList<Pair<String, String>>, params: CMPUrlParams) {
        queryParams.add("cmpsetpurposes" to params.addPurposes.joinToString("_"))
        if (!params.updateVendors) queryParams.add("cmpdontfixpurposes" to "")
        queryParams.add("cmpautoreject" to "")
        queryParams.add("cmpscreen" to "")
    }

    private fun appendEnableVendorsParams(queryParams: MutableList<Pair<String, String>>, params: CMPUrlParams) {
        queryParams.add("cmpsetvendors" to params.addVendors.joinToString("_"))
        queryParams.add("cmpautoaccept" to "1")
        queryParams.add("cmpscreen" to "")
    }

    private fun appendDisableVendorsParams(queryParams: MutableList<Pair<String, String>>, params: CMPUrlParams) {
        queryParams.add("cmpsetvendors" to params.addVendors.joinToString("_"))
        queryParams.add("cmpautoreject" to "")
        queryParams.add("cmpscreen" to "")
    }

    private fun appendAcceptParams(queryParams: MutableList<Pair<String, String>>) {
        queryParams.add("cmpautoaccept" to "1")
        queryParams.add("cmpscreen" to "")
    }

    private fun appendRejectParams(queryParams: MutableList<Pair<String, String>>) {
        queryParams.add("cmpautoreject" to "")
        queryParams.add("cmpscreen" to "")
    }

    @SuppressLint("DefaultLocale")
    private fun dateAndRandomNumberAsString(): String {
        val dateFormat = SimpleDateFormat("ddMMyyyyHHmmss", Locale.US)
        val dateString = dateFormat.format(Date())
        val randomNumber = (0..9999).random()
        return String.format("%s%04d", dateString, randomNumber)
    }
}