package net.consentmanager.cm_sdk_android_v3

import android.annotation.SuppressLint
import android.app.Activity
import android.graphics.Bitmap
import android.os.Build
import android.util.Log
import android.webkit.JavascriptInterface
import android.webkit.WebResourceError
import android.webkit.WebResourceRequest
import android.webkit.WebView
import android.webkit.WebViewClient
import androidx.annotation.Keep
import androidx.annotation.RequiresApi
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.jsonObject
import java.lang.ref.WeakReference

@Keep
interface WebViewManagerDelegate {
    fun didReceiveConsentMessage(consent: String, jsonObject: JsonObject)
    fun didReceiveOpenMessage()
    fun didNotNeedOpenConsentLayer()
    fun didReceiveError(error: String)
}

@Keep
class WebViewManager(
    private val userPreferences: CMPUserPreferencesService,
    private val coroutineScope: CoroutineScope = CoroutineScope(Dispatchers.Main),
    private val networkMonitor: NetworkMonitor
) {
    var delegate: WebViewManagerDelegate? = null
    private var webView: WebView? = null
    private var currentUseCase: UseCase? = null
    private var completionHandler: ((Result<Boolean>) -> Unit)? = null
    private var retryManager: RetryManager? = null
    private val json = Json { ignoreUnknownKeys = true }
    private var currentUrl: String? = null

    private fun createWebViewClient(): WebViewClient {
        return object : WebViewClient() {
            override fun onPageStarted(view: WebView?, url: String?, favicon: Bitmap?) {
                Log.d("CMPWebViewManager", "Page started loading: $url")
            }

            override fun onPageFinished(view: WebView?, url: String?) {
                Log.d("CMPWebViewManager", "Page finished loading: $url")
                injectJavaScript()
            }

            @RequiresApi(Build.VERSION_CODES.M)
            override fun onReceivedError(view: WebView?, request: WebResourceRequest?, error: WebResourceError?) {
                Log.e("WebViewManager", "Error loading page: ${error?.description}")
                delegate?.didReceiveError("Error loading page: ${error?.description}")
            }
        }
    }

    @SuppressLint("SetJavaScriptEnabled")
    fun createWebView(activityRef: WeakReference<Activity>): WebView? {
        Log.d("CMPWebViewManager", "Creating WebView")
        val activity = activityRef.get()
        if (activity == null) {
            Log.e("CMPWebViewManager", "Activity reference is null, cannot create WebView")
            return null
        }

        if (webView == null || webView?.isAttachedToWindow == false) {
            webView = WebView(activity).apply {
                settings.javaScriptEnabled = true
                settings.domStorageEnabled = true
                addJavascriptInterface(JavaScriptInterface(), "AndroidSDK")
                webViewClient = createWebViewClient()
            }
        }
        return webView
    }

    fun loadConsentURL(url: String, useCase: UseCase, completion: (Result<Boolean>) -> Unit) {
        Log.d("CMPWebViewManager", "Loading URL: $url")
        currentUrl = url
        currentUseCase = useCase
        completionHandler = completion

        webView?.loadUrl(url)
    }

    private fun injectJavaScript() {
        Log.d("CMPWebViewManager", "Injecting JavaScript")
        webView?.evaluateJavascript(CMPConstants.JavaScript.JAVASCRIPT_TO_EXECUTE) { result ->
            Log.d("CMPWebViewManager", "JavaScript injection result: $result")
            if (result == "null") {
                Log.d("CMPWebViewManager", "JavaScript injected successfully.")
                retryManager?.reset()
            } else {
                Log.d("CMPWebViewManager", "Error injecting JavaScript: $result")
                retryManager?.retry { success ->
                    if (!success) {
                        handleFailure(Exception("Failed to inject JavaScript"))
                    }
                }
            }
        }
    }

    private fun handleFailure(error: Exception) {
        coroutineScope.launch(Dispatchers.Main) {
            delegate?.didReceiveError(error.message ?: "Unknown error occurred")
            executeCompletion(Result.failure(error))
        }
    }

    fun handleConsentReceived(consent: String, jsonObject: JsonObject) {
        try {
            val jsonString = json.encodeToString(jsonObject)
            userPreferences.saveConsentData(jsonString, consent)
            delegate?.didReceiveConsentMessage(consent, jsonObject)
            completionHandler?.invoke(Result.success(false))

        } catch (e: Exception) {
            delegate?.didReceiveError("Error decoding JSON: ${e.message}")
        }
    }

    private fun executeCompletion(result: Result<Boolean>) {
        coroutineScope.launch(Dispatchers.Main) {
            completionHandler?.invoke(result)
            completionHandler = null
            currentUseCase = null
        }
    }

    fun cancelOperations() {
        webView?.stopLoading()
        completionHandler?.invoke(Result.failure(Exception("Operation cancelled")))
        completionHandler = null
        currentUseCase = null
        retryManager?.reset()
    }

    fun performRetry(completion: (Boolean) -> Unit) {
        webView.let { webView ->
            coroutineScope.launch(Dispatchers.Main) {
                if (webView != null) {
                    webView.url?.let { url ->
                        webView.loadUrl(url)
                        completion(true)
                    } ?: completion(false)
                }
            }
        }
    }

    fun isWebViewDestroyed(): Boolean {
        return webView == null
    }

    inner class JavaScriptInterface {
        @JavascriptInterface
        fun postMessage(type: String, message: String) {
            Log.d("CMPWebViewManager", "Received message: type=$type, message=$message")

            coroutineScope.launch(Dispatchers.Main) {
                when (type) {
                    "consent" -> {
                        Log.d("CMPWebViewManager", "Received consent message: $message")
                        val jsonObject = json.parseToJsonElement(message).jsonObject
                        val consent = jsonObject["cmpString"]?.toString() ?: return@launch
                        if (currentUseCase == UseCase.PERFORM_DRY_CHECK_CONSENT) {
                            executeCompletion(Result.success(false))
                        } else {
                            handleConsentReceived(consent, jsonObject)
                        }
                    }
                    "open" -> {
                        Log.d("CMPWebViewManager", "Received open message")
                        if (currentUseCase == UseCase.CHECK_CONSENT || currentUseCase == UseCase.PERFORM_DRY_CHECK_CONSENT) {
                            executeCompletion(Result.success(true))
                        } else {
                            executeCompletion(Result.success(false))

                            delegate?.didReceiveOpenMessage()
                        }
                    }
                    "error" -> {
                        delegate?.didReceiveError(message)
                        executeCompletion(Result.failure(Exception(message)))
                    }
                }
                retryManager?.reset()
            }
        }
    }
}