package net.consentmanager.cm_sdk_android_v3

import android.content.Context
import android.content.SharedPreferences
import android.util.Log
import androidx.annotation.Keep
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.boolean
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive
import kotlinx.serialization.json.put

@Keep
interface CMPUserPreferencesService {
    fun saveConsentData(json: String, consentString: String)
    fun getConsentData(): Pair<String?, String?>
    fun resetConsentManagementData()
}

@Keep
class CMPUserPreferencesServiceImpl(context: Context) : CMPUserPreferencesService {
    private val prefs: SharedPreferences = context.getSharedPreferences("CMPPrefs", Context.MODE_PRIVATE)
    private val jsonKey = "consentJson"
    private val consentStringKey = "consentString"

    override fun saveConsentData(json: String, consentString: String) {
        val jsonObject = Json.parseToJsonElement(json).jsonObject

        prefs.edit().apply {
            putString(jsonKey, json)
            putString(consentStringKey, consentString)
            putString("IABTCF_TCString", consentString)

            jsonObject["vendorConsents"]?.let { putString("IABTCF_VendorConsents", convertMapToBinaryString(it.jsonObject)) }
            jsonObject["purposeConsents"]?.let { putString("IABTCF_PurposeConsents", convertMapToBinaryString(it.jsonObject)) }
            jsonObject["specialFeatureOptins"]?.let { putString("IABTCF_SpecialFeaturesOptIns", convertMapToBinaryString(it.jsonObject)) }
            jsonObject["publisherConsent"]?.let { putString("IABTCF_PublisherConsent", it.jsonPrimitive.content) }
            jsonObject["purposeLI"]?.let { putString("IABTCF_PurposeLegitimateInterests", convertMapToBinaryString(it.jsonObject)) }
            jsonObject["vendorLI"]?.let { putString("IABTCF_VendorLegitimateInterests", convertMapToBinaryString(it.jsonObject)) }
            jsonObject["gdprApplies"]?.let { putString("IABTCF_gdprApplies", if (it.jsonPrimitive.boolean) "1" else "0") }
            jsonObject["cmpId"]?.let { putString("IABTCF_CmpSdkID", it.jsonPrimitive.content) }
            jsonObject["cmpVersion"]?.let { putString("IABTCF_CmpSdkVersion", it.jsonPrimitive.content) }
            jsonObject["tcfPolicyVersion"]?.let { putString("IABTCF_PolicyVersion", it.jsonPrimitive.content) }

            jsonObject["uspstring"]?.let { putString("IABUSPrivacy_String", it.jsonPrimitive.content) }
            jsonObject["addtlConsent"]?.let { putString("IABTCF_AddtlConsent", it.jsonPrimitive.content) }
        }.apply()

        Log.d("CMPUserPrefsService", "Consent data saved")
    }

    override fun getConsentData(): Pair<String?, String?> {
        val json = prefs.getString(jsonKey, null)
        val consentString = prefs.getString(consentStringKey, null)

        // If we don't have the full JSON, try to reconstruct it from individual keys
        return if (json == null && consentString != null) {
            Pair(reconstructJsonFromPrefs(), consentString)
        } else {
            Pair(json, consentString)
        }
    }

    private fun convertMapToBinaryString(jsonObject: JsonObject): String {
        val sortedKeys = jsonObject.keys.sortedBy { it.removePrefix("c").toIntOrNull() ?: Int.MAX_VALUE }
        return sortedKeys.joinToString("") { key ->
            if (jsonObject[key]?.jsonPrimitive?.boolean == true) "1" else "0"
        }
    }

    private fun reconstructJsonFromPrefs(): String {
        val jsonObject = buildJsonObject {
            put("cmpString", prefs.getString("IABTCF_TCString", ""))
            put("vendorConsents", convertBinaryStringToJsonObject(prefs.getString("IABTCF_VendorConsents", "")))
            put("purposeConsents", convertBinaryStringToJsonObject(prefs.getString("IABTCF_PurposeConsents", "")))
            put("specialFeatureOptins", convertBinaryStringToJsonObject(prefs.getString("IABTCF_SpecialFeaturesOptIns", "")))
            put("publisherConsent", prefs.getString("IABTCF_PublisherConsent", ""))
            put("purposeLI", convertBinaryStringToJsonObject(prefs.getString("IABTCF_PurposeLegitimateInterests", "")))
            put("vendorLI", convertBinaryStringToJsonObject(prefs.getString("IABTCF_VendorLegitimateInterests", "")))
            put("gdprApplies", prefs.getString("IABTCF_gdprApplies", "0") == "1")
            put("cmpId", prefs.getString("IABTCF_CmpSdkID", ""))
            put("cmpVersion", prefs.getString("IABTCF_CmpSdkVersion", ""))
            put("tcfPolicyVersion", prefs.getString("IABTCF_PolicyVersion", ""))
            put("uspstring", prefs.getString("IABUSPrivacy_String", ""))
            put("addtlConsent", prefs.getString("IABTCF_AddtlConsent", ""))
        }
        return jsonObject.toString()
    }

    override fun resetConsentManagementData() {
        prefs.edit().clear().apply()
        Log.d("CMPUserPrefsService", "Consent data cleared")
    }

    private fun convertBinaryStringToJsonObject(binaryString: String?): JsonObject {
        if (binaryString.isNullOrEmpty()) return JsonObject(emptyMap())
        return buildJsonObject {
            binaryString.forEachIndexed { index, char ->
                put("c${index + 1}", char == '1')
            }
        }
    }
}
