package net.consentmanager.cm_sdk_android_v3

import android.util.Log
import androidx.annotation.Keep
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.SerializationException
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonNull
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.JsonTransformingSerializer
import kotlinx.serialization.json.boolean
import kotlinx.serialization.json.booleanOrNull
import kotlinx.serialization.json.double
import kotlinx.serialization.json.doubleOrNull
import kotlinx.serialization.json.int
import kotlinx.serialization.json.intOrNull

@Serializable
sealed class FlexibleId {
    abstract val stringValue: String
    abstract val intValue: Int?

    @Serializable
    @SerialName("string")
    data class StringId(val value: String) : FlexibleId() {
        override val stringValue: String = value
        override val intValue: Int? = value.toIntOrNull()
    }

    @Serializable
    @SerialName("int")
    data class IntId(val value: Int) : FlexibleId() {
        override val stringValue: String = value.toString()
        override val intValue: Int = value
    }

    companion object {
        fun from(value: JsonElement): FlexibleId = when {
            value is JsonPrimitive && value.isString -> StringId(value.content)
            value is JsonPrimitive && value.intOrNull != null -> IntId(value.int)
            else -> throw SerializationException("Invalid ID format")
        }
    }
}

@Keep
@Serializable
sealed class MetadataValue {
    abstract val stringValue: String
    abstract val intValue: Int?

    @Serializable
    @SerialName("string")
    data class StringValue(val value: String) : MetadataValue() {
        override val stringValue: String = value
        override val intValue: Int? = value.toIntOrNull()
    }

    @Serializable
    @SerialName("int")
    data class IntValue(val value: Int) : MetadataValue() {
        override val stringValue: String = value.toString()
        override val intValue: Int = value
    }

    companion object {
        fun from(value: JsonElement): MetadataValue = try {
            when {
                value is JsonPrimitive && value.isString -> StringValue(value.content)
                value is JsonPrimitive && value.intOrNull != null -> IntValue(value.int)
                value is JsonPrimitive && value.doubleOrNull != null ->
                    // Handle numbers that might come as doubles
                    IntValue(value.double.toInt())
                value is JsonPrimitive && value.booleanOrNull != null ->
                    // Handle boolean values by converting to strings
                    StringValue(value.boolean.toString())
                value is JsonNull -> StringValue("")
                else -> StringValue(value.toString())
            }
        } catch (e: Exception) {
            // If anything goes wrong, return empty string value
            StringValue("")
        }
    }
}

// Update MetadataValueSerializer
object MetadataValueSerializer : JsonTransformingSerializer<MetadataValue>(
    MetadataValue.serializer()
) {
    override fun transformDeserialize(element: JsonElement): JsonElement {
        return try {
            JsonObject(mapOf(
                "type" to JsonPrimitive(
                    when {
                        element is JsonPrimitive && element.isString -> "string"
                        element is JsonPrimitive && (element.intOrNull != null || element.doubleOrNull != null) -> "int"
                        element is JsonNull -> "string"
                        else -> "string"
                    }
                ),
                "value" to when {
                    element is JsonPrimitive && element.doubleOrNull != null ->
                        JsonPrimitive(element.double.toInt())
                    element is JsonNull -> JsonPrimitive("")
                    else -> element
                }
            ))
        } catch (e: Exception) {
            // If anything goes wrong, return empty string value
            JsonObject(mapOf(
                "type" to JsonPrimitive("string"),
                "value" to JsonPrimitive("")
            ))
        }
    }
}

@Keep
@Serializable
data class GPPData(
    val applicableSections: List<Int>? = null,
    val cmpDisplayStatus: String? = null,
    val cmpId: Int? = null,
    val cmpStatus: String? = null,
    val gppString: String? = null,
    val gppVersion: String? = null,
    val parsedSections: Map<String, List<TCFSection>>? = null,
    val sectionList: List<String>? = null,
    val signalStatus: String? = null,
    val supportedAPIs: List<String>? = null
)

@Keep
@Serializable
data class CMPConsentModel(
    @SerialName("cmpString") val cmpStringBase64Encoded: String? = null,
    @SerialName("addtlConsent") val googleAdditionalConsent: String? = null,
    @SerialName("consentstring") val consentString: String? = null,
    val gdprApplies: Boolean? = null,
    val googleVendorConsents: Map<String, Boolean>? = null,
    val hasGlobalScope: Boolean? = null,
    val publisherCC: String? = null,
    val regulation: Int? = null,
    val regulationKey: String? = null,
    @SerialName("tcfcompliant") val tcfCompliant: Boolean? = null,
    @SerialName("tcfversion") val tcfVersion: Int? = null,
    val lastButtonEvent: Int? = null,
    @SerialName("tcfcaversion") val tcfcaVersion: Int? = null,
    @SerialName("gppversions") val gppVersions: List<String>? = null,
    @SerialName("uspstring") val uspString: String? = null,
    val vendorsList: List<Vendor>? = null,
    val purposesList: List<Purpose>? = null,
    val purposeLI: Map<String, Boolean>? = null,
    val vendorLI: Map<String, Boolean>? = null,
    val vendorConsents: Map<String, Boolean>? = null,
    val purposeConsents: Map<String, Boolean>? = null,
    val metadata: List<CmpMetadata>? = null,
    val userChoiceExists: Boolean? = null,
    val purModeActive: Boolean? = null,
    val purModeLoggedIn: Boolean? = null,
    val purModeLogic: Int? = null,
    val consentExists: Boolean? = null,
    @SerialName("consentmode")
    @Serializable(with = ConsentModeMapSerializer::class)
    val consentMode: Map<ConsentType, ConsentModeStatus>? = null,
    val gppdata: GPPData? = null
) {
    fun hasPurposeConsent(id: String): Boolean =
        getStatusForPurpose(id).toBoolean()

    fun hasVendorConsent(id: String): Boolean =
        getStatusForVendor(id).toBoolean()

    internal fun getStatusForPurpose(id: String): ConsentStatus {
        val normalizedId = id.lowercase()
        return when {
            purposeConsents == null -> ConsentStatus.CHOICE_DOESNT_EXIST
            purposeConsents[normalizedId] == true -> ConsentStatus.GRANTED
            else -> ConsentStatus.DENIED
        }
    }

    internal fun getStatusForVendor(id: String): ConsentStatus {
        val normalizedId = id.lowercase()
        return when {
            vendorConsents == null -> ConsentStatus.CHOICE_DOESNT_EXIST
            vendorConsents[normalizedId] == true -> ConsentStatus.GRANTED
            else -> ConsentStatus.DENIED
        }
    }

    fun exportCMPInfo(): String =
        cmpStringBase64Encoded ?: ""

    fun getAllPurposesIDs(): List<String> =
        purposesList?.map { it.id.stringValue }.orEmpty()

    fun getEnabledPurposesIDs(): List<String> =
        purposeConsents?.keys?.toList() ?: emptyList()

    fun getDisabledPurposesIDs(): List<String> {
        val allPurposes = getAllPurposesIDs().toSet()
        val enabledPurposes = getEnabledPurposesIDs().toSet()
        return (allPurposes - enabledPurposes).toList()
    }

    fun getAllVendorsIDs(): List<String> =
        vendorsList?.map { it.id.stringValue }.orEmpty()

    fun getEnabledVendorsIDs(): List<String> =
        vendorConsents?.keys?.toList() ?: emptyList()

    fun getDisabledVendorsIDs(): List<String> {
        val allVendors = getAllVendorsIDs().toSet()
        val enabledVendors = getEnabledVendorsIDs().toSet()
        return (allVendors - enabledVendors).toList()
    }

    fun hasUserChoice(): Boolean =
        !consentString.isNullOrEmpty()

    companion object {
        private val json = Json {
            ignoreUnknownKeys = true
            isLenient = true
            coerceInputValues = true
        }

        fun CMPConsentModel.toUserConsentStatus(): UserConsentStatus {
            return UserConsentStatus(
                hasUserChoice = if (hasUserChoice()) UserChoiceStatus.CHOICE_EXISTS else UserChoiceStatus.CHOICE_DOESNT_EXIST,
                vendors = getAllVendorsIDs().associateWith { vendorId ->
                    getStatusForVendor(vendorId)
                },
                purposes = getAllPurposesIDs().associateWith { purposeId ->
                    getStatusForPurpose(purposeId)
                },
                tcf = consentString ?: "",
                addtlConsent = googleAdditionalConsent ?: "",
                regulation = regulationKey ?: ""
            )
        }

//        internal sealed class ConsentChoice {
//            object DoesntExist : ConsentChoice()
//            object Granted : ConsentChoice()
//            object Denied : ConsentChoice()
//
//            fun toBoolean(): Boolean = this is Granted
//        }
//
        @OptIn(ExperimentalSerializationApi::class)
        fun fromJson(jsonString: String): CMPConsentModel? {
            return try {
                json.decodeFromString(jsonString)
            } catch (e: Exception) {
                // Log the error but don't crash
                Log.e("CMPConsentModel", "Error parsing JSON: ${e.message}", e)
                null
            }
        }

        @OptIn(ExperimentalSerializationApi::class)
        fun toJson(consentModel: CMPConsentModel): String {
            return try {
                json.encodeToString(consentModel)
            } catch (e: Exception) {
                // Log the error and return empty string
                Log.e("CMPConsentModel", "Error converting to JSON: ${e.message}", e)
                ""
            }
        }
    }
}

@Keep
@Serializable
data class Purpose(
    @Serializable(with = FlexibleIdSerializer::class)
    val id: FlexibleId,
    val name: String
)

@Keep
@Serializable
data class Vendor(
    val googleid: Int = 0,
    val iabid: Int = 0,
    @Serializable(with = FlexibleIdSerializer::class)
    val id: FlexibleId,
    val name: String,
    val purposes: List<String>,
    val systemid: String
)

@Keep
@Serializable
data class CmpMetadata(
    val name: String,
    val type: String,
    @Serializable(with = MetadataValueSerializer::class)
    val value: MetadataValue
)

@Keep
@Serializable
enum class ConsentType {
    @SerialName("analytics_storage") ANALYTICS_STORAGE,
    @SerialName("ad_storage") AD_STORAGE,
    @SerialName("ad_user_data") AD_USER_DATA,
    @SerialName("ad_personalization") AD_PERSONALIZATION
}

@Keep
@Serializable
enum class ConsentModeStatus {
    @SerialName("granted") GRANTED,
    @SerialName("denied") DENIED;
}

object FlexibleIdSerializer : JsonTransformingSerializer<FlexibleId>(
    FlexibleId.serializer()
) {
    override fun transformDeserialize(element: JsonElement): JsonElement {
        return JsonObject(mapOf("type" to JsonPrimitive(
            when {
                element is JsonPrimitive && element.isString -> "string"
                element is JsonPrimitive && element.intOrNull != null -> "int"
                else -> throw SerializationException("Invalid ID format")
            }
        ), "value" to element))
    }
}

@Keep
@Serializable
data class TCFSection(
    @SerialName("Version") val version: Int = 0,
    @SerialName("ConsentLanguage") val consentLanguage: String = "",
    @SerialName("PublisherCC") val publisherCC: String = "",
    @SerialName("IsServiceSpecific") val isServiceSpecific: Boolean = false,
    @SerialName("VendorConsent") val vendorConsent: List<String> = emptyList(),
    @SerialName("TcfPolicyVersion") val tcfPolicyVersion: Int = 0,
    @SerialName("CmpVersion") val cmpVersion: Int = 0,
    @SerialName("VendorLegitimateInterest") val vendorLegitimateInterest: List<String> = emptyList(),
    @SerialName("CmpId") val cmpId: Int = 0,
    @SerialName("SpecialFeatureOptIns") val specialFeatureOptIns: List<String> = emptyList(),
    @SerialName("LastUpdated") val lastUpdated: String = "",
    @SerialName("Created") val created: String = "",
    @SerialName("VendorListVersion") val vendorListVersion: Int = 0,
    @SerialName("PurposeOneTreatment") val purposeOneTreatment: Boolean = false,
    @SerialName("ConsentScreen") val consentScreen: Int = 0,
    @SerialName("PurposesLITransparency") val purposesLITransparency: List<String> = emptyList(),
    @SerialName("PurposeConsent") val purposeConsent: List<String> = emptyList(),
    @SerialName("UseNonStandardStacks") val useNonStandardStacks: Boolean = false
)

@Keep
enum class UserChoiceStatus {
    CHOICE_EXISTS,
    CHOICE_DOESNT_EXIST
}

@Keep
enum class ConsentStatus {
    CHOICE_DOESNT_EXIST,
    GRANTED,
    DENIED;

    // This function helps maintain backward compatibility with the old boolean methods
    internal fun toBoolean(): Boolean = this == GRANTED
}

@Keep
data class UserConsentStatus(
    val hasUserChoice: UserChoiceStatus,  // Simpler than having a separate enum
    val vendors: Map<String, ConsentStatus>,
    val purposes: Map<String, ConsentStatus>,
    val tcf: String,
    val addtlConsent: String,
    val regulation: String
)
