package net.consentmanager.cm_sdk_android_v3

import kotlinx.serialization.KSerializer
import kotlinx.serialization.SerializationException
import kotlinx.serialization.descriptors.SerialDescriptor
import kotlinx.serialization.descriptors.buildClassSerialDescriptor
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder
import kotlinx.serialization.json.JsonDecoder
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive

/**
 * Custom serializer for the consentMode map in CMPConsentModel.
 * Transforms a JSON object with string keys and values to a Map<ConsentType, ConsentStatus>
 */
object ConsentModeMapSerializer : KSerializer<Map<ConsentType, ConsentModeStatus>> {
    override val descriptor: SerialDescriptor = buildClassSerialDescriptor("ConsentModeMap")

    override fun deserialize(decoder: Decoder): Map<ConsentType, ConsentModeStatus> {
        val jsonDecoder = decoder as? JsonDecoder
            ?: throw SerializationException("ConsentModeMapSerializer can only be used with JSON")

        val jsonObject = jsonDecoder.decodeJsonElement() as? JsonObject
            ?: throw SerializationException("Expected JSON object")

        return jsonObject.jsonObject.mapNotNull { (key, value) ->
            try {
                // Try to convert the key string to a ConsentType enum value
                val consentTypeKey = ConsentType.entries.firstOrNull {
                    it.name.equals(key, ignoreCase = true) ||
                            it.name.replace("_", "").equals(key.replace("_", ""), ignoreCase = true)
                } ?: return@mapNotNull null

                // Convert the value to a ConsentModeStatus enum value
                val statusValue = value.jsonPrimitive.content
                val consentStatus = when (statusValue.lowercase()) {
                    "granted" -> ConsentModeStatus.GRANTED
                    "denied" -> ConsentModeStatus.DENIED
                    else -> ConsentModeStatus.DENIED // Default to DENIED for safety
                }

                consentTypeKey to consentStatus
            } catch (e: Exception) {
                // If we can't parse this entry, skip it
                null
            }
        }.toMap()
    }

    override fun serialize(encoder: Encoder, value: Map<ConsentType, ConsentModeStatus>) {
        // We don't need to serialize this map back to JSON in our use case,
        // but if needed, we could implement this method
        throw NotImplementedError("Serializing ConsentMode map is not implemented")
    }
}