package net.consentmanager.cm_sdk_android_v3

import android.content.Context
import android.content.SharedPreferences
import android.preference.PreferenceManager
import android.util.Log
import androidx.annotation.Keep
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.intOrNull
import kotlinx.serialization.json.jsonArray
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive

@Keep
interface CMPUserPreferencesService {
    fun saveConsentData(json: String, consentString: String)
    fun getConsentData(): Pair<String?, String?>
    fun resetConsentManagementData()
    fun getMetadataValue(key: String): Any?
}

@Keep
class CMPUserPreferencesServiceImpl(context: Context) : CMPUserPreferencesService {
    private val prefs: SharedPreferences = PreferenceManager.getDefaultSharedPreferences(context)

    companion object {
        private const val KEY_CONSENT_JSON = "consentJson"
        private const val KEY_CONSENT_STRING = "consentString"
    }

    override fun saveConsentData(json: String, consentString: String) {
        try {
            val jsonObject = Json.parseToJsonElement(json).jsonObject

            prefs.edit().apply {
                putString(KEY_CONSENT_JSON, json)
                putString(KEY_CONSENT_STRING, consentString)

                // Persist metadata values based on their types
                jsonObject["metadata"]?.jsonArray?.forEach { metadataItem ->
                    val metadata = metadataItem.jsonObject
                    val name = metadata["name"]?.jsonPrimitive?.content ?: return@forEach
                    val type = metadata["type"]?.jsonPrimitive?.content
                    val value = metadata["value"]?.jsonPrimitive

                    when (type) {
                        "int" -> value?.intOrNull?.let { putInt(name, it) }
                        "string" -> value?.content?.let { putString(name, it) }
                    }
                }
            }.apply()

            Log.d("CMPUserPrefsService", "Consent data saved successfully")
        } catch (e: Exception) {
            Log.e("CMPUserPrefsService", "Error saving consent data", e)
        }
    }

    override fun getConsentData(): Pair<String?, String?> {
        return try {
            Pair(
                prefs.getString(KEY_CONSENT_JSON, null),
                prefs.getString(KEY_CONSENT_STRING, null)
            )
        } catch (e: Exception) {
            Log.e("CMPUserPrefsService", "Error retrieving consent data", e)
            Pair(null, null)
        }
    }

    override fun getMetadataValue(key: String): Any? {
        return try {
            when {
                prefs.contains(key) -> {
                    // Try different types in order
                    try {
                        prefs.getInt(key, -1).takeIf { prefs.contains(key) }
                    } catch (e: ClassCastException) {
                        try {
                            prefs.getString(key, null)
                        } catch (e: ClassCastException) {
                            try {
                                prefs.getBoolean(key, false).takeIf { prefs.contains(key) }
                            } catch (e: ClassCastException) {
                                null
                            }
                        }
                    }
                }
                else -> null
            }
        } catch (e: Exception) {
            Log.e("CMPUserPrefsService", "Error retrieving metadata value for key: $key", e)
            null
        }
    }

    override fun resetConsentManagementData() {
        try {
            prefs.edit().clear().apply()
            Log.d("CMPUserPrefsService", "Consent data cleared successfully")
        } catch (e: Exception) {
            Log.e("CMPUserPrefsService", "Error clearing consent data", e)
        }
    }
}
