package net.consentmanager.cm_sdk_android_v3

import android.annotation.SuppressLint
import android.app.Activity
import android.graphics.Bitmap
import android.os.Build
import android.util.Log
import android.webkit.JavascriptInterface
import android.webkit.WebResourceError
import android.webkit.WebResourceRequest
import android.webkit.WebView
import android.webkit.WebViewClient
import androidx.annotation.Keep
import androidx.annotation.RequiresApi
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonArray
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonNull
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.boolean
import kotlinx.serialization.json.booleanOrNull
import kotlinx.serialization.json.double
import kotlinx.serialization.json.doubleOrNull
import kotlinx.serialization.json.int
import kotlinx.serialization.json.intOrNull
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.long
import kotlinx.serialization.json.longOrNull
import java.lang.ref.WeakReference

@Keep
interface WebViewManagerDelegate {
    fun didReceiveConsentMessage(consent: String, jsonObject: Map<String, Any>)
    fun didReceiveOpenMessage()
    fun didReceiveError(error: String)
    fun getOnClickLinkCallback(): OnClickLinkCallback?
}

@Keep
class WebViewManager(
    private val userPreferences: CMPUserPreferencesService,
    private val coroutineScope: CoroutineScope = CoroutineScope(Dispatchers.Main),
    private val networkMonitor: NetworkMonitor
) {
    var delegate: WebViewManagerDelegate? = null
    private var webView: WebView? = null
    private var currentUseCase: UseCase? = null
    private var completionHandler: ((Result<Boolean>) -> Unit)? = null
    private var retryManager: RetryManager? = null
    private val json by lazy {
        Json {
            ignoreUnknownKeys = true
            isLenient = true
        }
    }
    private var currentUrl: String? = null

    private fun createWebViewClient(): WebViewClient {
        return object : WebViewClient() {
            override fun onPageStarted(view: WebView?, url: String?, favicon: Bitmap?) {
                Log.d("CMPWebViewManager", "Page started loading: $url")
            }

            override fun onPageFinished(view: WebView?, url: String?) {
                Log.d("CMPWebViewManager", "Page finished loading: $url")
                injectJavaScript()
            }

            @Deprecated("Deprecated in Java")
            override fun shouldOverrideUrlLoading(view: WebView?, url: String?): Boolean {
                if (url == null) return false

                return try {
                    delegate?.getOnClickLinkCallback()?.onClickLink(url) ?: false
                } catch (e: Exception) {
                    delegate?.didReceiveError("Error handling URL: ${e.message}")
                    false
                }
            }

            override fun onReceivedError(view: WebView?, request: WebResourceRequest?, error: WebResourceError?) {
                Log.e("WebViewManager", "Error loading page: ${error?.description}")
                delegate?.didReceiveError("Error loading page: ${error?.description}")
            }
        }
    }

    @SuppressLint("SetJavaScriptEnabled")
    fun createWebView(activityRef: WeakReference<Activity>): WebView? {
        Log.d("CMPWebViewManager", "Creating WebView")
        val activity = activityRef.get()
        if (activity == null) {
            Log.e("CMPWebViewManager", "Activity reference is null, cannot create WebView")
            return null
        }

        if (webView == null || webView?.isAttachedToWindow == false) {
            webView = WebView(activity).apply {
                settings.javaScriptEnabled = true
                settings.domStorageEnabled = true
                addJavascriptInterface(JavaScriptInterface(), "AndroidSDK")
                webViewClient = createWebViewClient()
            }
        }
        return webView
    }

    fun loadConsentURL(url: String, useCase: UseCase, completion: (Result<Boolean>) -> Unit) {
        try {
            Log.d("CMPWebViewManager", "Loading URL: $url")
            currentUrl = url
            currentUseCase = useCase
            completionHandler = completion

            webView?.loadUrl(url)
        } catch (e: Exception) {
            Log.e("CMPWebViewManager", "Error loading consent URL", e)
            delegate?.didReceiveError("Failed to load URL: ${e.message}")
            completion(Result.failure(e))
        }
    }

    private fun injectJavaScript() {
        Log.d("CMPWebViewManager", "Injecting JavaScript")
        webView?.evaluateJavascript(CMPConstants.JavaScript.JAVASCRIPT_TO_EXECUTE) { result ->
            Log.d("CMPWebViewManager", "JavaScript injection result: $result")
            if (result == "null") {
                Log.d("CMPWebViewManager", "JavaScript injected successfully.")
                retryManager?.reset()
            } else {
                Log.d("CMPWebViewManager", "Error injecting JavaScript: $result")
                retryManager?.retry { success ->
                    if (!success) {
                        handleFailure(Exception("Failed to inject JavaScript"))
                    }
                }
            }
        }
    }

    private fun handleFailure(error: Exception) {
        coroutineScope.launch(Dispatchers.Main) {
            delegate?.didReceiveError(error.message ?: "Unknown error occurred")
            executeCompletion(Result.failure(error))
        }
    }

    private fun handleConsentReceived(consent: String, jsonObject: JsonObject) {
        try {
            val jsonString = json.encodeToString(jsonObject)
            userPreferences.saveConsentData(jsonString, consent)
            delegate?.didReceiveConsentMessage(consent, jsonObject.toMap())
            completionHandler?.invoke(Result.success(false))
        } catch (e: Exception) {
            delegate?.didReceiveError("Error handling consent: ${e.message}")
        }
    }

    private fun executeCompletion(result: Result<Boolean>) {
        coroutineScope.launch(Dispatchers.Main) {
            completionHandler?.invoke(result)
            completionHandler = null
            currentUseCase = null
        }
    }

    fun cancelOperations() {
        try {
            webView?.stopLoading()
            completionHandler?.invoke(Result.failure(Exception("Operation cancelled")))
            completionHandler = null
            currentUseCase = null
            retryManager?.reset()
        } catch (e: Exception) {
            Log.e("CMPWebViewManager", "Error cancelling operations", e)
        }
    }

    fun performRetry(completion: (Boolean) -> Unit) {
        try {
            webView.let { webView ->
                coroutineScope.launch(Dispatchers.Main) {
                    if (webView != null) {
                        webView.url?.let { url ->
                            webView.loadUrl(url)
                            completion(true)
                        } ?: completion(false)
                    }
                }
            }
        } catch (e: Exception) {
            Log.e("CMPWebViewManager", "Retry failed", e)
            completion(false)
        }
    }

    fun isWebViewDestroyed(): Boolean {
        return webView == null
    }

    inner class JavaScriptInterface {
        @JavascriptInterface
        fun postMessage(type: String, message: String) {
            Log.d("CMPWebViewManager", "Received message: type=$type, message=$message")

            coroutineScope.launch(Dispatchers.Main) {
                when (type) {
                    "consent" -> {
                        handleConsentMessage(message)
                    }
                    "open" -> {
                        handleOpenMessage()
                    }
                    "error" -> {
                        delegate?.didReceiveError(message)
                        executeCompletion(Result.failure(Exception(message)))
                    }
                    else -> Log.w("WebViewManager", "Unknown message type: $type")
                }
                retryManager?.reset()
            }
        }
    }

    private fun handleOpenMessage() {
        runCatching {
            Log.d("CMPWebViewManager", "Received open message")
            if (currentUseCase == UseCase.CHECK_CONSENT || currentUseCase == UseCase.PERFORM_DRY_CHECK_CONSENT) {
                executeCompletion(Result.success(true))
            } else {
                executeCompletion(Result.success(false))

                delegate?.didReceiveOpenMessage()
            }
        }.onFailure { e ->
            Log.e("WebViewManager", "Consent message processing failed", e)
            delegate?.didReceiveError("Consent processing error: ${e.message}")
        }
    }

    private fun handleConsentMessage(message: String) {
        runCatching {
            Log.d("CMPWebViewManager", "Received consent message")
            val jsonObject = json.parseToJsonElement(message).jsonObject
            val consent = jsonObject["cmpString"]?.toString() ?: throw IllegalArgumentException("Invalid consent format")

            when (currentUseCase) {
                UseCase.PERFORM_DRY_CHECK_CONSENT -> executeCompletion(Result.success(false))
                else -> handleConsentReceived(consent, jsonObject)
            }
        }.onFailure { e ->
            Log.e("WebViewManager", "Consent message processing failed", e)
            delegate?.didReceiveError("Consent processing error: ${e.message}")
        }
    }

    // region Helper methods do deal with Json mapping
    private fun JsonElement.toAny(): Any = when (this) {
        is JsonObject -> toMap()
        is JsonArray -> map { it.toAny() }
        is JsonPrimitive -> when {
            isString -> content
            intOrNull != null -> int
            booleanOrNull != null -> boolean
            doubleOrNull != null -> double
            else -> content
        }
        JsonNull -> ""
    }

    private fun convertJsonObjectToMap(jsonObject: JsonObject): Map<String, Any> {
        return buildMap {
            jsonObject.forEach { (key, element) ->
                put(key, when (element) {
                    is JsonPrimitive -> element.convertPrimitive()
                    is JsonObject -> convertJsonObjectToMap(element)
                    is JsonArray -> element.map { it.convertElement() }
                    JsonNull -> null
                } ?: JsonNull)
            }
        }
    }

    private fun JsonPrimitive.convertPrimitive(): Any = when {
        this.isString -> this.content
        this.intOrNull != null -> this.int
        this.longOrNull != null -> this.long
        this.doubleOrNull != null -> this.double
        this.booleanOrNull != null -> this.boolean
        else -> this.content
    }

    private fun JsonElement.convertElement(): Any? = when (this) {
        is JsonPrimitive -> this.convertPrimitive()
        is JsonObject -> convertJsonObjectToMap(this)
        is JsonArray -> this.map { it.convertElement() }
        JsonNull -> null
    }

    private fun JsonObject.toMap(): Map<String, Any> = entries.associate { (key, value) ->
        key to value.toAny()
    }
    // endregion
}
