package net.consentmanager.cm_sdk_android_v3

import android.app.Activity
import android.content.Context
import android.util.Log
import android.webkit.WebView
import androidx.annotation.Keep
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.suspendCancellableCoroutine
import net.consentmanager.cm_sdk_android_v3.CMPConsentModel.Companion.toUserConsentStatus
import java.lang.ref.WeakReference
import java.util.concurrent.atomic.AtomicReference

@Keep
class JavaFriendlyResult<T> internal constructor(private val kotlinResult: Result<T>) {
    fun isSuccess(): Boolean = kotlinResult.isSuccess
    fun exceptionOrNull(): Throwable? = kotlinResult.exceptionOrNull()
}

/**
 * Main entry point for the Consent Management Platform SDK.
 *
 * This class handles user consent flows, consent status queries, and interaction with the consent UI.
 * It follows a singleton pattern - get an instance with getInstance().
 */
@Keep
class CMPManager private constructor(
    private val context: Context,
    private val urlConfig: UrlConfig,
    private val webViewConfig: ConsentLayerUIConfig,
    delegate: CMPManagerDelegate,
    private var onClickLinkCallback: OnClickLinkCallback? = null
) : WebViewManagerDelegate, NetworkMonitorDelegate,
    DefaultLifecycleObserver {

    private var delegateRef: WeakReference<CMPManagerDelegate> = WeakReference(delegate)
    private var delegate: CMPManagerDelegate
        get() = delegateRef.get() ?: throw IllegalStateException("Delegate reference is null. Set delegate before using CMPManager.")
        set(value) {
            delegateRef = WeakReference(value)
        }

    companion object {
        private val instanceRef: AtomicReference<CMPManager?> = AtomicReference(null)

        /**
         * Gets a singleton instance of the CMPManager.
         *
         * @param context Application context
         * @param urlConfig Configuration for backend connection
         * @param webViewConfig UI configuration settings
         * @param delegate Callback interface for consent events
         * @return The CMPManager instance
         */
        fun getInstance(
            context: Context,
            urlConfig: UrlConfig,
            webViewConfig: ConsentLayerUIConfig = ConsentLayerUIConfig(),
            delegate: CMPManagerDelegate
        ): CMPManager {
            return instanceRef.get()?.apply {
                // Update the delegate
                this.delegate = delegate
            } ?: synchronized(this) {
                instanceRef.get()?.apply {
                    // Update the delegate
                    this.delegate = delegate
                } ?: CMPManager(
                    context.applicationContext,
                    urlConfig,
                    webViewConfig,
                    delegate
                ).also {
                    instanceRef.set(it)
                }
            }
        }
    }

    private val coroutineScope = CoroutineScope(Dispatchers.Main)
    private val userPreferences: CMPUserPreferencesService = CMPUserPreferencesServiceImpl(context)
    private val webViewManager: WebViewManager
    private val networkMonitor: NetworkMonitor = NetworkMonitor(context)
    private val retryManager: RetryManager

    private var activityContext: WeakReference<Activity>? = null

    private var consentLayerDialog: ConsentLayerDialog? = null
    private var webView: WebView? = null
    private var webViewContainer: WebViewContainer? = null

    init {
        networkMonitor.delegate = this

        webViewManager = WebViewManager(userPreferences, coroutineScope, networkMonitor)
        webViewManager.delegate = this

        retryManager = RetryManager(maxRetryCount = 3, baseDelay = 2000L, retryable = webViewManager)

        setupWebView()
    }

    // region UI/UX setup

    /**
     * Sets the activity that will be used to display consent dialogs.
     * This method should be called whenever the hosting activity changes.
     *
     * @param activity The current activity
     */
    fun setActivity(activity: Activity) {
        activityContext = WeakReference(activity)
        setupWebView()
    }

    fun setWebView(webView: WebView) {
        this.webView = webView
        setupWebView()
    }

    private fun setupWebView() {
        val activity = activityContext?.get() ?: return

        webView = webViewManager.createWebView(WeakReference(activity))
        webViewContainer = webView?.let { WebViewContainer(it) }
    }

    // endregion

    // region Private methods

    private fun cleanUp() {
        activityContext?.clear()
        dismissConsentLayer()
        webViewContainer = null
    }

    private fun loadConsentURLWithUseCase(useCase: UseCase, additionalParams: Map<String, Any>? = null, completion: (Result<Boolean>) -> Unit) {
        val params = createUrlParams(useCase)
        additionalParams?.let { params.apply(it) }

        val urlString = CMPUrlBuilder.build(params)
        println("Debug: URL for $useCase - $urlString") // Add this line for debugging

        webViewManager.loadConsentURL(urlString, useCase) { result ->
            completion(result)
        }
    }

    private fun showConsentLayer() {
        Log.d("CMPManager", "Showing consent layer")
        val activity = activityContext ?: run {
            delegate.didReceiveError("No valid activity context available")
            return
        }
        val config = webViewConfig

        if (webView == null) {
            try {
                webView = webViewManager.createWebView(activity)
                webViewContainer = WebViewContainer(webView!!)
            } catch (e: Exception) {
                Log.e("CMPManager", "Failed to create WebView", e)
                return
            }
        }

        if (consentLayerDialog == null || consentLayerDialog?.isShowing() == false) {
            Log.d("CMPManager", "Creating new consent layer dialog")
            consentLayerDialog?.dismiss()
            consentLayerDialog = ConsentLayerDialog(activity.get()!!, config, webViewContainer!!, webViewManager)
            consentLayerDialog?.show()
        }
        delegate.didShowConsentLayer()
    }

    private fun dismissConsentLayer() {
        Log.d("CMPManager", "Dismissing consent layer")

        consentLayerDialog?.dismiss()
        consentLayerDialog = null
    }

    private fun createUrlParams(useCase: UseCase): CMPUrlParams {
        return CMPUrlParams(
            id = urlConfig.id,
            domain = urlConfig.domain,
            useCase = useCase,
            consent = getConsentModel()?.exportCMPInfo(),
            language = urlConfig.language,
            appName = urlConfig.appName,
            packageName = context.packageName,
            isDebugMode = false
        )
    }

    private fun getConsentModel(): CMPConsentModel? {
        val (jsonString, _) = userPreferences.getConsentData()
        return jsonString?.let { CMPConsentModel.fromJson(it) }
    }
    // endregion

    // region Methods handling persisted data

    /**
     * Gets the current consent status for different purposes and vendors, along with the regulation,
     * Additional consent and TCF string information .
     *
     * @return UserConsentStatus object containing detailed consent information
     */
    fun getUserStatus(): UserConsentStatus {
        return getConsentModel()?.toUserConsentStatus() ?: UserConsentStatus(
            hasUserChoice = UserChoiceStatus.CHOICE_DOESNT_EXIST,
            vendors = emptyMap(),
            purposes = emptyMap(),
            tcf = "",
            addtlConsent = "",
            regulation = ""
        )
    }

    /**
     * Gets the consent status for a specific purpose.
     *
     * @param id The purpose ID to check (e.g., "c53")
     * @return ConsentStatus indicating if consent is GRANTED, DENIED, or CHOICE_DOESNT_EXIST
     */
    fun getStatusForPurpose(id: String): ConsentStatus {
        return getConsentModel()?.getStatusForPurpose(id) ?: ConsentStatus.CHOICE_DOESNT_EXIST
    }

    /**
     * Gets the consent status for a specific vendor.
     *
     * @param id The vendor ID to check (e.g., "s2789")
     * @return ConsentStatus indicating if consent is GRANTED, DENIED, or CHOICE_DOESNT_EXIST
     */
    fun getStatusForVendor(id: String): ConsentStatus {
        return getConsentModel()?.getStatusForVendor(id) ?: ConsentStatus.CHOICE_DOESNT_EXIST
    }

    /**
     * Exports the current CMP info as a string that can be imported later.
     *
     * @return String representation of the current consent settings
     */
    fun exportCMPInfo(): String = getConsentModel()?.exportCMPInfo() ?: ""

    /**
     * Resets all stored consent data, removing all user choices.
     */
    fun resetConsentManagementData(): Unit = userPreferences.resetConsentManagementData()

    // Add this method to the CMPManager class in CMPManager.kt

    /**
     * Returns a Map of consent settings compatible with Google Consent Mode v2.
     * This map can be directly passed to Firebase Analytics' setConsent method.
     *
     * @return Map<String, String> where keys are consent settings (analytics_storage, ad_storage, etc.)
     *         and values are "granted" or "denied"
     */
    fun getGoogleConsentModeStatus(): Map<String, String> {
        Log.d("CMPManager", "Generating Google Consent Mode v2 settings")

        val consentModel = getConsentModel()
        val consentModeSettings = consentModel?.consentMode

        // If no consent model or settings exist, default all to denied as per Google's recommendations
        if (consentModel == null || consentModeSettings == null) {
            Log.w("CMPManager", "No consent settings found, defaulting all to denied")
            return ConsentType.entries.associate {
                it.name.lowercase() to "denied"
            }
        }

        return consentModeSettings.map { (type, status) ->
            type.name.lowercase() to when (status) {
                ConsentModeStatus.GRANTED -> "granted"
                else -> "denied"
            }
        }.toMap().also {
            Log.d("CMPManager", "Generated consent settings: $it")
        }
    }

    // endregion

    // region WebView Methods
    /**
     * Checks if consent is required and opens the consent UI if necessary.
     *
     * @param jumpToSettings Whether to jump directly to settings page
     * @param completion Callback for operation completion with Result<Unit>
     */
    fun checkAndOpen(jumpToSettings: Boolean = false, completion: (Result<Unit>) -> Unit) {
        if (activityContext?.get() == null) {
            completion(Result.failure(IllegalStateException("Activity reference is null. Call setActivity() before checking consent.")))
            return
        }

        loadConsentURLWithUseCase(UseCase.VERIFY_CONSENT_ON_INITIALIZE, additionalParams = mapOf("jumpToSettings" to jumpToSettings)) { result ->
            result.onSuccess { needsConsent ->
                if (needsConsent) {
                    showConsentLayer()
                } else {
                    delegate.didCloseConsentLayer()
                }
                completion(Result.success(Unit))
            }.onFailure { error ->
                completion(Result.failure(error))
            }
        }
    }

    /**
     * Async version of the method that checks if consent is required and opens the consent UI if
     * necessary.
     *
     * @param jumpToSettings Whether to jump directly to settings page
     * @param completion Callback for operation completion with Result<Unit>
     */
    suspend fun checkAndOpen(jumpToSettings: Boolean = false): Result<Unit> = suspendCancellableCoroutine { continuation ->
        checkAndOpen(jumpToSettings) { result ->
            continuation.resumeWith(kotlin.Result.success(result))
        }
    }

    /**
     * Force opens the consent UI regardless of current consent status.
     *
     * @param jumpToSettings Whether to jump directly to settings page
     * @param completion Callback for operation completion with Result<Unit>
     */
    fun forceOpen(jumpToSettings: Boolean = false, completion: (Result<Unit>) -> Unit) {
        Log.d("CMPManager", "Force opening consent layer")

        webView = activityContext?.let { webViewManager.createWebView(it) }
        webViewContainer = WebViewContainer(webView!!)

        loadConsentURLWithUseCase(UseCase.OPEN_CONSENT, mapOf("jumpToSettings" to jumpToSettings)) { result ->
            result.onSuccess { needsConsent ->
                if (needsConsent) {
                    showConsentLayer()
                } else {
                    delegate.didCloseConsentLayer()
                }
                completion(Result.success(Unit))
            }.onFailure { error ->
                completion(Result.failure(error))
            }
        }
    }

    /**
     * Imports previously exported consent information.
     *
     * @param cmpString The string representation of consent info from exportCMPInfo()
     * @param completion Callback for operation completion with Result<Unit>
     */
    fun importCMPInfo(cmpString: String, completion: (Result<Unit>) -> Unit) {
        loadConsentURLWithUseCase(UseCase.IMPORT_CONSENT, mapOf("consent" to cmpString)) { result ->
            completion(result.map {  })
        }
    }

    /**
     * Accepts (enables or grants consent to) specified vendors.
     *
     * @param vendors List of vendor IDs to enable
     * @param completion Callback for operation completion with Result<Unit>
     */
    fun acceptVendors(vendors: List<String>, completion: (Result<Unit>) -> Unit) {
        loadConsentURLWithUseCase(UseCase.ENABLE_CONSENT_VENDORS, mapOf("addVendors" to vendors)) { result ->
            completion(result.map {  })
        }
    }

    /**
     * Rejects (disables or denies consent to) specified vendors.
     *
     * @param vendors List of vendor IDs to disable
     * @param completion Callback for operation completion with Result<Unit>
     */
    fun rejectVendors(vendors: List<String>, completion: (Result<Unit>) -> Unit) {
        loadConsentURLWithUseCase(UseCase.DISABLE_CONSENT_VENDORS, mapOf("addVendors" to vendors)) { result ->
            completion(result.map {  })
        }
    }

    /**
     * Accepts (enables or grants consent to) specified purposes.
     *
     * @param purposes List of purpose IDs to enable
     * @param updateVendor Whether to update related vendors
     * @param completion Callback for operation completion with Result<Unit>
     */
    fun acceptPurposes(purposes: List<String>, updatePurpose: Boolean, completion: (Result<Unit>) -> Unit) {
        loadConsentURLWithUseCase(UseCase.ENABLE_CONSENT_PURPOSES, mapOf("addPurposes" to purposes, "updateVendors" to updatePurpose)) { result ->
            completion(result.map {  })
        }
    }

    /**
     * Rejects (disables or denies consent to) specified purposes.
     *
     * @param purposes List of purpose IDs to disable
     * @param updateVendor Whether to update related vendors
     * @param completion Callback for operation completion with Result<Unit>
     */
    fun rejectPurposes(purposes: List<String>, updateVendor: Boolean, completion: (Result<Unit>) -> Unit) {
        loadConsentURLWithUseCase(UseCase.DISABLE_CONSENT_PURPOSES, mapOf("addPurposes" to purposes, "updateVendors" to updateVendor)) { result ->
            completion(result.map {  })
        }
    }

    /**
     * Rejects (disables or denies consent to) all consent purposes and vendors.
     *
     * @param completion Callback for operation completion with Result<Unit>
     */
    fun rejectAll(completion: (Result<Unit>) -> Unit) {
        loadConsentURLWithUseCase(UseCase.REJECT_ALL_CONSENT, mapOf("rejectAll" to true)) { result ->
            completion(result.map {  })
        }
    }

    /**
     * Accepts (enables or grants consent to) all consent purposes and vendors.
     *
     * @param completion Callback for operation completion with Result<Unit>
     */
    fun acceptAll(completion: (Result<Unit>) -> Unit) {
        loadConsentURLWithUseCase(UseCase.ACCEPT_ALL_CONSENT, mapOf("acceptAll" to true)) { result ->
            completion(result.map {  })
        }
    }
    // endregion

    // region Delegate methods
    /**
     * Notifies the SDK that the application is being paused.
     * Should be called from Activity.onPause().
     */
    fun onApplicationPause() {
        webViewManager.cancelOperations()
        Log.d("CMPManager", "Application paused")
    }

    /**
     * Notifies the SDK that the application is resuming.
     * Should be called from Activity.onResume().
     */
    fun onApplicationResume() {
        Log.d("CMPManager", "Application resumed")
    }

    /**
     * Called when consent is received from the server and update on the device.
     *
     * @param consent The consent string in compressed format
     * @param jsonObject A JSON object with the full consent details
     */
    override fun didReceiveConsentMessage(consent: String, jsonObject: Map<String, Any>) {
        Log.d("CMPManager", "Consent message received")

        // Pretty print JSON object
//        val prettyJsonString = try {
//            JSONObject(jsonObject).toString(2)
//        } catch (e: Exception) {
//            jsonObject.toString()
//        }

//        Log.d("CMPManager", "Json body:\n$prettyJsonString")

        delegateRef.get()?.didReceiveConsent(consent, jsonObject)
        dismissConsentLayer()
    }

    /**
     * Called when the consent layer is shown to the user because a consent was needed or a consent
     * should be updated (due to the adding of a new vendor, for example).
     */
    override fun didReceiveOpenMessage() {
        Log.d("CMPManager", "Open message received")
        coroutineScope.launch(Dispatchers.Main) {
            showConsentLayer()
        }
        delegateRef.get()?.didShowConsentLayer()
    }

    override fun didReceiveError(error: String) {
        Log.e("CMPManager", "Error received: $error")
        delegateRef.get()?.didReceiveError(error)
    }

    override fun networkStatusDidChange(isConnected: Boolean) {
        if (isConnected) {
            // Retry any pending operations if the network is back
            retryManager.retry { /* Handle retry result if needed */ }
        }
    }

    override fun onDestroy(owner: LifecycleOwner) {
        Log.d("CMPManager", "Activity is being destroyed")
        if (owner.lifecycle.currentState.isAtLeast(Lifecycle.State.RESUMED)) {
            Log.d("CMPManager", "Configuration is changing, not cleaning up")
            return
        }
        cleanUp()
    }

    override fun onResume(owner: LifecycleOwner) {
        Log.d("CMPManager", "Activity resumed")
        // Reinitialize WebView if needed
        val activity = activityContext
        if (activity != null && webViewManager.isWebViewDestroyed()) {
            val webView = webViewManager.createWebView(activity)
            webViewContainer = webView?.let { WebViewContainer(it) }
        }
    }

    fun onActivityDestroyed() {
        activityContext = null
        dismissConsentLayer()
        webView = null
        webViewContainer = null
    }

    /**
     * Sets a callback for handling links clicked within the consent layer.
     *
     * @param callback Callback that will be invoked when links are clicked
     */
    fun setOnClickLinkCallback(callback: OnClickLinkCallback?) {
        onClickLinkCallback = callback
    }

    override fun getOnClickLinkCallback(): OnClickLinkCallback? = onClickLinkCallback

    // endregion

    // region Deprecated Methods to be removed
    @Deprecated(
        message = "Use checkAndOpen() instead. This method will be removed in the next version.",
        replaceWith = ReplaceWith("chekcAndOpen()")
    )
    fun checkIfConsentIsRequired(completion: (Boolean) -> Unit) {
        loadConsentURLWithUseCase(UseCase.PERFORM_DRY_CHECK_CONSENT) { result ->
            result.onSuccess { needsConsent ->
                completion(needsConsent)
            }.onFailure {
                completion(true)
            }
        }
    }

    @Deprecated(
        message = "Use getUserStatus() instead. This method will be removed in the next version.",
        replaceWith = ReplaceWith("getUserStatus()")
    )
    fun hasUserChoice(): Boolean = getConsentModel()?.hasUserChoice() ?: false

    @Deprecated(
        message = "Use getStatusForPurpose() instead. This method will be removed in the next version.",
        replaceWith = ReplaceWith("getStatusForPurpose(id: String)")
    )
    fun hasPurposeConsent(id: String): Boolean = getConsentModel()?.hasPurposeConsent(id) ?: false
    @Deprecated(
        message = "Use getStatusForVendor() instead. This method will be removed in the next version.",
        replaceWith = ReplaceWith("getStatusForVendor(id: String)")
    )
    fun hasVendorConsent(id: String): Boolean = getConsentModel()?.hasVendorConsent(id) ?: false

    @Deprecated(
        message = "Use getUserStatus() instead. This method will be removed in the next version.",
        replaceWith = ReplaceWith("getUserStatus()")
    )
    fun getAllPurposesIDs(): List<String> = getConsentModel()?.getAllPurposesIDs() ?: emptyList()
    @Deprecated(
        message = "Use getUserStatus() instead. This method will be removed in the next version.",
        replaceWith = ReplaceWith("getUserStatus()")
    )
    fun getEnabledPurposesIDs(): List<String> = getConsentModel()?.getEnabledPurposesIDs() ?: emptyList()
    @Deprecated(
        message = "Use getUserStatus() instead. This method will be removed in the next version.",
        replaceWith = ReplaceWith("getUserStatus()")
    )
    fun getDisabledPurposesIDs(): List<String> = getConsentModel()?.getDisabledPurposesIDs() ?: emptyList()
    @Deprecated(
        message = "Use getUserStatus() instead. This method will be removed in the next version.",
        replaceWith = ReplaceWith("getUserStatus()")
    )
    fun getAllVendorsIDs(): List<String> = getConsentModel()?.getAllVendorsIDs() ?: emptyList()
    @Deprecated(
        message = "Use getUserStatus() instead. This method will be removed in the next version.",
        replaceWith = ReplaceWith("getUserStatus()")
    )
    fun getEnabledVendorsIDs(): List<String> = getConsentModel()?.getEnabledVendorsIDs() ?: emptyList()
    @Deprecated(
        message = "Use getUserStatus() instead. This method will be removed in the next version.",
        replaceWith = ReplaceWith("getUserStatus()")
    )
    fun getDisabledVendorsIDs(): List<String> = getConsentModel()?.getDisabledVendorsIDs() ?: emptyList()

    @Deprecated(
        message = "Use forceOpen() instead. This method will be removed in the next version.",
        replaceWith = ReplaceWith("forceOpen()")
    )
    fun openConsentLayer(completion: (Result<Unit>) -> Unit) {
        Log.d("CMPManager", "Force opening consent layer")

        webView = activityContext?.let { webViewManager.createWebView(it) }
        webViewContainer = WebViewContainer(webView!!)

        loadConsentURLWithUseCase(UseCase.OPEN_CONSENT) { result ->
            result.onSuccess {
                showConsentLayer()
                delegate.didShowConsentLayer()
                completion(Result.success(Unit))
            }.onFailure { error ->
                Log.e("CMPManager", "Error in forceOpenConsentLayer: ${error.message}")
                completion(Result.failure(error))
            }
        }
    }

    @Deprecated(
        message = "Use checkAndOpen() instead",
        replaceWith = ReplaceWith("checkAndOpen(false, completion)")
    )
    fun checkWithServerAndOpenIfNecessary(completion: (Result<Unit>) -> Unit) {
        if (activityContext?.get() == null) {
            completion(Result.failure(IllegalStateException("Activity reference is null. Call setActivity() before checking consent.")))
            return
        }

        loadConsentURLWithUseCase(UseCase.VERIFY_CONSENT_ON_INITIALIZE) { result ->
            result.onSuccess { needsConsent ->
                if (needsConsent) {
                    showConsentLayer()
                } else {
                    delegate.didCloseConsentLayer()
                }
                completion(Result.success(Unit))
            }.onFailure { error ->
                completion(Result.failure(error))
            }
        }
    }

    // endregion
}
