package net.consentmanager.cm_sdk_android_v3

import android.app.Activity
import android.app.Dialog
import android.graphics.Color
import android.graphics.drawable.GradientDrawable
import android.os.Build
import android.util.Log
import android.view.Gravity
import android.view.View.generateViewId
import android.view.ViewGroup
import android.view.Window
import androidx.annotation.Keep
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.core.graphics.drawable.toDrawable
import androidx.core.view.ViewCompat
import androidx.core.view.WindowCompat
import androidx.core.view.WindowInsetsCompat
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner

@Keep
class ConsentLayerDialog(
    private val activity: Activity,
    private val config: ConsentLayerUIConfig,
    private val webViewContainer: WebViewContainer,
    private val webViewManager: WebViewManager,
    private val dialogFactory: DialogFactory = DefaultDialogFactory()
) : DefaultLifecycleObserver {
    private var dialog: Dialog? = null

    companion object {
        private const val CONTAINER_ELEVATION = 8f
    }

    fun show() {
        dismiss()

        dialog = dialogFactory.createDialog(activity).apply {
            val rootLayout = createRootLayout()
            setContentView(rootLayout)
            setCancelable(config.isCancelable)

            window?.apply {
                setLayout(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT)
                setBackgroundDrawable(Color.TRANSPARENT.toDrawable())
                applyBackgroundStyle()

                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
                    WindowCompat.setDecorFitsSystemWindows(this, false)
                }

                when (config.position) {
                    ConsentLayerUIConfig.Position.HALF_SCREEN_BOTTOM -> {
                        setGravity(Gravity.BOTTOM)
                        setupContainer(rootLayout, isTop = false)
                    }
                    ConsentLayerUIConfig.Position.HALF_SCREEN_TOP -> {
                        setGravity(Gravity.TOP)
                        setupContainer(rootLayout, isTop = true)
                    }
                    else -> {
                        setGravity(Gravity.CENTER)
                        setupContainer(rootLayout, isFullScreen = true)
                    }
                }
            }

            setOnDismissListener {
                Log.d("CMPConsentLayerDialog", "Dialog dismissed")
            }
            show()
        }

        if (activity is LifecycleOwner) {
            activity.lifecycle.addObserver(this)
        }
    }

    private fun createRootLayout() = ConstraintLayout(activity).apply {
        layoutParams = ViewGroup.LayoutParams(
            ViewGroup.LayoutParams.MATCH_PARENT,
            ViewGroup.LayoutParams.MATCH_PARENT
        )
        id = generateViewId()
    }

    private fun createContainerBackground(isTop: Boolean = false, isFullScreen: Boolean = false): GradientDrawable {
        return GradientDrawable().apply {
            if (!isFullScreen) {
                cornerRadii = when {
                    isTop -> floatArrayOf(
                        0f, 0f, // top-left
                        0f, 0f, // top-right
                        config.cornerRadius, config.cornerRadius, // bottom-right
                        config.cornerRadius, config.cornerRadius  // bottom-left
                    )
                    else -> floatArrayOf(
                        config.cornerRadius, config.cornerRadius, // top-left
                        config.cornerRadius, config.cornerRadius, // top-right
                        0f, 0f, // bottom-right
                        0f, 0f  // bottom-left
                    )
                }
            } else {
                cornerRadius = config.cornerRadius
            }
            setColor(Color.WHITE)
        }
    }

    private fun setupContainer(rootLayout: ConstraintLayout, isTop: Boolean = false, isFullScreen: Boolean = false) {
        val container = webViewContainer.createContainer(activity)

        if (isFullScreen) {
            container.layoutParams = ConstraintLayout.LayoutParams(
                ConstraintLayout.LayoutParams.MATCH_PARENT,
                ConstraintLayout.LayoutParams.MATCH_PARENT
            ).apply {
                topToTop = ConstraintLayout.LayoutParams.PARENT_ID
                bottomToBottom = ConstraintLayout.LayoutParams.PARENT_ID
                startToStart = ConstraintLayout.LayoutParams.PARENT_ID
                endToEnd = ConstraintLayout.LayoutParams.PARENT_ID
            }

            if (config.respectsSafeArea) {
                ViewCompat.setOnApplyWindowInsetsListener(container) { view, insets ->
                    val systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars())
                    val displayCutout = insets.getInsets(WindowInsetsCompat.Type.displayCutout())
                    val topInset = maxOf(systemBars.top, displayCutout.top)
                    val bottomInset = maxOf(systemBars.bottom, displayCutout.bottom)
                    val leftInset = maxOf(systemBars.left, displayCutout.left)
                    val rightInset = maxOf(systemBars.right, displayCutout.right)

                    view.setPadding(leftInset, topInset, rightInset, bottomInset)
                    insets
                }
            }
        } else {
            container.layoutParams = ConstraintLayout.LayoutParams(
                ConstraintLayout.LayoutParams.MATCH_PARENT,
                activity.resources.displayMetrics.heightPixels / 2
            ).apply {
                if (isTop) {
                    topToTop = ConstraintLayout.LayoutParams.PARENT_ID
                } else {
                    bottomToBottom = ConstraintLayout.LayoutParams.PARENT_ID
                }
                startToStart = ConstraintLayout.LayoutParams.PARENT_ID
                endToEnd = ConstraintLayout.LayoutParams.PARENT_ID
            }

            if (config.respectsSafeArea) {
                ViewCompat.setOnApplyWindowInsetsListener(rootLayout) { view, insets ->
                    val systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars())
                    val displayCutout = insets.getInsets(WindowInsetsCompat.Type.displayCutout())

                    val topInset = maxOf(systemBars.top, displayCutout.top)
                    val bottomInset = maxOf(systemBars.bottom, displayCutout.bottom)
                    val leftInset = maxOf(systemBars.left, displayCutout.left)
                    val rightInset = maxOf(systemBars.right, displayCutout.right)

                    view.setPadding(leftInset, topInset, rightInset, bottomInset)
                    insets
                }
            }
        }

        container.apply {
            background = createContainerBackground(isTop, isFullScreen)
            elevation = CONTAINER_ELEVATION
        }

        rootLayout.addView(container)
    }

    private fun Window.applyBackgroundStyle() {
        when (val style = config.backgroundStyle) {
            is ConsentLayerUIConfig.BackgroundStyle.Dimmed -> {
                val dimColor = Color.argb(
                    (style.alpha * 255).toInt(),
                    Color.red(style.color),
                    Color.green(style.color),
                    Color.blue(style.color)
                )
                setBackgroundDrawable(dimColor.toDrawable())
            }
            is ConsentLayerUIConfig.BackgroundStyle.Solid -> {
                setBackgroundDrawable(style.color.toDrawable())
            }
            is ConsentLayerUIConfig.BackgroundStyle.None -> {
                setBackgroundDrawable(Color.TRANSPARENT.toDrawable())
            }
        }
    }

    fun dismiss() {
        dialog?.dismiss()
        dialog = null
        (activity as? LifecycleOwner)?.lifecycle?.removeObserver(this)
    }

    fun isShowing(): Boolean = dialog?.isShowing == true

    override fun onDestroy(owner: LifecycleOwner) {
        dismiss()
    }
}