package net.consentmanager.cm_sdk_android_v3

import android.content.Context
import android.util.Log
import androidx.annotation.Keep
import java.lang.reflect.InvocationTargetException

@Keep
class FirebaseConsentManager {
    companion object {
        private const val TAG = "FirebaseConsentManager"

        /**
         * Checks whether Firebase Analytics SDK exists in the app.
         *
         * @return true if Firebase Analytics SDK is present, false otherwise
         */
        fun appHasFirebaseAnalyticsSdk(): Boolean {
            return try {
                Class.forName("com.google.firebase.analytics.FirebaseAnalytics")
                true
            } catch (exception: ClassNotFoundException) {
                false
            }
        }

        /**
         * Sets consent settings in Firebase Analytics based on CMP consent model.
         *
         * @param context The application or activity context
         * @param consentSettings Map of consent settings to apply (key: consent type, value: "granted" or "denied")
         * @return true if consent was successfully set, false otherwise
         */
        fun setFirebaseAnalyticsConsent(context: Context, consentSettings: Map<String, String>): Boolean {
            if (!appHasFirebaseAnalyticsSdk()) {
                return false
            }

            var analyticsClass: Class<*>? = null
            var analyticsInstance: Any? = null

            try {
                // Get Firebase Analytics instance
                analyticsClass = Class.forName("com.google.firebase.analytics.FirebaseAnalytics")
                val getInstance = analyticsClass.getDeclaredMethod("getInstance", Context::class.java)
                analyticsInstance = getInstance.invoke(null, context)
            } catch (exception: Exception) {
                when (exception) {
                    is ClassNotFoundException -> Log.w(TAG, "FirebaseAnalytics SDK is not linked in")
                    is NoSuchMethodException -> Log.w(TAG, "Unable to invoke FirebaseAnalytics getInstance method")
                    is IllegalAccessException -> Log.w(TAG, "Unable to invoke FirebaseAnalytics getInstance method")
                    is InvocationTargetException -> Log.w(TAG, "Unable to invoke FirebaseAnalytics getInstance method")
                    else -> Log.w(TAG, "Unknown error when getting FirebaseAnalytics instance: ${exception.message}")
                }
                return false
            }

            // Get consent type and status classes via reflection
            val consentTypeClass: Class<*>
            val consentStatusClass: Class<*>

            try {
                consentTypeClass = Class.forName("com.google.firebase.analytics.FirebaseAnalytics\$ConsentType")
                consentStatusClass = Class.forName("com.google.firebase.analytics.FirebaseAnalytics\$ConsentStatus")
            } catch (exception: ClassNotFoundException) {
                Log.w(TAG, "FirebaseAnalytics SDK does not support setConsent API")
                return false
            }

            // Create a map to store consent settings
            val firebaseConsentMap = mutableMapOf<Any?, Any?>()

            // Convert CMP consent settings to Firebase Analytics consent settings
            try {
                consentSettings.forEach { (consentType, consentValue) ->
                    // Convert consent type string to Firebase ConsentType enum
                    val firebaseConsentType = when (consentType) {
                        "analytics_storage" -> consentTypeClass.getDeclaredField("ANALYTICS_STORAGE")[null]
                        "ad_storage" -> consentTypeClass.getDeclaredField("AD_STORAGE")[null]
                        "ad_user_data" -> consentTypeClass.getDeclaredField("AD_USER_DATA")[null]
                        "ad_personalization" -> consentTypeClass.getDeclaredField("AD_PERSONALIZATION")[null]
                        else -> null
                    }

                    // Convert consent value string to Firebase ConsentStatus enum
                    val firebaseConsentStatus = when (consentValue) {
                        "granted" -> consentStatusClass.getDeclaredField("GRANTED")[null]
                        "denied" -> consentStatusClass.getDeclaredField("DENIED")[null]
                        else -> null
                    }

                    // Add to consent map if both type and status are valid
                    if (firebaseConsentType != null && firebaseConsentStatus != null) {
                        firebaseConsentMap[firebaseConsentType] = firebaseConsentStatus
                    }
                }
            } catch (exception: Exception) {
                when (exception) {
                    is NoSuchFieldException -> Log.w(TAG, "FirebaseAnalytics SDK does not support required consent type")
                    is IllegalAccessException -> Log.w(TAG, "Unable to access consent fields")
                    else -> Log.w(TAG, "Error setting up consent map: ${exception.message}")
                }
                return false
            }

            // Set consent in Firebase Analytics
            try {
                val setConsent = analyticsClass.getDeclaredMethod("setConsent", MutableMap::class.java)
                setConsent.invoke(analyticsInstance, firebaseConsentMap)
                Log.i(TAG, "Successfully set consent via reflection")
                return true
            } catch (exception: Exception) {
                when (exception) {
                    is NoSuchMethodException -> Log.w(TAG, "FirebaseAnalytics SDK does not support setConsent API")
                    is IllegalAccessException -> Log.w(TAG, "Unable to invoke FirebaseAnalytics setConsent method")
                    is InvocationTargetException -> Log.w(TAG, "Unable to invoke FirebaseAnalytics setConsent method")
                    else -> Log.w(TAG, "Error setting consent: ${exception.message}")
                }
                return false
            }
        }
    }
}