package net.dongliu.commons.collection;

import java.util.*;
import java.util.function.Consumer;
import java.util.function.UnaryOperator;

/**
 * Array wrapper as collection
 *
 * @author Liu Dong {@literal <im@dongliu.net>}
 */
public class Array<T> extends AbstractList<T> implements RandomAccess, java.io.Serializable,
        ExList<T> {
    private static final long serialVersionUID = -2764017481108945198L;
    private final T[] array;

    public Array(T[] array) {
        this.array = Objects.requireNonNull(array);
    }

    @SafeVarargs
    public static <T> Array<T> of(T... values) {
        return new Array<>(values);
    }

    @Override
    public int size() {
        return array.length;
    }

    @Override
    public Object[] toArray() {
        return array.clone();
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T1> T1[] toArray(T1[] a) {
        int size = size();
        if (a.length < size)
            return Arrays.copyOf(this.array, size,
                    (Class<? extends T1[]>) a.getClass());
        System.arraycopy(this.array, 0, a, 0, size);
        if (a.length > size)
            a[size] = null;
        return a;
    }

    @Override
    public T get(int index) {
        return array[index];
    }

    @Override
    public T set(int index, T element) {
        T oldValue = array[index];
        array[index] = element;
        return oldValue;
    }

    @Override
    public int indexOf(Object o) {
        T[] a = this.array;
        if (o == null) {
            for (int i = 0; i < a.length; i++)
                if (a[i] == null)
                    return i;
        } else {
            for (int i = 0; i < a.length; i++)
                if (o.equals(a[i]))
                    return i;
        }
        return -1;
    }

    @Override
    public boolean contains(Object o) {
        return indexOf(o) != -1;
    }

    @Override
    public Spliterator<T> spliterator() {
        return Spliterators.spliterator(array, Spliterator.ORDERED);
    }

    @Override
    public void forEach(Consumer<? super T> action) {
        Objects.requireNonNull(action);
        for (T e : array) {
            action.accept(e);
        }
    }

    @Override
    public void replaceAll(UnaryOperator<T> operator) {
        Objects.requireNonNull(operator);
        T[] a = this.array;
        for (int i = 0; i < a.length; i++) {
            a[i] = operator.apply(a[i]);
        }
    }

    @Override
    public void sort(Comparator<? super T> c) {
        Arrays.sort(array, c);
    }

    @Override
    public ExList<T> subList(int fromIndex, int toIndex) {
        return new ForwardingList<>(super.subList(fromIndex, toIndex));
    }

}