package net.dongliu.commons.collection;

import java.lang.reflect.Array;
import java.util.Collection;
import java.util.Collections;
import java.util.StringJoiner;
import java.util.function.*;
import java.util.stream.StreamSupport;

/**
 * Expanded collection
 *
 * @author Liu Dong
 */
public interface ExCollection<T> extends Collection<T> {

    /**
     * If c is not EnhancedCollection, wrap to EnhancedCollection
     */
    static <T> ForwardingCollection<T> wrap(Collection<T> c) {
        if (c instanceof ForwardingCollection) {
            return (ForwardingCollection<T>) c;
        }
        return new ForwardingCollection<>(c);
    }

    @Override
    default ExStream<T> stream() {
        return ExStream.wrap(StreamSupport.stream(spliterator(), false));
    }

    @Override
    default ExStream<T> parallelStream() {
        return ExStream.wrap(StreamSupport.stream(spliterator(), true));
    }


    /**
     * Create immutable view of this Set.
     */
    default Collection<T> immutable() {
        return Collections.unmodifiableCollection(this);
    }

    /**
     * Add values to this collection
     */
    default void addAll(T... values) {
        Collections.addAll(this, values);
    }

    /**
     * Convert to array
     *
     * @param cls the array's element type
     */
    default T[] toArray(Class<T> cls) {
        @SuppressWarnings("unchecked")
        T[] array = (T[]) Array.newInstance(cls, size());
        int i = 0;
        for (T value : this) {
            array[i++] = value;
        }
        return array;
    }

    /**
     * Join element to String
     */
    default String join() {
        return join("");
    }

    /**
     * Join element to String
     */
    default String join(String delimiter) {
        return join(delimiter, "", "");
    }

    /**
     * Join element to String
     */
    default String join(String delimiter, String prefix, String suffix) {
        StringJoiner joiner = new StringJoiner(delimiter, prefix, suffix);
        for (T e : this) {
            joiner.add(e == null ? "null" : e.toString());
        }
        return joiner.toString();
    }

    /**
     * Returns a stream consisting of the results of applying the given function to the elements of this stream.
     */
    default <R> ExStream<R> map(Function<? super T, ? extends R> mapper) {
        return ExStream.wrap(stream().map(mapper));
    }

    /**
     * Returns an {@code IntStream} consisting of the results of applying the
     * given function to the elements of this stream.
     */
    default ExIntStream mapToInt(ToIntFunction<? super T> mapper) {
        return ExIntStream.of(stream().mapToInt(mapper));
    }

    /**
     * Returns an {@code LongStream} consisting of the results of applying the
     * given function to the elements of this stream.
     */
    default ExLongStream mapToLong(ToLongFunction<? super T> mapper) {
        return ExLongStream.of(stream().mapToLong(mapper));
    }

    /**
     * Returns an {@code DoubleStream} consisting of the results of applying the
     * given function to the elements of this stream.
     */
    default ExDoubleStream mapToDouble(ToDoubleFunction<? super T> mapper) {
        return ExDoubleStream.of(stream().mapToDouble(mapper));
    }

    /**
     * Returns a stream consisting of the elements of this stream that match the given predicate.
     */
    default ExStream<T> filter(Predicate<? super T> predicate) {
        return ExStream.wrap(stream().filter(predicate));
    }

}
