package net.dongliu.commons.collection;

import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.function.*;
import java.util.stream.*;

/**
 * Expanded stream with toList(), toSet() and other reduce method.
 *
 * @author Liu Dong
 */
public interface ExStream<T> extends Stream<T> {

    static <T> ExStream<T> wrap(Stream<T> stream) {
        if (stream instanceof ForwardingStream) {
            return (ForwardingStream<T>) stream;
        } else {
            return new ForwardingStream<>(stream);
        }
    }

    /**
     * Create stream from array or multi values
     */
    @SafeVarargs
    static <T> ExStream<T> wrap(T... stream) {
        return wrap(Arrays.stream(stream));
    }

    @Override
    ExStream<T> filter(Predicate<? super T> predicate);

    @Override
    <R> ExStream<R> map(Function<? super T, ? extends R> mapper);

    @Override
    ExIntStream mapToInt(ToIntFunction<? super T> mapper);

    @Override
    ExLongStream mapToLong(ToLongFunction<? super T> mapper);

    @Override
    ExDoubleStream mapToDouble(ToDoubleFunction<? super T> mapper);

    @Override
    <R> ExStream<R> flatMap(Function<? super T, ? extends Stream<? extends R>> mapper);

    @Override
    ExIntStream flatMapToInt(Function<? super T, ? extends IntStream> mapper);

    @Override
    ExLongStream flatMapToLong(Function<? super T, ? extends LongStream> mapper);

    @Override
    ExDoubleStream flatMapToDouble(Function<? super T, ? extends DoubleStream> mapper);

    @Override
    ExStream<T> distinct();

    @Override
    ExStream<T> sorted();

    @Override
    ExStream<T> sorted(Comparator<? super T> comparator);

    @Override
    ExStream<T> peek(Consumer<? super T> action);

    @Override
    ExStream<T> limit(long maxSize);

    @Override
    ExStream<T> skip(long n);

    @Override
    ExStream<T> sequential();

    @Override
    ExStream<T> parallel();

    @Override
    ExStream<T> unordered();

    @Override
    ExStream<T> onClose(Runnable closeHandler);

    /**
     * Collect items to a list
     */
    default ExList<T> toList() {
        return toArrayList();
    }

    /**
     * Collect items to array list
     */
    default ExArrayList<T> toArrayList() {
        return collect(new CollectorImpl<>((Supplier<List<T>>) ExArrayList::new, List::add,
                (left, right) -> {
                    left.addAll(right);
                    return left;
                },
                CollectorImpl.CH_ID));
    }

    /**
     * Collect items to linked list
     */
    default ExArrayList<T> toLinkedList() {
        return collect(new CollectorImpl<>((Supplier<List<T>>) ExLinkedList::new, List::add,
                (left, right) -> {
                    left.addAll(right);
                    return left;
                },
                CollectorImpl.CH_ID));
    }

    /**
     * Collect item to a set
     */
    default ExSet<T> toSet() {
        return toHashSet();
    }


    /**
     * Collect item to a hash set
     */
    default ExHashSet<T> toHashSet() {
        return collect(new CollectorImpl<>((Supplier<Set<T>>) ExHashSet::new, Set::add,
                (left, right) -> {
                    left.addAll(right);
                    return left;
                },
                CollectorImpl.CH_UNORDERED_ID));
    }

    /**
     * Collect item to a linked hash set
     */
    default ExLinkedHashSet<T> toLinkedHashSet() {
        return collect(new CollectorImpl<>((Supplier<Set<T>>) ExLinkedHashSet::new, Set::add,
                (left, right) -> {
                    left.addAll(right);
                    return left;
                },
                CollectorImpl.CH_UNORDERED_ID));
    }

    /**
     * Collect item to a linked hash set
     */
    default ExTreeSet<T> toTreeSet() {
        return collect(new CollectorImpl<>((Supplier<Set<T>>) ExTreeSet::new, Set::add,
                (left, right) -> {
                    left.addAll(right);
                    return left;
                },
                CollectorImpl.CH_UNORDERED_ID));
    }


    /**
     * Collect item to a map.
     * If the mapped keys contains duplicates, later value will override the older value.
     *
     * @param keyMapper   get map key
     * @param valueMapper get map value
     * @param <K>         key type
     * @param <U>         value type
     * @return the map
     */
    default <K, U> ExMap<K, U> toMap(Function<? super T, ? extends K> keyMapper,
                                           Function<? super T, ? extends U> valueMapper) {
        return toHashMap(keyMapper, valueMapper);
    }

    /**
     * Collect item to a map.
     *
     * @param keyMapper   get map key
     * @param valueMapper get map value
     * @param merger      a merge function, used to resolve collisions between
     *                    values associated with the same key
     * @param <K>         key type
     * @param <U>         value type
     * @return the map
     */
    default <K, U> ExMap<K, U> toMap(Function<? super T, ? extends K> keyMapper,
                                           Function<? super T, ? extends U> valueMapper,
                                           BinaryOperator<U> merger) {
        return toHashMap(keyMapper, valueMapper, merger);
    }

    /**
     * Collect item to a hash map.
     * If the mapped keys contains duplicates, later value will override the older value.
     *
     * @param keyMapper   get map key
     * @param valueMapper get map value
     * @param <K>         key type
     * @param <U>         value type
     * @return the map
     */
    default <K, U> ExHashMap<K, U> toHashMap(Function<? super T, ? extends K> keyMapper,
                                                   Function<? super T, ? extends U> valueMapper) {
        return toHashMap(keyMapper, valueMapper, (older, newer) -> newer);
    }

    /**
     * Collect item to a hash map.
     *
     * @param keyMapper   get map key
     * @param valueMapper get map value
     * @param merger      a merge function, used to resolve collisions between
     *                    values associated with the same key
     * @param <K>         key type
     * @param <U>         value type
     * @return the map
     */
    default <K, U> ExHashMap<K, U> toHashMap(Function<? super T, ? extends K> keyMapper,
                                                   Function<? super T, ? extends U> valueMapper,
                                                   BinaryOperator<U> merger) {
        return collect(CollectorImpl.toMap(keyMapper, valueMapper, merger, ExHashMap::new));
    }

    /**
     * Collect item to a linked hash map.
     * If the mapped keys contains duplicates, later value will override the older value.
     *
     * @param keyMapper   get map key
     * @param valueMapper get map value
     * @param <K>         key type
     * @param <U>         value type
     * @return the map
     */
    default <K, U> ExLinkedHashMap<K, U> toLinkedHashMap(
            Function<? super T, ? extends K> keyMapper,
            Function<? super T, ? extends U> valueMapper) {
        return toLinkedHashMap(keyMapper, valueMapper, (older, newer) -> newer);
    }

    /**
     * Collect item to a linked hash map.
     *
     * @param keyMapper   get map key
     * @param valueMapper get map value
     * @param merger      a merge function, used to resolve collisions between
     *                    values associated with the same key
     * @param <K>         key type
     * @param <U>         value type
     * @return the map
     */
    default <K, U> ExLinkedHashMap<K, U> toLinkedHashMap(
            Function<? super T, ? extends K> keyMapper,
            Function<? super T, ? extends U> valueMapper,
            BinaryOperator<U> merger) {
        return collect(CollectorImpl.toMap(keyMapper, valueMapper, merger, ExLinkedHashMap::new));
    }

    /**
     * Collect item to a tree map.
     * If the mapped keys contains duplicates, later value will override the older value.
     *
     * @param keyMapper   get map key
     * @param valueMapper get map value
     * @param <K>         key type
     * @param <U>         value type
     * @return the map
     */
    default <K, U> ExTreeMap<K, U> toTreeMap(
            Function<? super T, ? extends K> keyMapper,
            Function<? super T, ? extends U> valueMapper) {
        return toTreeMap(keyMapper, valueMapper, (older, newer) -> newer);
    }

    /**
     * Collect item to a tree map.
     *
     * @param keyMapper   get map key
     * @param valueMapper get map value
     * @param merger      a merge function, used to resolve collisions between
     *                    values associated with the same key
     * @param <K>         key type
     * @param <U>         value type
     * @return the map
     */
    default <K, U> ExTreeMap<K, U> toTreeMap(
            Function<? super T, ? extends K> keyMapper,
            Function<? super T, ? extends U> valueMapper,
            BinaryOperator<U> merger) {
        return collect(CollectorImpl.toMap(keyMapper, valueMapper, merger, ExTreeMap::new));
    }


    /**
     * Returns a concurrent {@code Collector} that accumulates elements into a
     * {@code ConcurrentMap} whose keys and values are the result of applying
     * the provided mapping functions to the input elements.
     * If the mapped keys contains duplicates, later value will override the older value.
     *
     * @param keyMapper   get map key
     * @param valueMapper get map value
     * @param <K>         key type
     * @param <U>         value type
     * @return the map
     */
    default <K, U> ConcurrentMap<K, U> toConcurrentMap(Function<? super T, ? extends K> keyMapper,
                                                       Function<? super T, ? extends U> valueMapper) {
        return collect(Collectors.toConcurrentMap(keyMapper, valueMapper, (older, newer) -> newer));
    }


    /**
     * Returns a concurrent {@code Collector} that accumulates elements into a
     * {@code ConcurrentMap} whose keys and values are the result of applying
     * the provided mapping functions to the input elements.
     *
     * @param keyMapper   get map key
     * @param valueMapper get map value
     * @param merger      a merge function, used to resolve collisions between
     *                    values associated with the same key
     * @param <K>         key type
     * @param <U>         value type
     * @return the map
     */
    default <K, U> ConcurrentMap<K, U> toConcurrentMap(Function<? super T, ? extends K> keyMapper,
                                                       Function<? super T, ? extends U> valueMapper,
                                                       BinaryOperator<U> merger) {
        return collect(Collectors.toConcurrentMap(keyMapper, valueMapper, merger));
    }

    /**
     * join list items to string
     */
    default String join() {
        return map(e -> e == null ? "null" : e.toString()).collect(Collectors.joining());
    }

    /**
     * join list items with delimiter to string
     */
    default String join(CharSequence delimiter) {
        return map(e -> e == null ? "null" : e.toString()).collect(Collectors.joining(delimiter));
    }

    /**
     * join list items with delimiter, prefix, suffix to string
     */
    default String join(CharSequence delimiter,
                        CharSequence prefix,
                        CharSequence suffix) {
        return map(e -> e == null ? "null" : e.toString())
                .collect(Collectors.joining(delimiter, prefix, suffix));
    }

}
