package net.dongliu.commons.concurrent;

import net.dongliu.commons.collection.TupleUtils;
import net.dongliu.commons.collection.*;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;

/**
 * utils for CompletableFuture
 *
 * @author Dong Liu dongliu@wandoujia.com
 */
public class FutureUtils {

    /**
     * create a future with exception
     */
    public static <T> CompletableFuture<T> exceptionalFuture(Throwable t) {
        CompletableFuture<T> future = new CompletableFuture<>();
        future.completeExceptionally(t);
        return future;
    }

    /**
     * combine two futures and return one future with result as Tuple2
     */
    public static <A, B> CompletableFuture<Tuple2<A, B>> combine(
            CompletableFuture<A> af, CompletableFuture<B> bf) {
        return af.thenCombine(bf, TupleUtils::of);
    }

    /**
     * combine three futures and return one future with result as Tuple3
     */
    public static <A, B, C> CompletableFuture<Tuple3<A, B, C>> combine(
            CompletableFuture<A> af, CompletableFuture<B> bf, CompletableFuture<C> cf) {
        return combine(af, bf).thenCombine(cf, (t2, c) -> new Tuple3<>(t2._1(), t2._2(), c));
    }

    /**
     * combine four futures and return one future with result as Tuple4
     */
    public static <A, B, C, D> CompletableFuture<Tuple4<A, B, C, D>> combine(
            CompletableFuture<A> af, CompletableFuture<B> bf, CompletableFuture<C> cf,
            CompletableFuture<D> df) {
        return combine(af, bf, cf)
                .thenCombine(df, (t3, d) -> new Tuple4<>(t3._1(), t3._2(), t3._3(), d));
    }

    /**
     * combine five futures and return one future with result as Tuple5
     */
    public static <A, B, C, D, E> CompletableFuture<Tuple5<A, B, C, D, E>> combine(
            CompletableFuture<A> af, CompletableFuture<B> bf, CompletableFuture<C> cf,
            CompletableFuture<D> df, CompletableFuture<E> ef) {
        return combine(af, bf, cf, df)
                .thenCombine(ef, (t4, e) -> new Tuple5<>(t4._1(), t4._2(), t4._3(), t4._4(), e));
    }

    /**
     * combine six futures and return one future with result as Tuple6
     */
    public static <A, B, C, D, E, F> CompletableFuture<Tuple6<A, B, C, D, E, F>> combine(
            CompletableFuture<A> af, CompletableFuture<B> bf, CompletableFuture<C> cf,
            CompletableFuture<D> df, CompletableFuture<E> ef, CompletableFuture<F> ff) {
        return combine(af, bf, cf, df, ef)
                .thenCombine(ff, (t, f) -> new Tuple6<>(t._1(), t._2(), t._3(), t._4(), t._5(), f));
    }

    /**
     * combine seven futures and return one future with result as Tuple7
     */
    public static <A, B, C, D, E, F, G> CompletableFuture<Tuple7<A, B, C, D, E, F, G>> combine(
            CompletableFuture<A> af, CompletableFuture<B> bf, CompletableFuture<C> cf,
            CompletableFuture<D> df, CompletableFuture<E> ef, CompletableFuture<F> ff,
            CompletableFuture<G> gf) {
        return combine(af, bf, cf, df, ef, ff)
                .thenCombine(gf, (t, g) -> new Tuple7<>(t._1(), t._2(), t._3(), t._4(), t._5(),
                        t._6(), g));
    }

    /**
     * combine multi futures, return one future with result as list
     */
    @SafeVarargs
    public static <T> CompletableFuture<List<T>> combine(CompletableFuture<T>... futures) {
        return CompletableFuture.allOf(futures)
                .thenApply(e -> ExStream.wrap(futures).map(f -> f.getNow(null)).toList());
    }

    /**
     * combine multi futures, return one future with result as list
     */
    public static <T>
    CompletableFuture<List<T>> combine(Collection<CompletableFuture<T>> futures) {
        return combine(futures.toArray((CompletableFuture<T>[]) new CompletableFuture[futures.size()]));
    }
}
