package net.dongliu.commons;

import java.util.function.Function;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

/**
 * Supplier that only compute value only once.
 * <p>
 * If error occurred when compute value, the exception would be thrown, and the next call will run the code again.
 * If computed value is null, a NPE would be thrown.
 * </p>
 *
 * @param <T> the value type
 */
public interface Lazy<T> extends Supplier<T> {
    /**
     * Create one new Lazy instance.
     *
     * @param supplier provider the value
     * @param <T>      the value type
     * @return the created lazy value
     */
    static <T> Lazy<T> of(Supplier<T> supplier) {
        requireNonNull(supplier);
        if (supplier instanceof Lazy) {
            return (Lazy<T>) supplier;
        }
        return new LazyImpl<>(requireNonNull(supplier));
    }

    @Override
    T get();

    /**
     * Create a new lazy value, with value is calculated using function
     *
     * @param function the function to calculate value
     * @param <R>      new value type
     * @return the new lazy value
     */
    <R> Lazy<R> map(Function<? super T, ? extends R> function);
}
