package net.dongliu.commons;

import java.security.SecureRandom;
import java.util.Objects;
import java.util.Random;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.DoubleStream;
import java.util.stream.IntStream;
import java.util.stream.LongStream;

import static java.nio.charset.StandardCharsets.US_ASCII;

/**
 * Wrap random for more convenient methods
 */
public class RichRandom {

    private final Random random;

    private static final ThreadLocal<RichRandom> richRandom = ThreadLocal.withInitial(
            () -> new RichRandom(ThreadLocalRandom.current()));

    public RichRandom() {
        this(new Random());
    }

    public RichRandom(Random random) {
        this.random = Objects.requireNonNull(random);
    }

    /**
     * Return a new RichRandom wrap SecureRandom.
     */
    public static RichRandom newSecureRandom() {
        return new RichRandom(new SecureRandom());
    }

    /**
     * Return a RichRandom wrap ThreadLocalRandom. The returned random is cached by thread.
     */
    public static RichRandom currentThreadLocalRandom() {
        return richRandom.get();
    }

    /**
     * return a random int value in range [begin, end).
     *
     * @param begin random int range begin (inclusive)
     * @param end   random int range end (excluded)
     */
    public int nextInt(int begin, int end) {
        if (begin >= end) {
            throw new IllegalArgumentException("invalid range bounds, begin: " + begin + ", end: " + end);
        }
        if (Integer.MAX_VALUE + begin < end) {
            throw new IllegalArgumentException("range overflows int, begin: " + begin + ", end: " + end);
        }
        return begin + random.nextInt(end - begin);
    }

    /**
     * Returns a pseudorandom, uniformly distributed {@code int} value between 0 (inclusive) and the specified value (exclusive).
     *
     * @param bound the upper bound (exclusive).  Must be positive.
     * @return the next pseudorandom, uniformly distributed {@code long}
     * value between zero (inclusive) and {@code bound} (exclusive)
     * from this random number generator's sequence
     * @throws IllegalArgumentException if bound is not positive
     */
    public long nextLong(long bound) {
        if (bound <= 0) {
            throw new IllegalArgumentException("invalid bound: " + bound);
        }

        if (bound < Integer.MAX_VALUE) {
            return nextInt((int) bound);
        }

        long r = nextLong() >>> 1;
        long m = bound - 1;
        if ((bound & m) == 0) {
            // i.e., bound is a power of 2
            //NOTE: the lower bits maybe not uniformly distributed
            return r & m;
        } else {
            for (long u = r; u - (r = u % bound) + m < 0; u = nextLong() >>> 1) {
            }
        }
        return r;
    }

    /**
     * return a random long value in range [begin, end).
     *
     * @param begin random long range begin (inclusive)
     * @param end   random long range end (excluded)
     */
    public long nextLong(long begin, long end) {
        if (begin >= end) {
            throw new IllegalArgumentException("invalid range bounds, begin: " + begin + ", end: " + end);
        }
        if (Long.MAX_VALUE + begin < end) {
            throw new IllegalArgumentException("range overflows int, begin: " + begin + ", end: " + end);
        }
        return begin + nextLong(end - begin);
    }

    /**
     * Return a byte array with len, filled by random data.
     */
    public byte[] bytes(int len) {
        if (len < 0) {
            throw new IllegalArgumentException("invalid byte array len: " + len);
        }
        if (len == 0) {
            return new byte[0];
        }
        byte[] bytes = new byte[len];
        random.nextBytes(bytes);
        return bytes;
    }

    /**
     * Generate random string only contains ascii printable([20,127)) chars with specific len. The DEL char is excluded.
     */
    public String asciiString(int len) {
        checkStringLen(len);
        if (len == 0) {
            return "";
        }

        byte[] chars = new byte[len];
        for (int i = 0; i < chars.length; i++) {
            chars[i] = (byte) (random.nextInt(127 - 20) + 20);
        }
        return new String(chars, US_ASCII);
    }

    private static byte[] alphanumeric = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz".getBytes(US_ASCII);

    /**
     * Generate random string only contains ascii alphabet and number chars([48,58),[65, 91),[97,123) ]) with specific len.
     */
    public String alphanumericString(int len) {
        checkStringLen(len);
        if (len == 0) {
            return "";
        }

        byte[] chars = new byte[len];
        for (int i = 0; i < chars.length; i++) {
            int index = random.nextInt(alphanumeric.length);
            chars[i] = alphanumeric[index];
        }
        return new String(chars, US_ASCII);
    }

    private void checkStringLen(int len) {
        if (len < 0) {
            throw new IllegalArgumentException("len less than zero: " + len);
        }
    }


    public void nextBytes(byte[] bytes) {
        random.nextBytes(bytes);
    }

    public int nextInt() {
        return random.nextInt();
    }

    public int nextInt(int bound) {
        return random.nextInt(bound);
    }

    public long nextLong() {
        return random.nextLong();
    }

    public boolean nextBoolean() {
        return random.nextBoolean();
    }

    public float nextFloat() {
        return random.nextFloat();
    }

    public double nextDouble() {
        return random.nextDouble();
    }

    public void setSeed(long seed) {
        random.setSeed(seed);
    }

    public double nextGaussian() {
        return random.nextGaussian();
    }

    public IntStream ints(long streamSize) {
        return random.ints(streamSize);
    }

    public IntStream ints() {
        return random.ints();
    }

    public IntStream ints(long streamSize, int randomNumberOrigin, int randomNumberBound) {
        return random.ints(streamSize, randomNumberOrigin, randomNumberBound);
    }

    public IntStream ints(int randomNumberOrigin, int randomNumberBound) {
        return random.ints(randomNumberOrigin, randomNumberBound);
    }

    public LongStream longs(long streamSize) {
        return random.longs(streamSize);
    }

    public LongStream longs() {
        return random.longs();
    }

    public LongStream longs(long streamSize, long randomNumberOrigin, long randomNumberBound) {
        return random.longs(streamSize, randomNumberOrigin, randomNumberBound);
    }

    public LongStream longs(long randomNumberOrigin, long randomNumberBound) {
        return random.longs(randomNumberOrigin, randomNumberBound);
    }

    public DoubleStream doubles(long streamSize) {
        return random.doubles(streamSize);
    }

    public DoubleStream doubles() {
        return random.doubles();
    }

    public DoubleStream doubles(long streamSize, double randomNumberOrigin, double randomNumberBound) {
        return random.doubles(streamSize, randomNumberOrigin, randomNumberBound);
    }

    public DoubleStream doubles(double randomNumberOrigin, double randomNumberBound) {
        return random.doubles(randomNumberOrigin, randomNumberBound);
    }

    /**
     * Return the delegated random.
     */
    public Random getRandom() {
        return random;
    }
}
