package net.dongliu.commons.collection;


import net.dongliu.commons.exception.TooManyElementsException;
import net.dongliu.commons.function.IndexedConsumer;
import net.dongliu.commons.function.LastAwareConsumer;
import org.checkerframework.checker.nullness.qual.Nullable;

import java.util.*;
import java.util.function.Function;
import java.util.function.IntFunction;
import java.util.function.Predicate;
import java.util.function.Supplier;

import static java.lang.Math.addExact;
import static java.util.Collections.unmodifiableList;
import static java.util.Objects.requireNonNull;
import static net.dongliu.commons.collection.Collections2.convertToList;
import static net.dongliu.commons.collection.Collections2.partitionToList;

/**
 * Utils method for List
 */
public class Lists {

    private static final int INIT_SIZE = 16;

    /**
     * If list is null, return immutable empty list; else return list self.
     *
     * @param list the list
     * @param <T>  the element type
     * @return non-null list
     */
    public static <T> List<T> nullToEmpty(@Nullable List<T> list) {
        if (list == null) {
            return Lists.of();
        }
        return list;
    }

    /**
     * Create new empty ArrayList
     */
    public static <T> ArrayList<T> newArrayList() {
        return new ArrayList<>();
    }

    /**
     * Create new ArrayList.
     */
    public static <T> ArrayList<T> newArrayList(T v) {
        ArrayList<T> list = new ArrayList<>();
        list.add(v);
        return list;
    }

    /**
     * Create new ArrayList.
     */
    public static <T> ArrayList<T> newArrayList(T v1, T v2) {
        ArrayList<T> list = new ArrayList<>();
        list.add(v1);
        list.add(v2);
        return list;
    }

    /**
     * Create new ArrayList.
     */
    public static <T> ArrayList<T> newArrayList(T v1, T v2, T v3) {
        ArrayList<T> list = new ArrayList<>();
        list.add(v1);
        list.add(v2);
        list.add(v3);
        return list;
    }

    /**
     * Create new ArrayList.
     */
    public static <T> ArrayList<T> newArrayList(T v1, T v2, T v3, T v4) {
        ArrayList<T> list = new ArrayList<>();
        list.add(v1);
        list.add(v2);
        list.add(v3);
        list.add(v4);
        return list;
    }

    /**
     * Create new ArrayList.
     */
    public static <T> ArrayList<T> newArrayList(T v1, T v2, T v3, T v4, T v5) {
        ArrayList<T> list = new ArrayList<>();
        list.add(v1);
        list.add(v2);
        list.add(v3);
        list.add(v4);
        list.add(v5);
        return list;
    }

    /**
     * Create new array List.
     */
    @SafeVarargs
    public static <T> ArrayList<T> newArrayList(T... values) {
        return newList(ArrayList::new, values);
    }

    /**
     * For easy list creation with initial values.
     *
     * @param supplier the list supplier
     * @param values   the elements to add into list
     * @param <T>      element type
     * @return the list
     */
    @SafeVarargs
    public static <T, R extends List<T>> R newList(Supplier<R> supplier, T... values) {
        R list = supplier.get();
        Collections.addAll(list, values);
        return list;
    }

    /**
     * Create new immutable empty List
     */
    public static <T> List<T> of() {
        return Collections.emptyList();
    }

    /**
     * Create new immutable List.
     */
    public static <T> List<T> of(T v) {
        return Collections.singletonList(v);
    }

    /**
     * Create new immutable List.
     */
    public static <T> List<T> of(T v1, T v2) {
        return Collections.unmodifiableList(newArrayList(v1, v2));
    }

    /**
     * Create new immutable List.
     */
    public static <T> List<T> of(T v1, T v2, T v3) {
        return Collections.unmodifiableList(newArrayList(v1, v2, v3));
    }

    /**
     * Create new immutable List.
     */
    public static <T> List<T> of(T v1, T v2, T v3, T v4) {
        return Collections.unmodifiableList(newArrayList(v1, v2, v3, v4));
    }

    /**
     * Create new immutable List.
     */
    public static <T> List<T> of(T v1, T v2, T v3, T v4, T v5) {
        return Collections.unmodifiableList(newArrayList(v1, v2, v3, v4, v5));
    }

    /**
     * Create new immutable List.
     */
    public static <T> List<T> of(T v1, T v2, T v3, T v4, T v5, T v6) {
        return Collections.unmodifiableList(newArrayList(v1, v2, v3, v4, v5, v6));
    }

    /**
     * Create new immutable List.
     */
    @SafeVarargs
    public static <T> List<T> of(T... values) {
        return Collections.unmodifiableList(Arrays.asList(values));
    }

    /**
     * Return a new immutable list equals the origin list
     *
     * @param list cannot be null
     */
    public static <T> List<T> copy(List<T> list) {
        return copyOf(list);
    }

    /**
     * Return a new immutable list equals the origin list
     *
     * @param list cannot be null
     */
    public static <T> List<T> copyOf(List<T> list) {
        if (list.isEmpty()) {
            return Lists.of();
        }
        return Collections.unmodifiableList(new ArrayList<>(list));
    }

    /**
     * Convert origin list to new immutable List, the elements are converted by mapper.
     *
     * @param mapper function to convert elements
     * @return list contains the result.
     */
    public static <S, T> List<T> convert(List<S> list, Function<? super S, ? extends T> mapper) {
        requireNonNull(list);
        return convertToList(list, mapper);
    }

    /**
     * Convert origin list to new immutable List, the elements are converted by specific function.
     *
     * @param mapper function to convert elements
     * @return list contains the result.
     * @deprecated use {@link #convert(List, Function)}
     */
    @Deprecated
    public static <S, T> List<T> convertTo(List<S> list, Function<? super S, ? extends T> mapper) {
        requireNonNull(list);
        return convertToList(list, mapper);
    }

    /**
     * Filter list, return a new list which contains the elements in origin list which accepted by predicate.
     *
     * @return new list
     */
    public static <T> List<T> filter(List<T> list, Predicate<? super T> predicate) {
        requireNonNull(list);
        if (list.isEmpty()) {
            return of();
        }
        List<T> newList = new ArrayList<>(Math.min(INIT_SIZE, list.size()));
        for (T e : list) {
            if (predicate.test(e)) {
                newList.add(e);
            }
        }
        return unmodifiableList(newList);
    }


    /**
     * Return a new reversed immutable List.
     *
     * @param list the list
     * @param <T>  the element type
     * @return reversed List
     */
    public static <T> List<T> reverse(List<T> list) {
        requireNonNull(list);
        List<T> newList = new ArrayList<>(list);
        Collections.reverse(newList);
        return unmodifiableList(newList);
    }

    /**
     * Return a new sorted immutable List, the element is compared by comparator.
     *
     * @param list       the list
     * @param comparator the function to get Comparable values from list elements.
     * @param <T>        the element type
     * @return sorted List
     */
    public static <T, R extends Comparable<R>> List<T> sort(List<? extends T> list, Function<? super T, R> comparator) {
        requireNonNull(list);
        List<T> newList = new ArrayList<>(list);
        newList.sort(Comparator.comparing(comparator));
        return unmodifiableList(newList);
    }

    /**
     * Return a new sorted immutable List.
     *
     * @param list the list
     * @param <T>  the element type
     * @return sorted List
     */
    public static <T extends Comparable<T>> List<T> sort(List<T> list) {
        return sort(list, s -> s);
    }

    /**
     * Concat two lists, to one new immutable list.
     *
     * @param list1 list1
     * @param list2 list2
     * @param <T>   element type
     * @return new List
     */
    public static <T> List<T> concat(List<T> list1, List<T> list2) {
        requireNonNull(list1);
        requireNonNull(list2);
        int totalSize = addExact(list1.size(), list2.size());
        List<T> list = new ArrayList<>(totalSize);
        list.addAll(list1);
        list.addAll(list2);
        return unmodifiableList(list);
    }

    /**
     * Concat multi lists, to one new immutable List.
     *
     * @param list1      list1 first List to concat
     * @param list2      list2 second List to concat
     * @param otherLists other lists to concat
     * @param <T>        element type
     * @return new List
     */
    @SafeVarargs
    public static <T> List<T> concat(List<? extends T> list1, List<? extends T> list2, List<? extends T>... otherLists) {
        requireNonNull(list1);
        requireNonNull(list2);
        requireNonNull(otherLists);
        int totalSize = addExact(list1.size(), list2.size());
        for (List<? extends T> list : otherLists) {
            requireNonNull(list);
            totalSize = addExact(totalSize, list.size());
        }

        List<T> list = new ArrayList<>(totalSize);
        list.addAll(list1);
        list.addAll(list2);
        for (List<? extends T> otherList : otherLists) {
            list.addAll(otherList);
        }
        return unmodifiableList(list);
    }

    /**
     * Split list, into multi subLists, each subList has the specified subSize, except the last one.
     *
     * @return List of SubLists
     */
    public static <T> List<List<T>> split(List<T> list, int subSize) {
        requireNonNull(list);
        if (subSize <= 0) {
            throw new IllegalArgumentException("SubList size must large than 0, but got: " + subSize);
        }

        int size = list.size();
        int count = (size - 1) / subSize + 1;
        List<List<T>> result = new ArrayList<>(count);
        for (int i = 0; i < count; i++) {
            result.add(list.subList(i * subSize, Math.min(size, (i + 1) * subSize)));
        }
        return unmodifiableList(result);
    }

    /**
     * Divide list to two immutable Lists, the first list contains elements accepted by predicate,
     * the other contains other elements.
     *
     * @param list can not be null
     * @return two list
     */
    public static <T> PartitionResult<List<T>> partition(List<T> list, Predicate<? super T> predicate) {
        requireNonNull(list);
        return partitionToList(list, predicate);
    }

    /**
     * Return the element in list, if list has one and only one element.
     * Throw an exception otherwise.
     *
     * @throws NoSuchElementException   if iterable has no element
     * @throws TooManyElementsException if iterable has more than one element
     */
    public static <T> T getOneExactly(List<T> list) throws NoSuchElementException, TooManyElementsException {
        requireNonNull(list);
        int size = list.size();
        if (size == 0) {
            throw new NoSuchElementException();
        }
        if (size > 1) {
            throw new TooManyElementsException("iterable has more than one elements");
        }
        return list.get(0);
    }

    /**
     * Fetch the first element of list.
     * The element should not be null, or NullPointerException will be thrown.
     *
     * @param list can not be null
     * @return Optional
     */
    public static <T> Optional<T> first(List<T> list) {
        requireNonNull(list);
        if (list.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(list.get(0));
    }

    /**
     * Fetch the first element of list.
     *
     * @param list can not be null
     * @return The first element. If list is empty, return null.
     */
    @Nullable
    public static <T> T firstOrNull(List<T> list) {
        requireNonNull(list);
        if (list.isEmpty()) {
            return null;
        }
        return list.get(0);
    }

    /**
     * Fetch the last element of list.
     * The element should not be null, or NullPointerException will be thrown.
     *
     * @param list can not be null
     * @return Optional
     */
    public static <T> Optional<T> last(List<T> list) {
        if (list.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(list.listIterator(list.size()).previous());
    }

    /**
     * Fetch the first element accepted by predicate in list.
     * The element should not be null, or NullPointerException will be thrown.
     *
     * @param list can not be null
     * @return Optional
     */
    public static <T> Optional<T> find(List<T> list, Predicate<? super T> predicate) {
        return Iterables.find(list, predicate);
    }

    /**
     * Fetch the first element accepted by predicate in list.
     *
     * @param list can not be null
     * @return The first accepted element. If list is empty, return null.
     */
    @Nullable
    public static <T> T findOrNull(List<T> list, Predicate<? super T> predicate) {
        return Iterables.findOrNull(list, predicate);
    }

    /**
     * Fetch the last element of list.
     * The element should not be null, or NullPointerException will be thrown.
     *
     * @param list can not be null
     * @return Optional
     */
    public static <T> Optional<T> reverseFind(List<T> list, Predicate<? super T> predicate) {
        if (list.isEmpty()) {
            return Optional.empty();
        }
        ListIterator<T> iterator = list.listIterator(list.size());
        while (iterator.hasPrevious()) {
            T value = iterator.previous();
            if (predicate.test(value)) {
                return Optional.of(value);
            }
        }
        return Optional.empty();
    }

    /**
     * Convert list to array. This method using method reference, work better than the API in Collection. Call this method as:
     * <pre>
     *     List&lt;String&gt; list = Lists.of("1", "2", "3");
     *     String[] array = Lists.toArray(list, String[]::new);
     * </pre>
     *
     * @param list  the list
     * @param maker create the target array
     * @param <T>   element type
     * @return the target array containing elements in collection
     * @deprecated using {@link Collection#toArray(IntFunction)}
     */
    @Deprecated
    public static <T> T[] toArray(List<? extends T> list, IntFunction<T[]> maker) {
        return Collections2.toArray(list, maker);
    }

    /**
     * Traverse on a list.
     *
     * @param list     the list
     * @param consumer the consumer
     * @param <T>      the data type
     */
    public static <T> void forEachLastAware(List<T> list, LastAwareConsumer<? super T> consumer) {
        requireNonNull(list);
        requireNonNull(consumer);
        Iterables.forEachLastAware(list, consumer);
    }

    /**
     * Traverse on a list, with index for each element.
     *
     * @param list     the list
     * @param consumer the consumer
     * @param <T>      the data type
     */
    public static <T> void forEachIndexed(List<T> list, IndexedConsumer<? super T> consumer) {
        requireNonNull(list);
        requireNonNull(consumer);
        Iterables.forEachIndexed(list, consumer);
    }

    /**
     * If collection is random access list, return it self;
     * else return a new random access list contains the element from collection.
     * There is not guaranty whether returned list is immutable or not.
     *
     * @param c   the list
     * @param <T> the element type
     * @return a random accessed list
     */
    public static <T> List<T> randomAccessed(Collection<T> c) {
        requireNonNull(c);
        if (c instanceof List && c instanceof RandomAccess) {
            return (List<T>) c;
        }
        return new ArrayList<>(c);
    }
}
