package net.dongliu.commons.retry;

import net.dongliu.commons.collection.Lists;
import net.dongliu.commons.exception.MaxRetryTimesExceedException;
import net.dongliu.commons.exception.RetryInterruptedException;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Predicate;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

/**
 * A utils class for do retry. This class can be build from{@link Builder},  maxRetryTimes and retryBackOff must be set.
 * An example:
 * <pre>{@code
 *      private static final Retryer retryer = Retryer.newBuilder()
 *                 .maxRetryTimes(5)
 *                 .waitBeforeRetry(100)
 *                 .build();
 *
 *      public void myMethod() {
 *          String result = retryer.call(() -> "result");
 *      }
 * }</pre>
 * <p>
 */
public class Retry {

    private final int maxRetryTimes;

    private final Predicate<Throwable> exceptionPredicate;

    private final RetryBackOff retryBackOff;

    // listeners
    private final List<RetryListener> retryListeners;

    private Retry(Builder builder) {
        this.maxRetryTimes = builder.maxRetryTimes;
        this.exceptionPredicate = builder.exceptionPredicate;
        this.retryListeners = Lists.copyOf(builder.retryListeners);
        this.retryBackOff = builder.retryBackOff;
    }

    /**
     * return a new Builder
     */
    public static Builder newBuilder() {
        return new Builder();
    }

    /**
     * Create a builder with setting of current retry instance.
     */
    public Builder toBuilder() {
        Builder builder = new Builder();
        builder.maxRetryTimes = maxRetryTimes;
        builder.exceptionPredicate = this.exceptionPredicate;
        builder.retryListeners = new ArrayList<>(this.retryListeners);
        builder.retryBackOff = this.retryBackOff;
        return builder;
    }

    /**
     * retry run a block of code.
     *
     * @param runnable the code to be run
     */
    public void run(Runnable runnable) {
        call(() -> {
            runnable.run();
            return null;
        }, v -> false);
    }

    /**
     * Retry call a block of code, util success or max retry times exceeded.
     *
     * @param supplier the code to be run
     * @return value returned by supplier
     * @throws MaxRetryTimesExceedException if exceed max retry times
     * @throws RetryInterruptedException    if is interrupt
     */
    public <T> T call(Supplier<T> supplier) {
        return call(supplier, v -> false);
    }


    /**
     * Retry call a block of code, util success or max retry times exceeded.
     *
     * @param supplier             the code to be run
     * @param returnValuePredicate If retry if return given value. If returnValuePredicate return true, will do retry even if no exception throws.
     * @return value returned by supplier
     * @throws MaxRetryTimesExceedException if exceed max retry times
     * @throws RetryInterruptedException    if is interrupt
     */
    public <T> T call(Supplier<T> supplier, Predicate<? super T> returnValuePredicate) {
        long lastInterval = 0;
        long lastSecondInterval = 0;
        for (int i = 0; i <= maxRetryTimes; i++) {
            notifyRetryBegin(i);
            try {
                T v = supplier.get();
                if (!returnValuePredicate.test(v)) {
                    return v;
                }
            } catch (Throwable e) {
                if (!exceptionPredicate.test(e)) {
                    throw e;
                }
                if (i == maxRetryTimes) {
                    notifyMaxRetryTimesExceeded();
                    throw e;
                }
            } finally {
                notifyRetryEnd(i);
            }

            long interval = retryBackOff.nextIntervalOf(i + 1, lastInterval, lastSecondInterval);
            lastSecondInterval = lastInterval;
            lastInterval = interval;
            if (interval >= 0) {
                try {
                    Thread.sleep(interval);
                } catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    throw new RetryInterruptedException();
                }
            }

        }
        notifyMaxRetryTimesExceeded();
        throw new MaxRetryTimesExceedException();
    }

    private void notifyRetryBegin(int retryTimes) {
        if (retryTimes > 0) {
            for (RetryListener retryListener : retryListeners) {
                try {
                    retryListener.onRetryBegin(retryTimes);
                } catch (Throwable e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private void notifyRetryEnd(int retryTimes) {
        if (retryTimes > 0) {
            for (RetryListener retryListener : retryListeners) {
                try {
                    retryListener.onRetryEnd(retryTimes);
                } catch (Throwable e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private void notifyMaxRetryTimesExceeded() {
        for (RetryListener retryListener : retryListeners) {
            try {
                retryListener.onMaxRetryTimesExceeded();
            } catch (Throwable e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Builder
     */
    public static final class Builder {
        private int maxRetryTimes;
        private Predicate<Throwable> exceptionPredicate = e -> true;
        private List<RetryListener> retryListeners = new ArrayList<>();
        private RetryBackOff retryBackOff;

        private Builder() {
        }

        /**
         * max retry times
         */
        public Builder maxRetryTimes(int times) {
            this.maxRetryTimes = checkTimes(times);
            return this;
        }

        /**
         * if retry when thrown given exception
         */
        public Builder retryIfExceptionMatch(Predicate<Throwable> predicate) {
            this.exceptionPredicate = requireNonNull(predicate);
            return this;
        }

        /**
         * Retry if thrown match the given exception
         */
        public Builder retryIfExceptionIs(Class<? extends Throwable> cls) {
            return this.retryIfExceptionMatch(cls::isInstance);
        }

        /**
         * Add {@link RetryListener}
         */
        public Builder addRetryListener(RetryListener listener) {
            this.retryListeners.add(requireNonNull(listener));
            return this;
        }

        /**
         * set back off strategy
         *
         * @see RetryBackOffs
         */
        public Builder retryBackOff(RetryBackOff backOff) {
            this.retryBackOff = requireNonNull(backOff);
            return this;
        }

        /**
         * set back off with fix delay
         */
        public Builder waitBeforeRetry(long interval) {
            return this.retryBackOff(RetryBackOffs.newFixBackOff(interval));
        }

        public Retry build() {
            if (this.maxRetryTimes < 0) {
                throw new RuntimeException("max retry times not set");
            }
            if (this.retryBackOff == null) {
                throw new RuntimeException("backOff not set");
            }
            return new Retry(this);
        }

        private static int checkTimes(int times) {
            if (times < 0) {
                throw new IllegalArgumentException("invalid retry times");
            }
            return times;
        }
    }
}