package net.dongliu.commons.retry;

/**
 * A utils class to get {@link RetryBackOff}
 */
public final class RetryBackOffs {
    private RetryBackOffs() {
    }

    /**
     * backoff with fix time
     */
    public static RetryBackOff newFixBackOff(long delayMills) {
        return new FixDelayBackOff(delayMills);
    }

    /**
     * return an exponential back off impl.
     *
     * @param initialDelayMillis the wait time for first retry
     * @param maxDelayMillis     the max retry time.
     */
    public static RetryBackOff newExponentialBackOff(long initialDelayMillis, long maxDelayMillis) {
        return new BinaryExponentialBackOff(initialDelayMillis, maxDelayMillis);
    }

    private static class FixDelayBackOff implements RetryBackOff {
        private final long delay;

        private FixDelayBackOff(long delay) {
            this.delay = checkDelay(delay);
        }

        @Override
        public long nextIntervalOf(int retryTimes, long lastDelayMills, long lastSecondDelayMillis) {
            return delay;
        }
    }

    /**
     * binary exponential impl
     */
    private static class BinaryExponentialBackOff implements RetryBackOff {
        private final long initialDelay;
        private final long maxDelay;

        private BinaryExponentialBackOff(long initialDelay, long maxDelay) {
            this.initialDelay = checkDelay(initialDelay);
            this.maxDelay = checkDelay(maxDelay);
        }

        @Override
        public long nextIntervalOf(int retryTimes, long lastDelayMills, long lastSecondDelayMillis) {
            if (lastDelayMills == 0) {
                return Math.min(initialDelay, maxDelay);
            }
            if (lastDelayMills >= maxDelay) {
                return lastDelayMills;
            }
            return Math.min(lastDelayMills * 2, maxDelay);
        }
    }

    private static long checkDelay(long delayMillis) {
        if (delayMillis < 0) {
            throw new IllegalArgumentException("delay time less than 0: " + delayMillis);
        }
        return delayMillis;
    }
}
